/*----------------------------------------------------------------------------
/ standalone bootstrap :
/
/   This bootstrap program is loaded by the PROM into the xf0000
/   memory (the (last)16th 64k chuck of memory of the 1st meg byte)
/
/    o check memory up to 16 meg bytes, check 0x?00000 in each meg bytes
/        board (?= 0 thru f)
/        use logical address 0x020000 to check memory
/    o load standalone or kernal code in first available memory
/        0x?00000
/    o prompt user for bootstrp loading, it can either load
/        standalone program or unix kernel
/    o use map at logical address 0x020000 to map to physical
/        address 0x?20000 for standalone program (text then data)
/        (need prom to be in)
/    o use map at logical address 0x020000 for kernel text & data
/        (data start at new page)
/      then move the map at log addr 20000 to 0, turn off prom, disable
/      interrupt and jump to logical addr (filhdr.entry)
/
/ NOTE: Variable names contain 'hsdt' don't exactly correspond to HSDT
/       but are generic name for all supported types of disk controllers.
/       These names have historical values.
/
/ 8/20/88	Peter Logan
/ Added code to pass the proper root major device number to the kernel when
/ using scsi drives
/
/ April 1988    Craig J. Kim
/ Added SCSI support
/
/ 12/8/86       Peter Logan
/ Changed the version number to E4.00 because this is to be run
/ with "mfs".  Also, added autoboot code
/
/ 8/21/86       Peter Logan
/ Added new arbiter ram values for the dmc4/2 board
/
/----------------------------------------------------------------------------*/

#ifdef MFGSYS
#define PVERSION    "rev M3.0 "
#else
#define PVERSION    "rev 3.0 "
#endif

#define SA_BOOTS

/*---------------------------------------------------- includes ------------*/
#include "a.out.h"
#include "dmc.h"
#include "dk.h"
#include "iop.h"
#include "misc.h"
#include "globlvars.h"
#include "b.out.h"
#include "ccpu.h"
#include "maps.h"
#include "icb.h"
#include "icbcmd.h"
#include "vreg.h"
#include "cpu.h"
#include "sysconf.h"
#include "disksect.h"

/*---------------------------------------------------- defines -------------*/
#ifndef TRUE
#define TRUE                1
#define FALSE               0
#endif

#define PAGESIZE            0x1000      /* 4k bytes */
#define BSIZE               0x400
#define ADDR20K             0x020000
#define ADDR160K            0x160000
#define ADDR100K            0x100000
#define RDAMT               0x40000
#define DBCNT               0x100
#define PAGECNT             0x40
#define VERS                0xc00
#define ROOTMAJ             0xc04
#define VERSION             3
#define SWAPDEV             0x19
#define INTDISK             0xe
#define BOOTLOC             0x100       /* starting sector for boot program */
#define ARGMAGIC            0x10203040  /* magic number on stack for arglist */
#define BOOT_SECTOR         8

#define ERR_NOERROR         0           /* all is well */
#define ERR_MEMORY          1           /* unable to find sufficient memory */
#define ERR_BOOTINFO        2           /* unable to read boot info */
#define ERR_RDBOOT          3           /* unable to open boot device */

#define NOREWIND            FALSE       /* read from current buffer pointer */
#define REWIND              TRUE        /* reset buffer pointer */

#define NOTHING             0           /* unknown */
#define DISK                1           /* fixed disk */
#define TAPE                2           /* tape - cartridge/nine-track */
#define RDISK               3           /* raw disk */

/*---------------------------------------------------- externs -------------*/
extern int open();
extern int close();
extern int read();
extern int getl();
extern int nintf();
extern char *gets();
extern int strncmp();
extern char *strchr();
extern void sprintf();
extern int isscsi();

extern struct devq diskrq;              /* from rp.c */
extern unsigned int nine_trk;           /* from mt.c */
extern unsigned char twenty_fiveM;      /* from SYS.c */
extern int type;                        /* from SYS.c - type of device opened */
extern int MAP_BASE;                    /* from SYS.c */

/*---------------------------------------------------- forwards ------------*/
int  main();                            /* entry point */
int  sys_init();                        /* initialize system boards */
int  bfile();                           /* load and execute */
int  loadfile();                        /* */
int  ldrelfile();                       /* load relative file */
int  ldabsfile();                       /* load absolute file */
void memload();                         /* load memory from device */
void clrphys();                         /* clear memory to zero */
char *checkname();                      /* check user input */
int  ld1file();                         /* load UNIX kernel */
int  loaddtc();                         /* load hsdt/edt/scsi */
int  loadsa();                          /* load hsdt/edt/scsi to memory */
int  getml();                           /* read an integer from tape */
int  getdl();                           /* read an integer from disk */
void parse();                           /* build argc and argv */
void wait_for_rew();                    /* wait for tape to rewind */

/*---------------------------------------------------- globals -------------*/
char        *AUTOBOOT_REG;
#ifdef _NOTUSED
char        valnum[] = "BAD";
#endif
char        buf[0x400];
unsigned short rtblist[2];
int         fd;
int         *desaddr;                   /* pointer to memory long word */
int         bootpg;
int         txtpg;
int         datpg;
int	    bsspg;
struct bhdr filhdr;

char        user_command[80];           /* what user types in */
char        *argv[25];                  /* points to user_command[] */
int         argc;                       /* how many args */
char        bootname[] = "c0d0s0syst";
struct devq *dskrq;
unsigned short dtc_vers;

#ifdef DEBUG
char        ans[80];
#endif
long        ctlbuf[4];


long        txtstrt;       /* start of text for common obj file format files */
long        datstrt;       /* start of data for common obj file format files */
int         newformat;

char        *bootab[] = {
            "rv", "rp", "mt", "c0", "c1", "c2", "c3", 0, };
char        sastrt[10] = "c0d0r";       /* fixed disk */
char        mtstrt[10] = "c0d0";        /* tape drive */
char        hsdtcode[] = "hsdtcode";    /* HSDT code signature */
char        edtcode[] = "edtcode";      /* EDT code signature */
char        scsicode[] = "scsicode";    /* SCSI code signature */

extern unsigned char dtc_array[];

struct fheadr {
    FILHDR  filehdr;
    AOUTHDR saouthdr;
    SCNHDR  sctnhdr[6];
} fhdr;

int         dtcbds = 0;                 /* how many controller boards */
int         (*newfunc)();               /* function to use to read */
int         buff_empty = FALSE;         /* buffering status */
int         directory = FALSE;
unsigned int bootsect = 0x100;
unsigned int boot_size = 0x8000;
unsigned int amt_loaded;

unsigned short arb_buf[] = {
    0,8,1,8,0,8,1,8,0,8,1,8,0,8,4,8,1,8,0,
    8,1,8,0,8,1,8,0,8,1,8,5,8,0,1,2,3,0,1,
    2,4,0,1,2,3,0,1,2,5,0,1,2,3,0,1,2,6,0,
    1,2,3,0,1,2,7,0,1,2,0,0,1,2,4,0,1,2,0,
    0,1,2,5,0,1,2,0,0,1,2,6,0,1,2,0,0,1,2,
    7,0,4,1,5,0,4,1,5,0,4,1,5,0,4,1,5,0,4,
    1,5,0,4,1,5,0,4,1,5,0,4,1,5             };


/*==================================================== main() ==============*/
main()
{
    int retcode;

    if ((retcode = sys_init()) != ERR_NOERROR)
        exit(retcode);
    bfile();
}

/*---------------------------------------------------- sys_init() ------------
/ locate and load HSDT, EDT, and SCSI boards.
/---------------------------------------------------------------------------*/
int sys_init()
{
    register int *addr1, *addr2, i, data1, data2;
    register int *maploc;
    register int progpg;
    struct icbcmdhdr ich;
    struct slot_info *slotptr;
    char *csaptr;
    char *hptr;
    int hsize, j, first;
    unsigned short *arbptr, *avalptr, arb_val;

    /*----------------------------------------------------------------
    / find the first meg byte by checking 0x?00000 & 0x?00004
    / use MAP logical address 0x020000 to map to different page number
    /----------------------------------------------------------------*/

    addr1 = (int *) ADDR20K;
    addr2 = (int *) ADDR20K+4;

    maploc = (int *) (SMAP(addr1));

    /*-----------------------------------------------------------------
    / don't allow memory address error interrupt
    / turn off interrupt bit
    / maybe setup own interrupt service routine to handle memory addr
    / error (bus error)
    /----------------------------------------------------------------*/

    bootpg = 0x000;
    data1 = 0xa5a50f0f;                 /* set data pattern */
    data2 = 0x5a5af0f0;

    /* check throught 0-16 meg bytes */

    for (i = 0; i < 16; i++) {
        *maploc = bootpg << 16;         /* map value, only map 1k */
        *addr1 = data1;
        *addr2 = data2;
        if (*addr1 == data1 && *addr2 == data2 )
            break;
        bootpg += 0x100;                /* next meg */
    }

    if (bootpg > 0xfff) {               /* do we have enough memory? */
        printf("no mem!");
        return(ERR_MEMORY);
    }

#ifdef DEBUG
    printf("Boot page = 0x%x--<cr> ", bootpg);
    gets(ans);

    if (twenty_fiveM)
        printf("25MHz CPU\n");
    else
        printf("12.5MHz CPU\n");
#endif

    printf("Boot (%s)\n", PVERSION);

    if (SYSCONF->bootdev == DTDISK) {
        ich.command = IBTINFO;          /* get boot info from hsdt */
        ich.status = 0;
        ich.source = GVP->SlotPtr->bdptr->bd_slot;
        ich.destination = SYSCONF->bslotptr->bdptr->bd_slot;
        ich.information = SYSCONF->bootunit & 0xf;
        ich.count = 0;
        sendicb(&ich);
        bootsect = ich.information;     /* starting sector of boot programs */
        if (bootsect == 0) {
            printf("Bad boot info\n");
            return(ERR_BOOTINFO);
        }
    }

    maploc = (int *) (SMAP(ADDR100K));
    progpg = bootpg + 0x100;

    for (i = 0; i < PAGECNT; i++) {
        j = progpg++;
        *maploc++ = j << 16;
    }

    /*-----------------------------------------------------------------
    / figure out the boot device name - human form that is (e.g. c0d0)
    /----------------------------------------------------------------*/

    if (SYSCONF->bootunit) {            /* if not c0d0 */
        sprintf(mtstrt, "c%dd%d",  (SYSCONF->bootunit >> 4) & 0x03,
                                    SYSCONF->bootunit       & 0x0f);
        sprintf(sastrt, "c%dd%dr", (SYSCONF->bootunit >> 4) & 0x03,
                                    SYSCONF->bootunit       & 0x0f);
    }

#ifdef DEBUG
    printf("Boot device is %s.\n", SYSCONF->bootdev == DTDISK ? sastrt : mtstrt);
#endif

    /*-----------------------------------------------------------------
    / Loading HSDT/EDT/SCSI controller codes from 'bootimage'
    /
    / Find_and_load_HSDT();
    / Find_and_load_EDT();
    / Find_and_load_SCSI();
    /----------------------------------------------------------------*/

    buff_empty = TRUE;                  /* file buffer is empty */

    /*-----------------------------------------------------------------
    / Find HSDT boards in the system and load all of them
    /----------------------------------------------------------------*/

#ifdef HSDTBDTYP
    slotptr = SYSCONF->slotinfo;
    first = TRUE;

    for (i = 0; i < NUMSLOTS; i++, slotptr++) {
        if (slotptr->flags & OCCUPIED
                && (((slotptr->slotaddr->bdtypreg &BDTYPMSK) == HSDTBDTYP))
                && !(slotptr->flags & BOARDDOWN)) {
            dtcbds++;
            hptr = hsdtcode;
            hsize = sizeof (hsdtcode);
            loaddtc(slotptr, bootsect, hptr, hsize, i, first);
            first = FALSE;
        }
    }
#endif

    /*-----------------------------------------------------------------
    / Find EDT boards in the system and load all of them
    /----------------------------------------------------------------*/

#ifdef EDTBDTYP
    slotptr = SYSCONF->slotinfo;
    first = TRUE;

    for (i = 0; i < NUMSLOTS; i++, slotptr++) {
        if (slotptr->flags & OCCUPIED
                && (((slotptr->slotaddr->bdtypreg & BDTYPMSK) == EDTBDTYP))
                && !(slotptr->flags & BOARDDOWN)) {
            dtcbds++;
            hptr = edtcode;
            hsize = sizeof (edtcode);
            loaddtc(slotptr, bootsect, hptr, hsize, i, first);
            first = FALSE;
        }
    }
#endif

    /*-----------------------------------------------------------------
    / Locate all SCSI-EDT boards and load them
    /----------------------------------------------------------------*/

#ifdef SCSIBDTYP
    slotptr = SYSCONF->slotinfo;
    first = TRUE;

    for (i = 0; i < NUMSLOTS; i++, slotptr++) {
        if (slotptr->flags & OCCUPIED
                && (((slotptr->slotaddr->bdtypreg & BDTYPMSK) == SCSIBDTYP))
                && !(slotptr->flags & BOARDDOWN)) {
            dtcbds++;
            hptr = scsicode;
            hsize = sizeof (scsicode);
            loaddtc(slotptr, bootsect, hptr, hsize, i, first);
            first = FALSE;
        }
    }
#endif

#ifdef DEBUG
    if (!(dtcbds))
        printf("What?! No hsdt, edt, or scsi boards found.\n");
#endif

    /*-----------------------------------------------------------------
    / close the boot device
    /----------------------------------------------------------------*/

    if (fd)
        close(fd);
    fd = 0;

    /*-----------------------------------------------------------------
    / Load the new values for the dmc arbiter ram
    /----------------------------------------------------------------*/

    arbptr = DMCARB0;
    arb_val = 0;
    *(DMCINTREG) = (0x480 | (MERRIENA | EDACENA));
    slotptr = SYSCONF->slotinfo;

    for (i = 0; i < NUMSLOTS; i++, slotptr++) {
        if (slotptr->flags & OCCUPIED
                && ((slotptr->slotaddr->bdtypreg & BDTYPMSK) == DMC4BDTYP)) {
            arb_val = (dtcbds == 3) ? 64 : 32;
            break;
        }
        else if (slotptr->flags & OCCUPIED
                && ((slotptr->slotaddr->bdtypreg & BDTYPMSK) == DMC42BDTYP)) {
            arb_val = 96;
            break;
        }
    }

    for (i = 0; i < 8; i++) {
        for (j = 0, avalptr = &arb_buf[arb_val]; j < 32; j++)
            *arbptr++ = *avalptr++;
    }

    return(ERR_NOERROR);
}

/*---------------------------------------------------- bfile() ---------------
/ prompt user to enter filename
/ using map at logical address 0x020000-0x03ffff
/ There are four types of files to be loaded:
/   o   410 files: kernel
/        loaded at physical memory ?0 0000
/   o   407 files: standalone program
/       loaded at physical memory ?0 0000
/   o   570 common object file format, standalone program
/   o   575 common object file format, kernel
/--------------------------------------------------------------------------*/
int bfile()
{
    register int i, mapvalue, *maploc, *maploc1;
    register char *nameptr;
    register int (* exec)();
    char *ptr, *cptr, *nptr, *cp;
    int *vptr, *lptr;
    int rval, num, sval, j;
    int initial_boot = 1;
    short dshift = 0;
    short cshift = 0;
    short lshift = 0;
    struct boot_list *bptr;

    /* mark map at logical addr 0x020000-0x02ffff invalid
       any other maps not used are already invalid by the PROM
     */
    mapvalue = M_INVALID;
    maploc  = (int *) (SMAP(ADDR20K));
    for (i = 0; i < 16; i++) {
        *maploc++ = mapvalue;
    }

    AUTOBOOT_REG = (MAP_BASE == 0x480000) ? (char *)0x4e0000 : (char *)0x4e0001;

    do {
        if (fd) {
            close(fd);
            fd = 0;
        }
        printf("x-1000 Boot\n:");       /* our prompt here */

        if ((*AUTOBOOT_REG & 0x10) && initial_boot) {
            initial_boot = 0;           /* allow this only once */
            printf("Autoboot\n");       /* announce we are autobooting */
            argc = 0;                   /* fake to "empty carriage return" */
        }
        else {
            gets(user_command);         /* see what the user wants */
            parse(user_command);        /* do "argc-argv" */
            nine_trk = 0;
        }

        if (argc == 0) {                /* empty carriage return */
            printf(": %s\n", bootname);
            nameptr = bootname;
            fd = open(bootname, 0);
            newfunc = getl;
        }
        else {
            nameptr = checkname(argv[0], 2);
            if (*nameptr) {
                if (!strcmp(argv[0], hsdtcode) || !strcmp(argv[0], edtcode)
                        || !strcmp(argv[0], scsicode)) {
                    printf("Cannot load %s\n", argv[0]);
                    fd = -1;
                    continue;
                }
            }
            else
                continue;

            if ((fd = open(nameptr, 0)) < 0) {
                printf("Cannot open device %s\n", nameptr);
                continue;
            }

            if (type == DISK || type == RDISK) {
                lseek(fd, BOOT_SECTOR * BSIZE, 0);
                if (read(fd, buf, BSIZE) < 0) {
                    printf("Cannot read boot sector\n");
                    exit(1);
                }
                bptr = (struct boot_list *) buf;
                bootsect = bptr->boot_strt;
                boot_size = bptr->boot_size;
                amt_loaded = 0;
                lseek(fd, bootsect * BSIZE, 0);
            }
            if (nine_trk)
                newfunc = getdl;
            if (directory)
                printf("\nDirectory of %s\n\n", nameptr);
        }
    } while (fd < 0 || loadfile() < 0);

    if (filhdr.fmagic == AOUT2MAGIC )  /* 407 type file */

        /*---------------------------------------------------
        / exec at 0x020000 because still need prom
        /--------------------------------------------------*/

        exec = (int (*)())filhdr.entry;
    else {

        /*----------------------------------------------------
        / Load the root, swap and pipe data for the kernel
        /
        / Device Major and Minor numbers are designated such
        / that for SMD devices, the major number is always
        / zero and for SCSI, the controller number plus one.
        /
        / SMD:    Major:   always and must be zero
        /         Minor:   76543210
        /                  \/\/\__/
        /                   | |  |
        /                   | |  `---- logical disk number (0..15)
        /                   | `------- drive number (0..3)
        /                   `--------- controller number (0..3)
        /
        / SCSI:   Major:   controller number + 1 (1..4)
        /         Minor:   76543210
        /                  \__/\__/
        /                    |   |
        /                    |   `---- logical disk number (0..15)
        /                    `-------- drive number (0..15)
        /
        /                                         cjk 880614
        /---------------------------------------------------*/

        vptr = (int *) (ADDR20K + VERS);
        ptr = (char *) (ADDR20K + ROOTMAJ);
        if (*nameptr == 'r') {
            nptr = argv[0];
            while (*nptr != '(')
                nptr++;
            cptr = ++nptr;
            while (*nptr != ',')
                nptr++;
            *nptr = '\0';
            rval = (*cptr < 0x30) ? *cptr : atoi(cptr);
        }
        else if (*nameptr == 'c') {
            cp = nameptr;
            cp++;                       /* skip over 'c' */
            i = atoi(cp);               /* controller number */
            if (isscsi(i)) {            /* scsi controller? */
                rval = dtc_array[i] << 8;    /* controller number as major device */
                if (cp = strchr(cp, 'd')) {
                    cp++;               /* skip over 'd' */
                    rval |= atoi(cp) << 4;
                    if (cp = strchr(cp, 's'))
                        rval |= atoi(cp + 1);
                }
            }
            else
		/* check for two digit slice number */
		if(*(nameptr + 6) >= '0' && *(nameptr + 6) <= '9')
			rval = ((*(nameptr + 1) - 0x30) << 6)
                         | ((*(nameptr + 3) - 0x30) << 4)
                         | (((*(nameptr + 5) & 0xf)*10)+(*(nameptr + 6) & 0xf));
		else
                   	rval = ((*(nameptr + 1) - 0x30) << 6)
                         | ((*(nameptr + 3) - 0x30) << 4)
                         | (*(nameptr + 5) & 0xf);
        }
        else {
            cptr = argv[0];
            rval = (*cptr < 0x30) ? *cptr : atoi(cptr);
        }

        *vptr = VERSION;                /* version # */
        *ptr++ = rval & 0xff;           /* root minor device*/
        *ptr++ = (rval & ~0xff) >> 8;   /* root major device*/

        /*------------------------------------------------------
        / try to locate the SWAP logical disk among all drives
        / if SCSI, check all 14 drives; if SMD, check 4 drives
        /-----------------------------------------------------*/

        lptr = (long *) ctlbuf;
        for (i = 0; i < dtcbds; i++) {
            int numdrv = isscsi(cshift) ? 14 : 4;

            for (j = 0; j < numdrv; j++, dshift++) {
                int fd;
                static char fnm[20] = "c0d0r";

                sprintf(fnm, "c%dd%dr", cshift, j);
                if ((fd = open(fnm, 0)) < 0)
                    continue;
                                        /* get the swap device ld num */
                if ((num = ioctl(fd, SWAPDEV, ctlbuf)) != 0) {
                    printf("%s ioctl error (%d)\n", fnm, num);
                    return;
                }
                close(fd);

                lshift = *lptr & 0xff;
                if (lshift >= 0 && lshift < 0x10)
                    goto gotit;
            }
            cshift++;
            dshift = 0;
        }

        printf("Scanned %d controllers.  Cannot find a swap device\n", dtcbds);
        return;

gotit:
        if (isscsi(cshift))             /* add one to controller */
            sval = (dtc_array[cshift] << 8) | (dshift << 4) | lshift;
        else
            sval = (cshift << 6)       | (dshift << 4) | lshift;


        *ptr++ = sval & 0xff;           /* swap minor device */
        *ptr++ = (sval & ~0xff) >> 8;   /* swap major device */
        *ptr++ = rval & 0xff;           /* pipe minor device */
        *ptr++ = (rval & ~0xff) >> 8;   /* pipe major device */
        *ptr++ = 1;                     /* ECC on */
        *ptr++ = 1;                     /* CACHE on */
        lptr = (long *) ptr;
        *lptr = ctlbuf[1];              /* size of the swap area */
        ptr += sizeof (long);
        *(short *) ptr = SYSCONF->bootbaud;
        exec = (int (*)()) filhdr.entry;

#ifdef DEBUG
        printf("Entry 0x%x--<cr> ", filhdr.entry);
        gets(ans);
#endif

        maploc = (int *) (SMAP(ADDR20K));
        maploc1 = (int *) (SMAP(0));
        for (i = 0; i < txtpg; i++)
            *maploc1++ = *maploc++ | CODEPAGE | CACHEABLE | WRITPROT;

	if ( (txtpg + datpg) > 0x14f )
	{
		printf( "$#*@ Text + Data overruns boot code\n" );
	        printf( "Can't initialize MMUs for all data\n" );
	}

	/* init the rest of the mmu entries up to code */
        for ( ; i < 0x14f; i++)	
            *maploc1++ = ( i << 16 ) | CACHEABLE;


        /* turn off prom and map is already enabled */
        chgint(6);
        if (twenty_fiveM)
            _set_vbr();
        *cpu_ctl_reg = 1 << prom_dis;

        /* We never come back from here so no need to return after the exec */
        (*exec)();
    }

    (*exec)(ARGMAGIC, argc, argv);
}

/*---------------------------------------------------- loadfile() ----------*/
loadfile()
{
    register int i;
    int ldret, filemagic;

    /* clear the first 128k bytes memory where boot is going to be loaded
     * use data map logical addr 0x020000 */

    for (i = bootpg; i < (bootpg + 32); i++)
        clrphys(i);

    /* pick up exec header */

    filhdr.fmagic   = (*newfunc)(fd, FALSE);
    filemagic = (filhdr.fmagic >> 16) & 0xffff;

    if (filemagic == M68AWRMAGIC || filemagic == M68AROMAGIC) {
        if ((ldret = ld1file()) < 0)
            return(-1);
    }
    else if (filemagic == M68ASAMAGIC) {
        if (loadsa(argv[0], strlen(argv[0]))) {
            if (directory)
                printf("\n");
            else
                printf("Cannot load %s\n", argv[0]);
            return(-1);
        }
    }
    else if (directory) {
        printf("No BOOTIMAGE found.\n\n");
        return(-1);
    }
    else {
        filhdr.tsize    = getl(fd);
        filhdr.dsize    = getl(fd);
        filhdr.bsize    = getl(fd);
        filhdr.ssize    = getl(fd);
        filhdr.rtsize   = getl(fd);
        filhdr.rdsize   = getl(fd);
        filhdr.entry    = getl(fd);
    }

#ifdef DEBUG
    printf("     magic number = 0x%x\n", filhdr.fmagic);
    printf("        text size = 0x%x\n", filhdr.tsize);
    printf("        data size = 0x%x\n", filhdr.dsize);
    printf("         bss size = 0x%x\n", filhdr.bsize);
    printf("symbol table size = 0x%x\n", filhdr.ssize);
    printf("       relocation = 0x%x\n", filhdr.rtsize);
    printf("      entry point = 0x%x ---<cr> ", filhdr.entry);
    gets(ans);
#endif

    if (filhdr.fmagic == AOUT2MAGIC) {
        if (ldrelfile() < 0)
            return(-1);
    }
    else if (filhdr.fmagic == AOUT1MAGIC) {
        if (ldabsfile() < 0)
            return(-1);
    }
    else {
        printf("Cannot load 0x%x type file\n", filhdr.fmagic);
        return(-1);
    }
    return(0);
}

/*---------------------------------------------------- ldrelfile() -----------
/ FMAGIC type file, (standalone sw)load text at physical address 0x?00000
/ followed by data.
/---------------------------------------------------------------------------*/
ldrelfile()
{
    register int i, j, pagenum, totpag, *maploc;

    /* text + data + bss */
    totpag = (filhdr.tsize + filhdr.dsize + filhdr.bsize
        + 0x10000 + PAGESIZE-1 ) >>12;
#ifdef DEBUG
    printf("ldrelfile totpag = 0x%x\n", totpag);
#endif

    /* setup data map at logical address 0x020000 to
     * physical address 0x?00000 (text first then data)
     */

    pagenum = bootpg + 0x20;
    maploc = (int *) (SMAP(ADDR20K));

    /* totpag = text+data */
    for (i = 0; i < (totpag + 1); i++)  {
        j = pagenum++;
        *maploc++ = j << 16;
    }

        /* read text code into logical addr 0x020000 */
    desaddr = (int *) ADDR20K;
#ifdef DEBUG
    printf("newformat = 0x%x   txtstrt = 0x%x\n", newformat, txtstrt);
#endif

    if (newformat) {
        for (i = 0; i < 32; i++)
            getl(fd);
    }
    memload(filhdr.tsize >> 2);

    /* read data after text code */
    memload(filhdr.dsize >> 2);

    newformat = 0;
    return(0);
}

/*---------------------------------------------------- ldabsfile() ---------*/
ldabsfile()
{
    register int i, j, pagenum, addr, *maploc;
    char *p;
    int rdamt,leftover,phaddr,nleft;

    txtpg = (filhdr.tsize + PAGESIZE-1) >> 12;
    bsspg = (filhdr.bsize + PAGESIZE -1) >> 12;
    datpg = (filhdr.dsize + PAGESIZE -1) >> 12;

#ifdef DEBUG
    printf("text page at 0x%x, data page at 0x%x--<cr> ", txtpg, datpg);
    gets(ans);
#endif


    /* phy address is 0x?00000 */
    pagenum = bootpg;
    maploc = (int *) (SMAP(ADDR20K));

    /* setup data map at logical address 0x020000 to physical
     * address 0x0000 for data
     */

    for (i = 0; i < (txtpg + datpg); i++) {
        j = pagenum++;
        *maploc++ = j << 16;
    }

    /* load text code */
    desaddr = (int *) ADDR20K;

    if (newformat)
        lseek(fd, txtstrt, 0);
    memload((BLKSIZE - txtstrt) >> 2);

    leftover = filhdr.tsize - (BLKSIZE - txtstrt);
    phaddr = (int) desaddr;
    rdamt = leftover / 0x400;
    leftover = leftover % 0x400;

    for (i = 0; i < rdamt; i++, phaddr += 0x400, desaddr += 0x100)
        read(fd, phaddr, 0x400);
    memload(leftover >> 2);

    /* load data */
    addr = (int) desaddr;

    desaddr = (int *) ((addr+(PAGESIZE-1)) & ~(PAGESIZE - 1));

    nleft = BLKSIZE - leftover;
    memload(nleft >> 2);

    phaddr = (int) desaddr;
    leftover = filhdr.dsize - nleft;
    rdamt = leftover / 0x400;
    leftover = leftover % 0x400;

    for (i = 0; i < rdamt; i++, phaddr += 0x400, desaddr += 0x100)
        read(fd, phaddr, 0x400);
    memload(leftover >> 2);

    newformat = 0;
    return(0);
}

/*---------------------------------------------------- memload() -------------
/ load memory pointed by 'desaddr' of given length in long words
/---------------------------------------------------------------------------*/
void memload(length)
int length;
{
    register int i;

#ifdef DEBUG
    printf("Loading 0x%x long words at memory addr 0x%x--<cr> ",
            length, desaddr);
    gets(ans);
#endif
    for (i = 0; i < length; i++)
        *desaddr++ = (*newfunc)(fd, NOREWIND);
}

/*---------------------------------------------------- clrphys() -------------
/ clear 0x1000 bytes of physical memory
/ use data map at logical address 0x020000
/---------------------------------------------------------------------------*/
void clrphys(pagen)
int pagen;
{   register int lcount, *maploc, *addr;

    maploc = (int *) (SMAP(ADDR20K));
    *maploc = pagen << 16;
    addr = (int *) ADDR20K;
    lcount = PAGESIZE >> 2;             /* # of long words */

    /* clear memory */
    while (lcount--)
        *addr++ = (long) 0;
}

/*---------------------------------------------------- checkname() ---------*/
char *checkname(ptr, count)
char *ptr;
register int count;
{
    char **pptr;                        /* points to device name prefix table */
    char *csp;

    directory = FALSE;
    if (csp = strchr(ptr, '/')) {       /* device prefix */
        if (*(csp - 1) == 'r')          /* c#d#r disk */
            newfunc = getdl;
        else                            /* tape */
            newfunc = getml;
        for (pptr = bootab; *pptr; pptr++)
            if (!strncmp(*pptr, ptr, count)) {
                *csp++ = NULL;
                argv[0] = csp;
                buff_empty = TRUE;
                return(ptr);
            }
    }
    else if (!strcmp(ptr, "dir")) {
        directory = TRUE;
        argv[0] = "**";
        if (argc > 1) {
            for (pptr = bootab; *pptr; pptr++)
                if (!strncmp(*pptr, argv[1], count)) {
                    buff_empty = TRUE;
                    if (argv[1][strlen(argv[1]) - 1] == 'r')
                        newfunc = getdl;
                    else
                        newfunc = getml;
                    return(argv[1]);
                }
            printf("Illegal device name %s\n", csp);
            return("");
        }
    }

    for (pptr = bootab; *pptr; pptr++)
        if (!strncmp(*pptr, ptr, count)) {
            newfunc = getl;
            return(ptr);
        }

    if (*ptr) {
        if (SYSCONF->bootdev == DTTAPE) {
            ptr = mtstrt;
            newfunc = getml;
        }
        else {
            ptr = sastrt;
            newfunc = getdl;
        }
        buff_empty = TRUE;
    }

    return(ptr);
}

/*---------------------------------------------------- ld1file() -------------
/ load UNIX kernel
/---------------------------------------------------------------------------*/
ld1file()
{
    int i, store;
    short vstamp;

    /* Now get four more long words to get to the end of the first header
    and to the beginning of the UNIX header */

    for (i = 0; i < 4; i++)
        store = getl(fd);

    /* read UNIX header */

    filhdr.fmagic   = getw(fd);
    vstamp          = getw(fd);
    filhdr.tsize    = getl(fd);
    filhdr.dsize    = getl(fd);
    filhdr.bsize    = getl(fd);
    filhdr.entry    = getl(fd);

    txtstrt = sizeof (struct naout) + sizeof (struct filehdr) +
            (3 * sizeof (struct scnhdr));

#ifdef DEBUG
    printf("Text start at 0x%x--<cr> ", txtstrt);
    gets(ans);
#endif

    datstrt  = txtstrt + filhdr.tsize;

    newformat++;

    return(0);
}

/*---------------------------------------------------- loaddtc() -------------
/ load hsdt, edt, or scsi controller code.
/---------------------------------------------------------------------------*/
int loaddtc(siptr, btsect, dtc_name, dtc_size, slotnum, first)
struct slot_info *siptr;                /* which slot */
int btsect;                             /* boot sector */
char *dtc_name;                         /* code name to load */
int dtc_size;                           /* code name size */
int slotnum;                            /* slot where the board is plugged in */
char first;                             /* load from device */
{
    static int ldamt;                   /* how much to load */
    register struct icbcmdhdr *p;
    register struct bd_desc *bddesc;
    register int i, j;
    register int *ptr, *dest;           /* from ... to */
    register int *maploc;
    register int pagenum;
    struct bd_desc **nbdptr;
    struct icb *icbptr;
    int oldint;                         /* previous interrupt level */

#ifdef DEBUG
    printf("Entering loaddtc() to load %s\n", dtc_name);
#endif

    icbptr = siptr->slotaddr;           /* icb slot addr */
    bddesc = siptr->bdptr;              /* board description */

    if (bddesc->bd_dwnld > 0)           /* already loaded */
        return;

    if (first) {                        /* first time for this code */

        if (fd)                         /* close the device if already open */
            close(fd);

        if (SYSCONF->bootdev == DTTAPE) {
#ifdef DEBUG
            printf("Opening boot device %s for read.\n", mtstrt);
#endif
            if ((fd = open(mtstrt, 0)) < 0) {
                printf("Cannot open %s\n", mtstrt);
                return(ERR_RDBOOT);
            }
#ifdef DEBUG
            printf("Reading the tape device - %s\n", mtstrt);
#endif
            if (nine_trk) {
                newfunc = getdl;
                wait_for_rew();
            }
            else
                newfunc = getml;
        }
        else {
#ifdef DEBUG
            printf("Opening boot device %s for read.\n", sastrt);
#endif
            if ((fd = open(sastrt, 0)) < 0 ) {
                printf("Cannot open %s\n", sastrt);
                return(ERR_RDBOOT);
            }
#ifdef DEBUG
            printf("Reading the disk device - %s\n", sastrt);
#endif
            lseek(fd, bootsect * BSIZE, 0);
            newfunc = getdl;
        }

        /* setup map, logical address 160k map to 0x?e0000 */

        desaddr = (int *) ADDR160K;
        maploc = (int *) SMAP(desaddr);

        pagenum = bootpg + 0x0e0;

        for (i = 0; i < 16; i++) {
            j = pagenum++;
            *maploc++ = j << 16;
        }
        filhdr.fmagic = newfunc(fd, REWIND);
        if (loadsa(dtc_name, dtc_size)) { /* locate our code */
            printf("Cannot load %s\n", dtc_name);
            return(-1);
        }

        printf("Loading %s code (VER %x)\n",
                 (dtc_name == hsdtcode) ? "HSDT" :
                ((dtc_name == edtcode)  ?  "EDT" : "SCSI/EDT"),
                dtc_vers);

        ldamt = filhdr.tsize >> 2;      /* size of code */

        /* load into main memory 0x?e0000 (0x0e0000 or 0x1e0000) */

        memload(ldamt);                 /* pointed by 'desaddr' */
    }

#ifdef DEBUG
    printf("sending icb commands\n");
#endif

    /*-----------------------------------------------------------------
    / we come here directly without device i/o when we know we have
    / the code already in memory, ie. subsequent loading of same code
    /----------------------------------------------------------------*/

    /* send icb command to HSDT/EDT/SCSI prom code to be in BOOT STATE */

    oldint = chgint(5);
    icbptr->intreg = (IDWNLD | siptr->intword);
    while (!(icbptr->bdtypreg & IOPIPND))
        ;
    i = icbptr->intreg & IDATMSK;
    chgint(oldint);
    if (i != IDWNLD) {
        printf("Downloading %s failed (0x%x)", dtc_name, i);
        return(-1);
    }

    /* find out where is the free space on hsdt/edt/scsi board */

    dest = (int *) ((int) bddesc->bd_memst + IOP_BASE(slotnum));
    p = (struct icbcmdhdr *) dest;
    dest = (int *) (((int) dest) + sizeof (struct icbcmdhdr));

    bddesc->bd_imcptr = (struct icbcmdhdr *) bddesc->bd_memst;

    p->command = IWJCMD;                /* fill in icb command header */
    p->information = 0x0;
    p->count =  filhdr.tsize;

    /* move data over to dual port */

    ptr = (int *) ADDR160K;             /* this where we'll find our code */
    for (i = 0; i < ldamt; i++)         /* load 'em to the board */
        *dest++ = *ptr++;


    /* tell HSDT/EDT/SCSI PROM to process memory resident command */
    /* in memory, there is a "write memory and jump command" */

    oldint = chgint(5);
    icbptr->intreg = (IMEMCMD | siptr->intword);
    while (!(icbptr->bdtypreg & IOPIPND))
        ;
    i = icbptr->intreg & IDATMSK;
    chgint(oldint);

    if (i != IMEMCMD) {
        printf ("Downloading %s failed (0x%x)", dtc_name, i);
loop8:
        goto loop8;                     /* endless loop! */
    }

    /*-----------------------------------------------------------------
    / temporary fix until PROM gets fixed - 20-Apr-88 cjk
    /----------------------------------------------------------------*/

    nbdptr = (struct bd_desc **) (IOP_BASE(slotnum) + BDVECT);
#ifdef DEBUG
    printf("Slot #%d - SYSCONF addr was 0x%x; BDVECT addr is 0x%x\n",
                slotnum,
                (int) SYSCONF->slotinfo[slotnum].bdptr,
                (int) *nbdptr);
    printf("          SYSCONF addr will be 0x%x--<cr>",
                (int) *nbdptr + IOP_BASE(slotnum));
    gets(ans);
#endif
    SYSCONF->slotinfo[slotnum].bdptr = (struct bd_desc *)
            ((int) *nbdptr + IOP_BASE(slotnum));

    return(0);
}

/*---------------------------------------------------- loadsa() --------------
/ search through 'bootimage' and locate an image named 'ptr'.  When located,
/ it returns 0; else non-zero
/---------------------------------------------------------------------------*/
int loadsa(code_name, name_length)
char *code_name;
unsigned int name_length;
{
    register long *lptr;
    register unsigned int drvptr, present;
    register i, totsize, j;
    unsigned int hsize, temp;
    struct fheadr *fhptr;
    int bytcnt, longcnt;

    lptr = (long *) &fhdr;
    fhptr = &fhdr;
    drvptr = BOOTLOC;

    while (((filhdr.fmagic >> 16) & 0xffff) == M68ASAMAGIC) {
        bytcnt = BLKSIZE;
        *lptr++ = filhdr.fmagic;
        hsize = sizeof (struct filehdr);
        for (i = 0; i < ((hsize - 4) / 4); i++, lptr++, bytcnt -= 4)
            *lptr = (*newfunc)(fd, NOREWIND);
        hsize = fhptr->filehdr.f_opthdr;
        for (i = 0; i < hsize / 4; i++, lptr++, bytcnt -= 4)
            *lptr = (*newfunc)(fd, NOREWIND);
        for (i = 0; i < fhptr->filehdr.f_nscns; i++) {
            for (j = 0; j < SCNHSZ / 4; j++, lptr++, bytcnt -= 4) {
                *lptr = (*newfunc)(fd, NOREWIND);
            }
        }

#ifdef DEBUG
        printf("Found '%s' while looking for '%s'--<cr> ",
                fhptr->saouthdr.boothdr.boot_name,
                code_name);
        gets(ans);
#endif
        if (directory)
            printf("%14s", fhptr->saouthdr.boothdr.boot_name);

        if (!strncmp(fhptr->saouthdr.boothdr.boot_name, code_name, name_length)) {
            filhdr.fmagic = fhptr->saouthdr.magic;
            filhdr.tsize = fhptr->saouthdr.tsize;
            filhdr.dsize = fhptr->saouthdr.dsize;
            filhdr.bsize = fhptr->saouthdr.bsize;
            filhdr.entry = fhptr->saouthdr.entry;
            dtc_vers = fhptr->saouthdr.boothdr.boothdr_version;
            return(0);
        }
        else {
            totsize = fhptr->filehdr.f_opthdr + sizeof (struct filehdr);
            for (i = 0; i < fhptr->filehdr.f_nscns; i++) {
                totsize += SCNHSZ;
                if (fhptr->sctnhdr[i].s_scnptr)
                    totsize += fhptr->sctnhdr[i].s_size;
            }
            present = drvptr;
            drvptr += (totsize / BLKSIZE);
            if (totsize % BLKSIZE)
                drvptr++;

#ifdef DEBUG
            printf("Skipping 0x%x bytes\n", totsize);
#endif
            if (directory)
                printf("  %8d (rev %x)\n",
                        totsize, fhptr->saouthdr.boothdr.boothdr_version);

            longcnt = bytcnt / 4;
            while (--longcnt)
                temp = (*newfunc)(fd, NOREWIND);

            while (++present < drvptr) {
                for (i = 0; i < BLKSIZE / 4; i++)
                    temp = (*newfunc)(fd, NOREWIND);
            }
            filhdr.fmagic = (*newfunc)(fd, NOREWIND);
            lptr = (long *) &fhdr;
        }
    }

#ifdef DEBUG
    printf("Header magic 0x%x\n", filhdr.fmagic);
#endif

    return(1);
}

/*---------------------------------------------------- getml() -------------*/
getml(fd, last)
int fd;
int last;
{
    static int *lwptr;
    static int lngcnt;
    static unsigned int totread;

    if (buff_empty) {
        buff_empty = FALSE;
        lwptr = (int *) ADDR100K;
        if ((totread = read(fd, ADDR100K, RDAMT)) < 0)
            lngcnt = 0;
        else
            lngcnt = totread / sizeof (int);
#ifdef DEBUG
        printf("Totread 0x%x lcnt 0x%x\n", totread, lngcnt);
#endif
    }

    if (last) {
        lngcnt = totread / sizeof (int);
        lwptr = (int *) ADDR100K;
    }

    lngcnt--;
    if (lngcnt == 0 && totread == RDAMT) {
        buff_empty = TRUE;
#ifdef DEBUG
        printf("lncount 0x%x\n", lngcnt);
#endif
    }
    if (lngcnt < 0) {
#ifdef DEBUG
        printf("lncount 0x%x\n", lngcnt);
#endif
        return(0);
    }

    return(*lwptr++);
}

/*---------------------------------------------------- getdl() -------------*/
getdl(fd, last)
int fd;
int last;
{
    register int i, btread;
    static int *lwptr;
    static int lngcnt = 0;
    static int totread = 0;
    char *cbptr;

    if (buff_empty) {                        /* file buffer is empty */
        buff_empty = FALSE;                  /* file buffer is not empty */
        totread = 0;
        cbptr = (char *) ADDR100K;
        for (i = 0; (i < DBCNT) && (amt_loaded < (boot_size * BSIZE));
                i++, cbptr += BSIZE) {
            if ((btread = read(fd,cbptr,BSIZE)) < 0) {
                totread = btread;
                break;
            }
            if (btread == 0)
                break;
            totread += btread;
            amt_loaded += btread;
#ifdef DEBUG
            printf("Totread 0x%x lcnt 0x%x\r", totread, lngcnt);
#endif
        }

#ifdef DEBUG
        printf("\n");
#endif

        last = TRUE;
    }

    if (last) {                         /* reset file buffer pointer */
        lngcnt = totread / sizeof (int);
        lwptr = (int *) ADDR100K;
    }

    if (--lngcnt <= 0)                  /* buffer is exhausted */
        buff_empty = TRUE;              /* file buffer is empty */
    if (totread < 0)                    /* this should not happen! */
        return(0);

    return(*lwptr++);                   /* return the integer of interest */
}

/*---------------------------------------------------- parse() ---------------
/ build 'argc' and 'argv' from the given string
/ spaces and tabs are delimiters and will be converted to null chars
/---------------------------------------------------------------------------*/
void parse(s)
register char *s;
{
    argc = 0;

    while (1) {
        while (isspace(*s))             /* skip over delimiters */
            *s++ = '\0';
        if (!*s) {                      /* end of command string */
            argv[argc] = (char *) 0;
            return;
        }
        argv[argc] = s;                 /* save the pointer */
        if (++argc > 18)                /* too many args */
            return;
        while (*s && !isspace(*s))
            s++;
    }
}

/*---------------------------------------------------- wait_for_rew() --------
/ waiting for the tape to rewind... so... we just waste CPU (really crude!)
/---------------------------------------------------------------------------*/
void wait_for_rew()
{
    unsigned short i, j;                /* slower the better */
    unsigned char k, m;                 /* prevent optimization */

    for (i = 0; i < 0x50; i++)
        for (m = 0, j = 0; j < 0x1000; j++)
            k = m;
}

/*----------------------------- End of boot.c ------------------------------*/
