/********************************************************************!HDR!*****
 * format.c
 *
 * description:
 * 
 * a disk is in 1 of 4 states as far as format is concerned:
 *	new disk
 *		can't read sector0
 *		each track has a media defect record
 *	formatted correctly
 *		sector0.id=="INIT", sector0.date[0] == format version(!=0xff),
 *		sector0.pd_ldmaxnum <= 16, sector0 == sector1
 *	formatted incorrectly
 *		can't read sector0 and can't read media defect info,  or
 *		physical cyls/heads/sectors disagree with those in sector0
 *	previous format operation interrupted
 *		sector.id=="INIT", sector0.date[0]==0xff
 *		there may be track media defect records
 *
 * format uses TrackBuffer and CompBuf as scratch areas
 * There is a limit to the number of media defects a disk can have.
 * At present it is equal to the number of cylinders on the disk.
 ********************************************************************!RDH!*****/

#include "disktest.h"
#include "hsdterror.h"


/* 1st version of disktest which does its own formatting */

#define FMTVER          2
#define FMTREL          2

#define MAXMEDFS        1000
#define MEDEFSEC        6
#define BADBSEC         16

int     cmpmed0(), cmpmed1();

struct medf *medfs;
struct medf Medf[MAXMEDFS + 1];

int     mdcount;
int     badcount;
char	abortmsg[] = {"%s aborted due to disk error - retcode 0x%x,%x\n"};

/********************************************************************!HDR!*****
 * format()
 * 	
 *
 ********************************************************************!RDH!*****/
format()
{	int i, j, firstbad, initonce, ts;

    medfs = (struct medf *) (((uint) &Medf[0] + 3) & ~0x3);
	if (setjmp(fabort)) {
		printf("\nFormat aborted.\nThe format operation must be run ");
		printf("again before this disk can be used.\n");
		exit(1);
	}
	/* BRENT
	 * MthrBlk.id[0] == 0 => initdrive() completely failed.
	 * we have no sector 0 info.
	 */
	if (MthrBlk.id[0] == 0) {
		initonce = NO;
		bzero(&MthrBlk, sectsiz);
		MthrBlk.bytesec = sectsiz;
		/* BRENT
		 * if we can't come up with physical parameters, then
		 * ask the human for some
		*/
		if (!hsized(&MthrBlk, YES))
			getdparm(&MthrBlk);
	}
	else	{
		struct sector0 *sp;
		sp = (struct sector0 *)TrackBuffer;
		bzero(sp, sizeof(*sp));
		sp->headcyl = MthrBlk.headcyl;
		initonce = YES;
		if (!hsized(sp, NO)) {
			printf("can't verify disk type\n");
		}
		else if (sp->cyldisk != MthrBlk.cyldisk
		    || sp->headcyl != MthrBlk.headcyl
		    || sp->sechead != MthrBlk.sechead
		    || sp->trkbpsg != MthrBlk.trkbpsg
		    || sp->ltrkbpsg != MthrBlk.ltrkbpsg) {
			printf("disk has been previously formatted incorrectly\n");
			MthrBlk.cyldisk = sp->cyldisk;
			MthrBlk.headcyl = sp->headcyl;
			MthrBlk.sechead = sp->sechead;
			MthrBlk.trkbpsg = sp->trkbpsg;
			MthrBlk.ltrkbpsg = sp->ltrkbpsg;
		}
	}
	/* BRENT
	 * if we are here, we know we can deal with the disk
	 */
	printf("\nWARNING--FORMATTING WILL DESTROY ALL OF THE DATA ON DISK c%dd%d !\n",
			UnitNum >> 4, DriveNum & 0xf);
    while (!(printdtype(&MthrBlk), getyn("Ok to format"))) {
        if (getyn("Do you want to change disk parameters"))
			getdparm(&MthrBlk);
        else if (getyn("Want to quit"))
			exit(1);
		sectsiz = MthrBlk.bytesec;
	}
	WrdsPerSect = sectsiz >> 1;
	TotEntries = 0;
	initMthr(&MthrBlk);
	settmreg();

	printf("\nreading media defect information...");
	/* try to read the initial media defects */
	mdcount = rdinitmd();
	if (initonce || mdcount==0) {
		if ((i = rdmedsec(initonce)) > 0) {
			mdcount += i;
			qsort(medfs, mdcount, sizeof(struct medf), cmpmed1);
		}
		badcount = rdbadbs();
		if (badcount &&
            !getyn("\nDo you want to preserve the old bad block list"))
			badcount = 0;
	}
	else
		badcount = 0;
	if (mdcount + badcount > MAXMEDFS) {
		printf("\n\ntoo many %s (%d)--maximum is %d\n",
			badcount ? "bad blocks" : "media defects",
			mdcount+badcount, MAXMEDFS);
		longjmp(fabort);
	}

	printf("\nformatting track 0...");
	init_buffer(TrackBuffer, sectsiz * MthrBlk.sechead, 0, 1, 0);
	MthrBlk.date[0] = 0xff;				/* set partial format */
	bcopy(&MthrBlk, TrackBuffer, sectsiz);
	sectdup(TrackBuffer, MthrBlk.sechead);

    if(debugflag && !getyn("About to format track 0. Ok")) exit(1);
	if (!writid(CompBuf, 0, 0, 0) || !writc0(TrackBuffer, 0, 0, 0)) {
        printf("Cannot write sector 0\n");
		longjmp(fabort);
	}
	reinit();
	ts = MthrBlk.sechead * sectsiz;
	if (!initonce && mdcount==0)		/* 1 last try to read MD list */
		mdcount = rdmedsec(NO);
	if (writidtr(0, 0) != MthrBlk.sechead 
	    || writtr(TrackBuffer, 0, 0) != ts
	    || readtr(CompBuf, 0, 0) != ts
	    || comp_bufs(TrackBuffer, CompBuf, ts) < ts) {
		printf("can't initialize track 0\n");
		longjmp(fabort);
	}

    printf("\nRecording the media defect list...");
	wrmedfs(TrackBuffer);
	firstbad = MthrBlk.nextsec;
	
	if (badcount > 0) {
		mdcount += badcount;
		badcount = 0;
		qsort(medfs, mdcount, sizeof(struct medf), cmpmed1);
	}
	wrbadbs(CompBuf, conv_defects(MthrBlk.sechead, MthrBlk.trkbpsg,
		MthrBlk.ltrkbpsg, 0, medfs, mdcount, CompBuf));

	init_buffer(TrackBuffer, ts, 0, 1, Pattern);
	j = ((MthrBlk.nextsec + MthrBlk.sechead-1) / MthrBlk.sechead) % MthrBlk.headcyl;
	if (debugflag) {
		char bb[8], *bbp;
        printf("\nStarting cylinder?");
		bbp = bb;
        gets(bb, 8);
		getlong(&bbp, &i);
        if(getyn("debug off"))
			debugflag=0;
	}
	else	i = j / MthrBlk.headcyl;

    printf("\nFormatting cylinder 0x%4x", i);
	for (; i < MthrBlk.cyldisk; i++, j=0) {
		register int r;
		if (usrabort()) {
            if (getyn("Abort format")) {
                printf("Last cylinder formatted 0x%x\n",i-1);
				longjmp(fabort);
			}
			printf("\nformatting cylinder 0x%4x", i);
		}
		printf("\b\b\b\b%4x", i);
		for (; j < MthrBlk.headcyl; j++) {
			if (writidtr(i, j) != MthrBlk.sechead)
				checkmedf(i, j, 0, YES);
			else if (writtr(TrackBuffer, i, j) != ts) {
				for (r=0; r<MthrBlk.sechead; r++) {
				    if (!writdata(TrackBuffer, i, j, r)) {
					checkmedf(i, j, r, YES);
					break;
				    }
				    else if (!readdata(TrackBuffer, i, j, r))
					checkmedf(i, j, r, NO);
				}
			}
		}
	}

    printf("\nRewriting sector 0...");
	if (badcount > 0)
		MthrBlk.nextsec = firstbad;
	MthrBlk.date[0] = version;
	MthrBlk.date[1] = release;
	if (!writdata(&MthrBlk, 0, 0, 0) || !writdata(&MthrBlk, 0, 0, 1)) {
        printf("Cannot rewrite sectors 0,1\n");
		longjmp(fabort);
	}
	reinit();

	if (badcount > 0) {
        printf("\nRewriting the bad block list...");
		mdcount += badcount;
		qsort(medfs, mdcount, sizeof(struct medf), cmpmed1);
		badcount = conv_defects(MthrBlk.sechead, MthrBlk.trkbpsg,
			MthrBlk.ltrkbpsg, 0, medfs, mdcount, CompBuf);
		xprintf("write %d bad blocks\n", badcount);
		wrbadbs(CompBuf, badcount);
	}
	checkrv();
    xprintf("\nFormatting successful!\n");
}

initMthr(sp)
struct sector0 *sp;
{
	strcpy(sp->id, "INIT");
	strcpy(&sp->date[2], "880204");
	sp->pd_ldmaxnum = LOGDR;
	sp->pd_ldnum = 0;
	sp->nextsec = 0x12;
	sp->seccyl = sp->sechead * sp->headcyl;
	sp->rv_strt = 0;
	sp->rv_size = 0x280;
	if (sp->sechead)			/* round to track */
		sp->rv_size += sp->sechead - sp->rv_size % sp->sechead;
	copytmregs(sp);
	bzero(sp->logdrive, sizeof(sp->logdrive));
}

writidtr(c, h)
{	struct devq rq;
	register struct wrsecid *idp;
	register int i, ii;
	int ch, cl;
	char tb[sizeof(struct wrsecid) * 256 + 3];

	idp = (struct wrsecid *)(((int)tb + 3) & ~3);
	bzero(idp, sizeof(struct wrsecid) * MthrBlk.sechead);
	rqsetup(&rq, idp, c, h, 0);
	ch = (c >> 8) & 0xff;
	cl = c & 0xff;
	for (i=0;  i < MthrBlk.sechead; i++, idp++) {
		idp->fmtf = 0;
		idp->cylhi = ch;
		idp->cyllo = cl;
		idp->head = h;
		idp->sec = i;
	}
	rq.q_cmd = WRITEID;
	rq.q_count = MthrBlk.sechead;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(0);
	}
	return(MthrBlk.sechead);
}

writtr(p, c, h)
char *p;
{	struct devq rq;
	rqsetup(&rq, p, c, h, 0);
	rq.q_cmd = PWRITPS;
	rq.q_count = sectsiz * MthrBlk.sechead;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(rq.q_count);
	}
	return(sectsiz * MthrBlk.sechead);
}

readtr(p, c, h)
char *p;
{	struct devq rq;
	rqsetup(&rq, p, c, h, 0);
	rq.q_cmd = PREADPS;
	rq.q_count = sectsiz * MthrBlk.sechead;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(0);
	}
	return(rq.q_count);
}

/* reinitialize the HSDT after sector 0 has been written
 */
reinit()
{	struct devq rq;
	rqsetup(&rq, 0, 0, 0, 0);
	rq.q_cmd = INIT;
	if (devreq(SlotPtr, &rq) || rq.rc1) {
		deverr(rq.rc1, rq.rc2);
		return(NO);
	}
	return(YES);
}


/* read the media defect list from records 6,7... and squash it in place
 * the copy-in-place uses a temp buffer since the source and target overlap
 */
rdmedsec(initonce)
{	register struct medf *sp, *dp;
	int c, next, i, seq, secdisk;
	static char badmd[] = {"bad media defect record in sector %d\n"};

	dp = (struct medf *)&medfs[mdcount];
	c = seq = 0;
	secdisk = MthrBlk.seccyl * MthrBlk.cyldisk;
	for (next=MEDEFSEC; next; ) {
		if (lsectop(PREADPS, next, dp, "") == -1
		    && lsectop(PREADPS, next+1, dp, "") == -1) {
			if (initonce)
				printf(badmd,next);
			return(c);
		}
		i = ((struct md_list *)dp)->nummedlist;
		if (i < 0 || i > MEDLSTSIZ
		    || ((struct md_list *)dp)->nextsec > secdisk
		    || ((struct md_list *)dp)->idfield != MEDSECID
		    || ((struct md_list *)dp)->seqnum != seq++) {
			if (initonce)
				printf(badmd,next);
			return(c);
		}
		sp = ((struct md_list *)dp)->medflist;
		next = ((struct md_list *)dp)->nextsec;
		for (; i--; sp++) {
			struct medf t;
			if ((sp->cyl & 0x7fff)>0x3ff || sp->defpos[0].len==0
			    || ((sp->cyl & 0x7fff)==0 && sp->head==0)) {
				if (initonce)
					printf(badmd,next);
				return(c);
			}
			if (inmedf(sp->cyl, sp->head, -1))
				continue;
			t = *sp;
			*dp++ = t;
			c++;
		}
	}
	xprintf("rdmedsec %d\n", c);
	return(c);
}

/* write the media defect list to tracks 0,1
 */
wrmedfs(mp)
register struct md_list *mp;
{	register struct medf *sp, *dp;
	struct sector0 *s0;
	int i, seq, lsect;

	if (mdcount == 0)
		printf("\nNo media defects--you may need to run defectlist\n");
	else
		printf("copying %d media defect entries\n", mdcount);
	sp = medfs;
	seq = 0;
	lsect = MEDEFSEC;
	i = mdcount;
	do	{
		bzero(mp, sizeof(*mp));
		mp->idfield = MEDSECID;
		mp->seqnum = seq++;
		mp->nummedlist = i;
		dp = mp->medflist; 
		while (i && dp < &mp->medflist[MEDLSTSIZ]) {
			*dp++ = *sp++;
			i--;
		}
		if (i) {
			mp->nummedlist = MEDLSTSIZ;
			mp->nextsec = inextrvsec();
		}
		else
			mp->nextsec = 0;
		lsectop(PWRITPS, lsect, mp, "");
		lsectop(PWRITPS, lsect+1, mp, "");
		bzero(mp+1, sizeof(*mp));
		if (lsectop(PREADPS, lsect, mp+1, "") == -1
		    || comp_bufs(mp, mp+1, sizeof(*mp)) < sizeof(*mp)
		    || lsectop(PREADPS, lsect+1, mp, "") == -1
		    || comp_bufs(mp, mp+1, sizeof(*mp)) < sizeof(*mp)) {
			printf("can't write media defect sector 0x%x\n", lsect);
			longjmp(fabort);
		}
		lsect = mp->nextsec;
	} while (i > 0);
}

/* read the bad block list and throw out entries on the media defect list
 */
rdbadbs()
{	register struct badlist *bp;
	register struct medf *mp;
	register struct sectid *sp;
	int seqnum, n, sec;

	seqnum = 0;
	sec = BADBSEC;
	bp = (struct badlist *)TrackBuffer;
	mp = &medfs[mdcount];
	do	{
		if (lsectop(PREADPS, sec, bp, "") == -1
		    && lsectop(PREADPS, sec+1, bp, "") == -1)
			break;
		if (*(unsigned *)bp != ASCIBAD || bp->idfield != BADSECID
		    || bp->seqnum != seqnum++
		    || (n=bp->numofentries) == 0 || n > BADLSTSIZ) {
			break;
		}
		for (sp=bp->badsctrinfo; n--; sp++) {
			if (sp->cyl==0 || sp->cyl > 0x3ff)
				continue;
			if (inmedf(sp->cyl, sp->head, sp->sector))
				continue;
			mp->cyl = sp->cyl;
			mp->head = sp->head;
			mp->defpos[0].pos = sp->sector * MthrBlk.trkbpsg / 8 + 1;
			mp->defpos[0].len = 8;
			mp++;
		}
		sec = bp->nextsec;
	} while (sec);
	return(mp - &medfs[mdcount]);
}

wrbadbs(sp, n)
register struct sectid *sp;
{	register struct badlist *bp;
	register struct sectid *dp;
	int seqnum, sec;

	seqnum = 0;
	sec = 0x10;
	bp = (struct badlist *)TrackBuffer;
	do	{
		bzero(bp, sizeof(struct badlist));
		*(unsigned *)bp = ASCIBAD;
		bp->idfield = BADSECID;
		bp->seqnum = seqnum++;
		bp->numofentries = n;
		for (dp=bp->badsctrinfo; dp < &bp->badsctrinfo[BADLSTSIZ];) {
			if (n-- <= 0)
				break;
			*dp++ = *sp++;
		}
		if (n > 0) {
			bp->nextsec = inextrvsec();
			bp->numofentries = BADLSTSIZ;
		}
		else
			bp->nextsec = 0;
		if (lsectop(PWRITPS, sec, bp, "bad block list") == -1
		    || lsectop(PWRITPS, sec+1, bp, "bad block list") == -1)
			longjmp(fabort);
		sec = bp->nextsec;
	} while (n>0);
}

sectdup(p, n)
char *p;
{	for (; n>1; n-=2) {
		bcopy(p, p+sectsiz, sectsiz);
		p += sectsiz * 2;
	}
}

/********************************************************************!HDR!*****
 * rdinitmd()
 *
 * look around for original media defect records
 * if the whole disk has been formatted skip the search
 ********************************************************************!RDH!*****/
rdinitmd()
{	register struct medf *mp;
	register int c, h, n;
	short nomd;
	short dd;

	mp = &medfs[mdcount];
	n = 0;
	settmreg();
	for (c=1; c < MthrBlk.cyldisk; ) {
		if (!readid(mp, c, 1, 0))
			break;
		c = (c + MthrBlk.cyldisk + 1) >> 1;
	}
	if (c >= MthrBlk.cyldisk)
		return(0);
	setmdtmreg();
	nomd = YES;
	dd=debugflag;
	debugflag=0;
	printf("scanning cylinder 0x0000");
	for (c=0, h=1; c < MthrBlk.cyldisk; c++, h=0) {
	    printf("\b\b\b\b%4x", c);
	    if (usrabort() || (nomd && c==2)) {
        if (c!=2 && getyn("\nAbort format")) {
			settmreg();
			longjmp(fabort);
		}
        if (nomd && getyn("Skip remaining media defect records"))
			break;
		printf("scanning cylinder 0x%4x", c);
	    }
	    for (; h < MthrBlk.headcyl; h++) {
		if (nomd) {
			settmreg();
			if (readid(mp, c, h, 0))
				continue;
			setmdtmreg();
		}
		if (readmd(mp, c, h)) {
			nomd = NO;
			if (mp->defpos[0].pos == 0 || mp->defpos[0].len == 0)
				continue;
			mp++;
			n++;
		}
		else if (errno == DER_NRDY || errno == DER_DEVCK) {
			printf("Drive went unready\n");
			longjmp(fabort);
		}
	    }
	}
debugflag=dd;
	xprintf("\nreturned %d md records\n",n);
	settmreg();
	return(n);
}

checkmedf(c, h, ns, trackbad)
{	struct medf *mp, *sp;

	if (errno == DER_NRDY || errno == DER_RDYCHG) {
		printf("\ndrive not ready\n");
		longjmp(fabort);
	}
	if (inmedf(c, h, trackbad ? -1 : ns))
		return;
	sp = &medfs[mdcount+badcount];
	sp->cyl = c | (trackbad ? 0x8000 : 0);
	sp->head = h;
	sp->defpos[0].pos = (ns * MthrBlk.trkbpsg / 8) + 1;
	sp->defpos[0].len = 8;
	badcount++;
}

/* see if the named sector is covered by a media defect entry
 */
inmedf(c, h, s)
{	struct medf me;
	static struct medf *mp;
	int i, pos;

	me.cyl = c & 0x7fff;
	me.head = h;
	if (mdcount == 0)
		return(NO);
	mp = (struct medf *)bsearch(&me, medfs, mdcount, sizeof(struct medf), cmpmed0);
	if (mp == 0)
		return(NO);
	if (s == -1 || mp->cyl & 0x8000)
		return(YES);
	s = s * MthrBlk.trkbpsg;
	for (i=0; i<4; i++) {
		if (mp->defpos[i].len == 0)
			break;
		if ((pos=mp->defpos[i].pos * 8) == 0)
			pos += 8;
		if (s + MthrBlk.trkbpsg < pos - 8)
			continue;
		if (s <= pos + mp->defpos[i].len)
			return(YES);
	}
	return(NO);
}

/* compare two media defect entries for searching the defect list
 */
cmpmed0(a,b)
register struct medf *a, *b;
{	int t;
	if ( t = (a->cyl & 0x3ff) - (b->cyl & 0x3ff) )
		return(t);
	return(a->head - b->head);
}

/* compare two media defect entries for sorting
 */
cmpmed1(a,b)
register struct medf *a, *b;
{	int t;
	if ( t = (a->cyl & 0x3ff) - (b->cyl & 0x3ff) )
		return(t);
	else if ( t = (a->head - b->head) )
		return(t);
	else
		return(a->defpos[0].pos - b->defpos[0].pos);
}

/* check the next logical sector from the reserved area
 * if the new sector crosses a track boundary format the track first
 */
inextrvsec()
{	int t;
	if ((MthrBlk.nextsec % MthrBlk.sechead) == 0) {
		t = MthrBlk.nextsec / MthrBlk.sechead;
		writidtr(t / MthrBlk.headcyl, t % MthrBlk.headcyl);
	}
	if ((++MthrBlk.nextsec % MthrBlk.sechead) == 0) {
		t = MthrBlk.nextsec / MthrBlk.sechead;
		writidtr(t / MthrBlk.headcyl, t % MthrBlk.headcyl);
	}
	t = getnextlsect();
	MthrBlk.nextsec = t + 2;
	return(t);
}

/* check for a bad track in the reserved area
 */
checkrv()
{	int i, c, h, s;

	h = MthrBlk.rv_size / MthrBlk.sechead;
	c = h / MthrBlk.headcyl;
	s = 0;
	for (i=0; i < mdcount; i++) {
		if ((medfs[i].cyl & 0x7fff) > c
		    || ((medfs[i].cyl & 0x7fff) == c && medfs[i].head > h)) {
			i = mdcount;
			break;
		}
		if (medfs[i].cyl & 0x8000)
			break;
		if (medfs[i].defpos[0].pos == (s * MthrBlk.trkbpsg / 8) + 1) {
			if (s++ == MthrBlk.sechead)
				break;
		}
		else
			s = 0;
	}
	if (i < mdcount) {
		printf("\n\nWARNING: an unformattable track lies within the reserved area\n");
		printf("     CCC.HH = 0x%3x.%2x, reserved area size = 0x%x\n",
			medfs[i].cyl & 0x3ff, medfs[i].head, MthrBlk.rv_size);
	}
}

#ifdef DEBUG
#undef devreq
xdevreq(sp, rp)
struct slot_info *sp;
struct devq *rp;
{
    int r;

	rp->rc1 = rp->rc2 = 0;

    if (debugflag) {
        printf("devreq($%x) unit $%x ", rp->q_cmd, rp->q_devnum);
        if (ScsiDrv)
            printf("block $%x func($%x)\n", rp->q_devun.block, rp->opr);
        else
            printf("chs $%x.%x.%x\n", rp->q_devun.pdisk.cyl,
                    rp->q_devun.pdisk.head, rp->q_devun.pdisk.sector);
    }
	if (r = devreq(sp, rp))
        xprintf("devreq = $%x\n", r);
    if (debugflag && rp->rc1)
        deverr(rp->rc1, rp->rc2);
    return(r);
}
#endif
