/*----------------------------------------------------------------------------
/ dset2.c - a portion of 'dsetup'
/
/ 4/28/86  Peter Logan
/ Fixed Sperry bug #303 (when modifying the protection of drives which
/ contained no bad sectors and that were followed by drives that had
/ both spares and skip tracks, the program would abort)
/ Change was made in "sizesection()"
/---------------------------------------------------------------------------*/

#include "disksect.h"
#include "dsetup.h"
#include "setjmp.h"


extern warn(),error();
struct sparedata *addspare();

extern int scsi_dtc;                    /* SCSI */


/*---------------------------------------------------- writeconfig() ---------
/ apply the selected configuration to disk
/ TODO: have some option to allow the user to suggest changes when
/       things don't fit
/---------------------------------------------------------------------------*/
writeconfig(setup)
struct setup *setup;
{
    register struct physbadsector *bads;
    struct sparedata *spare1,*spare;
    struct sprs *spare2;
    int count;
    int cbad;

    xprintf("writeconfig(),");
    if (!(setup->flag & SET_FIG))
        applyconfig(setup);

    /* cycle through bad sector table and make sure all sectors accounted for */
    spare1 = (struct sparedata *) (setup->memSpare->p);
    cbad = setup->cMemBad;
    for (bads = (struct physbadsector *) (setup->memBad->p);
            cbad > 0; cbad--, bads++) {
        if (!(bads->flag & (BS_HOLE | BS_SPARE | BS_UNSPARE)))
            software_bug("writeconfig", "Found unmarked bad sector");
        if ((bads->flag & (BS_HOLE |  BS_SPARE)) == (BS_HOLE | BS_SPARE))
            software_bug("writeconfig", "Doubly marked bad sector");
        if (bads->index != -1 && bads->index >= setup->cMemSpare)
            software_bug("writeconfig", "Spare index out of range [$%x]", bads->index);
        if (bads->flag & BS_SPARE) {
            if (bads->index == -1)
                software_bug("writeconfig", "No spare index assigned");
        }
        else if (bads->flag & BS_UNSPARE) {
            /* should check here */
        }
        else {
            if (bads->index != -1) { /* should be a bad alt sector */
                spare = &spare1[bads->index];
                if (VALCHS(spare->altcyl) != VALCHS(bads->cyl))
                    software_bug("writeconfig", "Bad spare index");
            }
        }
    }

    /* convert spare table in place, this destroys the old representation */
    spare2 = (struct sprs *) spare1;
    count = setup->cMemSpare;
    for ( ; count > 0 ; count--)
        memcpy(spare2++, &((spare1++)->cyl), sizeof (struct sprs));

    rvwriteconfig(setup);
    /* mark all the logical disks as protected */
    setup->flag |= SET_WRIT;
}

/*---------------------------------------------------- cmp_alt() -------------
/ for a new disk
/ Call assign start.  Then call applystart.
/ Assignstart determines the number of spare tracks required for each logical
/ disk.
/ Applystart sequentially scans the logical drives and assigns bad sectors.
/
/ old disks will have this stuff mostly setup already.  The existing spare
/ tables and skip track tables will have to be checked.  Since logical disks
/ will normally (hopefully) be contiguous with the spare sectors assigned to
/ them they can be pooled again.
/
/ The tradeoff of putting spare sectors on (or in) the actual bad track area
/ is best (the skip track lookup must go fast).
/----------------------------------------------------------------------------*/
cmp_alt(a,b)
struct sparedata *a,*b;
{
    return(*(long *)&a->altcyl - *(long *)&b->altcyl);
}

cmp_cyl(a,b)
struct sparedata *a,*b;
{
    return(*(long *)&a->cyl - *(long *)&b->cyl);
}

/*---------------------------------------------------- applyconfig() -------*/
applyconfig(setup)
struct setup *setup;
{
    register int i;
    register struct sparedata *spare,*spare2;

    xprintf("applyconfig(), ");
    setup->flag |= SET_FIG;
    assign1start(setup,1);
    i = 0;
    while (i++ < 10 && assign1start(setup, 0))
        ;
    if (i >= 10)
        software_bug("applyconfig", "Cannot configure after %d tries", i);
    if (setup->flag & SET_NEW) {        /* if freshly formatted */
        ssalloc(setup);                 /* set up spare/skip lists */
        setup->cMemSpare = 0;
        setup->cMemSkip = LOGDR + 1;
    }
    else if (!(setup->flag & SET_WBADS))
        getsslist(setup);
    clearprotect(setup);
    cleanss(setup);
    applystart(setup);

    { /* clean up the spare list */
    register int cspare;

    xprintf("applyconfig(): cleanupspare, ");
    cspare = setup->cMemSpare;
    spare = (struct sparedata *)(setup->memSpare->p);
    for (i=cspare;i>0;i--,spare++) {
        /* set alt of unused to -1, set alt of bad alt to alt number */
        if (VALCHS(spare->cyl) == -1) SETCHS(spare->altcyl,-1);
        if (VALCHS(spare->cyl) == 0) SETCHS(spare->cyl,VALCHS(spare->altcyl));
    }
    cleanss(setup);
    }

    { /* recalculate bads->index */
    register struct physbadsector *bads;
    register int cbad;
    struct sparedata sparekey;

    xprintf("applyconfig(): recalcbadindex, ");
    bads = (struct physbadsector *)(setup->memBad->p);
    spare = (struct sparedata *)(setup->memSpare->p);
    qsort((char *)spare,setup->cMemSpare, sizeof(struct sparedata), cmp_cyl);
    for (cbad = setup->cMemBad;cbad > 0;cbad--,bads++) {
        if (bads->index != -1) {
            SETCHS(sparekey.cyl,VALCHS(bads->cyl));
            spare2 = (struct sparedata *)bsearch(&sparekey,spare,
                setup->cMemSpare, sizeof(*spare), cmp_cyl);
            if (!spare2) errxit("applyconfig: bad with index has no spare");
            bads->index = spare2 - spare;
        }
    }
    }

    for (i = setup->cMemSpare; i > 0; i--, spare++) /* remark bad alts */
        if (VALCHS(spare->cyl) == VALCHS(spare->altcyl)) SETCHS(spare->cyl,0);
    checkapplysetup(setup, warn, errorbuf);
}

/*---------------------------------------------------- assign1start() --------
/ start handling protected disks
/
/ this procedure selects contiguous groups of non protected disks and passes
/ these sections of disk to assign2start
/ return 0 if no change made
/---------------------------------------------------------------------------*/
assign1start(setup, firstflag)
struct setup *setup;
{
    int change;
    int i,j;
    int avail;
    register struct confld *ld;
    register int nextstart;

    memset(&setup->cDiskBad, 0, sizeof (struct cbad));

    change = 0;
    i = RV_LOGDR;
    nextstart = 0;
    while (i < LOGDR) {
        ld = &setup->confld[i];
        if ((ld->flag & CFLD_PROTECT) || !ld->type) {
            if (firstflag && ld->type) {
                nextstart = ld->start + ld->cSect;
                switch(ld->type) {
                default:
                    if (i != RV_LOGDR) {
                        nextstart += ld->cBad.cTrack * PD0(setup)->sechead;
                        break;
                    }
                    /* else fall through */
                case LD_SWAP: case LD_USERSKIP:
                    nextstart += ld->cBad.cSkip;
                    break;
                }
            }
            setup->cDiskBad.cSkip += ld->cBad.cSkip;
            setup->cDiskBad.cTrack += ld->cBad.cTrack;
            i++;
            continue;
        }
        if (firstflag)
            ld->start = nextstart;
        j = writelasti(setup, i, PASS_EMPTY | EMPTY_OK | FREEZE_OK);
        avail = sizesection(setup,i,j);
        xprintf("assign1start: i=%d, j=%d, avail=%d, ",i,j,avail);
        change += assign2start(setup, i, j, avail, firstflag);
        i = j + 1;
    }
    xprintf("change = %d\n",change);
    return(change);
}

/*---------------------------------------------------- assign2start() --------
/ assign physical disk starting addresses
/ this is guessing at where to start
/
/   a clean setup is assumed.  ie. xx->start,
/   xx->cAssign, etc. are 0.
/
/ the spare handling seems awkward.  the tracks skipped by the skip track type
/ disks are not considered in calculating the spare requirements.  this is so
/ that spares can be relatively close to where the bad sectors are.
/
/ Some function which gives average distance to spare needs to be computed.
/ This function would be used to determine the exact placement and usage of
/ spare tracks.  A few extra spare tracks scattered here and there could do
/ much to guarentee a minimal distance to spare
/
/ Return 0 if no change made; else number of changes
/----------------------------------------------------------------------------*/
assign2start(setup, i, j, avail, firstflag)
register struct setup *setup;
{
    register struct confld *ld;
    register int nextdiskstart;
    register int ii;
    register int change = 0;
    register int oldskip;

    /* pick an initial assignment, start with the reserved area */
    /* dummy up a reserved area logical disk */
    xprintf("assign2start %d, ",firstflag);
    ii = i;
    ld = &setup->confld[ii];
    nextdiskstart = ld->start;
    for (; ii <= j; ld++, ii++)
        if (ld->type) {
            if (ld->flag & CFLD_PROTECT)
                software_bug("assign2start", "Protected logical disk encountered");
            /* part of a new sequence */
            if (ld->start != nextdiskstart) {
                ld->start = nextdiskstart;
                change++;
            }
            if (firstflag)
                ld->cAssignSect = ld->cSect;
            oldskip = ld->cBad.cSkip;
            ldcountbad(ld, setup);
            switch (ld->type) {
                case LD_UNIXFS:
                case LD_USERSPLIT:
                case LD_PWF:
                case LD_USERSPARE:
                case LD_USERHOLE:
                    /* is there a way to guarentee size if holes? frozen? ?keepsize */
                    ld->cBad.cSkip = roundtrack( setup, ld->cBad.cSpare)
                                        + ld->cBad.cTrack * PD0(setup)->sechead;
                    if (oldskip != ld->cBad.cSkip) change++;
                    break;

                case LD_SWAP:
                case LD_USERSKIP:
                    ld->cBad.cSkip = ld->cBad.cTrack * PD0(setup)->sechead;
                    if (oldskip != ld->cBad.cSkip) change++;
                    break;
            }
            setup->cDiskBad.cSpare += ld->cBad.cSpare;
            setup->cDiskBad.cHole += ld->cBad.cHole;
            setup->cDiskBad.cTrack += ld->cBad.cTrack;
            setup->cDiskBad.cSkip += ld->cBad.cSkip;
            avail -= (ld->cAssignSect + ld->cBad.cSkip);
            nextdiskstart += ld->cAssignSect + ld->cBad.cSkip;
        }

    if (avail) { /* sizes didn't quite match up */
        if (avail > 0)
            ii = writelasti(setup, i, PASS_EMPTY|FREEZE_OK|NOEMPTY_OK);
        else
            ii = writelasti(setup, i, PASS_EMPTY|NOFREEZE_OK|NOEMPTY_OK);
        ld = &setup->confld[ii];
        ld->cAssignSect += avail;
        ld->cSpareLim = calcspare(ld,setup);
        change++;
    }

    if (change)
        xprintf("%d changes\n", change);
    else
        xprintf("no change\n");

    return(change);
}

/*---------------------------------------------------- sizesection() ---------
/ determine the number of sectors available from ld'i' upto and including
/ ld'j'
/---------------------------------------------------------------------------*/
sizesection(setup,i,j)
register struct setup *setup;
{
    register struct confld *ld;
    register int beginblock, endblock;

    ld = &setup->confld[i];
    if (!ld->type)
        software_bug("sizesection", "First logical disk %d is empty", i);
    beginblock = ld->start;

    j++;
    if (j == LOGDR)
        endblock = setup->cTotalSect;
    else {
        ld = &setup->confld[j];
        if (!ld->type)
            software_bug("sizesection", "Second logical disk %d is empty", j);
        endblock = ld->start;
        if (ld->type != LD_SWAP && ld->type != LD_USERSKIP)
            endblock -= ld->cBad.cSkip - ld->cBad.cTrack*PD0(setup)->sechead;
    }
    return(endblock - beginblock);
}

cleanss(setup)
struct setup *setup;
{
    register struct sparedata *spare, *spare2, *sparelim;
    register struct badtrck *skip, *skip2;
    register int i;
    register int cspare;

    spare = (struct sparedata *)(setup->memSpare->p);
    cspare = setup->cMemSpare;
    qsort((char *)spare,cspare, sizeof(struct sparedata), cmp_alt);
    /* the unused sorted to the front, get rid of them */
    sparelim = &spare[cspare];
    for (spare2=spare;VALCHS(spare2->altcyl) == -1 && spare2<sparelim;spare2++);
    i = sparelim - spare2;
    setup->cMemSpare = i;
    for ( ; i > 0; i--)
        *spare++ = *spare2++;

    skip = (struct badtrck *)setup->memSkip->p;
    skip2 = skip;
    for (i = 0; i < LOGDR + 1; ) {
        while (VALCHS(skip2->cyl) == -1)
            skip2++;
        *skip = *skip2++;
        if (VALCHS(skip++->cyl) == 0)
            i++;
    }
    setup->cMemSkip = skip - (struct badtrck *) setup->memSkip->p;
}

/*---------------------------------------------------- clearprotect() --------
/ iterate through ld's and if a drive is not protected then clear the
/ associated bads, spares, and skips
/---------------------------------------------------------------------------*/
clearprotect(setup)
register struct setup *setup;
{
    register struct confld *ld;
    register struct badtrck *skip, *skipend;
    register int i;

    skip = (struct badtrck *)setup->memSkip->p;
    skipend = &skip[setup->cMemSkip];
    for (i = 0; i < LOGDR; i++) {            /* find skip list for rsv area */
        while (VALCHS(skip++->cyl))
            ;
    }
    for (i = RV_LOGDR, ld = &setup->confRsv; i < LOGDR; i++, ld++) {
        if (ld->type && !(ld->flag&CFLD_PROTECT)) {
            register struct physbadsector *bads =
                (struct physbadsector *)(setup->memBad->p);
            register int cbads = setup->cMemBad;
            register struct sparedata *spare;
            int hi,lo;

            hi = block_to_chs(setup,ld->start + ld->cAssignSect + ld->cBad.cSkip);
            lo = block_to_chs(setup,ld->start);
            for (;cbads>0 && VALCHS(bads->cyl) < hi;cbads--,bads++) {
                if (VALCHS(bads->cyl) < lo)
                    continue; /* skip until in range */
                if (bads->flag & BS_UNSPARE) {
                    bads->flag &= ~(BS_PROTECT|BS_SPARE);
                    continue;
                }
                bads->flag = 0;
                if (bads->index != -1) {
                    spare = ((struct sparedata *)(setup->memSpare->p));
                    spare = &spare[bads->index];
                    spare->flag = 0;
                    SETCHS(spare->altcyl,-1);
                }
                bads->index = -1;
            }
            while (skip < skipend && VALCHS(skip->cyl))
                SETCHS(skip++->cyl, -1);
            skip++;
        }
        else if (!(ld->type)){
            while (skip < skipend && VALCHS(skip->cyl))
                SETCHS(skip++->cyl, -1);
        }
        else {
            while (skip < skipend && VALCHS(skip++->cyl))
                ;
        }
        if (i == RV_LOGDR)
            skip = (struct badtrck *)setup->memSkip->p;
    }
}

applystart(setup)
register struct setup *setup;
{
    register struct confld *ld;
    register int nextdiskstart;
    register int i, bt;
    int rvskip;

    /* CHECK PROTECT */

    xprintf("applystart(), ");
    ld = &setup->confRsv;
    if (!(ld->flag & CFLD_PROTECT)) {
        applysparetracks(setup,ld->cAssignSect,
                         ld->cBad.cSkip - ld->cBad.cTrack*PD0(setup)->sechead);
        spareanybad(setup, 0, ld->cAssignSect);
    }
    applyskiptrack(setup, ld, ld->cAssignSect + ld->cBad.cSkip);
    nextdiskstart = ld->cAssignSect + ld->cBad.cSkip;
    for (ld = setup->confld, i = 0; i < LOGDR; ld++, i++) {
        bt = ld->cBad.cTrack * PD0(setup)->sechead;
        switch (ld->type) {
            case LD_UNIXFS:
            case LD_USERSPLIT:
            case LD_PWF:
            case LD_USERSPARE:
            case LD_USERHOLE:
                if (ld->flag & CFLD_PROTECT) {
                    if ((nextdiskstart += ld->cBad.cSkip)
                            != (ld->start + bt))
                        software_bug("applystart1",
                                "Logical disk %d, protect start match", i);
                    nextdiskstart += ld->cAssignSect;
                    applyskiptrack(setup, ld, ld->cSpareLim);
                    break;
                }
                /* else not protected */
                if (nextdiskstart != ld->start)
                    software_bug("applystart1",
                            "Logical disk %d, diskstart doesn't match", i);
                applysparetracks(setup, nextdiskstart,
                          ld->cBad.cSkip - bt);
                nextdiskstart += ld->cBad.cSkip;
                ld->start = nextdiskstart - bt;
                applyskiptrack(setup,ld,ld->cSpareLim + bt);
                spareanybad(setup, ld->start, ld->cSpareLim + bt);
                if (ld->cSpareLim < ld->cAssignSect) /* set the HOLES */
                    markanybad(setup,ld->start + ld->cSpareLim + bt,
                            ld->cAssignSect - (ld->cSpareLim + bt));
                nextdiskstart += ld->cAssignSect;
                break;
            case LD_SWAP:
            case LD_USERSKIP:
                if (nextdiskstart != ld->start)
                    software_bug("applystart2",
                            "Logical disk %d, diskstart doesn't match", i);
                applyskiptrack(setup,ld,ld->cAssignSect + ld->cBad.cSkip);
                nextdiskstart += (ld->cAssignSect + ld->cBad.cSkip);
                break;
            default: /* no logical disk marked */
                break;
        }
    }
}

cmp_bads(a,b)
struct physbadsector *a,*b;
{
    return(VALCHS(a->cyl) - VALCHS(b->cyl));
}

applysparetracks(setup,lo,count)
register struct setup *setup;
{
    register int i;
    register int sect;
    register struct sparedata *spare;
    register struct physbadsector *bads;
    struct physbadsector keybad;
    int chs;
    extern cmp_chs();

    checktrack(setup,lo);
    checktrack(setup,count);
    for (i = 0, sect = lo; i < count; i++, sect++) {
        spare = addspare(setup, chs = block_to_chs(setup,sect));
        SETCHS(keybad.cyl, chs);
        bads =  (struct physbadsector *) bsearch((char *)&keybad,
            setup->memBad->p, setup->cMemBad, sizeof(keybad), cmp_bads);
        if (bads) { /* a hole in a spare track */
            if (bads->flag &(BS_PROTECT|BS_HOLE|BS_SPARE))
                software_bug("applysparetracks", "Attempt to double mark");
            bads->flag = BS_HOLE;
            SETCHS(spare->cyl,0);
            bads->index = spare - ((struct sparedata *)(setup->memSpare->p));
        }
    }
}

/*
 *      MUSTDO:  fixup, this currently doesn't put the skip
 *              tracks into the available spare list.
 */
applyskiptrack(setup,ld,lim)
register struct setup *setup;
register struct confld *ld;
{
        register struct physbadsector *bads =
            (struct physbadsector *)(setup->memBad->p);
        register struct badtrck *skip;
        register int cbads = setup->cMemBad;
        register int hi,lo;
        register uint tlast;
        register uint lastskip = 0;
        register int tnewskip = 0;

        skip = (struct badtrck *)setup->memSkip->p;
        if ((lo = ld - setup->confld) == -1)
            lo = LOGDR;                                 /* rsvd area */
        while (lo--) {
            while (VALCHS(skip->cyl))
                skip++;
            skip++;
        }
        lo = block_to_chs(setup,ld->start);
        hi = block_to_chs(setup,ld->start + lim);
        for (; cbads > 0 && VALCHS(bads->cyl) < hi; cbads--, bads++) {
            if (VALCHS(bads->cyl) < lo)
                continue;
            if (bads->flag & (BS_UNSPARE|BS_SPARE))
                continue;
            if ((bads->flag & (BS_PROTECT|BS_HOLE)) == BS_HOLE)
                software_bug("applyskiptrack", "Attempt to double mark sector");
            if ((tlast=VALCH(bads->cyl)) == lastskip) {
                bads->flag |= BS_HOLE;
                continue;
            }
            else if (ld->type!=LD_SWAP && ld->type!=LD_USERSKIP && !trackbad(setup, bads))
                continue;
            lastskip = tlast;
            bads->flag |= BS_HOLE;
            if (bads->flag & BS_PROTECT) {
                if (lastskip != ((skip->cyl << 16) | (skip->head << 8)))
                    warn("old skip entry $%x is bad\n", VALCHS(skip->cyl));
            }
            else
                addskip(setup,lastskip,skip);
            tnewskip++;
            skip++;
        }
        if (ld->cBad.cTrack != tnewskip) {
            warn("Ld %d: incorrect number of skips", ld-setup->confld);
            if (!(ld->flag & CFLD_PROTECT))
                software_bug("applyskiptrack", "Not protected");
        }
}

/*---------------------------------------------------- spareanybad() ---------
/ iterate through bad block list and if a bad sector between 'lo' and
/ 'lo+count', then call sparebadsector.
/---------------------------------------------------------------------------*/
spareanybad(setup, lo, count)
register struct setup *setup;
{
    register struct physbadsector *bads = (struct physbadsector *)(setup->memBad->p);
    register int cbads = setup->cMemBad;
    int hi;

    hi = block_to_chs(setup, lo + count);
    lo = block_to_chs(setup, lo);
    for ( ; cbads>0 && VALCHS(bads->cyl) < hi; cbads--, bads++) {
        if (VALCHS(bads->cyl) < lo || (bads->flag & BS_UNSPARE))
            continue; /* skip until in range */
        if (trackbad(setup, bads)) {
            bads += PD0(setup)->sechead - 1;
            cbads -= PD0(setup)->sechead - 1;
        }
        else if (bads->flag & (BS_PROTECT|BS_HOLE|BS_SPARE))
            software_bug("spareanybad", "Attempt to double mark sector");
        else {
            bads->flag |= BS_SPARE;
            sparebadsector(setup,bads);
        }
    }
}

/*---------------------------------------------------- markanybad() ----------
/ iterate through bad block list and if a bad sector between 'lo' and
/ 'lo+count' then mark the bad sector as a hole.
/---------------------------------------------------------------------------*/
markanybad(setup,lo,count)
register struct setup *setup;
{
    register struct physbadsector *bads = (struct physbadsector *)(setup->memBad->p);
    register int cbads = setup->cMemBad;
    int hi;

    hi = block_to_chs(setup,lo + count);
    lo = block_to_chs(setup,lo);
    for ( ; cbads > 0 && VALCHS(bads->cyl) < hi; cbads--, bads++) {
        if (VALCHS(bads->cyl) < lo || (bads->flag & BS_UNSPARE))
            continue; /* skip until in range */
        if (bads->flag & (BS_PROTECT|BS_HOLE|BS_SPARE))
            software_bug("markanybad", "Attempt to double mark sector");
        bads->flag |= BS_HOLE;
    }
}

/*---------------------------------------------------- sparebadsector() ------
/ start at end of spare table and pick first empty.
/ find sector in bad list and mark it as spared.
/---------------------------------------------------------------------------*/
sparebadsector(setup,bads)
register struct setup *setup;
struct physbadsector *bads;
{
    register struct sparedata *spare;
    register int spareindex;

    spareindex = setup->cMemSpare-1;
    /* point "spare" to the last one */
    spare = &((struct sparedata *)(setup->memSpare->p))[spareindex];
    for (; spareindex >= 0; spare--, spareindex--)
        if (VALCHS(spare->cyl) == -1) {
            SETCHS(spare->cyl, VALCHS(bads->cyl));
            bads->index = spareindex;
            return;
        }
    software_bug("sparebadsector", "No available sector");
}

addsbad(setup,chs)
register struct setup *setup;
{
    register struct sparedata *sp, *se;
    register struct physbadsector *pb, *pe;
    struct logtype *ld;
    int i, flag, index;
    extern struct logtype *ownld();

    pe = &((struct physbadsector *) setup->memBad->p)[setup->cMemBad];
    for (pb = (struct physbadsector *) setup->memBad->p; pb < pe; pb++) {
        if (VALCHS(pb->cyl) >= chs)
            break;
    }
    if (VALCHS(pb->cyl) == chs) {
        if (!(pb->flag & BS_UNSPARE))
            error("block $%x is already on bad list\n", chs);
    }
    else if (((char *) pe - (char *) setup->memBad->p) >= setup->memBad->c)
        error("no room in bad block list. Write the current configuration first\n");
    ld = ownld(setup, chs);
    /* take care of sparing/skipping */
    switch (ld->ld_type) {
        case 0:         /* reserved area */
        case LD_UNIXFS:
        case LD_PWF:
        case LD_USERSPARE:
            i = block_to_chs(setup, ld->ld_strt + ld->ld_size);
            sp = (struct sparedata *)setup->memSpare->p;
            se = sp + setup->cMemSpare;
            xprintf("spare count %d\n", setup->cMemSpare);
            for (; sp < se; sp++) {              /* find end for this LD */
                if (VALCHS(sp->cyl) > i)
                    break;
            }
            if (setup->cMemSpare > 0)
                sp--;                           /* back up to last */
            if ( VALCHS(sp->altcyl) == 0)
                goto reconfig;
            i = block_to_chs(setup, chs_to_block(setup, VALCHS(sp->altcyl)) - 1);
            if ( block_to_chs(setup, ld->ld_strt-ld->ld_skip) > i )
                goto reconfig;
            for (; se > sp; se--) {
                SETCHS(se->cyl, VALCHS((se-1)->cyl));
                SETCHS(se->altcyl, VALCHS((se-1)->altcyl));
            }
            SETCHS(sp->altcyl, i);
            SETCHS(sp->cyl, chs);
            setup->cMemSpare++;
            flag = BS_SPARE;
            index = sp - (struct sparedata *)setup->memSpare->p;
            break;
        case LD_SWAP:
        case LD_USERSKIP:
        reconfig:
            i = ld - PD0(setup)->logdrive;
            printf("size will decrease\n");
            if (setup->confld[i].flag & CFLD_PROTECT)
                error("Ldisk %d must be unprotected\n", i);
            setup->confld[i].cSpareLim = calcspare(&setup->confld[i], setup);
            checkusersetup(setup, warn, errorbuf);
            /* fall through */
        case LD_USERHOLE:
        default:
            flag = BS_HOLE;
            index = -1;
            break;
    }

    /* insert the new bad block in the list */
    for (; pe > pb; pe--) {
        *pe = *(pe-1);
        if (index!=-1 && pe->index!=-1)
            pe->index++;
    }
    SETCHS(pb->cyl, chs);
    pb->index = index;
    pb->flag = flag;
    setup->cMemBad++;
    setup->flag |= SET_WBADS;
}


remsbad(setup, chs)
register struct setup *setup;
{
    register struct physbadsector *pb, *pe;
    int i;

    pb = (struct physbadsector *)setup->memBad->p;
    pe = &pb[setup->cMemBad];
    for (; pb < pe; pb++) {
        if (VALCHS(pb->cyl) >= chs)
            break;
    }
    if (VALCHS(pb->cyl) != chs || (pb->flag & BS_UNSPARE))
        error("$%x not on the bad block list\n", chs);

    i = ownld(setup, chs) - PD0(setup)->logdrive;
    if ((setup->confld[i].flag & CFLD_PROTECT) && !(pb->flag & BS_SPARE))
        error("Ldisk %d is protected\n", i);

    pb->flag = BS_UNSPARE;
    if (pb->index == -1) {
        printf("is not spared\n", chs);
        pb->flag |= BS_HOLE;
    }

    setup->flag |= SET_WBADS;
}

struct sparedata *addspare(setup, alt)
register struct setup *setup;
{
    register struct sparedata *spare;
    register int maxSpare = setup->memSpare->c / sizeof(*spare);
    register int iSpare = setup->cMemSpare;

    if (iSpare >= maxSpare)
        software_bug("addspare", "Too many spares");
    spare = &((struct sparedata *)(setup->memSpare->p))[iSpare];
    spare->flag = 0;
    SETCHS(spare->cyl,-1);
    SETCHS( spare->altcyl, alt);
    setup->cMemSpare++;
    return(spare);
}

addskip(setup, l, s)
register struct setup *setup;
struct sectid l;
struct badtrck *s;
{
    register struct sectid *chs = &l;
    register struct badtrck *badt;

    if (setup->cMemSkip >=  setup->memSkip->c / sizeof(struct badtrck))
        software_bug("addskip", "Out of skip table");
    badt = &((struct badtrck *)(setup->memSkip->p))[setup->cMemSkip];
    for (; badt > s; badt--)
        badt[0] = badt[-1];
    badt->cyl = chs->cyl;
    badt->head = chs->head;
    setup->cMemSkip++;
}

checktrack(setup,bno)
struct setup *setup;
{
    if (bno % PD0(setup)->sechead)
        errxit("checktrack: count not multiple of tracksize");
}

/*---------------------------------------------------- ldcountbad() ----------
/ traverse the bad block list based on the start disk address and number of
/ blocks of "ld" based on the requested size (cSect).
/ count up #holes, #spare, #skip based on the type of the logical disk.
/
/ The three values can be interpreted differently based on the type.
/              LD_SWAP,LD_USERSKI
/                      #track = number of tracks with some bad
/                      #skip = #track * sect/track
/                      #spare = 0
/                      #hole = total number of bad ( #hole <= #skip )
/              LD_everything_else
/                      #track = number of tracks with some bad
/                      #skip = roundup of #spares
/                      #spare = # of bad sector which need spares allocated.
/                      #hole = # of baddies which don't need spares.
/----------------------------------------------------------------------------*/
ldcountbad(ld,setup)
register struct confld *ld;
register struct setup *setup;
{
    register struct physbadsector *bads = (struct physbadsector *)(setup->memBad->p);
    register int cbads = setup->cMemBad;
    register int lo,hi,sparelim;
    register uint lastskip,tlast;
    register int t;

    lastskip = 0;
    t = ld->cBad.cSkip;
    memset(&ld->cBad, 0, sizeof (struct cbad));
    lo = block_to_chs(setup,ld->start);
    hi = block_to_chs(setup,ld->start + ld->cAssignSect + t);
    if (ld->type != LD_SWAP && ld->type != LD_USERSKIP)
        sparelim = block_to_chs(setup,ld->start + ld->cSpareLim + t);
    else
        sparelim = 0;
    for ( ; cbads > 0 && VALCHS(bads->cyl) < hi; cbads--, bads++) {
        if (VALCHS(bads->cyl) < lo || (bads->flag & BS_UNSPARE))
            continue; /* skip until in range */
        if (sparelim <= VALCHS(bads->cyl)) {
            ld->cBad.cHole++;
        }
        else if (trackbad(setup, bads)) {
            ld->cBad.cTrack++;
            cbads -= PD0(setup)->sechead - 1;
            bads += PD0(setup)->sechead - 1;
            lastskip = VALCHS(bads->cyl);
        }
        else {
            ld->cBad.cSpare++;
        }
        if ((tlast=VALCH(bads->cyl)) == lastskip)
            continue;
        lastskip = tlast;
        if (ld->type==LD_SWAP || ld->type==LD_USERSKIP) {
            ld->cBad.cTrack++;
        }
    }
}

printapplysetup(setup, rsvtoo)
struct setup *setup;
{
    register int i;
    struct confld *ld;
    struct cbad *cb;

    cb = &setup->cDiskBad;
    printf("    %d sectors available\n", setup->cAvail);
    printf("%d tracks skipped, %d holes, %d spares, %d skips\n",
        cb->cTrack, cb->cHole, cb->cSpare, cb->cSkip);
    printf(" ld type st    size   clean   start(CCC.HH)  skip trak spare holes\n");
    if (rsvtoo)
        printapplyld(setup, -1, &setup->confRsv);
    for (i = 0, ld = setup->confld; i < LOGDR; i++,ld++)
        if (ld->type)
            printapplyld(setup, i, ld);
    checkapplysetup(setup, warn, 0);
}

printapplyld(setup, i, ld) /* ADD VERBOSE FLAG */
struct setup *setup;
struct confld *ld;
{
    register int c, h;

    printf(" %2d %4s ", i, ascldtype[ld->type]);
    if (ld->flag & CFLD_PROTECT)
        printf(" P");
    else if (ld->flag & CFLD_FREEZE)
        printf(" F");
    else
        printf("  ");
    printf(" %7d  %6d", ld->cAssignSect, ld->cSpareLim);

    h = ld->start / PD0(setup)->sechead;
    c = h / PD0(setup)->headcyl;
    h = h % PD0(setup)->headcyl;
    printf(" %7d(%03x.%02x)", ld->start, c, h);
    printf(" %4d %4d  %4d  %4d\n",
            ld->cBad.cSkip, ld->cBad.cTrack, ld->cBad.cSpare, ld->cBad.cHole);
}

printbad(setup)
struct setup *setup;
{
    struct physbadsector *bads;
    struct confld *ld;
    int count, i, hh, ss;
    unsigned ldendchs;

    if (!setup->memBad)
        software_bug("printbad",
                "Should be bad blocks if physical disk selected");
    printf("%d BAD SECTORS\n", setup->cMemBad);
    bads = (struct physbadsector *)(setup->memBad->p);
    ld = &setup->confRsv;
    hh = PD0(setup)->headcyl;
    ss = PD0(setup)->sechead;
    ldendchs = 0;
    for (count=i=0; count<setup->cMemBad; count++,bads++) {
        if (bads->flag & BS_UNSPARE)
            continue;
        while (VALCHS(bads->cyl) >= ldendchs) {
            if (ld->start) {
                i = ld->start + ld->cAssignSect + ld->cBad.cSkip;
            }
            else {
                i = ld->cSect + (ldendchs & 0xff);
                i += ((ldendchs >> 8) & 0xff) * ss;
                i += (ldendchs >> 16) * ss * hh;
            }
            ldendchs = i % ss;
            i = i / ss;
            ldendchs += (i % hh) << 8;
            ldendchs += (i / hh) << 16;
            if (VALCHS(bads->cyl) < ldendchs)
                printf((ld == &setup->confRsv) ? "\nreserved area" :
                        "\nlogical disk %d", ld - setup->confld);
            ld++;
            i = 4;
        }
        if ( !(i++ & 3) )
            printf("\n     ");
        printf("($%x.%02x.%02x)[%d],",
                bads->cyl, bads->head, bads->sector, bads->index);
    }
    printf("\n");
}

printspare(setup)
struct setup *setup;
{
    struct sparedata *spare;
    int count, i;
    int used = 0;
    int bad = 0;
    int avail = 0;

    if (!setup->memSpare) {
        printf("spare sectors have not been calculated\n");
        return;
    }
    spare = (struct sparedata *)(setup->memSpare->p);
    printf("\n  %d ALTERNATE SECTORS\n\t",setup->cMemSpare);
    for (count = i = 0; count < setup->cMemSpare; count++, spare++) {
        if (VALCHS(spare->cyl) == -1)
            avail++;
        else {
            if (VALCHS(spare->cyl) == 0)
                bad++;
            used++;
            if (i++ == 3) {
                printf("\n     ");
                i = 1;
            }
            printf("$%x.%02x.%02x:$%x.%02x.%02x, ",
                    spare->cyl, spare->head, spare->sector,
                    spare->altcyl, spare->althead, spare->altsector);
        }
    }
    printf("\n%d alternates used, %d alternates bad, %d alternates avail\n",
        used, bad, avail);
    if (used + avail != count)
        warn("OUCH!!, alternate count != totals");
    printf("\n");
}

printskip(setup)
struct setup *setup;
{
    struct badtrck *skip;
    int count, i;

    if (!setup->memSkip) {
        printf("skip tracks have not been calculated\n");
        return;
    }
    skip = (struct badtrck *)(setup->memSkip->p);
    i = setup->cMemSkip-(LOGDR+1);
    printf("  %d SKIP TRACK elements,", i>0 ? i : 0);
    for (count=i=0;count<LOGDR+1;count++,skip++) {
        if (VALCHS(skip->cyl)) {
            printf(count==LOGDR ? "\nreserved area" : "\n LDisk %d:",count);
            i = 7;
        }
        while (VALCHS(skip->cyl)) {
            if (i++ == 7) {
                printf("\n     ");
                i = 1;
            }
            printf("$%x.%02x, ",skip->cyl, skip->head);
            skip++;
        }
    }
    printf("\n");
}

checkapplysetup(setup,reportproc,escapebuf)
register struct setup *setup;
jmp_buf escapebuf;
int (*reportproc)();
{
    int i;
    register struct confld *ld;
    register int check, allok;
    int tsum;
    int countSWAP = 0;
    int countPWF = 0;

    if ((setup->flag & SET_FIG) == 0)
        return;
    /* check out reserved first */
    ld = &setup->confRsv;
    tsum = PD0(setup)->sechead * 3;
    if (ld->cAssignSect < tsum) {
        reperr(reportproc,"reserved are size too small, at least %d",tsum);
        allok = 0;
    }
    if ( ld->cAssignSect > MAX_SIZE_RV) {
        reperr(reportproc, "reserved size too big, %d max", MAX_SIZE_RV);
        allok = 0;
    }
    check = ld->cAssignSect + ld->cBad.cSkip;
    allok = 1;
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++) {
        if (ld->type) {
            int tlim;

            if (ld->type == LD_SWAP)
                countSWAP++;
            else if (ld->type == LD_PWF)
                countPWF++;
            if (ld->cAssignSect < MIN_LD_SIZE) {
                reperr(reportproc, "check: ld %d, bad size %d",i,ld->cAssignSect);
                allok = 0;
            }
            if (ld->cAssignSect % PD0(setup)->sechead) {
                reperr(reportproc, "check: ld %d, bad size modulo %d",i,ld->cSect);
                allok = 0;
            }
            if (ld->cSpareLim > ld->cAssignSect) {
                reperr(reportproc, "check: ld %d, impossible clean (inode) size %d",
                        i, ld->cSpareLim);
                allok = 0;
            }
            switch (ld->type) {
                case LD_SWAP:
                case LD_USERSKIP:
                    if (ld->start != check)
                        reperr(reportproc,"ld %d starting address inconsistent",i);
                    break;
                default:
                    if (ld->start + ld->cBad.cTrack * PD0(setup)->sechead - ld->cBad.cSkip
                            != check)
                        reperr(reportproc, "ld %d starting address inconsistent", i);
                    break;
            }
            tlim = calcspare(ld, setup);
            if (!(ld->flag & CFLD_PROTECT))
                switch(ld->type) {
                    case LD_USERSPLIT:
                    case LD_UNIXFS:
                        if (ld->cSpareLim != tlim)
                            warn("non standard clean/inode bounds on logical disk %d",i);
                        break;
                    default:
                        if (tlim != ld->cSpareLim) {
                            allok = 0;
                            reperr(reportproc,
                                    "ld type %s cannot have modified clean/inode bounds",
                                    ascldtype[ld->type]);
                        }
                        break;
                }
            check += ld->cAssignSect + ld->cBad.cSkip;
            if (ld->cAssignSect != ld->cSect)
                warn("check: ld %d, size changed from %d to %d",
                        i, ld->cSect, ld->cAssignSect);
        }
    }

    if (countSWAP > 1) {
        allok = 0;
        reperr(reportproc,"at most one SWAP area per disk");
    }
    if (countPWF > 1) {
        allok = 0;
        reperr(reportproc,"at most one PWRF area per disk");
    }
    if (setup->cTotalSect != check) {
        allok = 0;
        reperr(reportproc,"total sectors (%d) != assigned sectors (%d)",
                setup->cTotalSect,check);
        if (check > setup->cTotalSect)
            errxit("more sectors allocatd than available");
    }
    if (setup->cDiskBad.cTrack > 1000) {         /* > HSDT table space? */
        allok = 0;
        reperr(reportproc, "%d skip tracks--too many for controller",
                setup->cDiskBad.cTrack);
    }
    else if (setup->cDiskBad.cTrack > 250)
        warn("skip track list may be too long for controller\n");
    if ((!allok) && escapebuf)
        longjmp(escapebuf, 1);
    return(allok);
}

pr_sp1(setup)
struct setup *setup;
{
    int count = setup->cMemSpare;
    struct sparedata *spare1 = (struct sparedata *)setup->memSpare->p;

    for (; count > 0; count--) {
        printf("%4x   %4x %2x %2x   %4x %2x %2x\n",
                spare1->flag, spare1->cyl, spare1->head, spare1->sector,
                spare1->altcyl, spare1->althead, spare1->altsector );
        spare1++;
    }
}

pr_sp2(setup)
struct setup *setup;
{
    int count = setup->cMemSpare;
    struct sprs *spare2 = (struct sprs *)setup->memSpare->p;

    for (;count > 0; count--) {
        printf("       %4x %2x %2x   %4x %2x %2x\n",
                spare2->cyl, spare2->head, spare2->sector,
                spare2->altcyl, spare2->althead, spare2->altsector );
        spare2++;
    }
}

struct logtype *ownld(setup, chs)
register struct setup *setup;
{
    int ld, bno;

    bno = chs_to_ldb(setup, chs);
    ld = (bno >> 24) & 0x1f;
    bno = (bno < 0) ? -(bno & 0xffffff) : (bno & 0xffffff);
    printf("$%x(chs) is block %d(decimal) of ", chs, bno);
    if (bno < 0)
        printf(" (spare space) ");
    if (ld == (LOGDR+1)) {
        ld = -1;
        printf("reserved area\n");
    }
    else
        printf("LDisk %d\n", ld);
    return( &PD0(setup)->logdrive[ld] );
}

trackbad(setup, bads)
struct setup *setup;
register struct physbadsector *bads;
{
    int i, s;

    if (bads->sector != 0)
        return(0);
    i = setup->cMemBad - (bads - (struct physbadsector *)setup->memBad->p);
    if (i < PD0(setup)->sechead)
        return(0);
    s = VALCHS(bads->cyl);
    for (i=PD0(setup)->sechead; i--; bads++)
        if (VALCHS(bads->cyl) != s++ || (bads->flag & BS_UNSPARE))
                return(0);
    return(1);
}

/*----------------------------- End of dset2.c -----------------------------*/
