/*----------------------------------------------------------------------------
/ dsetsmd.c - smd controller functions
/
/ April 1988    Craig J. Kim
/
/ NOTE:  dsetup makes use of setjmp(), longjmp() pair to recover from serious
/   error conditions.  setjmp() is executed once at the top of dsetup()
/   function and longjmp() is called from many error handling functions.
/---------------------------------------------------------------------------*/

#include "disksect.h"

#if defined(UNIXSTANDALONE) | defined(RO_UNIX)
#include <stdio.h>
#endif

#include "dsetup.h"
#include "diskconfig.h"

#ifndef TRUE
#define TRUE    1
#define FALSE   0
#endif

extern warn();
extern char *strlwr();                  /* library function */

/*---------------------------------------------------- forwards ------------*/
int smd_dsetup();
static void gbno();
static void setupselectclear();
static void setupapplyclear();
static void setupsave();
static void a_cmd();
static void b_c_cmd();
static void b_s_cmd();
static void b_u_cmd();
static void b_r_cmd();
static void b_w_cmd();
static void c_cmd();
static void d_cmd();
#if 0
static void i_cmd();
#endif
static void l_b_cmd();
static void l_n_cmd();
static void l_o_cmd();
static void l_p_cmd();
static void l_t_cmd();
static void l_c_cmd();
static void m_c_cmd();
static void m_p_cmd();
static void m_r_cmd();
static void m_s_cmd();
static void m_t_cmd();
static void n_cmd();
int q_cmd();
static void r_cmd();
static void s_cmd();
void u_cmd();
static void w_cmd();
static void printhelp();

/*---------------------------------------------------- globals -------------*/
extern jmp_buf errorbuf;                /* so we can bail us out */
extern struct setup oldsetup;           /* as found on disk */
extern struct setup prevsetup;          /* undo with this one */
extern struct setup newsetup;           /* working copy */
extern struct setup tempsetup;          /* newsetup just before any changes */

extern int debugflag;                   /* can toggle with z command */

extern char cmdbuf[200];                /* i hope there is a limit */
extern char *pcmd;                      /* command pointer */

extern int warn_level_flag;
extern int scsi_dtc;                    /* scsi type controller */

extern struct conftable gConf[];
extern char *ascldtype[];
extern char *brief_ascldtype[];


/*---------------------------------------------------- smd_dsetup() --------*/
int smd_dsetup(drv)
int drv;
{
    char cmdchar1[10], cmdchar2[10];
    int cmdint1, cmdint2, cmdint3;
    char *p;
    int help_request = FALSE;

    while (1) {
        if (setjmp(errorbuf)) {         /* the error subr comes here */
            xprintf("errorjump ");
            if (tempsetup.flag & SET_USED) {
                printf("No change was made.\n");
                if (debugflag)
                    l_n_cmd();
                if (tempsetup.flag & SET_FIG)
                    setupapplyclear(&tempsetup);
                if (newsetup.flag & SET_FIG)
                    setupapplyclear(&newsetup);
                newsetup = tempsetup;
                tempsetup.flag = 0;
            }
            drv = -1;
        }

        if (drv >= 0) {                 /* first time into loop */
            s_cmd(drv);
            drv = -1;
        }

        if (help_request) {             /* display help then accept a command */
            printhelp(cmdbuf);
            help_request = FALSE;
        }
        else {
            printf("dsetup-> ");        /* this is our prompt */
            gets(cmdbuf);               /* see what user says */
        }
        pcmd = strlwr(cmdbuf);          /* we'll need to parse */
        if (!gstr(cmdchar1))            /* didn't type anything */
            continue;

        switch (cmdchar1[0]) {          /* we'll look at only first char */
            case '?':                   /* (?) Help! */
                help_request = TRUE;
                break;
            case 'a':                   /* (a) apply a setup */
                a_cmd();
                break;
            case 'b':                   /* (b) one of the bad block commands */
                if (!gstr(cmdchar2)) {
                    l_b_cmd();
                    break;
                }                       /* get block addr */

                gbno(&cmdint1, &cmdint2, &cmdint3);
                switch (cmdchar2[0]) {  /* examine only first char */
                    case 'c':           /* (b c) convert 'chs' to 'bno' */
                        b_c_cmd(cmdint1, cmdint2, cmdint3);
                        break;
                    case 's':           /* (b s) spare a block */
                        b_s_cmd(cmdint1, cmdint2, cmdint3);
                        break;
                    case 'u':           /* (b u) unspare a block */
                        b_u_cmd(cmdint1, cmdint2, cmdint3);
                        break;
                    case 'r':           /* (b r) read a block */
                        b_r_cmd(cmdint1,cmdint2,cmdint3);
                        break;
                    case 'w':           /* (b w) write a block */
                        b_w_cmd(cmdint1,cmdint2,cmdint3);
                        break;
                    default:            /* huh? */
                        goto parserr;
                }
                break;
            case 'c':                   /* (c) pick a configuration */
                if (!gnum(&cmdint1))
                    cmdint1 = -1;
                c_cmd(cmdint1);
                break;
            case 'd':                   /* (d) delete a logical disk */
                if (!gnum(&cmdint1))
                    goto parserr;
                if (!gnum(&cmdint2))
                    cmdint2 = cmdint1 - 1;
                d_cmd(cmdint1, cmdint2);
                break;
#if 0
            case 'i':                   /* (i) make an empty entry at i */
                if (!gnum(&cmdint1))
                    goto parserr;
                i_cmd(cmdint1);
                break;
#endif
            case 'l':                   /* (l) one of a variety of list commands */
                if (!gstr(cmdchar2))
                    cmdchar2[0] = 'n';  /* default to list new */
                switch (cmdchar2[0]) {
                    case 'b':           /* (l b) list bad blocks */
                        l_b_cmd();
                        break;
                    case 'c':           /* (l c) list predefined configurations */
                        l_c_cmd();
                        break;
                    case 'n':           /* (l n) list new configuration */
                        l_n_cmd();
                        break;
                    case 'o':           /* (l o) list old configuration */
                        l_o_cmd();
                        break;
                    case 'p':           /* (l p) list physical property */
                        l_p_cmd();
                        break;
                    case 't':           /* (l t) list possible logical disk types */
                        l_t_cmd();
                        break;
                    default:            /* huh? */
                        goto parserr;
                }
                break;
            case 'm':                   /* (m) one of a variety of modify commands */
                if (!gstr(cmdchar2))    /* nothing follows 'm' */
                    goto parserr;
                switch (cmdchar2[0]) {
                    case 'c':           /* (m c) modify size of clean/inode area */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            goto parserr;
                        m_c_cmd(cmdint1, cmdint2);
                        break;
                    case 'p':           /* (m p) reset the protection */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        m_p_cmd(cmdint1);
                        break;
                    case 'r':           /* (m r) modify reserved area size */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            cmdint2 = -1;
                        m_r_cmd(cmdint1, cmdint2);
                        break;
                    case 's':           /* (m s) modify size */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            goto parserr;
                        /* the following not quite right, non numeric garbage
                            will look like a nonexistent thing, sigh */
                        if (!gnum(&cmdint3))
                            cmdint3 = -1;
                        m_s_cmd(cmdint1,cmdint2,cmdint3);
                        break;
                    case 't':           /* (m t) modify type */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            goto parserr;
                        m_t_cmd(cmdint1, cmdint2);
                        break;
                    default:            /* huh? */
                        goto parserr;
                }
                break;
            case 'n':                   /* (n) select config 0 */
                n_cmd(0);
                break;
            case 'q':                   /* (q) quit */
                if (q_cmd())
                    return(-1);
                break;
            case 'r':                   /* (r) remove a logical disk */
                if (!gnum(&cmdint1))
                    goto parserr;
                if (!gnum(&cmdint2))
                    cmdint2 = cmdint1-1;
                r_cmd(cmdint1,cmdint2);
                break;
            case 's':                   /* (s) select physical disk */
                if (!gdrive(&cmdint1))
                    goto parserr;
                if (check_drive(cmdint1))
                    return(cmdint1);
                break;
            case 'u':                   /* (u) undo */
                u_cmd();
                break;
            case 'v':                   /* (v) print current version */
                pversion();
                printf("\n");
                break;
            case 'w':                   /* (w) calculate and write it out */
                w_cmd();
                break;
            case 'x':                   /* (x) split a disk, spread the table out */
                if (!gnum(&cmdint1))
                    goto parserr;
                if (!gnum(&cmdint2))
                    goto parserr;
                m_s_cmd(cmdint1,cmdint2,cmdint1);
                break;
            case 'z':                   /* (z) hidden debug command */
                debugflag ^= 1;
                printf("Debug %s\n", debugflag ? "on" : "off");
                break;
            default:                    /* huh? */
                goto parserr;
        }
        if (tempsetup.flag & SET_USED)
            prevsetup = tempsetup;
        tempsetup.flag = 0;
        continue;
parserr:
        printf("Unrecognized command.  Type '?' for help.\n\n");
    }
}

/*---------------------------------------------------- gbno() ----------------
/ get a block address.  A block address can be one of two formats:
/
/       CCCHHSS         in hexadecimal
/                       where CCC = cylinder, HH = head, SS = sector
/ or
/
/       ldisk:[[0]x]block  in decimal
/                       where ldisk = logical disk, block = block offset
/                           if starts with an '0x', hexadecimal, else decimal
/---------------------------------------------------------------------------*/
static void gbno(pchs, pld, pbn)
int *pchs, *pld, *pbn;
{
    int chs, ld, bn;
    char *pp = pcmd;

    while (*pp && *pp != ':')   /* skip to ':' */
            pp++;
    if (*pp != ':'){              /* didn't find a colon */
        ghex(&chs);                  /* try 'CCCHHSS' format first */
        bn = chs_to_ldb(&newsetup, chs);
        ld = (bn >> 24) & 0x1f;
        if (ld == LOGDR + 1)
            ld = -1;                    /* reserved area */
        bn = (bn < 0) ? -(bn & 0xffffff) : bn & 0xffffff;
    }
    else {                              /* must be in 'ldisk:block' format */
	pp = pcmd;
        ld = atoi(pcmd);                /* ldisk */
        while (*pcmd && *pcmd != ':')   /* skip to ':' */
            pcmd++;
        if (*pcmd != ':')               /* didn't find a colon */
            ld = 99;                    /* make it fail */
        pcmd++;                         /* skip over the colon */
        if (*pcmd == 'x')               /* get block number */
            bn = ghex(++pcmd);          /* in hex - x##### format */
        else if (pcmd[0] == '0' && pcmd[1] == 'x')
            bn = ghex(pcmd += 2);       /* in hex - 0x#### format */
        else
            bn = atoi(pcmd);            /* in decimal */
        while (*pcmd)                   /* exhaust input string */
            pcmd++;
                                        /* now convert to 'chs' */
        chs = ldb_to_chs(&newsetup, ld, bn);
        if (chs == -1)                  /* hmm... didn't make it */
            error("%s: invalid block number specification\n", pp);
    }

    if (pchs)                           /* pass back the results */
        *pchs = chs;                    /* 'chs' */
    if (pld)
        *pld = ld;                      /* ldisk */
    if (pbn)
        *pbn = bn;                      /* block addr */
}

/*---------------------------------------------------- setupselectclear() --*/
static void setupselectclear(setup)
struct setup *setup;
{
    if (setup->fd)                      /* close related file */
        close(setup->fd);
    if (setup->memPhysdisk)             /* free any used memory */
        memfree(setup->memPhysdisk);
    setup->cMemBad = 0;                 /* this will make setupapplyclear()
                                           call faster as it doesn't have to
                                           act foolish - no need to try to
                                           manipulate bad block lists */
    if (setup->flag & SET_FIG)          /* has it been changed in anyway */
        setupapplyclear(setup);

    if (setup->memBad)                  /* free bad block list */
        memfree(setup->memBad);
    memset(setup, 0, sizeof (*setup));  /* clean up the structure */
}

/*---------------------------------------------------- setupapplyclear() ---*/
static void setupapplyclear(s)
struct setup *s;
{
    register struct physbadsector *bads;
    register int cbads = s->cMemBad;
    register struct confld *ld;

    bads = (struct physbadsector *) (s->memBad->p);
    if (s->memSpare)                    /* free used memory */
        memfree(s->memSpare);
    /*NOWARN: else warn("no spares to clear"); */

    if (s->memSkip)
        memfree(s->memSkip);
    /*NOWARN: else warn("no skips to clear"); */

    s->memSpare = s->memSkip = 0;
    s->cMemSkip = s->cMemSpare = 0;

    s->flag &= ~(SET_FIG | SET_WRIT);   /* clear apply and write flags */

    if (bads)                           /* must clean up from a figued command */
        for (; cbads > 0; cbads--, bads++) {
            if (!(bads->flag & BS_PROTECT)) {
                bads->flag &= ~(BS_HOLE | BS_SPARE);
                bads->index = -1;
            }
        }
    else
        software_bug("setupapplyclear", "No bad sectors to clear");

    /* clear the computed values */
    memset(&s->cDiskBad, 0, sizeof (struct cbad));

    for (ld = s->confld; ld < &s->confld[LOGDR]; ld++)
        memset(&ld->cBad, 0, sizeof (struct cbad));

    /* copy the assigned values to the requested valued????? */
}

/*---------------------------------------------------- setupsave() ---------*/
static void setupsave(d, s)
struct setup *d, *s;
{
    if (!(s->flag & SET_USED))
        error("no disk is selected");

    /*
    if (s->memSkip || s->memSpare) setupapplyclear(s);
     */

    *d = *s;                            /* copy structure here */
    d->flag |= SET_USED;
    if (!(warn_level_flag++)) {
        struct sector0 *pd;

        pd = PD0(s);
        if (pd->DSETUP_LEVEL < LEVEL_1_DSETUP)
            warn("the disk tables are at old version level, compensating");
    }
}

/*---------------------------------------------------- a_cmd() ---------------
/ apply new configuration
/---------------------------------------------------------------------------*/
static void a_cmd()
{
    setupsave(&tempsetup, &newsetup);   /* copy newsetup to tempsetup */
    applyconfig(&newsetup);             /* let newsetup be the current */
    newsetup.flag |= SET_MOD;           /* set modify bit */
}

/*---------------------------------------------------- b_c_cmd() -------------
/ convert between chs and disk/block-number
/---------------------------------------------------------------------------*/
static void b_c_cmd(chs, ld, bno)
{
    extern char *ordinal();             /* library function */

    printf("$%x(chs) is %d%s block of ", chs, bno, ordinal(bno));
    printf(ld == -1 ? "reserved area.\n" : "logical disk %d.\n", ld);
}

/*---------------------------------------------------- b_s_cmd() -------------
/ spare a bad block
/---------------------------------------------------------------------------*/
static void b_s_cmd(chs, ld, bno)
{
    prevsetup.flag = 0;                 /* there is no undo */
    addsbad(&newsetup, chs);            /* add a bad block */
    applyconfig(&newsetup);             /* apply it */
    newsetup.flag |= SET_MOD;
}

/*---------------------------------------------------- b_u_cmd() -------------
/ unspare a spared block
/---------------------------------------------------------------------------*/
static void b_u_cmd(chs)
{
    prevsetup.flag = 0;                 /* there is no undo */
    remsbad(&newsetup, chs);            /* remove block from bad list */
    applyconfig(&newsetup);             /* go do it */
    newsetup.flag |= SET_MOD;
}

/*---------------------------------------------------- b_r_cmd() -------------
/ read a block by 'chs'
/---------------------------------------------------------------------------*/
static void b_r_cmd(chs)
{
    ldread(&oldsetup, chs);             /* read a block using old setup */
}

/*---------------------------------------------------- b_w_cmd() -------------
/ write a block by 'chs'
/---------------------------------------------------------------------------*/
static void b_w_cmd(chs)
{
    ldwrite(&oldsetup, chs);            /* write a block using old setup */
}

/*---------------------------------------------------- c_cmd() -------------*/
static void c_cmd(i)
{
    register struct sector0 *pd;        /* misspelling this causes compiler dump */

    pd = 0;
    if (i == -1) {
        if (IS_SYS0(&newsetup)) {
            if (newsetup.cTotalSect < SMALL_LIMIT)
                i = CONF_SMALL0;
            else if (newsetup.cTotalSect < MEDIUM_LIMIT)
                i = CONF_MED0;
            else
                i = CONF_LARGE0;
        }
        else {
            if (newsetup.cTotalSect < SMALL_LIMIT)
                i = CONF_SMALL;
            else if (newsetup.cTotalSect < MEDIUM_LIMIT)
                i = CONF_MED;
            else
                i = CONF_LARGE;
        }
        printf("Choosing configuration %d\n", i);
    }
    if (i < 0 || i >= NUM_DISKCONFS)
        error("%d is not an available configuration", i);
    setupsave(&tempsetup, &newsetup);

    if (i == 0 && (newsetup.flag & SET_OLD)) {
        pd = PD0(&newsetup);
        pd->pd_ldnum = 0;               /* force it to be treated as new */
        printf("New configuration, if BOOTIMAGE desired, must re-run ldsa.\n");
        warn("Logical drive data will no longer be accessable");
        confirm();
        newsetup.confRsv.flag = 0;
        newsetup.flag = SET_USED;
        newsetup.cSpareGuess =          /* this is somewhat bogus, sizes can change */
                roundtrack(&newsetup,newsetup.cTotalSect / FRACTIONBAD);
    }

    if (newsetup.flag & (SET_OLD | SET_MOD)) {
        warn("Logical drive data will no longer be accessable");
        confirm();
        setupapplyclear(&newsetup);

        pd = PD0(&newsetup);            /* set pointer, non zero is flag */
    }

    if (!configit(&newsetup, gConf[i].pconf))
        error("selected configuration doesn't fit on disk");
    newsetup.flag |= SET_MOD | SET_NEW;

    /*-----------------------------------------------------------------
    / the following because of a earlier bug in the stored format of
    / the spare table.  By turning off SET_OLD the old table is not
    / read.  Since the table is not read the reserved area must be
    / unprotected so that all the sectors will be accounted for.
    /----------------------------------------------------------------*/

    if (pd && pd->DSETUP_LEVEL < LEVEL_1_DSETUP) {
        newsetup.confRsv.flag &= ~CFLD_PROTECT;
        newsetup.flag &= ~ SET_OLD;
        warn("the disk tables are at old version level, compensating");
    }
    applyconfig(&newsetup);
}

/*---------------------------------------------------- d_cmd() -------------*/
static void d_cmd(p1,p2)
{
    m_s_cmd(p1, 0, p2);
}

#if 0
/*---------------------------------------------------- i_cmd() -------------*/
static void i_cmd(p1)
{
    setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    moveupld(&newsetup,p1);
}
#endif

/*---------------------------------------------------- l_b_cmd() -------------
/ list bad blocks
/---------------------------------------------------------------------------*/
static void l_b_cmd()
{
    printbad(&newsetup);
    printspare(&newsetup);
    printskip(&newsetup);
}

/*---------------------------------------------------- l_c_cmd() -------------
/ list predefined configurations to be used with 'c' command
/---------------------------------------------------------------------------*/
static void l_c_cmd()
{
    register int i;
    register int count;
    struct defconf *conf;
    int *p;

    printf("id  name                  detail\n\n");
    for (i = 0; i < NUM_DISKCONFS; i++) {
        printf("%2d  %20s", i, gConf[i].name);
        conf = gConf[i].pconf;
        printf("  RESERVED:%d    leftover disks { %s:%d }*\n     ",
                conf->cRsv, brief_ascldtype[conf->normaltype], conf->normal);
        for (count = 0, p = conf->defined;
                count < conf->definedcount;
                count++, p += 2)
            printf(" %s:%d", brief_ascldtype[p[1]], *p);
        printf("\n");
    }
}

/*---------------------------------------------------- l_n_cmd() -----------*/
static void l_n_cmd()
{
    if (!(newsetup.flag & SET_USED)) {
        warn("there is nothing to list");
        return;
    }
    printapplysetup(&newsetup, debugflag);
}

/*---------------------------------------------------- l_o_cmd() -----------*/
static void l_o_cmd()
{
    if (!(oldsetup.flag & SET_USED)) {
        warn("there is nothing to list");
        return;
    }
    printapplysetup(&oldsetup);
}

/*---------------------------------------------------- l_p_cmd() -----------*/
static void l_p_cmd()
{
    if (!(newsetup.flag & SET_USED)) {
        warn("there is nothing to list");
        return;
    }
    printphysdisk(&newsetup);
}

/*---------------------------------------------------- l_t_cmd() -------------
/ list possible logical disk types to used with 'm t' command
/---------------------------------------------------------------------------*/
static void l_t_cmd()
{
    register int i;
    register char **p = &ascldtype[1];

    for (i = 1; i < NUM_DISKTYPES; i++, p++)
        printf("   %3d  %s\n", i, *p);
}

/*---------------------------------------------------- m_c_cmd() -------------
/ (m c) modify clean/inode size
/---------------------------------------------------------------------------*/
static void m_c_cmd(p1, p2)
{
    setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected", p1);
    setinodesize(&newsetup,p1,p2);
    newsetup.flag |= SET_MOD;
    applyconfig(&newsetup);
}

/*---------------------------------------------------- m_p_cmd() -----------*/
static void m_p_cmd(p1)
{
    struct confld *ld;

    setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    ld = &newsetup.confld[p1];
    if (!(ld->flag & CFLD_PROTECT)){
        printf("Logical disk %d is not protected\n", p1);
	return;
    }
    ld->flag &= ~CFLD_PROTECT;
    ld->cSpareLim = calcspare(ld, &newsetup);
    checkusersetup(&newsetup,warn,errorbuf);
    newsetup.flag |= SET_MOD;
    applyconfig(&newsetup);
}

/*---------------------------------------------------- m_r_cmd() -----------*/
static void m_r_cmd(p1, p2)
{
    setupsave(&tempsetup, &newsetup);
    if (newsetup.confld[-1].flag & CFLD_PROTECT)
        error("Reserved area is protected");
    setldsize(&newsetup, RV_LOGDR, p1, p2);
    newsetup.flag |= SET_MOD;
    applyconfig(&newsetup);
}

/*---------------------------------------------------- m_s_cmd() -----------*/
static void m_s_cmd(p1,p2,p3)
{
    setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected", p1);

    setldsize(&newsetup, p1, p2, p3);
    newsetup.flag |= SET_MOD;
    applyconfig(&newsetup);
}

/*---------------------------------------------------- m_t_cmd() -----------*/
static void m_t_cmd(p1, p2)
{
    setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected", p1);
    if (p2 < 1 || p2 > NUM_DISKTYPES)
        error("Unknown logical disk type %d", p2);
    modifyldtype(&newsetup, p1, p2);
    newsetup.flag |= SET_MOD;
    applyconfig(&newsetup);
}

/*---------------------------------------------------- n_cmd() ---------------
/ (n) select configuration #0
/---------------------------------------------------------------------------*/
static void n_cmd()
{
    c_cmd(0);
}

/*---------------------------------------------------- q_cmd() -------------*/
int q_cmd()
{
#ifndef RO_UNIX
    if (!(newsetup.flag & SET_WRIT) && (newsetup.flag & SET_MOD)) {
        warn("setup not written, really exit?");
        confirm();
    }
#endif
    return(TRUE);
}

/*---------------------------------------------------- r_cmd() -------------*/
static void r_cmd(p1,p2)
{
    setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected", p1);
    setldsize(&newsetup,p1,0,p2);
    removeld(&newsetup,p1);
    newsetup.flag |= SET_MOD;
    applyconfig(&newsetup);
}

/*---------------------------------------------------- s_cmd() ---------------
/ select a disk
/---------------------------------------------------------------------------*/
static void s_cmd(pd)
int pd;                 /* disk number in binary "ccccdddd" format */
{
    struct icbdev dev;

    dev.dev_num = pd;                   /* save physical unit number */
    if (newsetup.flag & SET_MOD) {
        warn("Configuration has not been written to selected disk");
        confirm();
    }
    setupselectclear(&newsetup);        /* clear working area */
    setdisk(&newsetup, &dev);           /* go fill working area with current */
    if (!(newsetup.flag & SET_NEW)) {   /* is it new? */
        oldsetup = newsetup;
        getsslist(&newsetup);           /* go get spare and skip lists */
    }

    printf("SMD Controller %d, drive %d selected.\n", pd >> 4, pd & 0x0f);
    warn_level_flag = 0;
}

/*---------------------------------------------------- u_cmd() -------------*/
void u_cmd()
{
    if (tempsetup.flag & SET_USED)
        software_bug("u_cmd",
                "'tempsetup' should not be valid when undo given");

    if (newsetup.flag & SET_WRIT) {
        warn("cannot undo after write");
        return;
    }

    if (prevsetup.flag & SET_USED) {
        newsetup = prevsetup;
        prevsetup.flag = 0;
    }
    else
        warn("nothing to undo");
}

/*---------------------------------------------------- w_cmd() ---------------
/ write the setup to disk (Caution: no undo - permanent)
/---------------------------------------------------------------------------*/
static void w_cmd()
{
    struct icbdev dev;

    dev = newsetup.dev;                 /* save device number */
    prevsetup.flag = 0;                 /* there is no undo */
#ifdef RO_UNIX
    error("dsetup is in READ ONLY mode, cannot write from OS");
#else
    if (!(newsetup.flag & SET_FIG))
        applyconfig(&newsetup);
    writeconfig(&newsetup);             /* write to disk (permanent) */
    setupselectclear(&newsetup);        /* clear structure */
    s_cmd(dev.dev_num);                 /* start over */
#endif      /* RO_UNIX */
}

/*---------------------------------------------------- printhelp() ---------*/
static void printhelp(cmd)
char *cmd;
{
    static char *helpmsg[] = {
    "SELECT COMMANDS ----------------------------------------------------",
    "  s <#>            Select physical drive",
    "  c [#]            Select logical disk configuration",
    "BAD BLOCK COMMANDS -------------------------------------------------",
    "  b c <block#>     Convert between CCCHHSS <-> ldisk:block-number",
    "  b r <block#>     Read a block",
    "  b s <block#>     Spare a block",
    "  b u <block#>     Unspare a block",
    "  b w <block#>     Write a block",
    "  Note:  <block#> is CCCHHSS(hex) | ldisk:block(decimal)",
    "LIST COMMANDS ------------------------------------------------------",
    "  [l] b            List bad blocks",
    "  l c              List predefined Configurations, use with 'c'",
    "  l [n]            List New configuration",
    "  l o              List Old configuration",
    "  l p              List physical property",
    "  l t              List possible logical disk Types, use with 'm t'",
    "MODIFY COMMANDS ----------------------------------------------------",
    "  m c <ld> <size>  Modify clean/inode size to <size>",
    "  m p <ld>         Modify/reset protection of <ld>",
    "  m s <ld1> <size> [ld2]  Mod Size of ld1, difference from ld2",
    "  m t <ld> <type>  Mod Type of ld1 to <type>",
    "*",
    "OTHER COMMANDS -----------------------------------------------------",
    "  x <ld> <size>    Split ld into two chunks, expand the ld table",
    "  d <ld1> [ld2]    Delete <ld1>, sectors to <ld2>",
#if 0
    "  i <ld1>          Make an empty table entry, move entries down",
#endif
    "  q                Quit",
    "  r <ld1> [ld2]    Remove <ld1>, as delete except collapse table",
    "  u                Undo the previous change",
    "  w                Write the setup to disk (no undo)",
    0 };
    register char **p, *cp;

    for (p = helpmsg; *p; p++)
        if (**p == '*') {               /* page break */
            printf("\nPress [Return] for next page or enter a command-> ");
            gets(cmd);
            printf("\n");
            cp = cmd;
            while (isspace(*cp))        /* skip leading spaces */
                cp++;
            if (*cp)                    /* user types in a command */
                return;
        }
        else
            printf("%s\n", *p);

    printf("\n");
}

/*----------------------------- End of dsetsmd.c ---------------------------*/
