/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)ip_main.c        1.18      07:44:55 - 89/09/18 "

/*
 * Internet Protocol
 *
 * Ip.c deals with the manupulation of the stream
 * ends for four queues that are associated with the IP
 * module. IP module is a multiplexing module at both
 * the sides of the stream. Its primary bottom end
 * interface is to an LLI interface. At the top end
 * it has a M_CTL interface defined by the structure
 * ip_if (see ip.h).
 */

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#include "sys/immu.h"
#include "sys/fs/s5dir.h"
#include "sys/region.h"
#ifndef i386
#include "sys/psw.h"
#include "sys/pcb.h"
#endif
#else
#include "sys/dir.h"
#include "sys/page.h"
#include "sys/seg.h"
#endif /* XENIX */
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/signal.h"
#include "sys/file.h"
#include "sys/errno.h"
#include "sys/user.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/var.h"
#include "sys/lihdr.h"
#include "sys/tiuser.h"

#include "sys/inet.h"
#include "sys/socket.h"		/* added by mohsen */
#include "sys/if.h"		/* added by mohsen */
#include "sys/in.h"		/* Pull in all the IPPROTO_XXX */
#include "sys/in_var.h"		/* added by mohsen */
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/ip_icmp.h"
#include "sys/icmp_var.h"
#include "sys/inetioctl.h"
#include "sys/route.h"

#ifndef DL_X25
#define DL_X25	9
#endif

int in_interfaces;	/* number of external internet interfaces */

ulong myinaddr;			/* tu_bind */
time_t bootime;
struct ifnet *ipbnext;
struct ipstat ipstat;
extern struct icmpstat icmpstat;
int	_trmask = 0x0;
extern struct in_ifaddr *get_inaddr();
extern struct in_ifaddr in_addr[];
extern int in_cnt;
#define satosin(sa)	((struct sockaddr_in *)(sa))
extern struct in_ifaddr *ifptoia();
extern struct ifnet *ip_getif();
extern void headflush(), ip_setname(), ip_input();

/********************************
*	Table of Contents	*
********************************/

void		ipinit();
int		ipopen();		/* Top Read Queue Open		*/
int		ipclose();		/* Top Read Queue Close		*/
struct ifnet	*ipmuxq();
void		ipbclose();
int		ipoput();		/* Top Write Queue Put		*/
int		iposrv();		/* Top Write Queue Service 	*/
int		ipid_to_prot();
int		ipsend();
int		ipbiput();		/* Bottom Read Queue Put	*/
int		ipbisrv();		/* Bottom Read Queue Service	*/
void		ipdrint();
void		ip_ctlmsg();
#ifndef XENIX
void		loadav();
#else
void		loadavg();
#endif
int		ipmuxnotify();
void		ip_doioctl();
ipfcmd		ip_sockopt();
mblk_t 		*ip_getmyaddr();
mblk_t 		*ip_gethisaddr();

/*********  End of TOC  ********/

static struct module_info 	ip_info = {	/* top QUEUE info */
	IP_ID, "ip", 0, INFPSZ, 65535, 65535 
};
static struct module_info 	ib_info = {	/* Bottom QUEUE info */
	IP_ID, "ip", 0, INFPSZ, 65535, 65535 
};
static struct qinit 		iprinit = { 	/* top Read QUEUE */
	NULL, NULL, ipopen, ipclose, NULL, &ip_info, NULL 
};
static struct qinit 		ipwinit = { 	/* top Write QUEUE */
	ipoput, iposrv, NULL, NULL, NULL, &ip_info, NULL 
};
static struct qinit 		ipbrinit = { 	/* Bottom Read QUEUE */
	ipbiput, ipbisrv, NULL, NULL, NULL, &ib_info, NULL 
};
static struct qinit 		ipbwinit = { 	/* Bottom Write QUEUE */
	putq, NULL, NULL, NULL, NULL, &ib_info, NULL 
};
struct streamtab 	ipinfo = {  	/* link to cdevsw table */
	&iprinit, &ipwinit, &ipbrinit, &ipbwinit 
};

/*
 * ip initialization
 */
void
ipinit()
{
	;
}

/*ARGSUSED*/
ipopen(q, dev, flag, sflag)
	register queue_t *q;
	dev_t dev;
{
	register struct ipd *idp;
	extern time_t time;
	register int mdev;

	ASSERT(q);

	mdev = minor(dev);
	if (sflag == CLONEOPEN) {
		/*  Look for first available minor device */
		for (idp = &ipd[0]; idp < &ipd[ipd_cnt]; idp++) {
			if (idp->id_rdq == (queue_t *)NULL)
				break;
		}
		/*  Set and test validity of selected minor device */
		if ((mdev = idp - &ipd[0]) >= ipd_cnt) {
			u.u_error = ENODEV;
			return OPENFAIL;
		}

	/* NOT a CLONEOPEN */

	} else if (idp = (struct ipd *)q->q_ptr) {
		/* Requested minor dev already open */
		return(mdev);
	} else {
		/*  Requested minor dev not already open */
		if (mdev < 0 || mdev >= ipd_cnt) {
			u.u_error = ENODEV;
			return OPENFAIL;
		}
		/* Allocate a minor device and continue */
		idp = &ipd[mdev];
	}

	/*
	 * For the first time around set up the time
	 */
	if (bootime == 0) {
		bootime = time;
		iptimeout();
		ip_rtinit_list();	/* initialize routing table */
	}

	/* initialize data structure */
	idp->id_rdq = q;
	idp->id_flags = 0;
	q->q_ptr = (caddr_t)idp;
	WR(q)->q_ptr = (caddr_t)idp;
	return mdev;
}

/*
 * if closing control channel, clean up mux
 */
ipclose(q)
	queue_t *q;
{
	register struct ipd *idp = (struct ipd *)q->q_ptr;
	extern void ipuntimeout();
	int s = splni();

	ASSERT(q && q->q_ptr);

	/*
	 * If closing control channel, clean up any
	 * remaining routes in the route table, and
	 * clear timer.
	 */
	if (idp == &ipd[0]) {
		rtflush();
		ipuntimeout();
		bootime = 0;
	}
	bzero((caddr_t)idp, sizeof(struct ipd));
	(void) splx(s);

	return (0);
}	/* End of ipclose()  */

/*
 * ipmuxq - from the cookie, find the corresponding q in ipb structures
 * return ifp if o.k. and NULL if not found
 */
struct ifnet *
ipmuxq(cookie)
{
	register struct ifnet *ifp;

	for (ifp = ipbnext; ifp ; ifp = ifp->if_next)
		if (ifp->ib_cookie == cookie)
			return(ifp);

	return(NULL);
}	/*  End of ipmuxq()  */

/*
 * these two externs need to be zeroed out so that when the network is
 * brought up and down with rebooting the route table entries don't get messed
 * up.
 */

/*
 * internal close of the bottom stream
 */
void
ipbclose(ifp)

	register struct ifnet *ifp;
{
	register queue_t *bq;
	register s;
	register struct ifnet **p;
	register struct in_ifaddr *ia = ifptoia(ifp);
	struct sockaddr_in netaddr;
	extern void free_inaddr();
	mblk_t *mp;

	STRLOG(IP_ID, IPBCHAN(ifp), DPRI_LO, SL_TRACE,
		 "ipbclose: For %x - if_flagss=%x", ifp, ifp->if_flags);

	bq = ifp->ib_wrq;
	mp = ifp->ib_iocblk;
	s = splstr();
	if (bq) {
		if (ifp->ib_mstate == DL_IDLE) {
			ifp->ib_mstate = DL_WACK_U; /* deadlock avoidance */
			(void) dl_unbind(bq);
		}
		flushq(bq, FLUSHALL);
		flushq(RD(bq), FLUSHALL);
	}

	/*
	 * Delete any route assoicated with this interface 
	 */
	bzero((caddr_t)&netaddr, sizeof (netaddr));
	netaddr.sin_family = AF_INET;
	if (ia->ia_flags & IFA_ROUTE) {
		if (ifp->if_flags & IFF_LOOPBACK) {
			ip_rtinit(&ia->ia_addr, &ia->ia_addr,
			    (int)IPIOC_DELETEROUTE, RTF_HOST, RTR_OTHER, 0);
		}
		else if (ifp->if_flags & IFF_POINTOPOINT) {
			ip_rtinit(&ia->ia_dstaddr, &ia->ia_addr,
			    (int)IPIOC_DELETEROUTE, RTF_HOST, RTR_OTHER, 0);
			netaddr.sin_addr.s_addr =
			    in_netof(satosin(&ia->ia_dstaddr)->sin_addr);

			/* Datakit VCS Only */
			if (ifp->if_flags & IFF_IPTOVCS) /* Delete Network */
				ip_rtinit((struct sockaddr *)&netaddr,
				    &ia->ia_addr, IPIOC_DELETEROUTE,
				    RTF_GATEWAY, RTR_OTHER, 0);
		} else { /* Broadcast and IPTOX25 */
			netaddr.sin_addr =
				in_makeaddr(ia->ia_subnet, INADDR_ANY);
			ip_rtinit((struct sockaddr *)&netaddr, &ia->ia_addr,
				IPIOC_DELETEROUTE, 0, RTR_OTHER, 0);
		}
		ia->ia_flags &= ~IFA_ROUTE;
	}
	/* remove ifp from the active chain "ipbnext" */
	for (p = &ipbnext; *p; p = &((*p)->if_next))
		if (*p == ifp) {
			*p = ifp->if_next;
			break;
		}
#ifdef OUTFORDATAKIT
	if (ifp->ib_qtop)
		(void) putctl1(RD(ifp->ib_qtop)->q_next, M_ERROR, EPROTO);
#endif
	free_inaddr(ia);
	bzero((char *)ifp, sizeof (struct ifnet));
	(void) splx(s);
	
	/* 
	 * Ack any pending I_UNLINK ioctl which is 
	 * waiting for the closing of this interface.
	 */
	if (mp) {
		struct iocblk *iop;
		struct linkblk *lp;
		
		iop = (struct iocblk *)mp->b_rptr;
		lp = (struct linkblk *)mp->b_cont->b_rptr;
		iop->ioc_error = 0;
		iop->ioc_count = 0;
		MTYPE(mp) = M_IOCACK;
		ifp->ib_iocblk = (mblk_t *)NULL;
		qreply(lp->l_qtop, mp);
	}
}	/*  End of ipbclose()  */

/*
 * Called when TCP, UDP or RAW put stuff to us
 */
ipoput(q, bp)
	register queue_t *q;
	register mblk_t *bp;
{
	register struct ipd *idp;
	register chan;
	register struct ip_if *mp;
	register int i;

	ASSERT(q && q->q_ptr && bp);

	idp = (struct ipd *)q->q_ptr;
	chan = IPCHAN(idp);

	/*
	 * Deal with the various message types.
	 */
	switch(MTYPE(bp)) {

	case M_IOCTL:
		ip_doioctl(q, bp);
		break;

	case M_CTL:
		/*
		 * tcp/udp/raw ... sends 'ip_if' to you
		 */
		mp = (struct ip_if *)bp->b_rptr;
			
		switch (mp->ipf_type) {

		case IPIF_LINKING:
			if (idp->id_mstate != IPIF_UNLINKED)
				goto badctl;
			/*
			 * from id, find the protocol # for multiplexing
			 */
			if ((i = ipid_to_prot((int)mp->ipf_id)) == -1)
				goto badctl;
			idp->id_prot = (unchar) i;
			idp->id_mstate = IPIF_LINKED;
			idp->id_topid = mp->ipf_id;
			mp->ipf_type = IPIF_LINKED;
Reply:
			mp->ipf_id = q->q_qinfo->qi_minfo->mi_idnum;
			qreply(q, bp);
			break;
badctl:
			STRlog(IP_ID, chan, SL_te, DPRI_HI,
				"ipoput: M_CTL %x, %x", idp->id_mstate, mp->ipf_id);
			mp->ipf_type = IPIF_UNLINKED;
			goto Reply;

		case IPIF_UNLINKING:
			/*
			 * Upper Queue is breaking away.
			 */
			if (idp->id_mstate != IPIF_LINKED)
				goto badctl;
			mp->ipf_type = idp->id_mstate = IPIF_UNLINKED;
			goto Reply;

		case IPIF_CLOSED:
			/* upper q is closed, clean up now */
			idp->id_mstate = IPIF_UNLINKED;
			idp->id_rdq = NULL;
			freemsg(bp);
			break;


		case IPIF_GETMYADDR:
			(void) ip_getmyaddr(chan, bp);
			break;

		case IPIF_NGETMYADDR:
			bp = ip_getmyaddr(chan, bp);
			qreply(q, bp);
			break;

		case IPIF_GETHISADDR:
			(void) ip_gethisaddr(chan, bp);
			break;

		case IPIF_NGETHISADDR:
			bp = ip_gethisaddr(chan, bp);
			qreply(q, bp);
			break;

		case IPIF_OPTIONS:
			/*
			 * Set or Get options.
			 */
			STRLOG(IP_ID, chan, DPRI_LO, SL_TRACE,
				"ipoput: IPIF_OPTIONS");

			/* ipf_type will tell ULP of success or failure */
			mp->ipf_type = ip_sockopt(bp, chan);
			break;

		default:
			STRlog(IP_ID, chan, DPRI_HI, SL_te,
				"ipoput: M_CTL message?? %x",mp->ipf_type,0);
			freemsg(bp);
			break;
		}	/* M_CTL */
		break;

 	case M_FLUSH:
		endflush(q, bp);
		break;
		
	/* Let the service routine handle M_PROTO and M_DATA messages. */
	case M_PROTO:
	case M_DATA:
		putq(q, bp);
		break;

	default:
		STRlog(IP_ID, chan, DPRI_HI, SL_te,
			 "ipoput: unknown bp type %x", MTYPE(bp), 0);
		freemsg(bp);
		break;
	}
	return;

}	/*  End of ipoput()  */

/*
 * See getmyaddr below
 */
mblk_t *
ip_gethisaddr(chan, bp)
	mblk_t *bp;
{
	register struct ip_if *mp;
	register struct in_ifaddr *ia;

	mp = (struct ip_if *)bp->b_rptr;
			
	/*
	 * Check for INADDR_ANY and INADDR_BROADCAST destination addresses. 
	 * called from tcpsend(), not via any service routine.
	 */	 
	if (mp->ipf_addr == INADDR_ANY)
		mp->ipf_addr = satosaddr(in_ifaddr->ia_addr);
	else if (mp->ipf_addr == INADDR_BROADCAST)
		mp->ipf_addr = satosaddr(in_ifaddr->ia_broadaddr);

	STRLOG(IP_ID, chan, DPRI_LO, SL_TRACE,
		"ip_gethisaddr: GETHISADDR addr = %x", mp->ipf_addr);

	return (bp);
}

/*
 * This routine is called with a remote IP address.
 * We need to return the local interface address from the result of routing
 * that remote address, and we need to return a valid MTU for that destination.
 */	 
mblk_t *
ip_getmyaddr(chan, bp)
	mblk_t *bp;
{
	register struct ip_if *mp;
	register struct in_ifaddr *ia;
	struct route_dst rtd;
	struct in_addr dst;
	struct ifnet *ifp;
		
	mp = (struct ip_if *)bp->b_rptr;
			
	STRLOG(IP_ID, chan, DPRI_LO, SL_TRACE,
	    "ipoput: getmyaddr for address %x", ntohl(mp->ipf_addr));

/* XXXXX Byte swapping needed for WINTCP 3.0 release XXXXX */
#ifdef OLD3_0
	dst.s_addr = htonl(mp->ipf_addr);
#else
	dst.s_addr = mp->ipf_addr;
#endif
	rtd.rtd_dst = dst;
	if (rtlookup(&rtd, 1, mp->ipf_tos) == 0)
	{
		/*
		 * NETUNREACHABLE
		 */
		STRLOG(IP_ID, chan, DPRI_LO, SL_te,
		    "ip_getmyaddr: no route for %x", ntohl(mp->ipf_addr));
		mp->ipf_addr = 0;
	}
	else
	{
		if ((ia = ifptoia(ifp = rtd.rtd_ifp)) != NULL)
		{
			STRLOG(IP_ID, -1, DPRI_LO, SL_te,
			    "ip_getmyaddr: ib_myinaddr = 0x%x",
			     ntohl(satosaddr(ia->ia_addr)));
			mp->ipf_addr = satosaddr(ia->ia_addr); 
/* XXXXX Byte swapping needed for WINTCP 3.0 release XXXXX */
#ifdef OLD3_0
			mp->ipf_addr = ntohl(mp->ipf_addr);
#else
			mp->ipf_addr = mp->ipf_addr;
#endif
			if (in_localaddr(dst))
			    mp->ipf_mtu = ifp->if_mtu;
			else if (ifp->if_mtu > IP_MSS)
			    mp->ipf_mtu = IP_MSS;
			else
			    mp->ipf_mtu = ifp->if_mtu;
			if (ifp->if_mtu > IP_MSS)
			    mp->ipf_un.ipun_addr.ipad_mss = IP_MSS;
			else
			    mp->ipf_un.ipun_addr.ipad_mss = ifp->if_mtu;
			mp->ipf_un.ipun_addr.ipad_flags = ifp->if_flags;
			mp->ipf_un.ipun_addr.ipad_thruput = ifp->ib_speed;
		}
		else
		{
			STRLOG(IP_ID, chan, DPRI_LO, SL_te,
			    "ip_getmyaddr: no ia for ipb, address %x",
			     ntohl(mp->ipf_addr));
			mp->ipf_addr = 0;
		} 
	}

	return (bp);
}

/*
 * data coming from tcp or udp or raw or ...
 */
iposrv(q)
	register queue_t *q;
{
	register mblk_t *bp;
	register mblk_t *databp;
	register mblk_t *opt;
	struct ip *ip;
	int flags;

	/*
	 * The message type is assumed to be either M_PROTO or M_DATA;
	 * ipoput() should ensure this.
	 * 
	 * For message type of M_PROTO a number of IP_args structures and args
	 * are expected.  The continuuation block will contain the real data.
	 *
	 * For message type of M_DATA the data starts in the first block.
	 */			
	while (bp = getq(q))
	{
		/* Initialize options to none */
		opt = NULL;
		flags = 0;

		if (MTYPE(bp) == M_PROTO)
		{
			struct IP_args *ipargs;
			unchar *optstart, *optend;

			for (;;)
			{
				ipargs = (struct IP_args *)bp->b_rptr;
				bp->b_rptr += sizeof(struct IP_args);

				if (ipargs->IP_type == IPFLAGS) {
					bcopy(bp->b_rptr, (caddr_t)&flags,
						ipargs->IP_length);
				}
				else if (ipargs->IP_type == IPOPTS) {
					opt = bp;
					optstart = bp->b_rptr;
					optend = bp->b_rptr + ipargs->IP_length;
					MTYPE(bp) = M_DATA;
				}
				else
					break;

				bp->b_rptr += ipargs->IP_length;
			}

			databp = bp->b_cont;
			bp->b_cont = NULL;

			if (opt == NULL)
				freeb(bp);
			else {
				bp->b_rptr = optstart;
				bp->b_wptr = optend;
			}

		} else
			databp = bp;

		ipstat.ipOutRequests++;
		ip = mtod(databp, struct ip *);
/* XXXXX Byte swapping needed for WINTCP 3.0 release XXXXX */
#ifdef OLD3_0
		ip->ip_src.s_addr = htonl(ip->ip_src.s_addr);
		ip->ip_dst.s_addr = htonl(ip->ip_dst.s_addr);
#endif
		if (ip_output(databp, opt, flags))
		{
		 	/* send icmp to OTHERQ(q)->q_next if error */
			/* XXX: TO BE CONTINUE */
		}
	}

}	/*  End of iposrv()  */


#define	IPPROTTABSIZ	5
/*
 * map module id to protocol number
 */
static int ipid2prot_table[IPPROTTABSIZ][2] = {
	{ TCP_ID, IPPROTO_TCP },
	{ UDP_ID, IPPROTO_UDP },
	{ ICMP_ID, IPPROTO_ICMP },
	{ RAW_ID, IPPROTO_RAW },
	{ -1, -1 },
};

/* return the protocol # if I know the guys on top, else -1 */
ipid_to_prot(id)
	register int id;
{
	register int i;
	register int id1 = 0;

	for (i=0; id1 != -1; i++) {
		if ((id1 = ipid2prot_table[i][0]) == id)
			return(ipid2prot_table[i][1]);
	}
	return -1;

}	/*  End of ipid_to_prot()  */


/*
 * called from ip_output() to put the msg to hardware module
 * bp is the complete message without ethernet/starlan header
 * dst is the intermediate internet address (may not == ip_dst)
 * ifp is the bottom queue (from if_ifonnetof(dst))
 * return 0 if o.k., -1 if error
 * when error, the caller will free the message
 */
ipsend(ifp, bp, dst, len, ifflags)
	register struct ifnet *ifp;
	register mblk_t *bp;
	register struct sockaddr_in *dst;
	unsigned short len;
	int ifflags;
{
	u_long tmp_dst = dst->sin_addr.s_addr;
	mblk_t *bp1;

	ASSERT(bp && (IPBCHAN(ifp) < ipb_cnt));

	/*
	 * If packet too small for interface, tack on some extra stuff.
	 */
	len = ntohs(len);
	if (len < ifp->ib_minpsz)
	{
		len = ifp->ib_minpsz - len;
		if ((bp1 = allocb(len, BPRI_LO)) == NULL)
		{
			STRlog(IP_ID, -1, DPRI_LO, SL_te, "ipsend: NOBUFFS");
			freemsg(bp);
			++ipstat.ipOutDiscards;
			return (-1);
		}
		bp1->b_wptr += len;
		linkb(bp, bp1);
	}

	/*
	 * If for us, loop it back around.
	 */
	++ifp->if_opackets;
	if (in_ouraddr(dst->sin_addr))
	{
		if (canput(RD(ifp->ib_wrq)))
			putq(RD(ifp->ib_wrq), bp);
		else
		{
			++ifp->if_oerrors;
			++ipstat.ipOutDiscards;
			freemsg(bp);
			return (-1);
		}
		return (0);
	}

	/*
	 * Check that the driver stream is still there and ready.
	 */
	if ((ifp->ib_wrq == NULL) || ((ifp->if_flags & IFF_UP) == 0))
	{
		STRlog(IP_ID, IPBCHAN(ifp), DPRI_LO, SL_te,
			"ipsend: err dst %x, flag %x", tmp_dst,
			ifp->if_flags);
		freemsg(bp);
		++ipstat.ipOutDiscards;
		++ifp->if_oerrors;
		return (-1);
	}

	/*
	 * Drivers Like the loopback and Point to Point don't need the
	 * pseudo ethernet header.
	 */
	if ((ifp->if_flags & (IFF_LOOPBACK | IFF_POINTOPOINT)) != 0)
	{
		if (canput(ifp->ib_wrq->q_next))
			putnext(ifp->ib_wrq, bp);
		else
		{
			freemsg(bp);
			++ipstat.ipOutDiscards;
			++ifp->if_oerrors;
			return (-1);
		}
		return (0);
	}

	/*
	 * All other drivers get the header.
	 */
	if (ipdl_send(ifp->ib_wrq, ifp->ib_bottype, tmp_dst, bp, ifflags))
	{
		STRlog(IP_ID, IPBCHAN(ifp), DPRI_LO, SL_te,
			"ipsend: err dst %x, flag %x", tmp_dst,
			ifp->if_flags);
		++ifp->if_oerrors;
		++ipstat.ipOutDiscards;
		freemsg(bp);
		return (-1);
	}

	return (0);
}

ipbiput(q, bp)
	register queue_t *q;
	register mblk_t *bp;
{
	queue_t *cq;
	register union DL_primitives *p;
	register struct ifnet *ifp;
	struct in_ifaddr *ia;
	struct sockaddr_in netaddr;
	long type;

	ASSERT(q && bp);
	ifp = (struct ifnet *)q->q_ptr;
	ASSERT(ifp);
	ifp->if_ipackets++;

	switch(MTYPE(bp)) {

	case M_DATA:
		/*
		 * For drivers that don't talk LLI
		 */
		putq(q, bp);
		break;

	case M_PROTO:
	case M_PCPROTO:
		p = (union DL_primitives *)bp->b_rptr;
		STRLOG(IP_ID, IPBCHAN(ifp), DPRI_MED, SL_TRACE,
		  "ipbiput: get prim %x", p->prim_type);

		switch(p->prim_type) {
		case DL_BIND_ACK:
			if (ifp->if_flags & IFF_BROADCAST) {
			    ASSERT(p->bind_ack.LLC_sap == ifp->ib_sap);
			}
			if (!(ifp->if_flags & IFF_IPTOX25))
			    ifp->if_flags |= IFF_UP;
			ifp->ib_mstate = DL_IDLE;
			bzero((caddr_t)&netaddr, sizeof netaddr);
			netaddr.sin_family = AF_INET;
			bcopy((caddr_t)p + p->bind_ack.ADDR_offset,
				(caddr_t)&netaddr.sin_addr, sizeof (ulong));
			for (ia = in_ifaddr; ia; ia = ia->ia_next)
			{
				if (ia->ia_ifp == ifp)
				{
					ip_ifinit(ifp, ia, &netaddr, -1);
					break;
				}
			}
			if (!(ifp->if_flags & IFF_LOOPBACK) && 
			    !(ifp->if_flags & IFF_IPTOVCS))
				++in_interfaces;  
			break;

		case DL_OK_ACK:	/* ack of the UNBIND */
			if (p->ok_ack.CORRECT_prim == DL_UNBIND_REQ) {
				ifp->if_flags &= ~IFF_UP;
				ifp->ib_mstate = DL_UNBND;
				if (!(ifp->if_flags & IFF_LOOPBACK) && 
				    !(ifp->if_flags & IFF_IPTOVCS))
					--in_interfaces;  
				ipbclose(ifp);
			}
			break;

		case DL_ERROR_ACK:
		{
			struct DL_error_ack *ep = (struct DL_error_ack *)p;

			STRlog(IP_ID, IPBCHAN(ifp), DPRI_LO, SL_te,
			  "ipbiput: ERR_ACK, prim %x, LLC err %x, unix err %x",
			  ep->ERROR_prim, ep->LLC_error, ep->UNIX_error);
			if (ifp->if_flags & IFF_IPTOVCS)
				ipvcshangup(q, ifp);
			else
				ipbclose(ifp);
			break;
		}

		case DL_UDERROR_IND:
			/* non-fatal error, but record */
			STRlog(IP_ID, IPBCHAN(ifp), DPRI_LO, SL_te,
				"ipbiput: UDERROR, type %x",
				p->error_ind.ERROR_type);
			++ifp->if_oerrors;
			break;

		case DL_INFO_ACK:
			ifp->if_mtu = p->info_ack.SDU_max;
			ifp->ib_minpsz = p->info_ack.SDU_min;
			type = p->info_ack.SUBNET_type;

			switch(type) {

			case DL_LOOP:
				ifp->if_flags |= IFF_LOOPBACK;
				goto Fill;
				
			case DL_PTOP:
				ifp->if_flags |= IFF_POINTOPOINT;
				goto Fill;

			case DL_IPX25:
				ifp->if_flags |= IFF_IPTOX25;
				ifp->ib_bottype = type;
				ifp->ib_sap = IP_SAP;
				ifp->ib_speed = 56000;
				break;

			case DL_CSMACD:
			case DL_ETHER:
				ifp->if_flags |= IFF_BROADCAST;
			case DL_X25:
Fill:
				ifp->ib_bottype = type;
				ifp->ib_speed = 10000000;
				if (type == DL_ETHER)
				    ifp->ib_sap = ETHERPUP_IPTYPE;
				else if (type == DL_X25)
				{
				    ifp->ib_sap = 0xCC;		/* DCA-NSAP */
				    ifp->if_flags |= IFF_IPTOX25;
				    ifp->ib_speed = 56000;	/* XXX */
				}
				else
				    ifp->ib_sap = IP_SAP;
				if (ifp->if_flags & IFF_BINDNEED) {
				    ifp->if_flags &= ~IFF_BINDNEED;
				    ifp->ib_mstate = DL_WACK_B;
				    if (dl_bind(ifp->ib_wrq, (long)ifp->ib_sap))
					break;
				}
				break;

			default:
				STRlog(IP_ID, IPBCHAN(ifp), DPRI_LO, SL_te,
			  	"ipbiput: unknow info_ack type %x", type, 0);
				break;
			}
			break;

		case DL_UNITDATA_IND:
		{
			register mblk_t *bp1;
LABEL(ipbunitdata);
			STRlog(IP_ID, -1, DPRI_LO, SL_TRACE,
			  "ipbiput: case DL_UNITDATA_IND");
			bp1 = unlinkb(bp);
			freeb(bp);
			ASSERT(bp1);
			putq(q, bp1);
			return;
		}
			
		default:
			STRlog(IP_ID, -1, DPRI_LO, SL_te,
			  "ipbiput: unknown DL_prim %x", p->prim_type, 0);
			break;
		}

		freemsg(bp);
		break;

	case M_FLUSH:
		headflush(q, bp);
		break;

	case M_ERROR:
		/* get fatal-error from the provider */
		STRlog(IP_ID, IPBCHAN(ipb), DPRI_LO, SL_te,
		    "ipbiput: got M_ERROR code %x", *bp->b_rptr, 0);
 
		/* Do below only if interface is not IFF_IPTOVS */
		if (!(ifp->if_flags & IFF_IPTOVCS)) {
			if (ifp->ib_mstate != DL_WACK_U)
				ipbclose(ifp);
			freemsg(bp);
			break;
		}
		/* Fall through */

	case M_HANGUP:
		/* Do below only if interface is IFF_IPTOVS */
		if (ifp->if_flags & IFF_IPTOVCS)
			ipvcshangup(q, ifp);
		else if(ifp->if_flags & IFF_IPTOX25) {
			ifp->if_flags &= ~IFF_UP;
		} else {
			ifp->if_flags &= ~IFF_UP;
			ifp->ib_mstate = DL_WACK_U;
			(void) dl_unbind(ifp->ib_wrq);
			ifp->if_flags |= IFF_BINDNEED;
		}
		freemsg(bp);
		break;

	case M_STOP:
		/* Command to stop transmition (only) */
		ifp->if_flags &= ~IFF_UP;
		freemsg(bp);
		break;

	case M_START:
		/* Command to re-start transmition */
		if (ifp->if_flags & IFF_IPTOX25) {
			ifp->if_flags |= IFF_UP;
		} else {
			ifp->if_flags |= IFF_BINDNEED;
			(void) dl_inforeq(ifp->ib_wrq);
		}
		freemsg(bp);
		break;

	default:
		STRlog(IP_ID, IPBCHAN(ifp), DPRI_HI, SL_te,
		    "ipbiput: unknown bp %x, type %x", bp, MTYPE(bp));
		cq = ipd[IPCTLCHAN].id_rdq;	/* control chan's q */
		putnext(cq, bp);
		break;
	}

}	/*  End of ipbiput()  */


/*
 * just call ip_input() to process the ip options, checksum, fragment, etc
 * when a whole ip is ready, it is passed to the proper ip upper queues
 * can we allow timer comes in? protect by spl2()?
 */
ipbisrv(q)
	register queue_t *q;
{
	register mblk_t *bp;

	STRlog(IP_ID, -1, DPRI_LO, SL_te,
		  "ipbisrv: Entered. q=0x%x",q);
	while (bp = getq(q)) {
		(void) ip_input(bp, (struct ifnet *)q->q_ptr);
			 /* ip_input will deal with errors */
	}

}	/*  End of ipbisrv()  */

/*
 * this pseudo interrupt is called from either ip_input or
 * ip_raw_input when an ip packet is fully assembled.
 * 2nd argument is the protocol # found in the ip header, e.g. 6 for tcp
 * We now linearly search the ipd table to find the matching upper q
 * use hashing if there are lots of ipd devices
 */
void
ipdrint(bp, prot)
	register mblk_t *bp;
	register unsigned char prot;
{
	register queue_t *q;
	register struct ipd *idp, *rdp = NULL;
	register struct ipovly *ip;

	/* Setup IP overlay for transport protocol above (broadcast notify) */
	ip = mtod(bp, struct ipovly *);
#ifdef notdef
	ip->ih_next = ip->ih_prev = NULL;
#endif
	ip->ih_x1 = 0;
	if (in_broadcast(ip->ih_dst))
		ip->ih_x1 |= IPO_BROADCAST;
/* XXXXX Byte swapping needed for WINTCP 3.0 release XXXXX */
#ifdef OLD3_0
	ip->ih_src.s_addr = ntohl(ih->ip_src.s_addr);
	ip->ih_dst.s_addr = ntohl(ih->ip_dst.s_addr);
#endif

	/* Search for upper-level protocol */
	for (idp = &ipd[0]; idp < &ipd[ipd_cnt]; idp++)
	{
		/* Watch for RAW */
		if (idp->id_prot == IPPROTO_RAW)
		{
			rdp = idp;
			continue;
		}

		/* Check for exact upper queue match */
		if (idp->id_prot == prot)
		{
			if ((q = idp->id_rdq) && canput(q))
			{
				putnext(q, bp);
				return;
			}
			else
			{
				freemsg(bp);
				return;
			}
		}
	}

	/* Send everything else to the raw module */
	if (rdp)
	{
		if ((q = rdp->id_rdq) && canput(q))
		{
			putnext(q, bp);
			return;
		}
	}

	STRlog(IP_ID, -1, DPRI_LO, SL_te,
		 "ipdrint: droping protocol %x", prot, 0);
	ipstat.ipInUnknownProtos++;
	freemsg(bp);
	return;

}	/*  End of ipdrint()  */

/*
 * This routine is called to pass a control message to the
 * upper protocols. It consists of two message blocks. The
 * first message block contains the ip_if structure and the code.
 * the second the actual control message. The message is of
 * type M_CTL.
 */
void
ip_ctlmsg(code, protocol, addr)
	u_char protocol;
	struct sockaddr_in *addr;
{
	register struct ip_if *ipf;
	register mblk_t *bp;
	register struct ipd *idp;

	if (!(bp = allocb(sizeof(struct ip_if), BPRI_LO)))
		return;
	ipf = (struct ip_if *)bp->b_wptr;
	bp->b_wptr += sizeof(struct ip_if);
	ipf->ipf_type = IPIF_CTLERROR;
	ipf->ipf_errorcode = code;
	ipf->ipf_errordst = ntohl(addr->sin_addr.s_addr);
	bp->b_cont = NULL;
	MTYPE(bp) = M_CTL;
	for (idp = &ipd[0]; idp < &ipd[ipd_cnt]; idp++)
		if (idp->id_prot == protocol)
			if (idp->id_rdq)
			{
				putnext((idp->id_rdq), bp);
				return;
			} else
				break;

	TRACE((1<<24), "ip_ctlmsg: droping protocol %x\n", protocol);
	(void) freemsg(bp);
}	/*  End of ip_ctlmsg()  */

/*
 * notify lower stream with my addr
 */
ipmuxnotify(ifp, oldaddr)
	struct ifnet *ifp;
	u_long oldaddr;
{
	register mblk_t *bp, *bp1;
	register u_long *inaddr;
	struct in_ifaddr *ia;

	if ((bp = allocb(sizeof(u_long), BPRI_MED)) == NULL) 
		return(-1);
	MTYPE(bp) = M_CTL;
	inaddr = (u_long *)bp->b_wptr;
	ia = ifptoia(ifp);
	if(ia == (struct in_ifaddr *)0) 
		return (-1);
	*inaddr = satosaddr(ia->ia_addr); 
	bp->b_wptr += sizeof(u_long);
	if (oldaddr) {
		if ((bp1 = allocb(sizeof(u_long), BPRI_MED)) == NULL) {
			freemsg(bp);
			return -1;
		}
		MTYPE(bp1) = M_CTL;
		*(u_long *)bp1->b_wptr = oldaddr;
		bp1->b_wptr += sizeof(u_long);
		bp->b_cont = bp1;
	}
	putnext(ifp->ib_wrq, bp);
	return(0);
}	/*  End of ipmuxnotify()  */


#define fill_in_msg( msg, source, count ) \
/*	register mblk_t	*msg;		/* first block of target message */ \
/*	char		*source;	/* source data structure */	\
/*	int		count;		/* bytes to copy from source */ \
{	bcopy( (char *)(source), (msg)->b_rptr, count ); \
	(msg)->b_wptr = (msg)->b_rptr + count ; }

/*
 * ioctls are not crossed to lower modules
 * most ioctls can only be done to the control channel
 */
void
ip_doioctl(q, bp)
	register queue_t *q;
	register mblk_t *bp;
{
	register struct ipd *idp;
	register struct ifnet *ifp;
	struct iocblk *ip;
	register struct ifreq *ifr;
	register struct in_ifaddr *iap, *ia = 0;
	mblk_t *bp1;
	struct ifaddr *ifa;
	struct linkblk *lp;
	int cmd, chan;
	queue_t *bq;
	ulong tmp;
	struct rtentry *rt;
	int length;
	extern ifconf();

	idp = (struct ipd *)q->q_ptr;
	chan = IPCHAN(idp);
	ip = (struct iocblk *)bp->b_rptr;
	bp1 = bp->b_cont;
	cmd = ip->ioc_cmd;

	switch (cmd) {

	case I_LINK:
	case I_UNLINK:
		if ((bp1 == (mblk_t *)NULL) ||
		    (ip->ioc_count < sizeof(struct linkblk))) {
			STRlog(IP_ID, chan, DPRI_HI, SL_te,
				"ipoput: ioctl (un)link invalid? %d", 
				ip->ioc_count);
			ip->ioc_error = EINVAL;
			goto nack;
		}
		lp = (struct linkblk *)bp1->b_rptr;
		bq = lp->l_qbot;
		goto skip_check;

	case IPIOC_GETIFCONF:
		if (bp1 == (mblk_t *)NULL) { 
			ip->ioc_error = EINVAL;
			goto nack;
		}
		goto skip_check; 

	case IPIOC_ADDROUTE:
	case IPIOC_DELETEROUTE:
	case IPIOC_GETROUTE:
		length = sizeof(struct rtentry);
		break;

	case IPIOC_GETPATHS:
		length = sizeof(struct route_dst);
		break;

	case IPIOC_GETRTSTAT:
		length = sizeof rtstat;
		break;

	case IPIOC_GETIPSTAT:
		length = sizeof ipstat;
		break;

	case IPIOC_GETICMPSTAT:
		length = sizeof icmpstat;
		break;

	case IPIOC_GETIPB:
		length = sizeof(struct ifnet);
		break;

	case IPIOC_GETINADDR:
		length = sizeof(struct in_ifaddr);
		break;

	case IPIOC_GETIPTTL:
	case IPIOC_SETIPFORWARD:
	case IPIOC_GETIPFORWARD:
	case IPIOC_GETIPGATEWAY:
	case IPIOC_GETIPREDIRECT:
		length = sizeof(long);
		break;

	default:
		length = sizeof(struct ifreq);
		break;
	}

	if ((bp1 == (mblk_t *)NULL) ||
	    (ip->ioc_count < length) ||
	     !pullupmsg(bp1, length)) {
		ip->ioc_error = EINVAL;
		goto nack;
	}

skip_check:
		
	STRLOG(IP_ID, chan, DPRI_HI, SL_TRACE, "ip_doioctl: cmd 0x%x", cmd);

	/*
	 * Some ioctl commands are for root only!
	 */
	switch (cmd) {

	case IPIOC_GETIFCOUNTS:
	case IPIOC_GETIFMETRIC:
	case IPIOC_GETIFFLAGS:
	case IPIOC_GETIFCONF:
	case IPIOC_GETIFADDR:
	case IPIOC_GETIFBRDADDR:
	case IPIOC_GETIFDSTADDR:
	case IPIOC_GETIFNETMASK:
	case IPIOC_GETIPB:
	case IPIOC_GETINADDR:
	case IPIOC_GETIPSTAT:
	case IPIOC_GETICMPSTAT:
	case IPIOC_GETROUTE:
	case IPIOC_GETRTSTAT:
	case IPIOC_GETPATHS:
	case IPIOC_GETIPTTL:
	case IPIOC_GETIPFORWARD:
	case IPIOC_GETIPGATEWAY:
	case IPIOC_GETIPREDIRECT:
		break;

	default:
		if (ip->ioc_uid) {
			ip->ioc_error = EPERM;
			goto nack;
		}
	}

	switch (cmd) {

	case IPIOC_GETICMPSTAT:
	case IPIOC_GETIFCONF:
	case IPIOC_GETIPSTAT:
	case IPIOC_GETRTSTAT:
	case IPIOC_GETPATHS:
	case IPIOC_GETIPTTL:
	case IPIOC_GETIPFORWARD:
	case IPIOC_GETIPGATEWAY:
	case IPIOC_GETIPREDIRECT:
	case IPIOC_SETIPFORWARD:
	case I_LINK:
	case I_UNLINK:
		break;

	case IPIOC_ADDROUTE:
	case IPIOC_DELETEROUTE:
	case IPIOC_GETROUTE:
		rt = (struct rtentry *)bp1->b_rptr;
		break;

	case IPIOC_GETIPB:
		ifp = (struct ifnet *)bp1->b_rptr;
		break;

	case IPIOC_GETINADDR:
		iap = (struct in_ifaddr *)bp1->b_rptr;
		break;

	case IPIOC_SETNAME:
		ifr = (struct ifreq *)bp1->b_rptr;
		break;

	default:
		ifr = (struct ifreq *)bp1->b_rptr;
		if ((ifp = ip_getif(ifr->ifr_name)) == NULL)
			goto nack;
		for (ia = in_ifaddr; ia; ia = ia->ia_next) {
			if (ia->ia_ifp == ifp)
				break;
		}
		break;
	}

	switch (cmd) {

	case IPIOC_ADDROUTE:
	case IPIOC_DELETEROUTE:
	case IPIOC_GETICMPSTAT:
	case IPIOC_GETIFCONF:
	case IPIOC_GETIFCOUNTS:
	case IPIOC_GETINADDR:
	case IPIOC_GETIPB:
	case IPIOC_GETIPSTAT:
	case IPIOC_GETROUTE:
	case IPIOC_GETPATHS:
	case IPIOC_GETRTSTAT:
	case IPIOC_SETIFFLAGS:
	case IPIOC_SETNAME:
	case IPIOC_GETIPTTL:
	case IPIOC_SETIPFORWARD:
	case IPIOC_GETIPFORWARD:
	case IPIOC_GETIPGATEWAY:
	case IPIOC_GETIPREDIRECT:
	case I_LINK:
	case I_UNLINK:
		break;

	case IPIOC_SETIFADDR:
	case IPIOC_SETIFNETMASK:
	case IPIOC_SETIFDSTADDR:
		if (ia == (struct in_ifaddr *)0) {
			iap = get_inaddr();
			if (iap == (struct in_ifaddr *)0)
				goto nack;
			if (ia = in_ifaddr) {
				for( ; ia->ia_next; ia = ia->ia_next)
					;	
				ia->ia_next = iap;
			} else
				in_ifaddr = iap;
			ia = iap;
			if (ifa = ifp->if_addrlist) {
				for( ; ifa->ifa_next; ifa = ifa->ifa_next);
					;
				ifa->ifa_next = (struct ifaddr *)ia;
			} else
				ifp->if_addrlist = (struct ifaddr *)ia;
			ia->ia_ifp = ifp;
			IA_SIN(ia)->sin_family = AF_INET;
		} 
		break;

	default:
		if (ia == (struct in_ifaddr *)0) {
			ip->ioc_error = EADDRNOTAVAIL;
			goto nack; 
		}
		break;
	}
	

	switch(cmd) {

	case I_LINK:
		/*
		 * internal open of an unused bottom q
		 */
		for (ifp = &ipb[0]; ifp < &ipb[ipb_cnt]; ifp++)
			if (ifp->ib_wrq == NULL)
			{
				bq->q_ptr = RD(bq)->q_ptr = (char *)ifp;
				if ((iap = get_inaddr()) == NULL)
				    goto nack;
				if (ia = in_ifaddr) {
				    for( ; ia->ia_next; ia = ia->ia_next)
					;	
				    ia->ia_next = iap;
				} else
				    in_ifaddr = iap;
				ia = iap;
				ia->ia_ifp = ifp;
				IA_SIN(ia)->sin_family = AF_INET;
				ifp->if_addrlist = (struct ifaddr *)ia;
				ifp->ib_wrq = bq;
				ifp->ib_mstate = DL_UNBND;
				ifp->if_flags |= IFF_BINDNEED;
				/*
				 * we have to wait for info_ack to know what
				 * to bind.  we just pretend the linking is
				 * successful and return, but ifp->ib_mstate
				 * is still DL_UNBND, and will be changed to
				 * DL_IDLE when bind_ack comes back.
				 */
				ifp->ib_qtop = lp->l_qtop;
				ifp->ib_cookie = lp->l_index;
				ifp->if_next = ipbnext;
				ipbnext = ifp;
				if (dl_inforeq(ifp->ib_wrq)) {
					(void) ipbclose(ifp);
					goto nack;
				}
				goto ack0;
			}

		STRlog(IP_ID, chan, DPRI_HI, SL_te, "ipoput: no ipb");
		goto nack;

	case I_UNLINK:
		/* Don't do anything if the interface is already disconnected */
		if ((ifp = ipmuxq(lp->l_index)) == NULL)
		{
			STRlog(IP_ID, chan, DPRI_HI, SL_te,
			   "ipoput: I_UNLINK failed to find l_index 0x%x",
			   lp->l_index);
			goto nack;
		}
		ASSERT(bq == ifp->ib_wrq);
		if (ifp->ib_mstate != DL_IDLE)
		{
			/* If not VCS or unbound already, error */
			if ((ifp->ib_mstate != DL_UNBND) &&
			    !(ifp->if_flags & IFF_IPTOVCS))
			{
			    STRlog(IP_ID, chan, DPRI_HI, SL_te,
				"ipoput: I_UNLINK bad mstate value of %d",
				ifp->ib_mstate);
			    goto nack;
			}
		}
		else
		{
			/* Unbind interface.
			 * Wait for the unbind request to be acked, 
			 * then close the interface, and ack the ioctl.
			 */
			ifp->ib_mstate = DL_WACK_U;
			ifp->ib_iocblk = bp;
			if (dl_unbind(ifp->ib_wrq) < 0)
			{
			    STRlog(IP_ID, chan, DPRI_HI, SL_te,
				"ipoput: I_UNLINK unbind failed 0x%x", ifp);
			    ifp->ib_mstate = DL_IDLE;
			    goto nack;
			}
		}
		return;

	case IPIOC_SETNAME:
		/*
		 * first ioctl, use cookie to set the ifr_name
		 * later ioctls use the name as key (see ip_getif)
		 */
		if ((ifp = ipmuxq(ifr->ifr_cookie)) == NULL)
			goto nack;
		if (ip_getif(ifr->ifr_name) != NULL)
			goto nack;		/* name conflict */
		ip_setname(ifp, ifr->ifr_name);
		break;

	case IPIOC_SETIFADDR:
		if (ip_ifinit(ifp, ia, (struct sockaddr_in *)&ifr->ifr_addr,
			      chan) < 0) {
			if(ifp->if_flags & IFF_IPTOVCS)
				ipvcshangup(q, ifp);
			ip->ioc_error = EINVAL;
			goto nack;
		}
		if (ipmuxnotify(ifp, satosaddr(ifr->ifr_addr)) < 0)
			goto nack;
		break;

	case IPIOC_SETIFFLAGS:
		/* XXX do this later 
		if ((ifp->if_flags & IFF_UP) && (ifr->ifr_flags & IFF_UP) == 0) 
			if_down(ifp);
		*/
		ifp->if_flags = (ifp->if_flags & IFF_CANTCHANGE) |
			(ifr->ifr_flags &~ IFF_CANTCHANGE);	
		break;

	case IPIOC_SETIFDSTADDR:
	{
		struct sockaddr oldaddr;

		if ((ifp->if_flags & IFF_POINTOPOINT) == 0) {
			ip->ioc_error = EINVAL;
			goto nack;
		}
		oldaddr = ia->ia_dstaddr;
		satosaddr(ia->ia_dstaddr) = satosaddr(ifr->ifr_dstaddr);
		if (ia->ia_flags & IFA_ROUTE) {
			ip_rtinit(&oldaddr, &ia->ia_addr, 
				IPIOC_DELETEROUTE, RTF_HOST, RTR_OTHER, 0);
			ip_rtinit(&ia->ia_dstaddr, &ia->ia_addr,
				IPIOC_ADDROUTE, RTF_HOST | RTF_UP,
				RTR_OTHER, 0);
		}
		break;
	}

	case IPIOC_SETIFBRDADDR:
		if ((ifp->if_flags & IFF_BROADCAST) == 0) {
			ip->ioc_error = EINVAL;
			goto nack;
		}
		ia->ia_broadaddr = ifr->ifr_broadaddr;
		tmp = ntohl(satosaddr(ia->ia_broadaddr));
		if ((tmp &~ ia->ia_subnetmask) == ~ia->ia_subnetmask)
			tmp |= ~ia->ia_netmask;
		else if ((tmp &~ ia->ia_subnetmask) == 0)
			tmp &= ia->ia_netmask;
		ia->ia_netbroadcast.s_addr = htonl(tmp);
		break;

	case IPIOC_SETIFNETMASK:
		ia->ia_subnetmask = ntohl(satosaddr(ifr->ifr_addr));
		break;

	case IPIOC_SETIFMETRIC:
		ifp->if_metric = ifr->ifr_metric;
		break;

	case IPIOC_GETIFFLAGS:
		ifr->ifr_flags = ifp->if_flags;
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_GETIFADDR:
		ifr->ifr_addr = ia->ia_addr;
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_GETIFBRDADDR:
		if ((ifp->if_flags & IFF_BROADCAST) == 0) {
			ip->ioc_error = EINVAL;
			goto nack;
		}
		ifr->ifr_broadaddr = ia->ia_broadaddr;
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_GETIFDSTADDR:
		if ((ifp->if_flags & IFF_POINTOPOINT) == 0) {
			ip->ioc_error = EINVAL;
			goto nack;
		}
		ifr->ifr_dstaddr = ia->ia_dstaddr;
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_GETIFNETMASK:
		satosin(&ifr->ifr_addr)->sin_family = AF_INET;
		satosaddr(ifr->ifr_addr) = htonl(ia->ia_subnetmask);
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_GETIFMETRIC:
		ifr->ifr_metric = ifp->if_metric;
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_GETIFCONF:
		if (ifconf(bp) < 0)
		{
			ip->ioc_error = EINVAL;
			goto nack;
		}
		goto ack;

	case IPIOC_GETIFCOUNTS:
		ifr->ifr_ipackets = ifp->if_ipackets;
		ifr->ifr_opackets = ifp->if_opackets;
		ip->ioc_count = sizeof(struct ifreq);
		goto ack;

	case IPIOC_ADDROUTE:
	case IPIOC_DELETEROUTE:
		/*
		 * Add a route to the routing table.
		 */
		if (ip->ioc_error = rtioctl(cmd, (caddr_t)rt))
			goto nack;
		break;

/* following ioctls added January 1989 for ARIX
 * to eliminate use of /dev/kmem in netstat
 */
	case IPIOC_GETIPTTL:
	{
		extern int ip_ttl;

		ip->ioc_count = sizeof (int);
		fill_in_msg(bp1, &ip_ttl, sizeof (int));
		goto ack;
	}
	case IPIOC_SETIPFORWARD:
	{
		extern int ipforwarding;

		bcopy(bp1->b_rptr, (caddr_t)&ipforwarding, sizeof (int));
		goto ack;
	}
	case IPIOC_GETIPFORWARD:
	{
		extern int ipforwarding;

		ip->ioc_count = sizeof (int);
		fill_in_msg(bp1, &ipforwarding, sizeof (int));
		goto ack;
	}
	case IPIOC_GETIPGATEWAY:
	{
		extern int ipgateway_icmp;

		ip->ioc_count = sizeof (int);
		fill_in_msg(bp1, &ipgateway_icmp, sizeof (int));
		goto ack;
	}
	case IPIOC_GETIPREDIRECT:
	{
		extern int ipsendredirects;

		ip->ioc_count = sizeof (int);
		fill_in_msg(bp1, &ipsendredirects, sizeof (int));
		goto ack;
	}

	case IPIOC_GETIPSTAT:
	{
		ip->ioc_count = sizeof ipstat;
		fill_in_msg(bp1, &ipstat, sizeof ipstat);
		goto ack;
	}

	case IPIOC_GETICMPSTAT:
	{
		ip->ioc_count = sizeof icmpstat;
		fill_in_msg(bp1, &icmpstat, sizeof icmpstat);
		goto ack;
	}

	case IPIOC_GETRTSTAT:
	{
		ip->ioc_count = sizeof rtstat;
		fill_in_msg(bp1, &rtstat, sizeof rtstat);
		goto ack;
	}

	case IPIOC_GETPATHS:
	{
		struct route_dst rtd[16];
		int count, n;

		/*
		 * Setup search for IP destination.
		 */
		bcopy(bp1->b_rptr, &rtd[0].rtd_dst.s_addr, sizeof(long));
		count = msgdsize(bp1) / sizeof (struct route_dst);

		/*
		 * Lookup all possible routes to <IP,*> for user.
		 */
		if ((n = rtlookup(rtd, count, -1)) == 0)
		{
			ip->ioc_error = EADDRNOTAVAIL;
			goto nack; 
		}

		/*
		 * Copy out to user.
		 */
		ip->ioc_count = sizeof (struct route_dst) * n;
		fill_in_msg(bp1, rtd, ip->ioc_count);

		goto ack;
	}

	case IPIOC_GETROUTE:
	{
		extern struct rtentry route[];
		extern int route_cnt;
		int index;

		/*
		 * Get route table index value.
		 */
		index = rt->rt_hash;

		/*
		 * Verify and then return route to user
		 */
		if ((index < 0) || (index >= route_cnt))
		{
			ip->ioc_error = EADDRNOTAVAIL;
			goto nack; 
		}

		/* Return rtentry to user */
		ip->ioc_count = sizeof(struct rtentry);
		fill_in_msg(bp1, &route[index], sizeof(struct rtentry));
		goto ack;
	}

	case IPIOC_GETIPB:
	{
		/* if user passed in an invalid ipb address, then
		 * return the ipb at the list head 
		 */
		ifp = ifp->if_next;
		if ((ifp < &ipb[0]) ||
		    (ifp >= &ipb[ipb_cnt]) ||
		    (((char *)ifp - (char *)&ipb[0]) % sizeof (struct ifnet)))
		{
			ifp = ipbnext;
			if (ifp == NULL)
			{
				ip->ioc_error = EADDRNOTAVAIL;
				goto nack; 
			}
		}
		ip->ioc_count = sizeof(struct ifnet);
		fill_in_msg(bp1, ifp, sizeof(struct ifnet));
		goto ack;
	}

	case IPIOC_GETINADDR:
	{
		/* if user passed in an invalid in_addr address, then
		 * return the in_addr at the list head 
		 */
		ia = iap->ia_next;
		if ((ia < &in_addr[0]) ||
		    (ia >= &in_addr[in_cnt]) ||
		    (((char *)ia - (char *)&in_addr[0]) % sizeof *ia))
		{
			ia = in_ifaddr;
			if (ia == NULL)
			{
				ip->ioc_error = EADDRNOTAVAIL;
				goto nack; 
			}
		}
		ip->ioc_count = sizeof(struct in_ifaddr);
		fill_in_msg(bp1, ia, sizeof(struct in_ifaddr));
		goto ack;
	}

	/* add new ioctls here <<<< */

	default:
		/* unknown ioctl are not passed down */
		STRlog(IP_ID, chan, DPRI_HI, SL_te,
		   "ip_doioctl: ioctl 0x%x on cntl chan", cmd);
		ip->ioc_error = EINVAL;
		goto nack;
	}

ack0:
	ip->ioc_count = 0;	/* no return data */
ack:
	MTYPE(bp) = M_IOCACK;
	qreply(q, bp);
	return;
nack:
	MTYPE(bp) = M_IOCNAK;
	ip->ioc_count = 0;
	qreply(q, bp);
	return;

}

/*
 * Initialize an interface's internet address
 * and routing table entry.
 */
ip_ifinit(ifp, ia, sin, chan)
	register struct ifnet *ifp;
	register struct in_ifaddr *ia;
	struct sockaddr_in *sin;
{
	register u_long i = ntohl(sin->sin_addr.s_addr);
	struct sockaddr oldaddr;
	struct sockaddr_in netaddr;
	int s = splni();

	oldaddr = ia->ia_addr;
	ia->ia_addr = *(struct sockaddr *)sin;

	/*
	 * Delete any previous route for an old address.
	 */
	bzero((caddr_t)&netaddr, sizeof (netaddr));
	netaddr.sin_family = AF_INET;
	if (ia->ia_flags & IFA_ROUTE) {
		if (ifp->if_flags & IFF_LOOPBACK)
		     ip_rtinit(&oldaddr, &oldaddr, IPIOC_DELETEROUTE,
			RTF_HOST, RTR_OTHER, 0);
		else if (ifp->if_flags & IFF_POINTOPOINT)
		     ip_rtinit(&ia->ia_dstaddr, &oldaddr, IPIOC_DELETEROUTE,
			RTF_HOST, RTR_OTHER, 0);
		else { /* Broadcast and IPTOX25 */
			netaddr.sin_addr = in_makeaddr(ia->ia_subnet,
			    INADDR_ANY);
			ip_rtinit((struct sockaddr *)&netaddr, &oldaddr,
			    IPIOC_DELETEROUTE, 0, RTR_OTHER, 0);
		}
		ia->ia_flags &= ~IFA_ROUTE;
	}

	if (IN_CLASSA(i))
		ia->ia_netmask = IN_CLASSA_NET;
	else if (IN_CLASSB(i))
		ia->ia_netmask = IN_CLASSB_NET;
	else
		ia->ia_netmask = IN_CLASSC_NET;
	ia->ia_net = i & ia->ia_netmask;

	/*
	 * The subnet mask includes at least the standard network part,
	 * but may already have been set to a larger value.
	 */
	ia->ia_subnetmask |= ia->ia_netmask;
	ia->ia_subnet = i & ia->ia_subnetmask;
	if (ifp->if_flags & IFF_BROADCAST) {
		ia->ia_broadaddr.sa_family = AF_INET;
		satosin(&ia->ia_broadaddr)->sin_addr = 
			in_makeaddr(ia->ia_subnet, INADDR_BROADCAST);
		ia->ia_netbroadcast.s_addr =
		    htonl(ia->ia_net | (INADDR_BROADCAST & ~ia->ia_netmask));
	}

	/*
	 * Add route for the network.
	 */
	if (ifp->if_flags & IFF_LOOPBACK)
		ip_rtinit(&ia->ia_addr, &ia->ia_addr, (int)IPIOC_ADDROUTE,
		    RTF_HOST|RTF_UP, RTR_OTHER, 0);
	else if (ifp->if_flags & IFF_POINTOPOINT)
		ip_rtinit(&ia->ia_dstaddr, &ia->ia_addr, (int)IPIOC_ADDROUTE,
		    RTF_HOST|RTF_UP, RTR_OTHER, 0);
	else if (ifp->if_flags & IFF_IPTOX25) {
		/* The ACC card binds on the IP address (very strange!!!) */
		if (ifp->ib_bottype == DL_IPX25)
		{
		    if (ifp->if_flags & IFF_BINDNEED) {
			ifp->if_flags &= ~IFF_BINDNEED;
			ifp->ib_mstate = DL_WACK_B;
			if (dl_bind(ifp->ib_wrq, (long)satosaddr(ia->ia_addr)))
				goto nackit;
		    }
		}
		netaddr.sin_addr = in_makeaddr(ia->ia_subnet, INADDR_ANY);
		ip_rtinit((struct sockaddr *)&netaddr, &ia->ia_addr,
		    IPIOC_ADDROUTE, RTF_UP, RTR_OTHER, 0);
	} else { /* Broadcast */
		netaddr.sin_addr = in_makeaddr(ia->ia_subnet, INADDR_ANY);
		ip_rtinit((struct sockaddr *)&netaddr, &ia->ia_addr,
		    IPIOC_ADDROUTE, RTF_UP, RTR_OTHER, 0);
		STRLOG(IP_ID, chan, DPRI_HI, SL_TRACE,
		    "ip: add route %x:%x", satosaddr(netaddr),
		    satosaddr(ia->ia_addr));
	} 
	ia->ia_flags |= IFA_ROUTE;
	STRLOG(IP_ID, chan, DPRI_HI, SL_TRACE,
	    "ip: ip_ifinit: ifp = %x if_flags = %d", ifp, ifp->if_flags);
	(void) splx(s);
	return (0);

nackit:
	(void) splx(s);
	return (-1);
}	/*  End of ip_ifinit()  */

/*
 * Process IP set/get options which came down via the above protocol.
 */
ipfcmd
ip_sockopt(bp, chan)
	register mblk_t *bp;
	register int chan;
{
	register mblk_t *nbp = bp->b_cont;
	register struct winsopt *winsopt = (struct winsopt *)nbp->b_rptr;
	ipfcmd result;

	STRLOG(IP_ID, chan, DPRI_LO, SL_TRACE, "ip_sockopt: OPTIONS");
	
	nbp->b_rptr += sizeof(struct winsopt);

	result = (ipfcmd)ip_ctloutput(winsopt->flags, winsopt->level,
						winsopt->name, nbp);

	if (winsopt->flags == T_NEGOTIATE) {
		bp->b_cont = nbp->b_cont; /* options block pointer */
		nbp->b_cont = NULL;
		freemsg(nbp);
	}

	return result;

}	/*  End of ip_sockopt()  */

/*
 * Send back a hangup condition to the IFF_IPTOVS control stream for the
 * specific interface.
 */
ipvcshangup(q, ifp)
	register queue_t *q;
	register struct ifnet *ifp;
{
	register s;
	register mblk_t *bp;
	register struct ipmgr *ipm;

	STRLOG(IP_ID, IPBCHAN(ifp), DPRI_MED, SL_TRACE,
	    "ipvcshangup: ifp = %x", ifp);
	s = splstr();
	if (ifp->ib_qtop) {
		flushq(q, FLUSHALL);
		flushq(RD(q), FLUSHALL);
		if (bp = allocb(sizeof(struct ipmgr), BPRI_HI)) {
			MTYPE(bp) = M_PROTO;
			bp->b_wptr += sizeof(struct ipmgr);
			ipm = (struct ipmgr *) bp->b_rptr;
			ipm->ipm_code = IPM_HANGUP;
			ipm->ipm_error = 0;
			bcopy(ifp->if_name, ipm->ipm_ifname, IFNAMESIZ);
			putnext(RD(ifp->ib_qtop), bp);
		}
	}
	(void) splx(s);
}

/* end */
