/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)ip_output.c (TWG)  1.4     89/07/30 "

/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *	@(#)ip_output.c	7.10 (Berkeley) 4/7/88
 */

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/strlog.h"
#include "sys/errno.h"
#include "sys/tiuser.h"

#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/inetioctl.h"
#include "sys/route.h"

mblk_t *add_ipopt_security();
u_char *check_opt();
mblk_t *ip_insertoptions();
mblk_t *bp_copy();
int ipsend();

/*
 * IP output.  The packet in mbuf chain m contains a skeletal IP
 * header (with len, off, ttl, proto, tos, src, dst).
 * The mbuf chain containing the packet will be freed.
 * The mbuf opt, if present, will not be freed.
 */
ip_output(m0, opt, flags)
	mblk_t *m0;
	mblk_t *opt;
	int flags;
{
	register struct ip *ip, *mhip;
	register struct ifnet *ifp;
	register mblk_t *m = m0;
	register int hlen = sizeof (struct ip);
	int len, off, error = 0, sendflags = 0;
	struct sockaddr_in dst;
	struct route_dst rtd;
	extern u_char my_hosts_class, ip_ttl;

	/*
	 * check for word alignment and, pullup the entire IP header
	 * if not all in first buffer or not aligned.
	 * Note: Max IP header size is 64 bytes.
	 */
	if (((int)m->b_rptr & 03) && (pullupmsg(m, 64) == 0)) {
enobufs:
		STRlog(IP_ID, -1, DPRI_LO, SL_te, "ip_output: NOBUFFS");
		ipstat.ipOutDiscards++;
		freemsg(m);
		return(ENOBUFS);
	}

	/*
	 * NOTE: the IP Security option MUST be the first option, if more
	 * than one option is used, in the IP header.  This requirement is
	 * stated in the Blacker Interface Control Document (ICD) dated
	 * 22 July 1985, section 2.4.5.
	 * If my_hosts_security is 0, running without security.
	 * If forwarding, packet already has valid security and options
	 */
	if (my_hosts_class && !(flags & IP_FORWARDING)) {
		if ((m = add_ipopt_security(m)) == 0) {
			STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			    "ip_output: security error %x", error, 0);
			if (opt)
				freemsg(opt);
			goto enobufs;
		}
	}

	/*
	 * Process normal options
	 */
	if (opt) {
		m = ip_insertoptions(m, opt, &len);
		hlen = len;
	}

	/*
	 * Fill in IP header.
	 */
	ip = mtod(m, struct ip *);
	if ((flags & IP_FORWARDING) == 0) {
		ip->ip_v = IPVERSION;
		ip->ip_off &= IP_DF;
		ip->ip_id = htons(ip_id++);
		ip->ip_hl = hlen >> 2;
		ip->ip_ttl = ip_ttl;
		/*
		 * no need to check for routing if not more 44 bytes of IP
		 * most systems will run security only
		 */
		if (hlen > (sizeof (struct ip) + IPOPT_SECUR_LENGTH)) {
			/*
			 * If we are Strict or Loose IP routing
			 * First hop destination becomes ip->ip_dst
			 * move all other addresses up one address
			 * install final destination at end of list
			 */
			u_char *optptr;
			if ((optptr = check_opt(ip+1,IPOPT_SSRR)) == NULL)
				optptr = check_opt(ip+1,IPOPT_LSRR);
			if (optptr) {
				register u_char *tmp = optptr +
						optptr[IPOPT_OFFSET]-1;
				u_long dest;
#ifdef DEBUG_TRACE
				extern int _trmask;
				if ( _trmask & (1<<10) ) {
					for(off = 0;off < optptr[IPOPT_OLEN];off++)
						printf("%u,",optptr[off]);
					printf("\n");
				}
#endif /* DEBUG_TRACE */
				off = 4 + IPOPT_MINOFF - 1;

				dest = ip->ip_dst.s_addr;

/*PBJ*/				if (optptr[IPOPT_OFFSET] < optptr[IPOPT_OLEN])
					bcopy((caddr_t)tmp,(caddr_t)&ip->ip_dst,
						sizeof(struct in_addr));

				if (off < optptr[IPOPT_OLEN])
					bcopy(tmp + sizeof(struct in_addr), tmp,
						optptr[IPOPT_OLEN] - off);

				tmp += (optptr[IPOPT_OLEN] - off);

/*PBJ*/				if (optptr[IPOPT_OFFSET] < optptr[IPOPT_OLEN])
					bcopy(&dest,tmp,sizeof(struct in_addr));
			}
		}
	} else
		hlen = ip->ip_hl << 2;

	/*
	 * Route packet.
	 */
	dst.sin_family = AF_INET;
	dst.sin_addr = rtd.rtd_dst = ip->ip_dst;

	/*
	 * If routing to interface only,
	 * short circuit routing lookup.
	 */
	if (flags & IP_ROUTETOIF) {
		struct in_ifaddr *ia;

		ia = (struct in_ifaddr *)ifa_ifwithdstaddr((struct sockaddr *)&dst);
		if (ia == 0)
			ia = in_iaonnetof(in_netof(ip->ip_dst));
		if (ia == 0) {
			error = ENETUNREACH;
			ipstat.ipOutNoRoutes++;
			goto bad;
		}
		ifp = ia->ia_ifp;
	} else {
		if (rtlookup(&rtd, 1, ip->ip_tos) <= 0) {
			if (in_localaddr(ip->ip_dst))
				error = EHOSTUNREACH;
			else
				error = ENETUNREACH;
			STRlog(IP_ID, -1, DPRI_LO, SL_te,
				"rtlookup failed for %x", ip->ip_dst);
			ipstat.ipOutNoRoutes++;
			goto bad;
		}
		dst.sin_addr = rtd.rtd_dst;
		ifp = rtd.rtd_ifp;
	}

	/*
	 * If the source address is zero, then fill it in.  This is used
         * by upper level modules that don't want to do the IP_GETMYADDR stuff.
	 */
	if (ip->ip_src.s_addr == 0) {
		struct in_ifaddr *ia, *ifptoia();

		ia = ifptoia(ifp);
		ip->ip_src.s_addr = satosaddr(ia->ia_addr);
	}

	/*
	 * Look for broadcast address and
	 * and verify user is allowed to send
	 * such a packet.
	 */
	if (in_broadcast(dst.sin_addr)) {
		if ((ifp->if_flags & IFF_BROADCAST) == 0) {
			error = EADDRNOTAVAIL;
			STRlog(IP_ID, -1, DPRI_LO, SL_te,
			    "ip_ouput: Can't broadcast to %x",
			    dst.sin_addr.s_addr);
			goto bad;
		}
#ifdef notdef
		if ((flags & IP_ALLOWBROADCAST) == 0) {
			error = EACCES;
			goto bad;
		}
#endif
		/* don't allow broadcast messages to be fragmented */
		if (ip->ip_len > ifp->if_mtu) {
			error = EMSGSIZE;
			goto bad;
		}
		sendflags |= IFF_BROADCAST;
	}

	/*
	 * If small enough for interface, can just send directly.
	 */
	if (ip->ip_len <= ifp->if_mtu) {
		ip->ip_len = htons((u_short)ip->ip_len);
		ip->ip_off = htons((u_short)ip->ip_off);
		ip->ip_sum = 0;
#ifdef i386
		ip->ip_sum = ip_cksum(ip, hlen);
#else
		ip->ip_sum = in_cksum(m, hlen);
#endif
		STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
			"ip_output: ip_dst %x len(%d)",
			ntohl(ip->ip_dst.s_addr), ntohs(ip->ip_len));
		error = ipsend(ifp, m, &dst, ip->ip_len, sendflags);
		goto done;
	}

	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		 "ip_output: fragment len %d", ip->ip_len);

	/*
	 * Too large for interface; fragment if possible.
	 * Must be able to put at least 8 bytes per fragment.
	 */
	if (ip->ip_off & IP_DF) {
		error = EMSGSIZE;
		goto bad;
	}
	if (in_localaddr(dst.sin_addr))
		len = (ifp->if_mtu - hlen) &~ 7;
	else
		len = (IP_MSS - hlen) &~ 7;
	if (len < 8) {
		error = EMSGSIZE;
		ipstat.ipFragFails++;
		goto bad;
	}

    {
	int mhlen, firstlen = len;
	mblk_t **mnext = &m->b_next;

	/*
	 * Loop through length of segment after first fragment,
	 * make new header and copy data of each part and link onto chain.
	 */
	m0 = m;
	for (off = hlen + len; off < ip->ip_len; off += len) {
		mhlen = sizeof (struct ip);
		m = allocb(hlen, BPRI_MED);
		if (m == 0) {
			ipstat.ipFragFails++;
			STRlog(IP_ID, -1, DPRI_LO, SL_te,
				"ip_output: NOBUFFS");
			error = ENOBUFS;
			ipstat.ipOutDiscards++;
			goto bad;
		}
		mhip = mtod(m, struct ip *);
		*mhip = *ip;
		if (hlen > sizeof (struct ip)) {
			mhlen = ip_optcopy(ip, mhip) + sizeof (struct ip);
			mhip->ip_hl = mhlen >> 2;
		}
		m->b_wptr += mhlen;
		mhip->ip_off = ((off - hlen) >> 3) + (ip->ip_off & ~IP_MF);
		if (ip->ip_off & IP_MF)
			mhip->ip_off |= IP_MF;
		if (off + len >= ip->ip_len)
			len = ip->ip_len - off;
		else
			mhip->ip_off |= IP_MF;
		mhip->ip_len = htons((u_short)(len + mhlen));
		m->b_cont = bp_copy(m0, off, len);
		if (m->b_cont == 0) {
			freemsg(m);
			error = ENOBUFS;
			goto sendorfree;
		}
		mhip->ip_off = htons((u_short)mhip->ip_off);
		mhip->ip_sum = 0;
#ifdef i386
		mhip->ip_sum = ip_cksum(mhip, mhlen);
#else
		mhip->ip_sum = in_cksum(m, mhlen);
#endif
		*mnext = m;
		mnext = &m->b_next;
	}
	/*
	 * Update first fragment by trimming what's been copied out
	 * and updating header, then send each fragment (in order).
	 */
	(void) ipadjmsg(m0, hlen + firstlen - ip->ip_len);
	ip->ip_len = htons((u_short)(hlen + firstlen));
	ip->ip_off = htons((u_short)(ip->ip_off | IP_MF));
	ip->ip_sum = 0;
#ifdef i386
	ip->ip_sum = ip_cksum(ip, hlen);
#else
	ip->ip_sum = in_cksum(m0, hlen);
#endif
sendorfree:
	for (m = m0; m; m = m0) {
		m0 = m->b_next;
		m->b_next = 0;
		ip = mtod(m, struct ip *);
		if (error == 0) {
			ipstat.ipFragCreates++;
			error = ipsend(ifp, m, &dst, ip->ip_len, sendflags);
		} else {
			STRlog(IP_ID, -1, DPRI_LO, SL_te,
				"ip_output: NOBUFFS");
			error = ENOBUFS;
			ipstat.ipOutDiscards++;
			freemsg(m);
		}
	}
    }
	/*
	 * fragmentation is done, release original packet
	 */
    	if (error == 0)
		ipstat.ipFragOKs++;
	goto done;
bad:
	STRLOG(IP_ID, -1, DPRI_LO, SL_TRACE,
		 "ip_output: dropped error %x", error);
	freemsg(m0);
done:
	return (error);
}

/*
 * Insert IP options into preformed packet.
 * Adjust IP destination as required for IP source routing,
 * as indicated by a non-zero in_addr at the start of the options.
 */
mblk_t *
ip_insertoptions(m, opt, phlen)
	register mblk_t *m;
	mblk_t *opt;
	int *phlen;
{
	register struct ipoption *p = mtod(opt, struct ipoption *);
	mblk_t *n;
	register struct ip *ip = mtod(m, struct ip *);
	unsigned optlen;

	optlen = MLEN(opt) - sizeof(p->ipopt_dst);
	if (p->ipopt_dst.s_addr)
		ip->ip_dst = p->ipopt_dst;
	if ((m->b_datap->db_ref > 1) ||
	    ((m->b_rptr - m->b_datap->db_base) < optlen)) {
		n = allocb(sizeof(struct ip) + optlen, BPRI_MED);
		if (n == 0)
			return (m);
		m->b_rptr += sizeof(struct ip);
		n->b_cont = m;
		n->b_datap->db_type = m->b_datap->db_type;
		m = n;
		m->b_wptr += optlen + sizeof(struct ip);
		bcopy((caddr_t)ip, mtod(m, caddr_t), sizeof(struct ip));
	} else {
		m->b_rptr -= optlen;
		ovbcopy((caddr_t)ip, mtod(m, caddr_t), sizeof(struct ip));
	}
	ip = mtod(m, struct ip *);
	bcopy((caddr_t)p->ipopt_list, (caddr_t)(ip + 1), (unsigned)optlen);
	*phlen = sizeof(struct ip) + optlen;
	ip->ip_len += optlen;
	return (m);
}

/*
 * Copy options from ip to jp,
 * omitting those not copied during fragmentation.
 */
ip_optcopy(ip, jp)
	struct ip *ip, *jp;
{
	register u_char *cp, *dp;
	int opt, optlen, cnt;

	cp = (u_char *)(ip + 1);
	dp = (u_char *)(jp + 1);
	cnt = (ip->ip_hl << 2) - sizeof (struct ip);
	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[0];
		if (opt == IPOPT_EOL)
			break;
		if (opt == IPOPT_NOP)
			optlen = 1;
		else
			optlen = cp[IPOPT_OLEN];
		/* bogus lengths should have been caught by ip_dooptions */
		if (optlen > cnt)
			optlen = cnt;
		if (IPOPT_COPIED(opt)) {
			bcopy((caddr_t)cp, (caddr_t)dp, (unsigned)optlen);
			dp += optlen;
		}
	}
	for (optlen = dp - (u_char *)(jp+1); optlen & 0x3; optlen++)
		*dp++ = IPOPT_EOL;
	return (optlen);
}
