/*
 * @(#) Copyright 1988.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)raw.c (TWG)  1.12     89/09/18 "

/*
 * raw tli multiplexer
 */

#include "sys/param.h"
#include "sys/types.h"
#include "sys/inline.h"
#include "sys/errno.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/tiuser.h"
#include "sys/strlog.h"
#ifdef u3b2
#include "sys/psw.h"
#include "sys/pcb.h"
#include "sys/immu.h"
#include "sys/region.h"
#include "sys/proc.h"
#endif
#ifndef	XENIX
#include "sys/fs/s5dir.h"
#else
#include "sys/dir.h"
#endif
#include "sys/signal.h"
#include "sys/file.h"
#include "sys/user.h"
#include "sys/debug.h"

#include "sys/inet.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/somod.h"
#include "sys/inetioctl.h"
#include "sys/socket.h"
#include "sys/raw.h"


struct rawd *rawd_head = (struct rawd *)NULL;	
struct rawb rawb;		/* Bottom Queue structure - just one */

/* this is for debugging */
struct rstat {
	int rs_allocb_ir;	/* T_info_req allocb failure */
	int rs_allocb_ba;	/* T_bind_ack allocb failure */
	int rs_allocb_ub;	/* T_unbind_req allocb failure */
	int rs_allocb_hdr;	/* raw header allocb failure */
	int rs_allocb_arg;	/* IP "args" allocb failure */
	int rs_allocb_udi;	/* T_uderror_ind allocb failure */
	int rs_allocb_mux;	/* mux notify allocb failure */
	int rs_allocb_my;	/* getmyaddr allocb failure */
	int rs_allocb_ipopt;	/* IP options allocb failure */
	int rs_raw_output;	/* raw_ouput() failure */
	int rs_raw_operr;	/* error # for last raw_output() failure */
} rstat;

struct rawstat rawstat;

#define	WQREPLY(q, bp)	{q = RD(q); putnext(q, bp);}

/* for these see master.d file */
extern struct rawd rawd[];	/* Top Queue Structure - one per minor device */
extern int rawd_cnt;		/* number of allowed minor devices	      */
extern mblk_t *rawmsg[];   	/* used by raw_input() to point to copied
				 * or duplicated messages.
				 */
#ifdef DEBUG
mblk_t *raw_message;
#endif


/***********************
 *  Table of Contents  *
 **********************/

/* XXXX - may make most of thses static later */

int	rawopen();		/* top QUEUE function */
int	rawclose();		/* top QUEUE function */
void	_rawclose();
int	rawisrv();		/* top QUEUE function */
int	rawoput();		/* top QUEUE function */
int	rawosrv();		/* top QUEUE function */
int	raw_output();
void	rawmuxnotify();
int	rawbiput();		/* bottom QUEUE function */
void	rawmuxack();
int	rawbisrv();		/* bottom QUEUE function */
void	raw_input();
long	raw_bind();
void	raw_getmyaddr();
int	raw_sockopt();
int	raw_ipoptions();

/***** End of TOC *****/


static struct module_info 	raw_rinfo = {	/* Top Read info */
RAW_ID, "raw", 0, INFPSZ, 8192, 6144 };


static struct module_info 	raw_winfo = {	/* Top Write info */
RAW_ID, "raw", 0, INFPSZ, 8192, 6144 };


static struct module_info 	rawb_rinfo = {	/* Bottom Read info */
RAW_ID, "raw", 0, INFPSZ, 8192, 6144 };


static struct module_info 	rawb_winfo = {	/* Bottom Write info */
RAW_ID, "raw", 0, INFPSZ, 0, 0 };


static struct qinit 		rawrinit = { 	/* Top Read QUEUE */
NULL, rawisrv, rawopen, rawclose, NULL, &raw_rinfo, NULL };


static struct qinit 		rawwinit = { 	/* Top Write QUEUE */
rawoput, rawosrv, NULL, NULL, NULL, &raw_winfo, NULL };


static struct qinit 		rawbrinit = { 	/* Bottom Read QUEUE */
rawbiput, rawbisrv, NULL, NULL, NULL, &rawb_rinfo, NULL };


static struct qinit 		rawbwinit = { 	/* Bottom Write QUEUE */
NULL, NULL, NULL, NULL, NULL, &rawb_winfo, NULL };


struct streamtab 		rawinfo = {	/* Link to cdevsw table */ 
&rawrinit, &rawwinit, &rawbrinit, &rawbwinit };



/*
 * may have to open control channel first
 * if dev==CLONEOPEN, then select any free channel to open
 */

int
rawopen(q, dev, flag, sflag)

	register queue_t *q;
	dev_t dev;
{
	register struct rawd *raw;
	int mdev = minor(dev);

	ASSERT(q);

	/* we only want super user to use raw driver */
	if (u.u_uid != 0) {
		u.u_error = EPERM;
		return OPENFAIL;
	}

	if (sflag == CLONEOPEN) {

		/*  Look for first available minor device */
		for (raw = &rawd[0]; raw < &rawd[rawd_cnt]; raw++) {
		        if (raw->rawd_rdq == (queue_t *)NULL)
				break;
		}

		/*  Set and test validity of selected minor device */
		if ((mdev = (raw - &rawd[0])) >= rawd_cnt) {
			u.u_error = ENODEV;
			return OPENFAIL;
		}
	/* NOT a CLONEOPEN */
	} else {
		/*  Requested minor dev not already open */
		if (mdev < 0 || mdev >= rawd_cnt) {
			u.u_error = ENODEV;
			return OPENFAIL;
		}
		/* Allocate a minor device and continue */
		raw = &rawd[mdev];
	}

	/* Requested minor dev already open */
	if (raw == (struct rawd *)q->q_ptr) {
		ASSERT(raw->rawd_dev == mdev);
		return mdev;
	}

	/* initialize data structure */
	raw->rawd_rdq = q;
	raw->rawd_dev = mdev;
	raw->rawd_tstate = TS_UNBND;
	raw->rawd_flags = RAW_SUPERUSER; /* | SO_DONTROUTE; XXX set this too? */
	raw->rawd_next = rawd_head;
	raw->rawd_proto.sp_family = AF_INET;	   /* default address family */
	raw->rawd_proto.sp_protocol = IPPROTO_RAW; /* default protocol */
	raw->rawd_ipopt = NULL;		/* pointer to outgoing IP options */
	raw->rawd_ipoptin = NULL;	/* pointer to incoming IP options */
	raw->rawd_ttl = RAW_TTL;	/* default Time-to-Live */
	raw->rawd_tos = 0;		/* default Type-of-Service */
	rawd_head = raw;

	q->q_ptr = (caddr_t)raw;	/* set read queue's private pointer */
	WR(q)->q_ptr = (caddr_t)raw;	/* set write queue's private pointer */
	noenable(WR(q));
	STRlog(RAW_ID, mdev, DPRI_LO, SL_TRACE, "rawopen: open chan %x\n",mdev);
	return mdev;

}	/*  End of rawopen()  */


/*
 * if closing control channel, close all other chans and clean up mux
 */

int
rawclose(q)

	queue_t *q;
{
	register struct rawd *raw, *p;
	register struct rawb *rawbp = (struct rawb *)&rawb;

	ASSERT(q && q->q_ptr);

	raw = (struct rawd *)q->q_ptr;
	if (RAWCHAN(raw) == RAWCTLCHAN) {
		for (p = &rawd[0]; p < &rawd[rawd_cnt]; p++)
			if (p != raw && p->rawd_rdq)
				_rawclose(p);
		if (rawbp->rawb_wrq) {
			rawbp->rawb_mstate = IPIF_CLOSED;
			rawmuxnotify(rawbp);
			rawbp->rawb_wrq = NULL;
			if (rawbp->rawb_iocblk) {
				/* stream head timeout, don't care iocnak */
				freeb(rawbp->rawb_iocblk);
				rawbp->rawb_iocblk = NULL;
			}
		}
	}
	_rawclose(raw);

}	/*  End of rawclose()  */


/*
 * no need to send "disconnect" to user
 */

void
_rawclose(raw)

	struct rawd *raw;
{
	register struct rawd **p;

	STRlog(RAW_ID, RAWCHAN(raw), DPRI_LO, SL_TRACE,
		 "rawclose: %x\n", RAWCHAN(raw), 0);

	for (p = &rawd_head; *p; p = &((*p)->rawd_next)) {

		if (*p == raw) {
			*p = raw->rawd_next;
			break;
		}
	}

	if (raw->rawd_ipopt)
		freeb(raw->rawd_ipopt);

	if (raw->rawd_ipoptin)
		freeb(raw->rawd_ipoptin);

	bzero((char *)raw, sizeof(struct rawd));

}	/*  End of _rawclose()  */


/*
 * called from raw_input()
 * if canput, just pass the data upward
 * else do nothing (mesg is left in the current q)
 */

int
rawisrv(q)

	register queue_t *q;
{
	register struct rawd *raw;
	register mblk_t *bp;
	register int chan;

	raw = (struct rawd *)q->q_ptr;
	ASSERT(raw);
	if (raw->rawd_tstate != TS_IDLE) {
		chan = RAWCHAN(raw);
		flushq(q, 1);
		STRlog(RAW_ID, chan, DPRI_MED, SL_te,
			"rawisrv: Not idle, chan %x, state %x\n", 
			chan, raw->rawd_tstate);
		return;
	}

	while (canput(q->q_next) && (bp = getq(q))) {
#ifdef DEBUG
raw_message = bp;
LABEL(rawISRV);
#endif
		putnext(q, bp); /* send message upstream */
	}

}	/*  End of rawisrv()  */


static struct T_info_ack RAW_info_ack =
{
	T_INFO_ACK,		/* type */
	INFPSZ,			/* TSDU size */
	-2,			/* ETSDU size */
	0,			/* connect data size */
	0,			/* disconnect data size */
	RAW_ADDRLEN,		/* prot address size */
	0,			/* option size */
	65535,			/* TIDU size */
	T_CLTS,			/* orderly release support */
	TS_IDLE,		/* default current state */
};


/*
 * rawoput(q, bp) is the put procedure called by the higher level
 * modules (Closer to the user). 
 *
 * This routine determines the type of buffer passed to it.
 * The buffer types passed are typically of type M_PROTO
 * or M_PCPROTO. 
 * If the protocol message is any other than a unit data request, 
 * T_UNITDATA_REQ, it is processed immediately.
 *
 * The IOCTL I_LINK is used to link a driver beneath RAW,
 * this would typically be IP.  Only one driver is currently
 * allowed to be linked.  This is because of the fact that RAW
 * has intrinsic knowledge that IP exists beneath it.
 *
 * The IOCTL I_UNLINK is used to unlink a driver from beneath this
 * module.
 *
 * IOCTL's that are not recognized by this module are
 * not passed down to the driver beneath it. This would be a
 * breach in security.
 *
 * Protocol messages of the type T_UNITDATA_REQ are put on the
 * queue via the putq routine. 
 */

int
rawoput(q, bp)

	register queue_t *q;
	register mblk_t *bp;
{
	register union T_primitives *Tp;
	register struct rawd *raw;
	register unchar Ttype;
	register mblk_t *bp1;
	struct iocblk *ip;
	struct rawb *rawbp;
	int cmd;
	queue_t *bq;
	struct linkblk *lp;
	struct T_info_ack *Ti;
	int err_no = EPROTO;

	ASSERT(q && q->q_ptr && bp);

	raw = (struct rawd *)q->q_ptr;

	switch(MTYPE(bp)) {

	case M_PROTO:
	case M_PCPROTO:
		Tp = (union T_primitives *)bp->b_rptr;
		Ttype = (unchar)TTYPE(Tp);

		switch(Ttype) {

		case T_INFO_REQ:
			if (!REUSEABLE(bp, sizeof(struct T_info_ack))) {
				freemsg(bp);
				if (!(bp = allocb(sizeof(*Ti), BPRI_MED))) {
					err_no = ENOBUFS;
					rstat.rs_allocb_ir++;
					goto fatal;
				}
			}

			MTYPE(bp) = M_PCPROTO;
			Ti = (struct T_info_ack *)bp->b_wptr;
			*Ti = RAW_info_ack;
			Ti->CURRENT_state = raw->rawd_tstate;
			bp->b_wptr += sizeof(struct T_info_ack);
			putnext((RD(q)), bp);
			break;

		case T_UNITDATA_REQ:
			if (raw->rawd_tstate == TS_IDLE) {
#ifdef DEBUG
LABEL(rawDATAREQ);
#endif
				putq(q, bp); /* rawosrv() will be scheduled */
				qenable(q);
			} else
				freemsg(bp);
			break;

		case T_BIND_REQ:
		{
			struct T_bind_req *Tr;
			struct T_bind_ack *Ta;
			long retcode, raw_bind();
			struct tsap Tsap;

			if (raw->rawd_tstate != TS_UNBND)
				goto fatal;

			Tr = (struct T_bind_req *)Tp;

			if (BLEN(bp) < Tr->ADDR_offset + Tr->ADDR_length) {
				retcode = TBADADDR;
				goto nonfatal;
			}

			switch(Tr->ADDR_length) {

			case RAW_ADDRLEN:
				/* copy: because data not aligned */
				bcopy((char *)(bp->b_rptr+Tr->ADDR_offset),
					(char *)&Tsap, BINDLEN);
				break;

			case 0:
				Tsap.ta_addr = INADDR_ANY;
				Tsap.ta_port = 0;
				break;

			default:
				retcode = TBADADDR;
				goto nonfatal;
			}

			if ((retcode = raw_bind(&Tsap, raw->rawd_flags)) != 0)
				goto nonfatal;

			raw->rawd_la = Tsap;

			/* success, return T_bind_ack */
			if (!REUSEABLE(bp, sizeof(*Ta)+ RAW_ADDRLEN)) {
				freemsg(bp);
				if (!(bp = allocb(sizeof(*Ta)+ RAW_ADDRLEN,
					BPRI_LO))) {
					err_no = ENOBUFS;
					rstat.rs_allocb_ba++;
					goto fatal;
				}
			}

			Ta = (struct T_bind_ack *)bp->b_rptr;
			Ta->PRIM_type = T_BIND_ACK;
			Ta->ADDR_length = RAW_ADDRLEN;
			Ta->ADDR_offset = sizeof(struct T_bind_ack);
			Ta->CONIND_number = 0;
			bcopy((char *)&Tsap, (char *)(bp->b_rptr +
			   Ta->ADDR_offset), BINDLEN);
			MTYPE(bp) = M_PCPROTO;
			bp->b_wptr = bp->b_rptr + sizeof(struct T_bind_ack) 
				+ BINDLEN;
			/* zero out the last 8 bytes in sockaddr_in */
			bzero((char *)bp->b_wptr, 8);
			bp->b_wptr += 8;

			if (raw->rawd_laddr == INADDR_ANY) {
				/*
				 * If the local address is INADDR_ANY we
				 * need RAW_LADDR flag to be clear or else
				 * raw_input() won't send anything up our
				 * stream.
				 */
				raw->rawd_flags &= ~RAW_LADDR;
			} else {
				raw->rawd_flags |= RAW_LADDR;
			}

			raw->rawd_tstate = TS_IDLE;
			putnext((RD(q)), bp);
			break;

nonfatal:
			STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
				"rawoput: bind nonfatal retcode %x\n",
				retcode, 0);
			if (raw_error_ack(q, bp, T_BIND_REQ, retcode, 
			    TS_NOCHANGE)) {
					err_no = ENOBUFS;
					goto fatal;
			}
			break;

			}

		case T_UNBIND_REQ:
			if (upflush(q) == -1 || raw->rawd_tstate != TS_IDLE) {
fatal:
				if (bp1 = allocb(1, BPRI_HI)) {
					MTYPE(bp1) = M_ERROR;
					*bp1->b_wptr++ = err_no;
					putnext((RD(q)), bp1);
				} else {
					rstat.rs_allocb_ub++;
				}
				raw->rawd_tstate = TS_UNBND;
				raw->rawd_flags = 0;
				freemsg(bp);
				STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te, 
					"raw fatal unbind %x\n",
					raw->rawd_tstate, 0);
				break;
			}
			if (raw_ok_ack(q, bp, T_UNBIND_REQ, TS_UNBND) == 0)
				raw->rawd_tstate = TS_UNBND;
			break;

		default:
			STRlog(RAW_ID, RAWCHAN(raw), DPRI_LO, SL_te,
				"rawoput: unknown PROTO %x\n", TTYPE(Tp), 0);
			freemsg(bp);
			break;
		}
		break;	/* end of cases M_PROTO and M_PCPROTO */

	case M_IOCTL:
		ip = (struct iocblk *)bp->b_rptr;
		cmd = ip->ioc_cmd;

		rawbp = (struct rawb *)&rawb;

		switch(cmd) {

		case I_LINK:
			/*
			 * there is only one bottom q, i.e. for ip
			 */
			if ((raw->rawd_flags & RAW_SUPERUSER) == 0){
				ip->ioc_error = EPERM;
				goto nack;
			}
			lp = (struct linkblk *)bp->b_cont->b_rptr;
			if (rawbp->rawb_wrq) {
				STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
					"rawoput: link to ip again\n", 0, 0);
nack:
				if (ip->ioc_error == 0)
					ip->ioc_error = EINVAL;
				ip->ioc_count = 0;
				MTYPE(bp) = M_IOCNAK;
				iocreply(q, bp);
				break;
			}
			if (rawbp->rawb_iocblk) {
				STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
				 "raw bottom busy..\n", 0, 0);
				ip->ioc_error = EAGAIN;
				goto nack;
			}
			bq = lp->l_qbot;
			bq->q_ptr = (RD(bq))->q_ptr = (caddr_t)rawbp;
			rawbp->rawb_wrq = bq;
			rawbp->rawb_qtop = /* lp->l_qtop; */ q;
			rawbp->rawb_cookie = lp->l_index;
			rawbp->rawb_mstate = IPIF_LINKING;
			/*
			 * keep the M_IOCTL block and change to M_IOCACK
			 * when the M_CTL is acked
			 * actually the only thing needed to be saved is
			 * ioc_id. Saving the block to save time (don't need
			 * freemsg here and allocb later).
			 * Remember to free it when the chan is closing
			 */
			rawbp->rawb_iocblk = bp;
			rawmuxnotify(rawbp);
			break;

		case I_UNLINK:
			if (rawbp->rawb_wrq == NULL) {
				STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
				"rawoput: ioctl unlink what?\n", 0, 0);
				goto nack;
			}
			rawbp->rawb_mstate = IPIF_UNLINKING;
			rawbp->rawb_iocblk = bp;
			rawmuxnotify(rawbp);
			rawbp->rawb_wrq = NULL;
			break;

		case RAWIOC_GETMYSTATE:

			/* find out what TLI state we are */

			if ((bp->b_cont == NULL) ||
					(ip->ioc_count < sizeof(char)))
				goto nack;

			bp1 = bp->b_cont;

			*(mtod (bp1, char *)) = raw->rawd_tstate;
			ip->ioc_count = sizeof (char);
ack:
			ip->ioc_error = 0;
			MTYPE(bp) = M_IOCACK;
			qreply(q, bp);
			return;

		case RAWIOC_GETSOCKNAME:
		case RAWIOC_GETPEERNAME:
		{
			/*
		 	 * Return address bound to.
		 	 */
			struct tsap *tp;

			if ((bp->b_cont == (mblk_t *)NULL) ||
					(ip->ioc_count < RAW_ADDRLEN)) 
				goto nack;

			bp1 = bp->b_cont;

			tp = (struct tsap *)bp1->b_rptr;

			if (ip->ioc_cmd == RAWIOC_GETSOCKNAME)
				*tp = raw->rawd_la;
			else
				*tp = raw->rawd_fa;

			bp1->b_wptr = bp1->b_rptr + BINDLEN;
			bzero((char *)bp1->b_wptr, 8);
			bp1->b_wptr += 8;
			tp->ta_family = (short)raw->rawd_proto.sp_family;

			ip->ioc_count = RAW_ADDRLEN;
			goto ack;
		}

		case RAWIOC_PROTO:
		{
			/*
			 * we need this so the user can tell us which protocol
			 * and address family he is using.  We'll store this
			 * in our upper queue; it'll be matched against when
			 * packets eventually came back upstream to us.
			 */
			struct sockproto *rp;

			if ((bp->b_cont == NULL) ||
				    (ip->ioc_count < sizeof(struct sockproto)))
				goto nack;

			rp = (struct sockproto *)bp->b_cont->b_rptr;

	 		/*
			 * Allow anything but TCP or UDP.
			 */
			if ((rp->sp_protocol == IPPROTO_TCP) ||
			    (rp->sp_protocol == IPPROTO_UDP)) {
				ip->ioc_error = EPROTONOSUPPORT;
				goto nack;
			}

			raw->rawd_proto.sp_protocol = rp->sp_protocol;
			raw->rawd_proto.sp_family = rp->sp_family;
			ip->ioc_count = 0;
			goto ack;
		}

		case RAWIOC_CONNECT:
		{
			/*
			 * we need this so the user can "connect" to
			 * a foreign address.
			 */
			struct tsap *tp;

			if ((bp->b_cont == (mblk_t *)NULL) ||
					(ip->ioc_count < RAW_ADDRLEN)) 
				goto nack;

			bp1 = bp->b_cont;

			tp = (struct tsap *)bp1->b_rptr;

			/*
			 * Tell user if we're already connected.
			 * But do we really care?  Put in for now
			 * just to be BSD compatible.
			 */
			if (raw->rawd_flags & RAW_FADDR) {
				ip->ioc_error = EISCONN;
				goto nack;
			}

			raw->rawd_faddr = tp->ta_addr; /* set address */
			raw->rawd_flags |= RAW_FADDR; /* checked by raw_input */

			ip->ioc_count = 0;
			goto ack;
		}

		case RAWIOC_SOCKOPT:	/* handle all socket options */
			if ((ip->ioc_error = raw_sockopt(ip, raw, bp)) == 0)
			    goto ack;
			else
			    goto nack;

		case RAWIOC_SETTOS:	/* Set Type of service */
			ip->ioc_error = 0;
			if (ip->ioc_count != sizeof(unsigned char))
			    goto nack;
			raw->rawd_tos = *(unsigned char *)bp->b_cont->b_rptr;
			goto ack;

		case RAWIOC_GETTOS:	/* Get Type of service */
			ip->ioc_error = 0;
			if (ip->ioc_count != sizeof(unsigned char))
			    goto nack;
			*(unsigned char *)bp->b_cont->b_rptr = raw->rawd_tos;
			goto ack;

		case RAWIOC_SETTTL:	/* Set Time to live */
			ip->ioc_error = 0;
			if (ip->ioc_count != sizeof(unsigned char))
			    goto nack;
			raw->rawd_ttl = *(unsigned char *)bp->b_cont->b_rptr;
			goto ack;

		case RAWIOC_GETTTL:	/* Get Time to live */
			ip->ioc_error = 0;
			if (ip->ioc_count != sizeof(unsigned char))
			    goto nack;
			*(unsigned char *)bp->b_cont->b_rptr = raw->rawd_ttl;
			goto ack;

		case RAWIOC_GETFAMILY:	/* Get address family. */
			ip->ioc_error = 0;
			if (ip->ioc_count != sizeof(ushort))
			    goto nack;
			*(ushort *)bp->b_cont->b_rptr = AF_INET;
			goto ack;

		default:		/* unrecognised IOCTL - just nack it */
			ip->ioc_error = EINVAL;
			goto nack;
		}

		break;	/* end of case M_IOCTL */


 	case M_FLUSH:
		endflush(q, bp);
		break;
		
	case M_DATA:

		/*
		 * This is not allowed because the User
		 * did a write instead of send.
		 * (No Target address available.)
		 */

	default:
		STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
			 "rawoput: unknown bp type %x\n", MTYPE(bp), 0);
		freemsg(bp);
		break;
	}

}	/*  End of rawoput()  */


/*
 * Rawosrv(q) is the service routine for the top end write
 * queue of RAW.  The only messages that are put on
 * this queue are T_UNITDATA_REQ.  These are taken off
 * the queue and passed to the routine raw_output().
 */

/*
 * all M_PROTOs with T_UNITDATA_REQ
 */

int
rawosrv(q)

	register queue_t *q;
{
	register struct rawd *raw;
	register mblk_t *bp;
	int error;

	raw = (struct rawd *)q->q_ptr;
	ASSERT(raw);
	while (bp = getq(q)) {
		if (error = raw_output(raw, bp)) {
			rstat.rs_raw_output++;
			rstat.rs_raw_operr = error;
		}
	}

}	/*  End of rawosrv()  */


/*
 * Raw_output(raw, bp) processes the data that has to be sent out.
 * This routine allocates a rawip header, and fills in the various
 * fields. It then invokes the put procedure of the module downstream.
 */

/*
 * Send out Data to IP.
 * If we are sending to the same destinations
 * as the last time, then we don't have to
 * know our own source address.
 */

int
raw_output(raw, bp)

	struct rawd *raw;
	mblk_t *bp;
{
	register struct rawiphdr *ui;
	register struct T_unitdata_req *Tp;
	register queue_t *q;	/* bottom write q */
	register mblk_t *hbp, *dbp, *abp;
	register ushort len;
	struct tsap ta;
	long retcode;
	int ipflags;
	int error;

	Tp = (struct T_unitdata_req *)bp->b_rptr;
	ASSERT(Tp->PRIM_type == T_UNITDATA_REQ);

	q = ((struct rawb *)&rawb)->rawb_wrq;	/* only 1 bottom q */
	if (q == NULL || q->q_next == NULL || q->q_next->q_flag & QFULL) {
		STRlog(RAW_ID, 0, DPRI_MED, SL_te,
			"rawosrv: ip not linked/full, drop packet\n", 0, 0);
		error = 0x1;
		goto drop;
	}
	if ((BLEN(bp) < (Tp->DEST_offset + Tp->DEST_length))
	    || (Tp->DEST_offset < sizeof(*Tp))
	    || (Tp->DEST_length != RAW_ADDRLEN)) {
		retcode = TBADADDR;
		error = 0x2;
		goto nonfatal;
	}
	if((dbp = bp->b_cont) == NULL) {
		retcode = TNODATA;
		STRlog(RAW_ID, 0, DPRI_MED, SL_te,"raw_output: No Data.", 0, 0);
		error = 0x4;
		goto nonfatal;
	}
	if ((hbp = allocb(sizeof(struct rawiphdr), BPRI_LO)) == NULL) {
		retcode = TSYSERR;
		error = 0x8;
		rstat.rs_allocb_hdr++;
		goto nonfatal;
	}
	/* can't use pointer arithemetic: struct not aligned */
	bcopy((char *)(bp->b_rptr+Tp->DEST_offset), (char *)&ta, BINDLEN);
	freeb(bp);
	bp = (mblk_t *) NULL;
	len = (ushort)msgdsize(dbp);

	ui = (struct rawiphdr *)hbp->b_wptr;
	hbp->b_wptr += sizeof(struct rawiphdr);
	hbp->b_cont = dbp;
	ui->ui_next = ui->ui_prev = 0;
	ui->ui_x1 = 0;

	/*
	 * The user can set this to anything he likes via the RAWIOC_PROTO
	 * ioctl, but typically it'll be either IPPROTO_ICMP or IPPROTO_RAW.
	 */
	ui->ui_pr = raw->rawd_proto.sp_protocol;

	/* ui->ui_len = ui->ui_ulen = htons(len); */

	/*
	 * If retransmitting to the same destination as
	 * one previouly used, then reuse old address.
	 * If to a new destination then get my source address
	 * from IP.
	 */
	if (raw->rawd_laddr == INADDR_ANY || raw->rawd_faddr != ta.ta_addr) {
		/* set up the foreign address and local address */

		/*
		 * XXX - See BSD raw_usrreq() - case PRU_SEND
		 * BSD does other stuff which I don't think we need.
		 *
		 * Need to set RAW_FADDR flag or else raw_input() won't
		 * check the foreign address of incoming packets
		 */
		raw->rawd_flags |= RAW_FADDR;
		raw->rawd_faddr = ta.ta_addr;

		/*
		 * Now set up local address - get my source address from IP.
		 */
		raw_getmyaddr(raw, q);
		if (raw->rawd_laddr == 0) {
			/*
			 * Could not establish my address.
			 */
			retcode = TBADADDR;
			freemsg(hbp);
			STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
			    "raw_output: no src addr for %x\n", ta.ta_addr,0);
			error = 0x10;
			goto nonfatal;
		}

		raw->rawd_flags |= RAW_LADDR; /* checked by raw_input */
	}

	ui->ui_src = raw->rawd_laddr;
	ui->ui_dst = ta.ta_addr;
	/* ui->ui_sum = -1; XXX not required for RAW interface */

	((struct ip *)ui)->ip_len = len + sizeof(struct rawiphdr);
	((struct ip *)ui)->ip_hl = IPHLEN  >> 2 ;
	((struct ip *)ui)->ip_ttl = raw->rawd_ttl;
	((struct ip *)ui)->ip_tos = raw->rawd_tos;
	((struct ip *)ui)->ip_off = 0;

	/*
	 * See if we have any "arguments" for IP.
	 * Only ip options and ip flags at present.
	 */
	ipflags = (int)(raw->rawd_flags & (IP_ROUTETOIF | IP_ALLOWBROADCAST));
	if (raw->rawd_ipopt || ipflags) {
		/*
		 * Set up "arguments" for IP in a message block of type M_PROTO.
		 * The continuation pointer will point to the real data.
		 */
		struct IP_args *ipargs;
		int buflen;
#ifdef DEBUG
LABEL(IP_rawarg);
#endif
		if (raw->rawd_ipopt)
			buflen = 64;
		else
			buflen = 16;

		if ((abp = allocb(buflen, BPRI_LO)) == NULL) {
			retcode = TSYSERR;
			freemsg(hbp);
			STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
			    "raw_output: allocb failed for IP arguments");
			rstat.rs_allocb_arg++;
			error = 0x20;
			goto nonfatal;
		}

		MTYPE(abp) = M_PROTO;

		if (ipflags) {
			ipargs = (struct IP_args *)abp->b_wptr;
			ipargs->IP_type = IPFLAGS;
			ipargs->IP_length = sizeof(int);
			abp->b_wptr += sizeof(struct IP_args);
			bcopy((caddr_t)&ipflags, abp->b_wptr, sizeof(int));
			abp->b_wptr += sizeof(int);
		}

		if (raw->rawd_ipopt) {
			int size = BLEN(raw->rawd_ipopt);

			ipargs = (struct IP_args *)abp->b_wptr;
			ipargs->IP_type = IPOPTS;
			ipargs->IP_length = size;
			abp->b_wptr += sizeof(struct IP_args);
			bcopy(raw->rawd_ipopt->b_rptr, abp->b_wptr, size);
			abp->b_wptr += size;
		}

		ipargs = (struct IP_args *)abp->b_wptr;
		ipargs->IP_type = ipargs->IP_length = IPENDLIST;
		abp->b_wptr += sizeof(struct IP_args);

		abp->b_cont = hbp;
		
	} else
		abp = hbp;

#ifdef DEBUG
LABEL(rawTOIP);
#endif
	putnext(q, abp);
	/*
	 * It seems to be a problem in the
	 * Look up if we let the laddr be the same
	raw->rawd_faddr = INADDR_ANY;
	 */
	return 0;

nonfatal:
	if (hbp = allocb(sizeof(struct T_uderror_ind), BPRI_HI)) {
		struct T_uderror_ind *Tp1;

		Tp1 = (struct T_uderror_ind *)hbp->b_wptr;
		Tp1->PRIM_type = T_UDERROR_IND;
		Tp1->DEST_length = Tp->DEST_length;
		Tp1->DEST_offset = Tp->DEST_offset;
		Tp1->OPT_length = Tp->OPT_length;
		Tp1->OPT_offset = Tp->OPT_offset;
		Tp1->ERROR_type = retcode;
		hbp->b_wptr += sizeof(struct T_uderror_ind);
		MTYPE(hbp) = M_PROTO;	
		STRlog(RAW_ID, RAWCHAN(raw), DPRI_MED, SL_te,
			"raw_output: Non fatal error %d\n", retcode, 0);
		putnext(raw->rawd_rdq, hbp);
	} else {
		rstat.rs_allocb_udi++;
		error |= 0x80;
	}

drop:	/* drop the incoming message */
	if (bp)
		freemsg(bp);

	return error;

}	/*  End of raw_output()  */


/*
 * send "ip_if" to lower module telling you are raw
 */

void
rawmuxnotify(rawbp)

	register struct rawb *rawbp;
{
	register queue_t *bq;
	register mblk_t *bp;
	register struct ip_if *mp;

	ASSERT(rawbp && rawbp->rawb_wrq && rawbp->rawb_qtop && rawbp->rawb_iocblk);
	bq = rawbp->rawb_wrq;
	if ((bp = allocb(sizeof(struct ip_if), BPRI_MED)) == NULL) {
		/* send M_IOCNAK upstream */
		rstat.rs_allocb_mux++;
		if (bp = rawbp->rawb_iocblk) {
			MTYPE(bp) = M_IOCNAK;
			rawbp->rawb_iocblk = NULL;
			qreply(rawbp->rawb_qtop, bp);
		}
		return;
	}

	MTYPE(bp) = M_CTL;
	mp = (struct ip_if *)bp->b_wptr;
	mp->ipf_type = rawbp->rawb_mstate;
	mp->ipf_id = bq->q_qinfo->qi_minfo->mi_idnum;
	bp->b_wptr += sizeof(struct ip_if);
	putnext(bq, bp);

}	/*  End of rawmuxnotify()  */


/*
 * got data from ip
 */

int
rawbiput(q, bp)

	register queue_t *q;
	register mblk_t *bp;
{
	register struct rawb *rawbp;
	register struct ip_if *mp;
	register struct rawd *raw;
	mblk_t *bp1;

	ASSERT(q && bp);
	rawbp = (struct rawb *)q->q_ptr;
	ASSERT(rawbp);

	switch(MTYPE(bp)) {

	case M_PROTO:
		bp1 = bp;
		bp = unlinkb(bp);
		freeb(bp1);
		/* fall thru... */

	case M_DATA:
		putq(q, bp);
		return;

	case M_CTL:
		/*
		 * this is the ack for I_LINK/I_UNLINK from bottom module
		 */
		mp = (struct ip_if *)bp->b_rptr;
		switch(mp->ipf_type) {

		  case IPIF_NGETMYADDR:
		    raw = (struct rawd *)mp->ipf_priv;
		    raw->rawd_laddr = mp->ipf_addr;
		    break;

		case IPIF_LINKED:
			rawbp->rawb_botid = mp->ipf_id;
			/* fall thru */

		case IPIF_UNLINKED:
			rawmuxack(rawbp, mp->ipf_type);
			break;

		default:
			STRlog(RAW_ID, 0, DPRI_MED, SL_te,
			"rawbiput: unkown M_CTL type %x\n", mp->ipf_type, 0);
		}
		break;

	default:
		STRlog(RAW_ID, 0, DPRI_MED, SL_te,
		"rawbiput: unkown message type %x\n", MTYPE(bp), 0);
		break;
	}
	freemsg(bp);

}	/*  End of rawbiput()  */


/*
 * send M_IOCACK/M_IOCNAK upstream
 */

void
rawmuxack(rawbp, muxtype)

	register struct rawb *rawbp;
	char muxtype;
{
	register mblk_t *bp;
	register struct iocblk *ioc;

	bp = rawbp->rawb_iocblk;
	ASSERT(bp && rawbp->rawb_qtop);
	rawbp->rawb_iocblk = NULL;
	if ((muxtype == IPIF_LINKED && rawbp->rawb_mstate != IPIF_LINKING) ||
	   (muxtype == IPIF_UNLINKED && rawbp->rawb_mstate != IPIF_UNLINKING)) {
		rawbp->rawb_mstate = IPIF_UNLINKED;
		MTYPE(bp) = M_IOCNAK;
	} else {
		rawbp->rawb_mstate = muxtype;
		MTYPE(bp) = M_IOCACK;
	}
	ioc = (struct iocblk *)bp->b_rptr;
	ioc->ioc_count = ioc->ioc_error = 0;
	qreply(rawbp->rawb_qtop, bp);	/* top control queue */

}	/*  End of rawmuxack()  */


/*
 * Rawbisrv(q) is the service routine for the bottom end read queue.
 * Messages are basically data that is to be passed to the user.
 * This routine takes the messages off the queue and invokes the
 * raw_input() routine.
 */

int
rawbisrv(q)

	register queue_t *q;
{
	register mblk_t *bp;

	while (bp = getq(q)) { /* must all be M_DATA */
		ASSERT(MTYPE(bp) == M_DATA);
		raw_input(bp);
	}

}	/*  End of rawbisrv()  */


void
raw_input(bp)

	register mblk_t *bp;
{
	register struct rawiphdr *ui;
	register struct rawd *raw;
	register mblk_t *bp1 = NULL;
	register struct T_unitdata_ind *Tp;
	int errcode;
	struct tsap ta;
	int success = 0;
	int count = 0;
	struct ip *ip;

#define	UILOG(x)	{errcode = x; goto err;}

	/*
	 * Pull IP and RAW header together in first block.
	 * Don't strip IP options.
	 */
	if (BLEN(bp) < sizeof(struct rawiphdr)) {
		if (pullupmsg(bp, sizeof(struct rawiphdr)) == 0) {
			rawstat.raws_hdrops++;
			UILOG(RAWE_HDR);
		}
	}

	ui = (struct rawiphdr *)bp->b_rptr;
	ip = (struct ip *)bp->b_rptr;

	/*
	 * convert RAW protocol specific fields to host format.
	 */
	ta.ta_addr = ui->ui_dst;

#ifdef NOTDEF
	/* XXXX - NOTE - we shouldn't strip the IP header, however we
	 * should strip any raw header.  At present there is no raw
	 * header but there may be in the FUTURE so beware!
	 */
	/* drop raw & ip header */
	bp->b_rptr += sizeof(struct rawiphdr);
#endif /* NOTDEF */

#ifdef DEBUG
raw_message = bp;
LABEL(rawLOOKUP);
#endif
	for (raw = rawd_head; raw ; raw = raw->rawd_next) {

		count++;

		/*
		 * We don't want to send packets up to minor device 0
		 * as that's the reserved control channel.
		 */
		if (raw->rawd_dev == 0)
			continue;

		/* check if upper queue in use */
		if (raw->rawd_rdq == NULL)
			continue;

		/*
		 * XXX - should we check family first like BSD?
		 * I don't see the point as RAW is always on top of IP
		 * and therfore only applies to AF_INET.  To allow
		 * different address families we would need to make
		 * this into a many to many multiplexing driver;  this'll
		 * have to be done later if required.
		 *
		 * If we do this later, however, whoever calls raw_input()
		 * could set up the address family in ui->ui_x1 (see raw.h)
		 * Alternatively we could have a pseudo raw header like BSD.
		 *
		 * if (ui->ui_x1 != raw->rawd_proto.sp_family)
		 *	continue;
		 */

		/*
		 * Check if protocols match.  Typically only RAW and ICMP
		 * protocols are allowed, however, just to be BSD compatible
		 * we'll match on a protocol number of zero.
		 */
		if ((raw->rawd_proto.sp_protocol) &&
			(raw->rawd_proto.sp_protocol != ui->ui_pr))
				continue;

		/* check if source addresses match */
		if ((raw->rawd_flags & RAW_LADDR) &&
			(raw->rawd_laddr != ui->ui_dst))
				continue;

		/* check if destination addresses match */
		if ((raw->rawd_flags & RAW_FADDR) &&
			(raw->rawd_faddr != ui->ui_src))
				continue;

		if (raw->rawd_tstate != TS_IDLE) {
			/*
			 * Is this a problem?
			 */
			rawstat.raws_busy++;
			UILOG(RAWE_IDLE);
		}

		if (bp1 == NULL) {

#define LEN	sizeof(struct T_unitdata_ind) + RAW_ADDRLEN

			if ((bp1 = allocb(LEN, BPRI_LO)) == NULL) {
				rawstat.raws_allocb1++;
				UILOG(RAWE_ALLOCB1);
			}
			MTYPE(bp1) = M_PROTO;
			Tp = (struct T_unitdata_ind *)bp1->b_wptr;
			Tp->PRIM_type = T_UNITDATA_IND;
			Tp->SRC_length = RAW_ADDRLEN;
			Tp->SRC_offset = sizeof(struct T_unitdata_ind);
			Tp->OPT_length = 0;
			Tp->OPT_offset = sizeof(struct T_unitdata_ind);
			ta.ta_addr = ui->ui_src;
			ta.ta_port = 0;
			bcopy((char *)&ta, (char *)(bp1->b_wptr +
			    sizeof(struct T_unitdata_ind)), BINDLEN);

			bp1->b_wptr = bp1->b_rptr + 
				      sizeof(struct T_unitdata_ind) + BINDLEN;
			bzero((char *)bp1->b_wptr, 8);
			bp1->b_wptr += 8;
			bp1->b_cont = bp;
#undef LEN
		}

#ifdef DEBUG
raw_message = bp1;
LABEL(RAWsendup);
#endif
		STRLOG(RAW_ID, RAWCHAN(raw), DPRI_LO, SL_TRACE,
		  "raw_input: sent up from %x.%d\n", ta.ta_addr, ta.ta_port);

		/* duplicate message */
		if ((rawmsg[count-1] = dupmsg(bp1)) == NULL) {
			rawstat.raws_dupmsg++;
			UILOG(RAWE_DUPMSG);
		}

		/* send duplicated message upstream via top service routine */
		if (canput(raw->rawd_rdq) == 1) {
			/*
			 * Copy any IP options to raw's ipoptin.
			 * Required by getsockopt().
			 */
			if (ip->ip_hl > 5) {
				mblk_t *obp = raw->rawd_ipoptin;
				int olen = (ip->ip_hl << 2) - IPHLEN;

				/* This message is freed in _rawclose() */
				if (!obp) {
					obp = allocb(MAX_IPOPTLEN, BPRI_LO);
					if (!obp) {
						rawstat.raws_allocb2++;
						UILOG(RAWE_ALLOCB2);
					}
				}

				bzero(obp->b_rptr, MAX_IPOPTLEN);
				bcopy(bp->b_rptr + (ip->ip_hl << 2),
					obp->b_rptr, olen);
				obp->b_wptr = obp->b_rptr + olen;

				raw->rawd_ipoptin = obp;

			} else {
				/* zap old options */
				if (raw->rawd_ipoptin) {
					freeb(raw->rawd_ipoptin);
					raw->rawd_ipoptin = NULL;
				}
			}

			putq(raw->rawd_rdq, rawmsg[count-1]);
			success++;

		} else {
			freemsg(rawmsg[count-1]);
			rawstat.raws_qfull++;
			UILOG(RAWE_FULL);
		}

		/*
		 * Don't return yet as we may be sending message up to more
		 * than one queue - the user has to determine whether the
		 * packet is actually his.  Unlike UDP and TCP we DON'T match
		 * on port numbers.
		 */
	}

	if (success > 0) {	/* we found at least one upper queue */
		freemsg(bp1);	/* free message - duplicate was sent upstream */
		return;
	}

	rawstat.raws_lookup++;
	UILOG(RAWE_LOOKUP); /* no upper matching queue found */
err:
	STRlog(RAW_ID, 0, DPRI_MED, SL_te,
		"raw_input: log code %x\n", errcode, 0);

	/* drop the messages */

	if (bp1) {
		/*
		 * This'll free bp too.  Set bp to NULL so that we
		 *  we don't attempt to free it again later on.
		 */
		freemsg(bp1);
		bp = NULL;
	}

	if (bp)
		freemsg(bp);

	return;

}	/*  End of raw_input()  */


ushort rawport = RAWPORT_RESERVED;		/* XXX */
/*
 * Bind to an unused port number.
 * if none specified.
 */

/*
 * Raw_bind() is used to find the next available port number to
 * which a user can bind. If the user requested a non zero port,
 * the routine just checks to see if there is already a stream
 * bound to that port. If there is one, then an error is returned
 * unless the REUSEADDR flag is set.  If a non super user requests a port
 * which is less than RAW_PORTRESERVED then an error is returned.
 * If no any port is requested, this routine just picks any available
 * port number that is not in use.
 */

static long
raw_bind(ta, flags)

	register struct tsap *ta;
	register ushort flags;
{
	register struct rawd *raw;
	register int nxt;
	register ushort port;

	port = ntohs(ta->ta_port);

	/*
	 * Find an acceptable port.
	 * If user is root then allow lower than
	 * RAWPORT_RESERVED.
	 */
	if (port && (port < RAWPORT_RESERVED) &&
				((flags & RAW_SUPERUSER) == 0))
		return TACCES;

	if (++rawport == 0xffff)
		rawport = RAWPORT_RESERVED;

	nxt = htons(rawport);
	port = ntohs(port);
	for (raw = rawd_head; raw->rawd_next ; raw = raw->rawd_next) {
		/*
		 * Find the requested address.
		 */
		if ((port && raw->rawd_sport == port) &&
				(raw->rawd_laddr == ta->ta_addr))
			if ((flags & SO_REUSEADDR) == 0)
				return TNOADDR;

		if ((port == 0) && (nxt == raw->rawd_sport)) {
		    if (++rawport == 0xffff)
			rawport = RAWPORT_RESERVED;
		    nxt = htons(rawport);
		    }
	}

	if (port)
		ta->ta_port = port;
	else
		ta->ta_port = nxt;

	return 0;

}	/*  End of raw_bind()  */


#undef UILOG
/*
 * Format a M_CTL packet to IP
 * requesting my source address.
 */ 

void
raw_getmyaddr(raw, q)
     register struct rawd *raw;
     queue_t *q;
{
    register mblk_t *bp;
    register struct ip_if *tpf;

    raw->rawd_laddr = 0;
    if ((bp = allocb( sizeof(struct ip_if), BPRI_MED))==NULL){
	rstat.rs_allocb_my++;
	return;
	}
    tpf = (struct ip_if *)bp->b_wptr;
    bp->b_wptr += sizeof(struct ip_if);
    MTYPE(bp) = M_CTL;
    tpf->ipf_type = IPIF_NGETMYADDR;
    tpf->ipf_addr = raw->rawd_faddr;
    tpf->ipf_tos = raw->rawd_tos;
    tpf->ipf_priv = (char *)raw;
    putnext(q, bp);
    return;

}	/*  End of raw_getmyaddr()  */


/*
 *	raw_sockopt() allows users to set or get various "socket" options.
 *
 *	The beginning of bp->cont is expected to contain a winsopt 
 *	structure which comprises the following members:
 *
 *		1. level - this is the protocol level
 *		2. name  - this is the option name
 *		3. flags - this indicates the action to take (i.e. get or set)
 */
int
raw_sockopt(ip, raw, bp)
	register struct iocblk *ip;
	register struct rawd *raw;
	register mblk_t *bp;
{
	register mblk_t *bp1 = bp->b_cont;
	register struct winsopt *winsopt;
	int *tmp;
	int ans = EINVAL;
	int len;
	int ret;

	if (bp1 == NULL || BLEN(bp1) < sizeof(struct winsopt))
		goto nonfatal;

	winsopt = (struct winsopt *)bp1->b_rptr;

	/*
	 * Let's see if the option level is really for us. We'll
	 * accept SOCKET level and RAW level options.   IP level
	 * options will be sent down to IP using a M_CTL message.
	 */
	if (winsopt->level == IPPROTO_IP) {
		if ((ans = raw_ipoptions(bp, raw, winsopt->flags)) == 0) {
			bp1 = bp->b_cont;
			if (bp1)
				ip->ioc_count = BLEN(bp1);
			else
				ip->ioc_count = 0;
			return ans;
		}
		goto nonfatal;
	} else if (winsopt->level != IPPROTO_RAW &&
						winsopt->level != SOL_SOCKET) {
		/* reject option */
		ans = ENOPROTOOPT;
		goto nonfatal;
	}

	/* Only integers expected */
	if ((len = BLEN(bp1) - sizeof(struct winsopt)) != sizeof(int))
		goto nonfatal;

	if (winsopt->flags == T_NEGOTIATE) {
		/* set option */
		tmp = (int *)(bp1->b_rptr + sizeof(struct winsopt));

		switch (winsopt->name) {

		case SO_REUSEADDR:
		case SO_DONTROUTE:
		case SO_BROADCAST:
			/*
			 * Turn option on or off.
			 */

			if (*tmp)
			    raw->rawd_flags |= winsopt->name;  /* set flag   */
			else
			    raw->rawd_flags &= ~winsopt->name; /* reset flag */

			break;

		default:
			ans = ENOPROTOOPT;
			goto nonfatal;
		}

	} else if (winsopt->flags == T_DEFAULT) {
		/* get option */
		switch (winsopt->name) {

		case SO_REUSEADDR:
		case SO_DONTROUTE:
		case SO_BROADCAST:
			/*
			 * Return option setting.
			 */
			ret = (int)(winsopt->name & raw->rawd_flags);
			tmp = &ret;
			break;

		default:
			ans = ENOPROTOOPT;
			goto nonfatal;
		}

	} else
		/* unknown request */
		goto nonfatal;

	bcopy((caddr_t)tmp, (caddr_t)bp1->b_rptr, len);
	ip->ioc_count = len;
	bp1->b_wptr = bp1->b_rptr + len;
	return 0;

nonfatal:
	STRlog(RAW_ID, -1, DPRI_LO, SL_te,
			"raw_sockopt: nonfatal error %0x\n", ans);
	return ans;

}	/*  End of raw_sockopt()  */


/*
 * Options meant for IP - send them down.
 */
int
raw_ipoptions(bp, raw, flags)
	register mblk_t *bp;
	register struct rawd *raw;
	register int flags;
{
	register mblk_t *ifbp;
	register mblk_t *bp1 = bp->b_cont;
	ipfcmd *iftype;
	int ans = 0;
	extern char rawopt_out; /* see /etc/master.d/raw */

	if ((ifbp = allocb(sizeof(ipfcmd), BPRI_MED)) == NULL) {
		rstat.rs_allocb_ipopt++;
		STRlog(RAW_ID, -1, DPRI_HI, SL_te,
			"raw_sendipopts: out of buffer\n");
		return -1;
	}

	bp->b_cont = NULL;

	/*
	 * ifbp's continuation will tell IP what to do with the sent options.
	 * It contains the winsopt structure and data as sent down by the user.
	 */
	ifbp->b_cont = bp1;

	/*
	 * bp1's continuation pointer will point to the ip options message
	 * block. This could be NULL if there are currently no options set.
	 *
	 * If rawopt_out is set then get the outgoing IP options
	 * else get the incoming IP options.
	 */
	if ((rawopt_out) || flags == T_NEGOTIATE)
		bp1->b_cont = raw->rawd_ipopt; /* set (or get) outgoing opts. */
	else
		bp1->b_cont = raw->rawd_ipoptin; /* get incoming options */

	iftype = (ipfcmd *)ifbp->b_wptr;
	*iftype = IPIF_OPTIONS;
	ifbp->b_wptr += sizeof(ipfcmd);

	MTYPE(ifbp) = M_CTL;
	putnext(rawb.rawb_wrq, ifbp); /* send message to IP */

	/* iftype indicates success or failure on return */
	iftype = (ipfcmd *)ifbp->b_rptr;
	if (*iftype == 0) {
		if (flags == T_NEGOTIATE) {
			/* setting - update raw's option pointer */
			if (!raw->rawd_ipopt)
				raw->rawd_ipopt = ifbp->b_cont;
		} else {
			/* getting - update b_cont for user */
			bp->b_cont = ifbp->b_cont;
		}
	} else {
		STRlog(RAW_ID, -1, DPRI_HI, SL_te,
			"raw_ipoptions: error 0x%x from ip", *iftype);
		ans = (int)*iftype;
	}

	freeb(ifbp);
	return ans;

}	/*  End of raw_ipoptions()  */

/*
 * send a T_ok_ack upstream
 * q is the downward q, bp holds the T_ok_ack, and prim is the request type
 * on input, bp points to some T_primitives, we check it here to see if
 * we can use it for reply. If not, free it and allocb a new one
 * return -1 if can't allocb, 0 if o.k.
 */
raw_ok_ack(q, bp, prim, state)
	register queue_t *q;
	register mblk_t *bp;
	register long prim;
	unchar state;
{
	register struct T_ok_ack *p;
	struct rawd *raw;

	ASSERT(q && !(q->q_flag&QREADR) && bp);
	raw = (struct rawd *)q->q_ptr;
	ASSERT(raw);

	if (!REUSEABLE(bp, sizeof(struct T_ok_ack))) {
		freemsg(bp);
		if ((bp = allocb(sizeof(struct T_ok_ack), BPRI_MED)) == NULL)
			return(-1);
	}
	MTYPE(bp) = M_PCPROTO;
	p = (struct T_ok_ack *)bp->b_rptr;
	p->PRIM_type = T_OK_ACK;
	p->CORRECT_prim = prim;
	bp->b_wptr += sizeof(struct T_ok_ack);

	if (state != TS_NOCHANGE)
		raw->rawd_tstate = state;
	WQREPLY(q, bp);
	return(0);

}	/*  End of raw_ok_ack()  */

/*
 * send an T_error_ack upstream
 */
raw_error_ack(q, bp, prim, code, state)
	register queue_t *q;
	register mblk_t *bp;
	register long prim;
	register long code;
	unchar state;
{
	register struct T_error_ack *p;
	struct rawd *raw;

	ASSERT(q && !(q->q_flag&QREADR) && bp);
	raw = (struct rawd *)q->q_ptr;
	ASSERT(raw);

	if (!REUSEABLE(bp, sizeof(struct T_error_ack))) {
		freemsg(bp);
		if ((bp = allocb(sizeof(struct T_error_ack), BPRI_MED)) == NULL)
			return(-1);
	}
	MTYPE(bp) = M_PCPROTO;
	p = (struct T_error_ack *)bp->b_rptr;
	p->PRIM_type = T_ERROR_ACK;
	p->ERROR_prim = prim;
	p->TLI_error = code;
	bp->b_wptr += sizeof(struct T_error_ack);
	
	if (state != TS_NOCHANGE)
		raw->rawd_tstate = state;
	qreply(q, bp);
	return(0);

}	/*  End of raw_error_ack()  */

/* end */
