/*
 * @(#) Copyright 1988.  The Wollongong Group, Inc.  All Rights Reserved.
 *
 * Fixes: ID	SPR#	Descrition
 *  	  nw1	4799	TCP "PUSH flag" problem
 *	  cah1  	Always send a response for mss if flag is on.
 */

#ident "@(#)tcp_tw.c (TWG)  1.8     89/08/17 "

#include "sys/param.h"
#include "sys/types.h"
#ifndef XENIX
#include "sys/inline.h"
#endif
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#ifdef XENIX
#include "sys/assert.h"
#endif
#include "sys/inet.h"
#include "sys/socket.h"
#include "sys/if.h"
#include "sys/in.h"
#include "sys/in_var.h"
#include "sys/ip.h"
#include "sys/ip_var.h"
#include "sys/tcp.h"

/*
 * Initial options.
 */
unchar	tcp_initopt[4] = { TCPOPT_MAXSEG, 4, 0x0, 0x0, };

/*
 * TCP_NET utilities
 */

/***********************
 *  Table of Contents  *
 **********************/

int		tw_send();
void 		tw_drop();
void		tw_close();
void		tw_clean();
mblk_t		*tw_dupmsga();
void		tw_acked();
void		tw_rmtcb();
void		tw_drop_floater();

/***** End of TOC *****/

extern	int	tcp_maxrcvwin;
extern mblk_t *tcp_pcbcache;
extern struct tcpstat tcpstat;
/*
 * tcp low level send routine
 * tp: tcpcb, if NULL, use hbp (tp and hbp can't be both NULL)
 * hbp: tcpiphdr, if NULL, copy it from t_template
 * dbp: data followed the tcp header
 * seq: seq#, ack: ack#, flags: flag
 * return 0 if o.k., error code if failed
 * if failed, the message is NOT freed
 *
 * assume: ti_next = ti_prev = ti_x1 = ti_x2 = ti_urp = 0;
 *	ti_off = TCPHLEN >> 2;
 *	ti_pr = IPPROTO_TCP;
 */
int
tw_send(tp, hbp, dbp, seq, ack, flags)
     register struct tcpcb *tp;
     register mblk_t *hbp;
     mblk_t *dbp;
     ulong seq, ack;
     unchar flags;
{
    register mblk_t *obp, *abp;
    register int len = 0;
    register struct tcpiphdr *ti;
    register int win = 0,htlen = 0;
    int error;
    unchar *opt;
    unsigned optlen = 0; 
    struct tcpd *td;
    int ipflags;

    extern int tcpsend();
    extern void tcp_bestmss();
    
    if (hbp) {
	/*
	 * check sanity of header
	 */
	STRlog(TCP_ID, -1, DPRI_LO, SL_te,
	       "tw_send: hbp not null, len=%d\n",len);
	len = TCPIPHLEN - BLEN(hbp);
	if (len) {
	    STRlog(TCP_ID, -1, DPRI_LO, SL_te, "tw_send: hlen %d\n", len);
	    if (len > 0)
		goto errret;
	    if (adjmsg(hbp, len) == -1)
		goto errret;
	    }
	len = 0;
    } else {
	/*
	 * Copy the Source and Dest form tp->t_template
	 */
	if (tp->t_template == NULL){
	    STRlog(TCP_ID, -1, DPRI_LO, SL_te, "tw_send: no template\n");
	    goto errret;
	    }
	if ((hbp = allocb(TCPIPHLEN+htlen, BPRI_MED)) == NULL)
	    goto errret;
	hbp->b_wptr += TCPIPHLEN+htlen;
	bcopy(tp->t_template->b_rptr, hbp->b_rptr, TCPIPHLEN);
	}

    if (dbp) {
	hbp->b_wptr = hbp->b_rptr + TCPIPHLEN;
	len = msgdsize(dbp);
	hbp->b_cont = dbp;
	}

    /*
     * Before ESTABLISHED, force sending of initial options
     * unless TCP set to not do any options
     */
    if (tp)
    {
       opt = NULL;

/* I've checked and TF_NOOPT is never used elsewhere. So it seems */ 
/* that this check is meaningless. CAH - June 16, 1992            */

       if (flags & TH_SYN && (tp->t_flags & TF_NOOPT) == 0)
       {
          u_short mss;

	  tcp_bestmss(tp);
	  mss = min(t_rhiw(tp->t_tdp) / 2, tp->t_maxseg);
   
/* Always respond because the mss may be different    */
/* than our partner is using *cah1*                   */

	  opt = tcp_initopt;
	  optlen = sizeof(tcp_initopt);
	  *(u_short *)(opt + 2) = htons(mss);
	 
/*	    if (mss > IP_MSS - sizeof(struct tcpiphdr))
            {
		opt = tcp_initopt;
		optlen = sizeof(tcp_initopt);
		*(u_short *)(opt + 2) = htons(mss);
	    }
*/
		
	}
	if (opt) {
	    /*
	     * allocate a msgblk for the options
	     * update tcpip mblk to point to options blk
	     * then pull both into one mblk
	     * if there is data then update new block to point to it
	     */
	    if ((obp = allocb(optlen, BPRI_MED)) == NULL)
		goto errret;
	    obp->b_wptr += optlen;
	    bcopy (opt, obp->b_rptr, optlen);
	    hbp->b_cont = obp;
	    if (pullupmsg(hbp, TCPIPHLEN + optlen) == 0) {
		freemsg(obp);
		goto errret;
		}
	    if (dbp)
		hbp->b_cont = dbp;
	    }
	}

    ti = (struct tcpiphdr *)hbp->b_rptr;

    ti->ti_flags = flags;
    if (tp) {
	if (tp->t_tdp && (struct tcpcb *)(tp->t_tdp->td_tcpcb->b_rptr) == tp) {
	    /* else it is floating tcpcb, keep win = 0 */
	    win = t_rwin(tp->t_tdp);

	    /*
	     * Calculate receive window.  Don't shrink window,
	     * but avoid silly window syndrome.
	     */
	    if (win < (long)(t_rhiw(tp->t_tdp)/4) && win < (long)tp->t_maxseg)
		win = 0;
	    if (win > IP_MAXPACKET)
		win = IP_MAXPACKET;
	    if (win < (long)(tp->rcv_adv - tp->rcv_nxt))
		win = (long)(tp->rcv_adv - tp->rcv_nxt);
	    ti->ti_win = htons((u_short)win);
	    if (SEQ_GT(tp->snd_up, tp->snd_nxt)) {
		ti->ti_urp = htons((u_short)(tp->snd_up - tp->snd_nxt));
		flags |= TH_URG;				/* nw1 */
	    } else
		/*
		 * If no urgent pointer to send, then we pull
		 * the urgent pointer to the left edge of the send window
		 * so that it doesn't drift into the send window on sequence
		 * number wraparound.
		 */
		tp->snd_up = tp->snd_una;		/* drag it along */
	    }
	/*
	 * If anything to send and we can send it all, set PUSH.
	 * (This will keep happy those implementations which only
	 * give data to the user when a buffer fills or a PUSH comes in.)
	 */
	if (len && (tp->snd_nxt - tp->snd_una)+len == t_wcc(tp->t_tdp))
	    flags |= TH_PUSH;				/* nw1 */

	/*
	 * Put TCP length in extended header, and then
	 * checksum extended header and data.
	 */
	if (len + optlen)
	    ti->ti_len = htons((u_short)(sizeof(struct tcphdr) + optlen + len));
	/*
	 * In transmit state, time the transmission and arrange for
	 * the retransmit.  In persist state, just set snd_max.
	 */
	if (tp->t_force == 0 || tp->t_timer[TCPT_PERSIST] == 0) {
	    tcp_seq startseq = tp->snd_nxt;
	    
	    /*
	     * Advance snd_nxt over sequence space of this segment.
	     */
	    if (flags & TH_SYN)
		tp->snd_nxt++;
	    if (flags & TH_FIN) {
		tp->snd_nxt++;
		tp->t_flags |= TF_SENTFIN;
		}
	    tp->snd_nxt += len;
	    if (SEQ_GT(tp->snd_nxt, tp->snd_max)) {
		tp->snd_max = tp->snd_nxt;
		/*
		 * Time this transmission if not a retransmission and
		 * not currently timing anything.
		 */
		if (tp->t_rtt == 0) {
		    tp->t_rtt = 1;
		    tp->t_rtseq = startseq;
		    tcpstat.tcps_segstimed++;
		    }
		}
		
	    /*
	     * Set retransmit timer if not currently set,
	     * and not doing an ack or a keep-alive probe.
	     * Initial value for retransmit timer is smoothed
	     * round-trip time + 2 * round-trip time variance.
	     * Initialize shift counter which is used for backoff
	     * of retransmit time.
	     */
	    if (tp->t_timer[TCPT_REXMT] == 0 && tp->snd_nxt != tp->snd_una) {
		tp->t_timer[TCPT_REXMT] = tp->t_rxtcur;
		if (tp->t_timer[TCPT_PERSIST]) {
			tp->t_timer[TCPT_PERSIST] = 0;
			tp->t_rxtshift = 0;
		}
	    }
	} else
	    if (SEQ_GT(tp->snd_nxt + len, tp->snd_max))
		tp->snd_max = tp->snd_nxt + len;

	}
    ti->ti_off = (TCPHLEN + optlen) >> 2;
    ti->ti_flags = flags;
    ti->ti_win = htons((ushort)win);
    ti->ti_seq = htonl(seq);
    ti->ti_ack = htonl(ack);
    ti->ti_sum = 0; ti->ti_x2 = 0;
    ti->ti_len = htons(htlen + len + sizeof(struct tcphdr) +optlen);
    /* checksum includes psuedo ip header */
    ti->ti_sum = in_cksum(hbp, len + htlen + TCPIPHLEN + (int)optlen);

    /*
     * Fill in IP length and desired time to live and
     * send to IP level.
     */
    ((struct ip *)ti)->ip_len = sizeof (struct tcpiphdr) + optlen + len;

    /* extern to comply with user (administrator) modifiable options */
    ((struct ip *)ti)->ip_tos = tcp_tos;
    ((struct ip *)ti)->ip_ttl = tcp_ttl;
	    
    abp = hbp;
    /*
     * See if we have any "arguments" for IP.
     * Only ip options and ip flags at present.
     */
    if (tp && tp->t_tdp) {
	td = tp->t_tdp;
	((struct ip *)ti)->ip_tos = td->td_tos;
	((struct ip *)ti)->ip_ttl = td->td_ttl;
	ipflags = (int)(td->td_sockflags & IP_ROUTETOIF);
	STRLOG(TCP_ID, TCPCHAN(td), DPRI_LO, SL_TRACE,
	       "tw_send: td->td_ipopt=%x, td->td_sockflags=%x",
	       td->td_ipopt, td->td_sockflags);

	if (td->td_ipopt || ipflags) {
	    /*
	     * Set up "arguments" for IP in a message block of type M_PROTO.
	     * The continuation pointer will point to the real data.
	     */
	    struct IP_args *ipargs;
	    int buflen;

	    buflen = (td->td_ipopt) ? 64 : 16;
	    if ((abp = allocb(buflen, BPRI_LO)) == NULL)
		goto errret;
	
	    MTYPE(abp) = M_PROTO;

	    if (ipflags) {
		ipargs = (struct IP_args *)abp->b_wptr;
		ipargs->IP_type = IPFLAGS;
		ipargs->IP_length = sizeof(int);
		abp->b_wptr += sizeof(struct IP_args);
		bcopy((caddr_t)&ipflags, abp->b_wptr, sizeof(int));
		abp->b_wptr += sizeof(int);
		}
	    if (td->td_ipopt) {
		int size = BLEN(td->td_ipopt);
		
		ipargs = (struct IP_args *)abp->b_wptr;
		ipargs->IP_type = IPOPTS;
		ipargs->IP_length = size;
		abp->b_wptr += sizeof(struct IP_args);
		bcopy(td->td_ipopt->b_rptr, abp->b_wptr, size);
		abp->b_wptr += size;
		}

	    ipargs = (struct IP_args *)abp->b_wptr;
	    ipargs->IP_type = ipargs->IP_length = IPENDLIST;
	    abp->b_wptr += sizeof(struct IP_args);
	    abp->b_cont = hbp;
	    }
    }
    if (error = tcpsend(abp)){
	STRlog(TCP_ID, -1, DPRI_LO, SL_te, "tw_send: tcpsend error\n");
	return error;
	}

    if (tp && tp->t_rxt)
	tcpstat.tcpRetransSegs++;
    else
	tcpstat.tcpOutSegs++;
    tcpstat.tcps_sndtotal++;
    /*
     * Data sent (as far as we can tell).
     * If this advertises a larger window than any other segment,
     * then remember the size of the advertised window.
     * Any pending ACK has now been sent.
     */
    if (tp) {
	if ((win > 0) && SEQ_GT(tp->rcv_nxt+win, tp->rcv_adv))
	    tp->rcv_adv = tp->rcv_nxt + win;
	tp->t_flags &= ~(TF_ACKNOW|TF_DELACK);
	}
    return 0;

errret:
    STRlog(TCP_ID, -1, DPRI_LO, SL_te, "tw_send: err\n");
    if (hbp)
	freemsg(hbp);
    return(-1);

}	/*  End of tw_send()  */

/*
 * drop a TCP connection.
 * if connection is synchronized, i.e. in SYN_RCVD, ESTABLISHED,
 * FIN_W1, CLOSING, CLOSE_WAIT state, send a RST to peer.
 * ??? TO BE CONTINUED
 */
void
tw_drop(tp)
     register struct tcpcb *tp;
{
    ASSERT(tp);
    STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_LO, SL_TRACE,
	   "tw_drop: dropping.");

    if (TV_HAVERCVDSYN(tp->t_state)) {
	tp->t_state = TV_CLOSED;
	(void) tw_send(tp, (mblk_t *)NULL, (mblk_t *)NULL,
		       tp->snd_nxt, tp->rcv_nxt, TH_RST|TH_ACK);
	tcpstat.tcps_drops++;
    } else
	tcpstat.tcps_conndrops++;
    tw_close(tp);
    return;

}	/*  End of tw_drop()  */

/*
 * discard all resources
 * tell tli to send T_discon_ind upstream
 */
void
tw_close(tp)

	register struct tcpcb *tp;
{
	extern void tn_disconnected();

	ASSERT(tp);
	STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_LO, SL_TRACE,
	       "tw_close: closing");
	tw_clean(tp);
	tn_disconnected(tp);	/* Send MSG upstream */
	wakeup(tp->t_tdp);	/* Wake up anyone sleeping on close */
	tw_rmtcb(tp);		/* is it done later in tu_cleancb ?? */
	return;

}	/*  End of tw_close()  */

/*
 * discard all resource except tcpcb itself
 */
void
tw_clean(tp)

	register struct tcpcb *tp;
{
	register mblk_t *bp, *rsqbp, *bp1;

	ASSERT(tp);
	STRLOG(TCP_ID, TCPCHAN(tp->t_tdp), DPRI_LO, SL_TRACE,
	       "tw_clean : ");
	/*
	 * clean reassembly queue
	 */
	rsqbp = (mblk_t *)&tp->t_rsq;
	bp = rsqbp->b_next;
	while (bp != rsqbp) {
		bp1 = bp->b_next;
		c_deq(bp);
		freemsg(bp);
		bp = bp1;
	}
	if (tp->t_template) {
		freeb(tp->t_template);
		tp->t_template = NULL;
	}

	return;

}	/*  End of tw_clean()  */

/*
 * implementation error code: perhaps ASSERT
 */
#define	EOFF	1
#define ELEN	2
#define	EDUP	3
#define ENULL	4

/*
 * dup data 'len' byte from offset 'off'
 * return the new message if o.k, or NULL if failed
 * bp is usually q_first
 * DO cross to b_next messages
 */
mblk_t *
tw_dupmsga(bp, off, len)

	register mblk_t *bp;
	register int off;
	register int len;
{
	register mblk_t *bp1, *nbp, *last, *first;
	register int blen;
	int errcode;

	first = last = (mblk_t *)NULL;

	/* 
	 * find the place where the offset starts
	 * bp is the next pointer, bp1 is the cont pointer
	 */
	if (bp == NULL){
		errcode = ENULL;
		goto err;
	}
	do {
		bp1 = bp;
		do {
			blen = BLEN(bp1);
			if (blen > off)
				goto found;
			off -= blen;
		} while (bp1 = bp1->b_cont);
	} while (bp = bp->b_next);
	errcode = EOFF;
	goto err;
		
found:
	/* 
	 * found the place where the offset starts
	 * bp is the next, bp1 the cont ptr of the original message
	 * first and last are for the newly duped message
	 * now dup 'len' byte
	 */
	do {
		do {
			if ((nbp = dupb(bp1)) == NULL) {
				errcode = EDUP;
				goto err;
			}


	STRlog(TCP_ID, -1, DPRI_MED, SL_te,
		"tw_dupmsga: bp=%x nbp=%x rc=%x "
                ,bp1 , nbp, nbp->b_datap->db_ref );
			if (first == NULL) {
				first = nbp;
				first->b_rptr += off;
			} else
				last->b_cont = nbp;
			last = nbp;

			len -= BLEN(last);
			if (len <= 0) {		/* done */
				last->b_wptr += len;
				return(first);
			}
		} while (bp1 = bp1->b_cont);
	} while (bp1 = bp = bp->b_next);
	errcode = ELEN;

err:
	/*
	 * Probably done by set_persist.
	 * We don't have anything to force.
	 */
	if (first)
		freemsg(first);
	STRlog(TCP_ID, -1, DPRI_MED, SL_te,
		"dupmsga: err %x, len=%x, ", errcode, len);
	STRlog(TCP_ID, -1, DPRI_MED, SL_te, "off=%d\n", off);
	return(NULL);

}	/*  End of tw_dupmsga()  */

/*
 * drop 'len' character from the write q since they are acked
 * 'len' bytes may across message boundary
 */
void
tw_acked(tdp, len)

	register struct tcpd *tdp;
	register int len;
{
	register queue_t *q;		/* write q */
	register mblk_t *bp = (mblk_t *)NULL, *bp1;
	register i;

	ASSERT(tdp && len <= tdp->td_wcount);
	if ((q = tdp->td_rdq) == NULL)
		return;			/* may be ASSERT */
	else
		q = WR(q);
	tdp->td_wcount -= len;

	while (len > 0 && (bp = getq(q))) {
cont:
		i = min(BLEN(bp), len);
		bp->b_rptr += i;
		len -= i;
		if (bp->b_rptr < bp->b_wptr)	/* done: len ==0 */
			break;
		bp1 = bp->b_cont;
		bp->b_cont = NULL;
		freeb(bp);
		if (bp = bp1)
			goto cont;		/* next block */
	}					/* next message */

	ASSERT(len==0);
	/*
	 * If there is stuff left in the
	 * Buffer then put it back at the
	 * beginning.
	 */
	if (bp)
		putbq(q, bp);
#ifdef NOTDEF
	else if (tdp->td_wcount == 0 && tdp->td_flags & TD_CLOSING) {
		struct tcpcb *tp;
		/*
		 * If the user did a bunch of write and did a
		 * close. Then we will just nuke this guy in
		 * 1 clock tick.
		 */
		tp = mtod ((tdp->td_tcpcb), struct tcpcb *);
		tp->t_timer[TCPT_2MSL] = 1;
	}
#endif /* NOTDEF */

}	/*  End of tw_acked()  */

/*
 * Dump a tcpcb.
 * this may not be necessary: will be done later in tu_cleancb
 */
void
tw_rmtcb(tp)

	register struct tcpcb *tp;
{
	register mblk_t *bp;

	STRlog(TCP_ID, 0, DPRI_MED, SL_TRACE, "tw_rmtcb..");
	/**
	 ** Remove from the active list of tcpcb's
	 **/
	for (bp = (mblk_t *)tcpcb.b_next; bp != (mblk_t *)&tcpcb; bp = bp->b_next) {
		if ((struct tcpcb *)bp->b_rptr == tp) {
		    if (bp == tcp_pcbcache)
			tcp_pcbcache = (mblk_t *)NULL;
		    c_deq(bp);
		    goto free;
		}
	}
	STRlog(TCP_ID, 0, DPRI_MED, SL_te, "tw_rmtcb: can't find tcpcb\n");
	return;
free:
	if (tp->t_tdp) {
		tp->t_tdp->td_tcpcb = NULL;
		wakeup(tp->t_tdp);	/* If User is sleeping */
	}
	freeb(bp);
	return;

}	/*  End of tw_rmtcb()  */

/*
 * Drop a floating TCPCB
 */
void
tw_drop_floater(bp)

	mblk_t *bp;
{
	register struct tcpcb *listner, *tp;

	tp = mtod (bp, struct tcpcb *);
	listner = mtod ((tp->t_tdp->td_tcpcb), struct tcpcb *);
	tp->t_state = TV_CLOSED;
	(void) tw_send (tp, (mblk_t *)NULL, (mblk_t *)NULL, tp->snd_nxt,
			tp->rcv_nxt, TH_RST|TH_ACK);
	(void) tw_clean (tp);
	if (--listner->t_cqlen == 0)
		listner->t_tdp->td_tstate = TS_IDLE;
	if (bp == tcp_pcbcache)
	    tcp_pcbcache = (mblk_t *)NULL;
	c_deq (bp);
	(void) freeb (bp);

}	/*  End of tw_drop_floater()  */


/*
 * set persist timer, fixed time interval (no smooth function)
 */
void
tw_setpersist(tp)
     register struct tcpcb *tp;
{
    register t = ((tp->t_srtt >> 2) + tp->t_rttvar) >> 1;

    ASSERT(tp->t_timer[TCPT_REXMT] == 0);

    /*
     * Start/restart persistance timer.
     */
    TCPT_RANGESET(tp->t_timer[TCPT_PERSIST],
		  t * tcp_backoff[tp->t_rxtshift],
		  TCPTV_PERSMIN, TCPTV_PERSMAX);
    if (tp->t_rxtshift < TCP_MAXRXTSHIFT)
	tp->t_rxtshift++;
    
    return;

}	/*  End of tw_setpersist()  */
