/*
 * @(#) Copyright 1986.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)route.c (TWG)  1.4     89/07/30 "

#include <stdio.h>
#include <sys/types.h>
#include <sys/inet.h>
#include <sys/socket.h>
#include <sys/stream.h>
#include <sys/if.h>
#include <sys/in.h>
#include <sys/in_var.h>
#include <sys/stropts.h>
#include <sys/ip.h>
#include <sys/tcp.h>
#include <sys/route.h>
#include <sys/inetioctl.h>
#include <netdb.h>
#include <errno.h>

extern	int nflag;
extern	int fdip;
extern  char *strchr();
extern	char *routename(), *netname();

/*
 * Definitions for showing gateway flags.
 */
struct bits {
	short	b_mask;
	char	b_val;
} bits[] = {
	{ RTF_UP,	'U' },
	{ RTF_GATEWAY,	'G' },
	{ RTF_HOST,	'H' },
	{ 0 }
};

static 
print_route()
{
	register struct bits *p = bits;
	struct strioctl	strioctl;
	struct rtentry route;
	struct ifnet ifnet;
	char name[16], *flags;
	struct in_addr addr;
	int count = 0;

	while (1)
	{
		/* Get next entry */
		route.rt_hash = count++;
		strioctl.ic_cmd = IPIOC_GETROUTE;
		strioctl.ic_timout = 0;
		strioctl.ic_len = sizeof route;
		strioctl.ic_dp = (char *)&route;
		if (ioctl(fdip, I_STR, &strioctl) < 0)
		{
			if (errno != EADDRNOTAVAIL)
			{
				perror( "netstat: print_route");
				return;		
			}
			break;
		}

		/* Skip unused entries */
		if ((route.rt_flags & RTF_INUSE) == 0)
			continue;

		/* Pretty print entry */
		addr = route.rt_dst;
		printf("%-20.20s ",
			(route.rt_dst.s_addr == 0) ? "default" :
			(route.rt_flags & RTF_HOST) ?
			routename(addr) : netname(addr, 0L));
		addr = route.rt_gateway;
		printf("%-20.20s ", routename(addr));
		for (flags = name, p = bits; p->b_mask; p++)
			if (p->b_mask & route.rt_flags)
				*flags++ = p->b_val;
		*flags = '\0';
		printf("%-8.8s %-6d %-10d ", name, route.rt_tos, route.rt_use);
		if (route.rt_ifp == 0)
		{
			putchar('\n');
			continue;
		}
		ifnet.if_next = route.rt_ifp;	/* tell ioctl where to look */
		strioctl.ic_cmd = IPIOC_GETIPB;
		strioctl.ic_timout = 0;
		strioctl.ic_len = sizeof ifnet;
		strioctl.ic_dp = (char *)&ifnet;
		if (ioctl(fdip, I_STR, &strioctl) < 0)
		{
			perror( "netstat: print_route");
			continue;		
		}
		printf("%s\n", ifnet.if_name/* , ifnet.ib_unit */);
	}
}

/*
 * Print routing tables.
 */
routepr(rt_net, rt_host)
	off_t rt_host, rt_net;
{
	printf("Routing tables\n");
	printf("%-20.20s %-20.20s %-8.8s %-6.6s %-10.10s %s\n",
		"Destination", "Gateway", "Flags", 
		"Tos", "Use", "Interface");
	print_route();
}

char *
routename(in)
	struct in_addr in;
{
	register char *cp;
	static char line[50];
	struct hostent *hp;
	static char domain[50];
	static int first = 1;
	char *index();

	if (first) {
		first = 0;
		if (gethostname(domain, 50) == 0 &&
		    (cp = index(domain, '.')))
			(void) strcpy(domain, cp + 1);
		else
			domain[0] = 0;
	}
	cp = 0;
	if (!nflag) {
		hp = gethostbyaddr((char *)&in, sizeof (struct in_addr),
			AF_INET);
		if (hp) {
			if ((cp = index(hp->h_name, '.')) &&
			    !strcmp(cp + 1, domain))
				*cp = 0;
			cp = hp->h_name;
		}
	}
	if (cp)
		strncpy(line, cp, sizeof(line) - 1);
	else {
#define C(x)	((x) & 0xff)
		in.s_addr = ntohl(in.s_addr);
		sprintf(line, "%u.%u.%u.%u", C(in.s_addr >> 24),
			C(in.s_addr >> 16), C(in.s_addr >> 8), C(in.s_addr));
	}
	return (line);
}

/*
 * Return the name of the network whose address is given.
 * The address is assumed to be that of a net or subnet, not a host.
 */
char *
netname(in, mask)
	struct in_addr in;
	u_long mask;
{
	char *cp = 0;
	static char line[50];
	struct netent *np = 0;
	u_long net;
	register i;
	int subnetshift;

	i = ntohl(in.s_addr);
	if (!nflag && i) {
		if (mask == 0) {
			if (IN_CLASSA(i)) {
				mask = IN_CLASSA_NET;
				subnetshift = 8;
			} else if (IN_CLASSB(i)) {
				mask = IN_CLASSB_NET;
				subnetshift = 8;
			} else {
				mask = IN_CLASSC_NET;
				subnetshift = 4;
			}
			/*
			 * If there are more bits than the standard mask
			 * would suggest, subnets must be in use.
			 * Guess at the subnet mask, assuming reasonable
			 * width subnet fields.
			 */
			while (i &~ mask)
				mask = (long)mask >> subnetshift;
		}
		net = i & mask;
		while ((mask & 1) == 0)
			mask >>= 1, net >>= 1;
		np = getnetbyaddr(net, AF_INET);
		if (np)
			cp = np->n_name;
	}	
	if (cp)
		strncpy(line, cp, sizeof(line) - 1);
	else if ((i & 0xffffff) == 0)
		sprintf(line, "%u", C(i >> 24));
	else if ((i & 0xffff) == 0)
		sprintf(line, "%u.%u", C(i >> 24) , C(i >> 16));
	else if ((i & 0xff) == 0)
		sprintf(line, "%u.%u.%u", C(i >> 24), C(i >> 16), C(i >> 8));
	else
		sprintf(line, "%u.%u.%u.%u", C(i >> 24),
			C(i >> 16), C(i >> 8), C(i));
	return (line);
}

/*
 * Print routing statistics
 */
rt_stats(off)
	off_t off;
{
	struct strioctl strioctl;
	struct rtstat rtstat;

	strioctl.ic_cmd = IPIOC_GETRTSTAT;
	strioctl.ic_timout = 0;
	strioctl.ic_len = sizeof rtstat;
	strioctl.ic_dp  = (char *)&rtstat;
	if (ioctl(fdip, I_STR, &strioctl )<0) {
		if (errno != EADDRNOTAVAIL )
			perror( "netstat: rt_stats" );
		return;		
	}

	printf("routing:\n");
	printf("\t%d bad routing redirect%s\n",
		rtstat.rts_badredirect, plural(rtstat.rts_badredirect));
	printf("\t%d dynamically created route%s\n",
		rtstat.rts_dynamic, plural(rtstat.rts_dynamic));
	printf("\t%d new gateway%s due to redirects\n",
		rtstat.rts_newgateway, plural(rtstat.rts_newgateway));
	printf("\t%d destination%s found unreachable\n",
		rtstat.rts_unreach, plural(rtstat.rts_unreach));
	printf("\t%d use%s of a wildcard route\n",
		rtstat.rts_wildcard, plural(rtstat.rts_wildcard));
}
