/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)so_socket.c        1.2      09:28:21 - 89/06/29 "

#include "sys/types.h"
#include "sys/param.h"
#include "sys/stream.h"
#include "sys/inet.h"
#include "sys/inetioctl.h"
#include "sys/socket.h"
#include "sys/in.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#include "sys/errno.h"
#include "sys/somod.h"
#include "sys/un.h"

extern mblk_t *so_allocb();

/* 
 * called for each socket call.
 */
so_create(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct iocblk *ip;
	register struct S_sock_req *Sp;		
	int size, size2;
	
	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if ((mp->b_cont == NULL) ||
	    (ip->ioc_count < sizeof(struct S_sock_req))) {
		return (EINVAL);
	}

	Sp = (struct S_sock_req *)mp->b_cont->b_rptr;
	size2 = (Sp->domain == AF_INET) ?
		sizeof(struct sockaddr_in) : AFUADDRLEN;
	size = sizeof(union T_req) + size2;

	if ((so->so_ctlbuf = (mblk_t *)allocb(size),BPRI_LO) == NULL) 
		return (ENOBUFS);

	so->so_domain = Sp->domain;
	so->so_type = Sp->type;
	so->so_proto = Sp->proto;
	so->so_state = 0;
	so->so_options = 0;
	so->so_prim = 0;
	so->so_iocsave = NULL;
	so->so_conq = NULL;
	so->so_oob = NULL;
	/*
	 * For non-SOCK_STREAM sockets disable 
	 * the read queue service routine.
	 */
	if (Sp->type != SOCK_STREAM)
		noenable(RD(q));	
	ip->ioc_count = 0;
	return 0;
}

/*
 * get socket info 
 */
so_getinfo(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct iocblk *ip;
	register struct s_info *info;
	
	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if ((mp->b_cont == NULL) ||
	    (ip->ioc_count < sizeof(struct s_info))) {
		return (EINVAL);
	}

	info = (struct s_info *)mp->b_cont->b_rptr;
	info->domain = so->so_domain;
	info->type = so->so_type;
	info->proto = so->so_proto;
	info->flags = so->so_flags;
	info->state = so->so_state;
	info->options = so->so_options;
	info->somsgsz = somsgsz;
	return 0;
}

/*
 * Binds a socket by sending down a T_BIND_REQ to
 * the lower driver. It may either be called as the 
 * result of user initiated bind call or internally 
 * by somod module. 
 * q is the write queue.
 * mp is NULL for all ineternally generated bind calls. 
 */
so_bind(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct iocblk *ip;
	register struct T_bind_req *Tp;
	register struct so_socket *so;
	register mblk_t *bp;
	int size;

	ASSERT(q && q->q_ptr);
	so = (struct so_socket *)q->q_ptr;
	size = sizeof(struct T_bind_req);

	if (mp) {
		ip = (struct iocblk *)mp->b_rptr;
		size += ip->ioc_count;
	}

	if ((bp = so_allocb(so->so_ctlbuf, size)) == NULL) 
		return (ENOBUFS);

	Tp = (struct T_bind_req *)bp->b_wptr;
	Tp->PRIM_type = T_BIND_REQ;
	Tp->ADDR_offset = sizeof(struct T_bind_req);
	Tp->CONIND_number = 0;
	if (mp && mp->b_cont) {
		bcopy(mp->b_cont->b_rptr, (char *)Tp + Tp->ADDR_offset,
		      ip->ioc_count);
		Tp->ADDR_length = ip->ioc_count;
	} else
		Tp->ADDR_length = 0;

	bp->b_wptr += size;
	MTYPE(bp) = M_PROTO;
	so->so_prim = T_BIND_REQ;

	/*
	 * set these only if this is a user 
	 * initiated bind request.
	 */
	if (mp) {
		so->so_iocsave = mp;
		so->so_flags |= WAITIOCACK;
	}

	putnext(q, bp);
	return 0;
}

/*
 * so_listen - pass down the ioctl message 
 * to TCP to set the conection queue length
 */
so_listen(q, mp) 
	queue_t *q;
	mblk_t	*mp;
{
	register struct so_socket *so;

	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;
	if (so->so_type != SOCK_STREAM)
		return (EOPNOTSUPP);

	so->so_iocsave = mp;
	so->so_flags |= WAITIOCACK;

	/* bind if needed */
	if (!(so->so_state & SS_ISBOUND))
		return(so_bind(q,0));

	putnext(q, mp);
	return 0;
}

/* 
 * so_connect - initiate the connection process.
 */
so_connect(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct so_socket *so;

	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;
		
	if (so->so_state & (SS_ISCONNECTED | SS_ISCONNECTING)) 
		return(EISCONN);

	so->so_iocsave = mp;
	so->so_flags |= WAITIOCACK;

	/* bind if needed */
	if (!(so->so_state & SS_ISBOUND)) 
		return(so_bind(q, 0));

	if (so->so_type == SOCK_STREAM)
		return (so_connecting(q, mp));
	else 
		return (so_setingpeer(q, mp));
}


/*
 * so_connecting - send down a T_conn_req 
 */
so_connecting(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct iocblk *ip;
	register struct T_conn_req *Tp;
	register struct so_socket *so;
	register mblk_t *bp;
	int size;

	ASSERT(q && q->q_ptr && mp);

	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if ((mp->b_cont == NULL) ||
	    (ip->ioc_count < sizeof(struct sockaddr_in))) 
		return (EINVAL);
	
	size = sizeof(struct T_conn_req) + ip->ioc_count;

	if((bp = so_allocb(so->so_ctlbuf, size)) == NULL) 
		return (ENOBUFS);

	Tp = (struct T_conn_req *)bp->b_wptr;
	Tp->PRIM_type = T_CONN_REQ;
	Tp->DEST_length = ip->ioc_count;
	Tp->DEST_offset = sizeof(struct T_conn_req);
	Tp->OPT_length = Tp->OPT_offset = 0;
	bcopy(mp->b_cont->b_rptr, (char *)Tp + Tp->DEST_offset,
	      ip->ioc_count);
	bp->b_wptr += size;
	MTYPE(bp) = M_PROTO;
	so->so_prim = T_CONN_REQ;
	so->so_state &= ~(SS_ISCONNECTED | SS_ISDISCONNECTING);
	so->so_state |= SS_ISCONNECTING;
	putnext(q, bp);
	return 0;
}

/*
 * so_bindqptr - bind and get the queue pointer
 */ 
so_bindqptr(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct so_socket *so;

	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;

	so->so_iocsave = mp;
	so->so_flags |= WAITIOCACK;
	return(so_bind(q, 0));
}


/*
 * so_accept - accept incoming connections
 */
so_accept(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct iocblk *ip;
	register struct so_socket *so;

	ASSERT(q && q->q_ptr && mp);
	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if ((so->so_state & SS_NBIO) && so->so_conq == NULL)
		return (EWOULDBLOCK);

	so->so_iocsave = mp;
	so->so_flags |= WAITIOCACK;

	return (so_accepting(q, mp));
}

/*
 * so_accept - Accept the first connection indication on 
 * so_conq by sending down a T_conn_res for it to TCP.
 */
so_accepting(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct so_socket *so;
	register struct T_conn_res *Tp;
	struct iocblk *ip;
	struct T_conn_ind *tp;
	mblk_t	*bp;

	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if (so->so_conq == NULL)
		return 0;	

	/* check so_error XXX */

	if (so->so_state & SS_CANTRCVMORE) 
		return (ECONNABORTED);

	/*
	 * send down a T_conn_res for the 
	 * first T_conn_ind on the so_conq.
	 */

	tp = (struct T_conn_ind *)so->so_conq->b_rptr;
	if (tp->SRC_length > ip->ioc_count)
		return (EFAULT);

	if((bp = so_allocb(so->so_ctlbuf, sizeof(struct T_conn_res))) == NULL) 
		return (ENOBUFS);

	Tp = (struct T_conn_res *)bp->b_wptr;
	Tp->PRIM_type = T_CONN_RES;
	Tp->QUEUE_ptr = *(queue_t **)so->so_iocsave->b_cont->b_rptr; 
	Tp->OPT_length = Tp->OPT_offset = 0;
	Tp->SEQ_number = tp->SEQ_number;
	bp->b_wptr += sizeof(struct T_conn_res);
	MTYPE(bp) = M_PROTO;
	so->so_prim = T_CONN_RES;
	putnext(q, bp);
	return 0;
}


/*
 * so_setingpeer - sends down the SIOC_CONNECT
 * ioctl to the lower module.
 */
so_setingpeer(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct iocblk *ip;
	register struct so_socket *so;

	ASSERT(q && q->q_ptr && mp);

	so = (struct so_socket *)q->q_ptr;
	ip = (struct iocblk *)mp->b_rptr;

	if ((mp->b_cont == NULL) ||
	    (ip->ioc_count < sizeof(struct sockaddr_in))) 
		return (EINVAL);
	
	so->so_state &= ~SS_ISCONNECTED;
	so->so_state |= SS_ISCONNECTING;
	putnext(q, mp);
	return 0;
}


#define FREAD	1
#define FWRITE 	2
/*
 * shutdown
 */
so_shutdown(q, mp)
	queue_t *q;
	mblk_t	*mp;
{
	register struct so_socket *so;
	register int how;
	register mblk_t *bp;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;

	how = *(int *)mp->b_cont->b_rptr;
	if (how < 0 || how > 2)
		return (EINVAL);

	how++;
	if (how & FREAD) {
		/*
		 * shutdown the read side
		 * flush stream head read queue
		 */
		if (putctl1(RD(q)->q_next, M_FLUSH, FLUSHR) == 0) 
			return (ENOBUFS);

		if (so->so_type != SOCK_STREAM) {
			/*
			 * for UDP/RAW send up a zero length
			 * message, and remove any address
			 * (T_UNITDATA_IND parts) from the queue.
			 */
			if ((bp = allocb(4, BPRI_MED)) == NULL)
				return (ENOBUFS);
			putnext(RD(q), bp);
			flushq(RD(q), FLUSHALL);
		}
		so->so_state |= SS_CANTRCVMORE;
	}

	if (how & FWRITE) {
		/*
		 * shutdown the write side. 
		 * mark the connectionas being incapable 
		 * of further output.
		 */
		if (so->so_type == SOCK_STREAM) {
			/* send down an ord_rel request */
			
			if ((bp = allocb(sizeof(struct T_ordrel_req),
			     BPRI_MED)) == NULL) 
				return (ENOBUFS);
			MTYPE(bp) = M_PROTO;
			*(long *)bp->b_wptr = T_ORDREL_REQ;
			bp->b_wptr += sizeof(long);
			putnext(q, bp);
		}
		sowriteblock(q);
		so->so_state |= SS_CANTSENDMORE;
	}

	return 0;
}
#undef FREAD
#undef FWRITE
