/*
 * @(#) Copyright 1989.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)so_tp.c        1.1      12:32:50 - 89/06/19 "

#include "sys/types.h"
#include "sys/param.h"
#include "sys/signal.h"
#include "sys/stream.h"
#include "sys/inet.h"
#include "sys/inetioctl.h"
#include "sys/socket.h"
#include "sys/in.h"
#include "sys/stropts.h"
#include "sys/tihdr.h"
#include "sys/tiuser.h"
#include "sys/strlog.h"
#include "sys/debug.h"
#include "sys/errno.h"
#include "sys/somod.h"

/*
 * so_tp.c
 * Functions called upon the receipt of 
 * messages from the transport layer below.
 */
void so_proto();
void so_bind_ack();
void so_ok_ack();
void so_error_ack();
void so_disc_ind();
void so_merror();
void so_ud_ind();
void W_sel_happy();


void
so_proto(q, mp)
	register queue_t *q;
	register mblk_t *mp;
{
	register struct so_socket *so;
	register union T_primitives *Tp;
	register struct iocblk *ip;
	register mblk_t *bp;
	int error;

/* LABEL(soproto) */
	
	/* assert enough data to determine type */
	ASSERT((mp->b_wptr - mp->b_rptr) >= sizeof(long));

	so = (struct so_socket *)q->q_ptr;
	Tp = (union T_primitives *)mp->b_rptr;

	switch (Tp->type) {

	case T_EXDATA_IND:
		(void) so_exdata_ind(q, mp, DO_Q);
		return;

	case T_DATA_IND:
		if (msgdsize(mp) == 0) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"so_proto: got 0 len T_DATA_IND, dropping\n");
			freemsg(mp);
			return;
		}
		(void) so_data_ind(q, mp, DO_Q);
		return;

	case T_ORDREL_IND:
		/* Generate EOF */	
		MTYPE(mp) = M_DATA;
		mp->b_wptr = mp->b_rptr;
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_proto: got ORDREL, sending up 0 len message to gen EOF");
		(void) so_data_ind(q, mp, DO_Q);
		return;

	case T_DISCON_IND:
		so_disc_ind(q, mp);
		return;

	case T_ERROR_ACK:
		if (!(so->so_flags & WAITIOCACK) || 
		    (so->so_iocsave == NULL) ||
		    (so->so_prim != Tp->error_ack.ERROR_prim)){
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"so_proto: unexpected error_ack, dropping\n");
			freemsg(mp);
			return;
		}
		so_error_ack(q, mp);
		return;

	case T_OK_ACK:
		if (!(so->so_flags & WAITIOCACK) || 
		    (so->so_iocsave == NULL) ||
		    (so->so_prim != Tp->ok_ack.CORRECT_prim)){
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"so_proto: unexpected ok_ack, dropping\n");
			freemsg(mp);
			return;
		}
		so_ok_ack(q, mp);
		return;
	
	case T_BIND_ACK:
		if (!(so->so_flags & WAITIOCACK) || 
		    (so->so_iocsave == NULL) ||
		    (so->so_prim != T_BIND_REQ)) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"so_proto: unexpcted bind_ack, dropping\n");
			freemsg(mp);
			return;
		}
		so_bind_ack(q, mp);
		return;

	case T_CONN_CON:
		if (!(so->so_flags & WAITIOCACK) || 
		    (so->so_iocsave == NULL) ||
		    (so->so_prim != T_CONN_REQ)) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"so_proto: unexpcted T_conn_con, dropping\n");
			freemsg(mp);
			return;
		}
		so->so_state &= ~(SS_ISCONNECTING | SS_ISDISCONNECTING);
		so->so_state |= SS_ISCONNECTED;
		sowriteunblock(WR(q));	
		freemsg(mp);
		so_ack(q, 0);
		return;

	case T_CONN_IND:
		so_enq(&so->so_conq, mp);
		if (qsize(q->q_next) == 0) 	/* make select happy */
			W_sel_happy(q, mp);

		if (!(so->so_flags & WAITIOCACK) ||
		    (so->so_iocsave == NULL) ||
		    (SIOC_CMD(so) != SIOC_ACCEPT)) { /* XXX ? */
			/* not necessarly issued T_conn_res yet */
			return;
		}
		error = so_accepting(WR(q), so->so_iocsave);
		if (error) 
			so_ack(q, error);
		return;

	case T_UNITDATA_IND:
		so_ud_ind(q, mp);
		return;

	case T_UDERROR_IND:
		freemsg(mp);
		return;

	default:
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_proto: dropping T_primtive with type=%d\n", Tp->type);
		freemsg(mp);
		return;
	}
}


/*
 * so_bind_ack - Process the T_BIND_ACK
 */
void
so_bind_ack(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct iocblk *ip;
	register struct T_bind_ack *tp;
	union T_primitives *Tp;
	register mblk_t *bp;
	int error;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;
	
	if (!(so->so_flags & WAITIOCACK) || 
	    (so->so_iocsave == NULL) ||
	    (so->so_prim != T_BIND_REQ)) {
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_bind_ack: unexpcted bind_ack, dropping\n");
		freemsg(mp);
		return;
	}

	switch(SIOC_CMD(so)) {

	case SIOC_BIND:
	case SIOC_BINDQPTR:
		/*
		 * send up the bind ack in an ioctl ack.
		 */ 
		Tp = (union T_primitives *)mp->b_rptr;
		tp = &Tp->bind_ack;
		bp = so->so_iocsave;

		if (bp->b_cont == NULL) {
			freemsg(mp);
			so->so_state |= SS_ISBOUND;
			so_ack(q, 0);
			break;
		}
			
		ip = (struct iocblk *)bp->b_rptr;
		if (tp->ADDR_length > ip->ioc_count) {
			freemsg(mp);
			so_ack(q, EFAULT);
			break;
		}

		if (SIOC_CMD(so) == SIOC_BIND) 
			bcopy((char *)tp + tp->ADDR_offset,
			      bp->b_cont->b_rptr, tp->ADDR_length);
		else {
			/* return TCP queue pointer instead */
			queue_t *nq;
			
			for (nq = WR(q); nq->q_next; nq = nq->q_next);
			nq = RD(nq);
			*(queue_t **)bp->b_cont->b_rptr = nq;
			
			/* zero out the rest */
			bzero((caddr_t)(bp->b_cont->b_rptr + sizeof(queue_t *)),
			      tp->ADDR_length - sizeof(queue_t *));
		}

		if (tp->ADDR_length < ip->ioc_count)
			bp->b_cont->b_wptr = 
			bp->b_cont->b_rptr + tp->ADDR_length;

		ip->ioc_count = tp->ADDR_length;
		freemsg(mp);
		so->so_state |= SS_ISBOUND;
		so_ack(q, 0);
		break;

	case SIOC_LISTEN:
		freemsg(mp);
		so->so_state |= SS_ISBOUND;
		so->so_prim = 0;
		putnext(WR(q), so->so_iocsave);
		break;

	case SIOC_CONNECT:
		freemsg(mp);
		so->so_state |= SS_ISBOUND;
		so->so_prim = 0;
		if (so->so_type == SOCK_STREAM) 
			error = so_connecting(WR(q), so->so_iocsave);
		else
			error = so_setingpeer(WR(q), so->so_iocsave);

		if (error) 
			so_ack(q, error);
		break;

	case SIOC_SENDTO:
	case SIOC_SENDTO2:
		freemsg(mp);
		so->so_state |= SS_ISBOUND;
		so->so_prim = 0;
		error = so_sendingto(WR(q), so->so_iocsave);
		if (error)
			so_ack(q, error);
		break;

	default:
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_bind_ack: freeing T_bind_ack message !!\n");
		freemsg(mp);
		break;
	}
	
	return;
}

/*
 * so_ok_ack - process the ok_ack's 
 * received from downstream.
 */
void
so_ok_ack(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	queue_t *aq;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;

	switch (SIOC_CMD(so)) {

	case SIOC_CONNECT:
		freemsg(mp);
		if (so->so_state & SS_NBIO) {
			so_ack(q, EINPROGRESS);
			return;
		}
		/* nothing to do here. */
		break;

	case SIOC_ACCEPT:
	{
		register struct T_conn_ind *tp;
		register mblk_t *bp;
		struct iocblk *ip;
		queue_t *aq;

		freemsg(mp);
		bp = so->so_iocsave;
		/* get the qptr for the accepting somod channel */
		aq = *(queue_t **)bp->b_cont->b_rptr;	
/* LABEL(accept1) */
		aq = aq->q_next;

		/* record the source address from conq */
		ip = (struct iocblk *)bp->b_rptr;
		tp = (struct T_conn_ind *)so->so_conq->b_rptr;
		bcopy((char *)tp + tp->SRC_offset, 
		      bp->b_cont->b_rptr, tp->SRC_length);
		if (tp->SRC_length < ip->ioc_count) {
			bp->b_cont->b_wptr = bp->b_cont->b_rptr+tp->SRC_length;
			ip->ioc_count = tp->SRC_length;
		}

		/* remove it from the conq */
		so_deq(&so->so_conq, so->so_conq);
		if (so->so_conq && qsize(q->q_next) == 0) 
			W_sel_happy(q, so->so_conq);

		/* make the accepting channel writeable */
		so = (struct so_socket *)aq->q_ptr;
		so->so_state |= SS_ISCONNECTED;
		sowriteunblock(WR(aq));	
		so_ack(q, 0);
		break;
	}

	default:
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_ok_ack: No SIOC cmd for ok_ack, dropping\n");
		freemsg(mp);
	}

	return;
}


/*
 * so_error_ack - process the error_ack's 
 * received from downstream.
 */
void
so_error_ack(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct iocblk *ip;
	register union T_primitives *Tp;
	mblk_t *bp;
	int error;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;
	Tp = (union T_primitives *)mp->b_rptr;

	if (Tp->error_ack.TLI_error == TBADSEQ) {
		/*
		 * to cover the conn_ind for which we
		 * did a T_conn_res, but a T_disc_ind
		 * or TCP timers has zaped its 
		 * counterpart in TCP.
		 * Do not ack the accept ioctl.
		 */
		freemsg(mp);
		/* remove it from the conq */
		so_deq(&so->so_conq, so->so_conq);
		if (so->so_conq && qsize(q->q_next) == 0) 
			W_sel_happy(q, so->so_conq);
		return;
	}

	/* Set errno (and t_errno) via ioctl ack */
	error = Tp->error_ack.UNIX_error;
	if (error == 0)
		/*
		 * lower module didn't do its job 
		 * try to set it here.
		 */
		error = so_errset(q, Tp->error_ack.TLI_error);

	error |= Tp->error_ack.TLI_error << 8;
	freemsg(mp);
	so_ack(q, error);
	return;
}


/*
 * so_disc_ind - process the T_disc_ind 
 * received from downstream.
 */
void
so_disc_ind(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct T_discon_ind *tp, *tp1;
	union T_primitives *Tp;
	mblk_t *bp;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;
	Tp = (union T_primitives *)mp->b_rptr;
	tp = &Tp->discon_ind;

	if (tp->SEQ_number == -1) {
		if ((so->so_flags & WAITIOCACK) && 
		    (so->so_iocsave != NULL) &&
		    (so->so_prim == T_CONN_REQ)) {
			/*
			 * defensive measure. Currently tcp 
			 * sends up an M_ERROR prior to this.
			 */
			freemsg(mp);
			so->so_state &= ~(SS_ISCONNECTING|SS_ISCONNECTED);
			/*  also turn on SS_ISDISCONNECTING XXX */
			so_ack(q, ECONNREFUSED);     /* or ECONNRESET */
			return;
		}
		/* generate end of file */
		MTYPE(mp) = M_DATA;
		mp->b_wptr = mp->b_rptr;
		so->so_state |= SS_CANTSENDMORE; 
		(void) so_data_ind(q, mp, DO_Q);
		return;
	} else {
		/* remove this conn_ind from so_conq (if there) */
		for (bp = so->so_conq ; bp ; bp = bp->b_next) {
			Tp = (union T_primitives *)bp->b_rptr;
			tp1 = &Tp->discon_ind;
			if (tp1->SEQ_number == tp->SEQ_number) {
				/*
				 * if we have send up a 0 len msg for
				 * this, then try to flush the stream head
				 * read queue to take if off
				 */  
				if (bp->b_cont == NULL) 
					putctl1(q->q_next, M_FLUSH, FLUSHR);
					
				so_deq(&so->so_conq, bp);
				if (so->so_conq && qsize(q->q_next) == 0) 
					W_sel_happy(q, so->so_conq);
				break;
			}
		}
		return;
	}
}

/*
 * so_merror - process the M_ERROR
 * messages received from downstream.
 */
void
so_merror(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register int error;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;
	error = *mp->b_rptr;	

	STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_merror: received M_ERROR %d", error);
	if (so->so_flags & WAITIOCACK) {
		if (SIOC_CMD(so) == SIOC_CONNECT)  
			so->so_state &= ~(SS_ISCONNECTING|SS_ISCONNECTED);
		so_ack(q, error);
		freeb(mp); 	/* no need to send it up anymore */
		return;
	}
	putnext(q, mp);
}

/*
 * so_iocack - process ioctl acks coming up 
 * from downstream. for those that are captured in
 * somod clean up the state, flags, etc.
 */
void
so_iocack(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register int cmd;
	
	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;
	cmd = *(int *)mp->b_rptr;

	switch (cmd) {

	case SIOC_LISTEN:
		if (SIOC_CMD(so) != SIOC_LISTEN)
			goto bad;
		so->so_options |= SO_ACCEPTCONN;
		goto ackit;

	case SIOC_CONNECT:
		/*
		 * connect for non-SOCK_STREAM types
		 * make the channel writeable
		 */
		if (SIOC_CMD(so) != SIOC_CONNECT)
			goto bad;
		so->so_state |= SS_ISCONNECTED;
		sowriteunblock(WR(q));	
		goto ackit;	

	default:
		/* non-captured ioctls, pass up the ack */
		putnext(q, mp);
		return;
	}

bad:
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
		"so_iocack: unexpected ack for cmd type = %d, dropping\n", cmd);
		freemsg(mp);
		return;

ackit:
		so->so_prim = 0;
		so->so_iocsave = NULL;
		so->so_flags &= ~WAITIOCACK;
		putnext(q, mp);
		return;
}

void
so_iocnak(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register int cmd;
	
	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;
	cmd = *(int *)mp->b_rptr;

	switch (cmd) {

	case SIOC_LISTEN:
	case SIOC_CONNECT:
		if (SIOC_CMD(so) != cmd) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_te,
			"so_iocnak: unexpected nak for cmd type=%d, dropping\n",
			cmd);
			freemsg(mp);
		} else {
			so->so_prim = 0;
			so->so_iocsave = NULL;
			so->so_flags &= ~WAITIOCACK;
			putnext(q, mp);
		}
		return;
	
	default:
		/* non-captured ioctls, pass up the nak */
		putnext(q, mp);
		return;
	}
}


/*
 * unit data ind
 * we must have been bound or else 
 * we would not have received this
 */
void
so_ud_ind(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	register mblk_t *bp, *tmp;
	register struct so_socket *so;
	int error;
	
	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;

	/*
	 * sync with the stream head rdq
	 */
	so_sync_addr(q);

	/*
	 * queue the address part 
	 */
	bp = mp->b_cont;	
	mp->b_cont = NULL;
	insq(q, q->q_first, mp);

	/* 
	 * send up the data part
	 */
	if (BLEN(bp) == 0) {
		tmp = bp->b_cont;
		bp->b_cont = NULL;
		freemsg(bp);
		bp = tmp;
	}
	putnext(q, bp);
	
	/* activate blocked recvfrom call */
	if ((so->so_flags & WAITIOCACK) &&
	    (so->so_iocsave != NULL) &&
	    (SIOC_CMD(so) == SIOC_RECVFROM)) {
		error = so_recvit(q, so->so_iocsave);
		if (error)
			so_ack(q, error);
	}
	return;
}


/*
 * so_exdata_ind - called when T_exdata_ind 
 * are received or via sorsrv().  
 * generates M_SIG if it is the first one.
 * if q_flg is on messages blocked by flow 
 * control are queued. returns -1 if a 
 * message needs to be queued.
 */
int
so_exdata_ind(q, mp, q_flg)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register struct T_exdata_ind *tp;
	register mblk_t *bp;

	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;

	if ((so->so_flags & OOB) == 0) {
		/* 
		 * new out-of-band data is arrived
		 * sendup M_SIG to generate SIGURG
		 */
		if (mp->b_cont == NULL) {
			/* no data, (0 byte T_EXDTATA_IND from TCP) */
			MTYPE(mp) = M_PCSIG;
			*mp->b_rptr = SIGURG;
			mp->b_wptr = mp->b_rptr + 1;
			so->so_flags |= OOB;
			putnext(q, mp);
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
			"so_exdata_ind: got 0 t_exdata_ind, sendup M_SIG");
			return 0;
		} else {
			/*
			 * This should not happen, since TCP should
			 * send us a 0 byte T_EXDATA_IND first.
			 */	
			if ((bp = allocb(1, BPRI_MED)) == NULL) {
				/* let sorsrv try this later */
				if (q_flg)
					putq(q, mp);
				return -1;
			}
			MTYPE(mp) = M_PCSIG;
			*bp->b_rptr = SIGURG;
			bp->b_wptr = bp->b_rptr + 1;
			so->so_flags |= OOB;
			putnext(q, bp);
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
			"so_exdata_ind:got non-0 t_exdata_ind !!,sendup M_SIG");
		}
	} else if (so->so_state & SS_RCVATMARK) {
		/*
		 * new oob is arrived, queue it for
		 * later processing (SO_OOBINLINE)
		 */
		if (q_flg)
			putq(q, mp);
		return -1;
	}

	if (!(bp = mp->b_cont))
		return 0;    /* no data */
	
	/*
	 * This is the data ahead of the urgent,
	 * data, continue sending these up,
	 * holding on to the most recent one.
	 */
	if (!so->so_oob) {
		mp->b_cont = NULL;
		freemsg(mp);
		so->so_oob = bp;
		return 0;
	} else {
		if (canput(q->q_next)) {
			putnext(q, so->so_oob);
			mp->b_cont = NULL;
			freemsg(mp);
			so->so_oob = bp;
			return 0;
		} else {
			if (q_flg)
				putq(q, mp);
			return -1;
		}
	}
}

/*
 * so_data_ind - called when T_DATA_INDs are
 * received from downstream, as well as thru
 * sorsrv(). if q_flg is on messages blocked 
 * by flow control are queued. 
 * returns -1 if a message needs to be queued.
 */
int
so_data_ind(q, mp, q_flg)
	queue_t *q;
	mblk_t *mp;
{
	register struct so_socket *so;
	register mblk_t *bp;
	
	ASSERT(q && mp);
	so = (struct so_socket *)q->q_ptr;

	if (so->so_state & SS_CANTRCVMORE) {
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
		"so_data_ind: CANTRCVMORE - dropping T_DATA_IND");
		freemsg(mp);
		return 0;
	}	

	if (so->so_flags & OOB) {
		if ((so->so_state & SS_RCVATMARK) == 0) {
			/*
			 * send up the data in so_oob 
			 * minus the urgent (last) byte 
			 */
			if (!canput(q->q_next) || !(bp = dupmsg(so->so_oob))) {
				if (q_flg)
					putq(q, mp);
				return -1;
			}
			bp->b_wptr -= 1;
			putnext(q, bp);
			so->so_oob->b_rptr = so->so_oob->b_wptr - 1;
			so->so_state |= SS_RCVATMARK;
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
			"so_data_ind: reached at mark");
		}
		/* queue the incomining data */
		if (q_flg)
			putq(q, mp);
		return -1;
	}
		
	/* normal data, try to send it up */
	if (canput(q->q_next) && (!q_flg || !q->q_first)) {
		bp = mp;
		if (MTYPE(mp) == M_PROTO) {
			/* strip off the M_PROTO portion */	
			bp = mp->b_cont;
			mp->b_cont = NULL;
			freemsg(mp);
		}
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
		"so_data_ind: sending up %d bytes of data", msgdsize(bp));
		putnext(q, bp);
		return 0;
	} else {
		STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
		"so_data_ind: normal data - canput failed");
		if (q_flg) {
			STRlog(SOMOD_ID, SOCHAN(so), DPRI_LO, SL_TRACE,
			"so_data_ind: queuing %d bytes of data", msgdsize(mp));
			putq(q, mp);
		}
		return -1;
	}
}

void
W_sel_happy(q, mp)
	queue_t *q;
	mblk_t *mp;
{
	mblk_t *zp;

	zp = mp->b_cont;
	if (!zp) {
		STRlog(SOMOD_ID, -1, DPRI_LO, SL_te,
			"W_sel_happy: zp was NULL !!");
		return;
	}
	mp->b_cont = NULL;
	MTYPE(zp) = M_DATA;			   /* overkill */
	zp->b_rptr = zp->b_wptr = zp->b_datap->db_base;	
	putnext(q, zp);
}
