/*
 * @(#) Copyright 1988.  The Wollongong Group, Inc.  All Rights Reserved.
 */

#ident "@(#)rwhod.c (TWG)  1.3     89/09/19 "

#ifndef lint
static char sccsid[] = "@(#)rwhod.c 5.9 (Berkeley) 3/5/86";
#endif /* not lint */

#include <sys/types.h>
#include <sys/stat.h>

/* AT&T streams includes */
#include <tiuser.h>
#include <sys/stream.h>
#include <sys/stropts.h>

/* TWG public includes */
#include <sys/inet.h>
#include <sys/socket.h>
#include <sys/if.h>
#include <sys/in.h>
#include <sys/in_var.h>
#include <sys/ip.h>
#include <sys/udp.h>
#include <sys/inetioctl.h>
#include <netdb.h>
#include <syslog.h>

/* TWG private includes */
#include "rwhod.h"

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <utmp.h>
#include <ctype.h>
#include <sys/fcntl.h>

extern	int errno;
extern  char *sys_errlst[];

/* JTH The Arix kernel is typically called /arix instead of /unix */

#ifdef ARIX
#define SLASH_UNIX	"/arix"
#else
#define SLASH_UNIX	"/unix"
#endif /* ARIX */

/*
 * Alarm interval. Don't forget to change the down time check in ruptime
 * if this is changed.
 */
#define AL_INTERVAL (3 * 60)

#define	RWHODIR		"/usr/spool/rwho"
#define	WHDRSIZE	(sizeof (mywd) - sizeof (mywd.wd_we))
#define L_SET		0   /*  A BSD define for use with lseek */
#define	NL_AVENRUN	0
#define	NL_BOOTTIME	1
#define	NL_IPB		2

/*
 * We communicate with each neighbor in 
 * a list constructed by configure() at the time we're
 * started up.  Neighbors are currently 
 * directly connected via a hardware interface.
 */
struct	neighbor {
	struct	neighbor *n_next;
	char	*n_name;		/* interface name */
	char	*n_addr;		/* who to send to */
	int	n_addrlen;		/* size of address */
	int	n_flags;		/* should forward?, interface flags */
};

/* 
 * GLOBALS 
 */
char		  myname[32];
int		  s, utmpf, kmemf = -1;
struct	neighbor *neighbors = NULL;
struct 	whod 	  mywd;
struct 	servent   *sp;

#ifndef XENIX
struct	nlist nl[] = {
#ifndef	NOLOADAVG
	{ "avenrun" },
#endif
	{ "bootime" },
	{ "ipbnext" },
	0
};
#else
struct	nlist nl[] = {
#ifndef	NOLOADAVG
	{ "_avenrun",0,0},
#endif
	{ "_bootime",0,0},
	{ "_ipb",0,0},
	0
};
#endif /* XENIX */
int	onalrm();
char	*strcpy(), *malloc();
long	lseek();
int	getkmem();
struct utmp *getutent();

main()
{
	char path[64];
	struct	sockaddr_in sin;
	struct  sockaddr_in from;
	char *cp;
	extern char *index();
	int cnt, i;
#ifdef notdef
	int on = 1; 
#endif /* not notdef */

	if (getuid()) {
		fprintf(stderr, "rwhod: not super user\n");
		exit(1);
	}
	sp = getservbyname("who", "udp");
	if (sp == 0) {
		fprintf(stderr, "rwhod: udp/who: unknown service\n");
		exit(1);
	}
#ifndef DEBUG
	if (fork())
		exit(0);
	{ int fd;
	  for (fd = 0; fd < 10; fd++)
		(void) close(fd);
	  (void) open("/dev/console", 0);
	  (void) dup2(0, 1);
	  (void) dup2(0, 2);
	  setpgrp();
	}
#endif /* not DEBUG */
	if (chdir(RWHODIR) < 0) {
		perror(RWHODIR);
		exit(1);
	}
	(void) signal(SIGHUP, getkmem);
	openlog("rwhod", LOG_PID|LOG_NDELAY, LOG_USER);
	/*
	 * Establish host name as returned by system.
	 */
	if (gethostname(myname, sizeof (myname) - 1) < 0) {
		syslog(LOG_ERR, "gethostname: %m");
		exit(1);
	}
	/* 
	 * If domain name, truncate domain portion.
	 */
	if ((cp = index(myname, '.')) != NULL)
		*cp = '\0';
	strncpy(mywd.wd_hostname, myname, sizeof (myname) - 1);

	/*
	 * Make sure we have a utmp file to work with.
	 */
	utmpf = open("/etc/utmp", O_RDONLY);
	if (utmpf < 0) {
		(void) close(creat("/etc/utmp", 0644));
		utmpf = open("/etc/utmp", O_RDONLY);
	}
	if (utmpf < 0) {
		syslog(LOG_ERR, "/etc/utmp: %m");
		exit(1);
	}
	/*
	 * Make sure we're working with the latest /unix.
	 */
	getkmem();
	/*
	 * Prepare for incoming packets.
	 */
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		syslog(LOG_ERR, "socket: %m");
		exit(1);
	}

	sin.sin_family = AF_INET;
#ifdef notdef
	if (setsockopt(s, SOL_SOCKET, SO_BROADCAST, &on, sizeof (on)) < 0) {
		syslog(LOG_ERR, "setsockopt SO_BROADCAST: %m");
		exit(1);
	}
#else
	sin.sin_addr.s_addr = INADDR_ANY;
#endif /* not notdef */
	sin.sin_port = sp->s_port;
	if (bind(s, &sin, sizeof (sin)) < 0) {
		syslog(LOG_ERR, "bind: %m");
		exit(1);
	}
	/*
	 * Establish which networks to send packets to.
	 */
	if (!configure()) {
		syslog(LOG_ERR, "configuration failed");
		exit(1);
	}
	/*
	 *  Periodically emit packets.
	 */
	signal(SIGALRM, onalrm);
	onalrm();
	/*
	 * Handle incoming packets.
	 *	(1) grab them from the net;
	 *	(2) make sure they're acceptable;
	 *	(3) save their data in the appropriate RWHODIR files.
	 */
	for (;;) {
		struct whod wd;
		int cc, whod, len = sizeof (from);

		cc = recvfrom(s, (char *)&wd, sizeof (struct whod), 0,
			&from, &len);
		if (cc <= 0) {
			if (cc < 0 && errno != EINTR)
				syslog(LOG_WARNING, "recv: %m");
			continue;
		}
		if (from.sin_port != sp->s_port) {
			syslog(LOG_WARNING, "%d: bad 'from' port",
				from.sin_port);
			continue;
		}
#ifdef notdef
		if (gethostbyname(wd.wd_hostname) == 0) {
			syslog(LOG_WARNING, "%s: unknown host",
				wd.wd_hostname);
			continue;
		}
#endif /* notdef */
		if (wd.wd_vers != WHODVERSION) {
			syslog(LOG_WARNING, "bad whod version from %x",
				from.sin_addr);
			continue;
		}
		if (wd.wd_type != WHODTYPE_STATUS) {
			syslog(LOG_WARNING, "bad whod type from %x",
				from.sin_addr);
			continue;
		}
		if (!verify(wd.wd_hostname)) {
			syslog(LOG_WARNING, "malformed host name from %x",
				from.sin_addr);
			continue;
		}
		(void) sprintf(path, "whod.%s", wd.wd_hostname);
		whod = open(path, O_WRONLY | O_CREAT | O_TRUNC, 0644);
		if (whod < 0) {
			syslog(LOG_WARNING, "%s: %m", path);
 		continue;
		}
		wd.wd_sendtime = ntohl(wd.wd_sendtime);
		wd.wd_loadav[0] = ntohl(wd.wd_loadav[0]);
		wd.wd_loadav[1] = ntohl(wd.wd_loadav[1]);
		wd.wd_loadav[2] = ntohl(wd.wd_loadav[2]);
		wd.wd_boottime = ntohl(wd.wd_boottime);
		cnt = sizeof(wd.wd_we) / sizeof(struct whoent);
		for (i = 0; i < cnt; i++) {
		    wd.wd_we[i].we_utmp.out_time = ntohl(wd.wd_we[i].we_utmp.out_time);
		    wd.wd_we[i].we_idle	= ntohl(wd.wd_we[i].we_idle);
	}
		
		(void) time(&wd.wd_recvtime);
		(void) write(whod, (char *)&wd, cc);
		(void) close(whod);
	}
}	/* end of main() */

/*
 * Check out host name for unprintables
 * and other funnies before allowing a file
 * to be created.  Sorry, but blanks aren't allowed.
 */
int
verify(name)
	register char *name;
{
	register int size = 0;

	while (*name) {
		if (!isascii(*name) || !(isalnum(*name) || ispunct(*name)))
			return (0);
		name++, size++;
	}
	return (size > 0);

}	/* end of verify() */

/*
 *  ONALRM():  At regular intervals, build a datagram with current user/usage 
 *  data and broadcast it.  This function has been muched changed from BSD.
 *  It now uses the SV utmpent() family of functions and no longer caches the
 *  whole of the /etc/utmp file in memory.  It is also heavily commented!
 */

int	utmptime = 0;
int	wd_we_size  = 0;
struct	utmp *utmp;
int	alarmcount;

int
onalrm()
{
	register int i;
	register struct whoent *we, *wlast;
	register struct neighbor *np;
	struct stat stb;
	int cc;
#ifndef	NOLOADAVG
	long avenrun[3];
#endif
	time_t now = time(0);

	signal(SIGALRM, onalrm);  /* For repeatability */
	/*
	 *  Periodically ensure we're working with the current UNIX,
	 *  since we look at kmem for load average info.
	 */
	if ((alarmcount %= 10) == 0)
		getkmem();
	alarmcount++;
	/*
	 *  As a precautionary measure, initialize volatile elements of 
	 *  the structure in which we build the rwhod output data.
	 *  (Don't reset hostname or boottime, set elsewhere.)
	 */
	mywd.wd_vers   = '\0';
	mywd.wd_type   = '\0';
	mywd.wd_pad[0] = '\0';
	mywd.wd_sendtime = 0;
	mywd.wd_recvtime = 0;
	mywd.wd_loadav[0] = 0;
	mywd.wd_loadav[1] = 0;
	mywd.wd_loadav[2] = 0;
	wd_we_size = sizeof(mywd.wd_we) / sizeof(struct whoent);
	for (i = 0; i < wd_we_size; i++) {
		mywd.wd_we[i].we_utmp.out_line[0] = '\0';
		mywd.wd_we[i].we_utmp.out_name[0] = '\0';
		mywd.wd_we[i].we_utmp.out_time    =   0;
		mywd.wd_we[i].we_idle	          =   0;
	}
	/*
	 *  Get and stash data for rwhod output.
	 *
	 *  Fill the individual local user data array (wd_we) in mywd.
	 *  Leave the loop when end of file is reached on /etc/utmp or
	 *  when we run out of room in wd_we[].  Note that there is no
	 *  check to see whether there might be more legitimate user entries
	 *  in /etc/utmp once we reach the end of the statically-sized wd_we[].
	 */
	we = mywd.wd_we;
	wlast = &mywd.wd_we[wd_we_size - 1];
	(void) setutent();
	while ((utmp = getutent()) != NULL && we <= wlast) {
		if (utmp->ut_type != USER_PROCESS)
			continue;
		bcopy(utmp->ut_line,we->we_utmp.out_line,sizeof(utmp->ut_line));
		bcopy(utmp->ut_user,we->we_utmp.out_name,sizeof(utmp->ut_user));
		we->we_utmp.out_time = htonl(utmp->ut_time);
		we++;
	}
	(void) endutent();
	wlast = we;
	/*
	 * This test looks silly---after all, if no one is
	 * logged on, why worry about efficiency?---but is useful on
	 * (e.g.) compute servers.
	 */
	if (mywd.wd_we[0].we_utmp.out_line[0] && chdir("/dev")) {
		syslog(LOG_ERR, "chdir(/dev): %m");
		exit(1);
	}
	/*
	 *  Set idle time for each local user.
	 */
	for (we = mywd.wd_we; we < wlast; we++) {
		if (stat(we->we_utmp.out_line, &stb) >= 0)
			we->we_idle = htonl(now - stb.st_atime);
	}
#ifndef	NOLOADAVG
	/*
	 *  Go to kmem for ruptime load average info.
	 */
#ifdef i386
	(void) lseek(kmemf, (long)nl[NL_AVENRUN].n_value, L_SET);
#else
	(void) lseek(kmemf, (long)nl[NL_AVENRUN].n_value&0x7fffffff, L_SET);
#endif /* i386 */
	(void) read(kmemf, (char *)avenrun, sizeof (avenrun));
	/*
	 *  Set load average.  Multiplied by 1000 in TWG driver, so scaled down.
	 */
	for (i = 0; i < 3; i++)
		mywd.wd_loadav[i] = htonl((u_long)(avenrun[i] / 10));
#else
	for (i = 0; i < 3; i++)
		mywd.wd_loadav[i] = (u_long)0;
#endif
	mywd.wd_sendtime = htonl(time(0));
	mywd.wd_vers = WHODVERSION;
	mywd.wd_type = WHODTYPE_STATUS;
	cc = WHDRSIZE + (wlast - mywd.wd_we) * sizeof(struct whoent);
	/*
	 *  Broadcast local data.
	 */
	for (np = neighbors; np != NULL; np = np->n_next) {
#ifndef DEBUG
		sendto(s,(char *)&mywd,cc,0,np->n_addr,np->n_addrlen);
#else
		_sendto(s, (char *)&mywd, cc, 0, np->n_addr, np->n_addrlen);
#endif /* DEBUG */
	}
	if (mywd.wd_we[0].we_utmp.out_line[0] && chdir(RWHODIR)) {
		syslog(LOG_ERR, "chdir(%s): %m", RWHODIR);
		exit(1);
	}
done:
	(void) alarm(AL_INTERVAL);
	return;

}	/* end of onalrm() */

int
getkmem()
{
	static ino_t unixino;
	static time_t unixctime;
	struct stat sb;
	struct nlist *nlp;

	if (stat( SLASH_UNIX, &sb) < 0) {
		if (unixctime)
			return;
	} else {
		if (sb.st_ctime == unixctime && sb.st_ino == unixino)
			return;
		unixctime = sb.st_ctime;
		unixino= sb.st_ino;
	}
	if (kmemf >= 0)
		(void) close(kmemf);
loop:
	for (nlp = &nl[sizeof (nl) / sizeof (nl[0]) ] ; --nlp >= nl; )  {
		nlp->n_value = 0;
		nlp->n_type = 0;
	}
	if (nlist( SLASH_UNIX, nl)) {
		syslog(LOG_WARNING, "/unix namelist botch");
		sleep(300);
		goto loop;
	}
	kmemf = open("/dev/kmem", O_RDONLY);
	if (kmemf < 0) {
		syslog(LOG_ERR, "/dev/kmem: %m");
		exit(1);
	}
	if (nl[NL_BOOTTIME].n_value == 0)
		syslog(LOG_WARNING, "no boottime");
#ifdef i386
	(void) lseek(kmemf, (long)nl[NL_BOOTTIME].n_value, L_SET);
#else
	(void) lseek(kmemf, (long)nl[NL_BOOTTIME].n_value&0x7fffffff, L_SET);
#endif /* i386 */
	(void) read(kmemf, (char *)&mywd.wd_boottime,
	    sizeof (mywd.wd_boottime));
	mywd.wd_boottime = htonl(mywd.wd_boottime);

}	/* end of getkmem() */

/*
 * Figure out device configuration and select
 * networks which deserve status information.
 */

int
configure()
{
	char buf[BUFSIZ];
	struct ifconf *ifc;
	struct ifreq ifreq, *ifr;
	struct sockaddr_in *sin;
	register struct neighbor *np;
	int ipfd, n;
	struct strioctl ioc;

	if ((ipfd = open("/dev/ip",O_RDWR)) < 0) {
		syslog(LOG_ERR, "open /dev/ip failed; unable to configure");
		return (0);
	}
		
	ifc = (struct ifconf *)buf;
	ifc->ifc_len = sizeof (buf) - 4;
	ioc.ic_cmd = IPIOC_GETIFCONF;
	ioc.ic_timout = 60;
	ioc.ic_len = sizeof (buf);
	ioc.ic_dp = (char *)ifc;

	if (ioctl(ipfd, I_STR, (char *)&ioc) < 0) {
		syslog(LOG_ERR, "ioctl (get interface configuration)");
		return (0);
	}
	ifr = (struct ifreq *)&ifc->ifc_buf;
	for (n = ifc->ifc_len / sizeof (struct ifreq); n > 0; n--, ifr++) {
		for (np = neighbors; np != NULL; np = np->n_next)
			if (np->n_name &&
			    strcmp(ifr->ifr_name, np->n_name) == 0)
				break;
		if (np != NULL)
			continue;
		ifreq = *ifr;
		np = (struct neighbor *)malloc(sizeof (*np));
		if (np == NULL)
			continue;
		np->n_name = malloc(strlen(ifr->ifr_name) + 1);
		if (np->n_name == NULL) {
			free((char *)np);
			continue;
		}
		strcpy(np->n_name, ifr->ifr_name);
		np->n_addrlen = sizeof (ifr->ifr_addr);
		np->n_addr = malloc(np->n_addrlen);
		if (np->n_addr == NULL) {
			free(np->n_name);
			free((char *)np);
			continue;
		}
		bcopy((char *)&ifr->ifr_addr, np->n_addr, np->n_addrlen);

		ioc.ic_cmd = IPIOC_GETIFFLAGS;
		ioc.ic_timout = 60;
		ioc.ic_len = sizeof(ifreq);
		ioc.ic_dp = (char *)&ifreq;
		if (ioctl(ipfd, I_STR, (char *)&ioc) < 0) {
			syslog(LOG_ERR, "ioctl (get interface flags)");
			free((char *)np);
			continue;
		}
		if (((ifreq.ifr_flags & IFF_UP) == 0) ||
		    (ifreq.ifr_flags & IFF_IPTOX25) ||
		    (ifreq.ifr_flags & (IFF_BROADCAST|IFF_POINTOPOINT) == 0)) {
			free((char *)np);
			continue;
		}
		np->n_flags = ifreq.ifr_flags;
		if (np->n_flags & IFF_POINTOPOINT) {
			ioc.ic_cmd = IPIOC_GETIFDSTADDR;
			ioc.ic_timout = 60;
			ioc.ic_len = sizeof(ifreq);
			ioc.ic_dp = (char *)&ifreq;
			if (ioctl(ipfd, I_STR, (char *)&ioc) < 0) {
				syslog(LOG_ERR, "ioctl (get dstaddr)");
				free((char *)np);
				continue;
			}
			/* we assume addresses are all the same size */
			bcopy((char *)&ifreq.ifr_dstaddr,
			  np->n_addr, np->n_addrlen);
		}
		if (np->n_flags & IFF_BROADCAST) {
			ioc.ic_cmd = IPIOC_GETIFBRDADDR;
			ioc.ic_timout = 60;
			ioc.ic_len = sizeof(ifreq);
			ioc.ic_dp = (char *)&ifreq;
			if (ioctl(ipfd, I_STR, (char *)&ioc) < 0) {
				syslog(LOG_ERR, "ioctl (get broadaddr)");
				free((char *)np);
				continue;
			}
			/* we assume addresses are all the same size */
			bcopy((char *)&ifreq.ifr_broadaddr,
			  np->n_addr, np->n_addrlen);
		}
		/* gag, wish we could get rid of Internet dependencies */
		sin = (struct sockaddr_in *)np->n_addr;
		sin->sin_family = AF_INET;
		sin->sin_port = sp->s_port;
		sin->sin_addr.s_addr = sin->sin_addr.s_addr;
		np->n_next = neighbors;
		neighbors = np;
	}
	close(ipfd);
	return (1);

}	/* end of configure() */

#ifdef DEBUG
_sendto(sock, buf, cc, flags, to, tolen)
	int sock;
	char *buf;
	int cc, flags;
	char *to;
	int tolen;
{
	register struct whod *w = (struct whod *)buf;
	register struct whoent *we;
	struct sockaddr_in *sin = (struct sockaddr_in *)to;
	char *interval();

	printf("_sendto %x.%d\n", ntohl(sin->sin_addr), ntohs(sin->sin_port));
	printf("hostname %s %s\n", w->wd_hostname,
	   interval(ntohl(w->wd_sendtime) - ntohl(w->wd_boottime), "  up"));
	printf("load %4.2f, %4.2f, %4.2f\n",
	    ntohl(w->wd_loadav[0]) / 100.0, ntohl(w->wd_loadav[1]) / 100.0,
	    ntohl(w->wd_loadav[2]) / 100.0);
	cc -= WHDRSIZE;
	for (we = w->wd_we, cc /= sizeof (struct whoent); cc > 0; cc--, we++) {
		time_t t = ntohl(we->we_utmp.out_time);
		printf("%-8.8s %s:%s %.12s",
			we->we_utmp.out_name,
			w->wd_hostname, we->we_utmp.out_line,
			ctime(&t)+4);
		we->we_idle = ntohl(we->we_idle) / 60;
		if (we->we_idle) {
			if (we->we_idle >= 100*60)
				we->we_idle = 100*60 - 1;
			if (we->we_idle >= 60)
				printf(" %2d", we->we_idle / 60);
			else
				printf("   ");
			printf(":%02d", we->we_idle % 60);
		}
		printf("\n");
	}
}	/* end of _sendto() */

char *
interval(time, updown)
	int time;
	char *updown;
{
	static char resbuf[32];
	int days, hours, minutes;

	if (time < 0 || time > 3*30*24*60*60) {
		(void) sprintf(resbuf, "   %s ??:??", updown);
		return (resbuf);
	}
	minutes = (time + 59) / 60;		/* round to minutes */
	hours = minutes / 60; minutes %= 60;
	days = hours / 24; hours %= 24;
	if (days)
		(void) sprintf(resbuf, "%s %2d+%02d:%02d",
		    updown, days, hours, minutes);
	else
		(void) sprintf(resbuf, "%s    %2d:%02d",
		    updown, hours, minutes);
	return (resbuf);

}	/* end of interval() */

#endif /* DEBUG */
