/*#######################################################################
#									#
#				NOTICE:					#
#									#
#    The source code contained in this listing is a proprietary trade	#
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and	#
#    is also subject to copyright protection as an unpublished work	#
#    pursuant to Section 104(a) of Title 17 of the United States Code.	#
#    Unauthorized copying, adaptation, distribution, use or display is	#
#    prohitibed by law and may be subject to civil and criminal		#
#    penalties.								#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  Filename:  PICODE03.C						#
#									#
#   Purpose:  Contains screen manipulation routines, command line	#
#	      argument parsing routine, and related utility routines.	#
#									#
#  Maintenance Log:							#
#									#
#  Ver	Date	  Who  SPR/APR	Change/Comments				#
#  ---	--------  ---  -------	--------------------------------------	#
#  1.0	09/14/87  DS	 N/A	Created.				#
#  1.1	10/02/87  DS	 N/A	Added PParseOptions.			#
#  1.2	10/30/87  WeD	 N/A	Fixed action items in code-review list. #
#  1.3	11/11/87  WeD	 N/A	Fixed 2 command-line option problems.	#
#  1.4	11/14/87  WeD	 N/A	Revised input/output routines.		#
#  1.5	11/20/87  WeD	 N/A	Fixed TimeDate routine.			#
#  1.6  11/30/87  WeD	 N/A	Added default to PConfirm.		#
#  1.7	12/01/87  WeD	 N/A	Now space-out getstring field at start. #
#  1.8	12/02/87  WeD	 N/A	NOECHO mode in input routine only now.	#
#######################################################################*/

/*#######################################################################
#									#
#  Outline of General Structures or Algorithms used in the code:	#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  The following functions are contained in this file:			#
#									#
#	 void  PGoToXY()						#
#	 void  PClearDisplay()						#
#	 void  PEraseDisplayToEnd()					#
#	 void  PEraseLineToEnd()					#
#	 void  PClearRectangle()					#
#	 void  PSaveCursorPosition()					#
#	 void  PRestoreCursorPosition()					#
#	 ULONG PInitScreenMenu()					#
#	 ULONG PUnInitScreenMenu()					#
#	 ULONG PGetTimeDate()						#
#	 int   PGetString()						#
#	 int   PGetNum()						#
#	 int   PDisplay()						#
#	 int   PConfirm()						#
# static int   PGetSwitchChar()						#
#	 ULONG PParseOptions()						#
#									#
#  The function PDisplay depends upon the standard C calling		#
#  convention -- arguments are pushed in reverse order, and the		#
#  callers of PDisplaypop its arguments from the stack after it		#
#  it returns.	This allows PDisplay to be called with a varying	#
#  number of arguments.							#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  Included Files:							#
#									#
#######################################################################*/

#include <stdio.h>

#include <portab.h>
#include <string.h>
#include <stdlib.h>

#include <time.h>
#include "pistruct.h"
#include "piprotos.h"

/*#######################################################################
#									#
#  Local Structure Definitions						#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  Static Variables							#
#									#
#######################################################################*/

static	WORD		OldSMode = 0;	   /* Screen Mode Flags		*/
static	WORD		OldKMode = 0;	   /* Keyboard Mode Flags	*/
static	WORD		CharacterSize = 0; /* 8- or 16-Bit Characters	*/
static  ConsoleTable	Console;	   /* Current console setup	*/

/*#######################################################################
#									#
#  External Functions and locations:					#
#									#
#    LONG	s_get();	--  High C Runtime Library		#
#    LONG	s_set();	--  High C Runtime Library		#
#    LONG	s_define();	--  High C Runtime Library		#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  External Variables:							#
#									#
#######################################################################*/

extern	WORD	PMaxRow;	/* Defined in PIGLOBAL.C file */
extern	WORD	PMaxColumn;	/* Defined in PIGLOBAL.C file */

/*#######################################################################
#									#
# Function Name:  PGoToXY						#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  PLocation    Where					#
#									#
#   Description:  Function assumes that the coordinates of the point	#
#		  are within limits and positions the cursor to the	#
#		  point ( Where.Row, Where.Column ).			#
#									#
#######################################################################*/

void PGoToXY( PLocation Where )
{

    if ( CharacterSize == EIGHT_BIT )
	fprintf( stdout, "\033Y%c%c", Where.Row + 31, Where.Column + 31 );
    else
	fprintf( stdout, "%d%d", Where.Row + 0x3001, Where.Column + 0x4001 );

}/* PGoToXY */

/*#######################################################################
#									#
# Function Name:  PClearDisplay						#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  Function clears the display console using escape	#
#		  sequences.						#
#									#
#######################################################################*/

void PClearDisplay( void )
{

    if ( CharacterSize == EIGHT_BIT )
	fprintf( stdout, "\033E" );
    else
	fprintf( stdout, "%d", 0x2045 );

}/* PClearDisplay */

/*#######################################################################
#									#
# Function Name:  PEraseDisplayToEnd					#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  PLocation   From					#
#									#
#   Description:  Function assumes that the point From is within limits #
#		  and clears the display starting at the position	#
#		  (From.Row,From.Column) to the end of the display.	#
#									#
#######################################################################*/

void PEraseDisplayToEnd( PLocation From )
{

    PGoToXY( From );

    if ( CharacterSize == EIGHT_BIT )
	fprintf( stdout, "\033J" );
    else
	fprintf( stdout, "%d", 0x204A );

}/* PEraseDisplayToEnd */

/*#######################################################################
#									#
# Function Name:  PEraseLineToEnd					#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  PLocation  From					#
#									#
#   Description:  Function assumes the point From is within limits and	#
#		  clears from the position (From.Row,From.Column) to	#
#		  the end of the line.					#
#									#
#######################################################################*/

void PEraseLineToEnd( PLocation From )
{

    PGoToXY( From );
    if ( CharacterSize == EIGHT_BIT )
	fprintf( stdout, "\033K" );
    else
	fprintf( stdout, "%d", 0x204B );

}/* PEraseLineToEnd */

/*#######################################################################
#									#
# Function Name:  PClearRectangle					#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  Function clears lines 9 through 16.			#
#									#
#######################################################################*/

void PClearRectangle( void )
{
    PLocation  Line;
    
    Line.Row = 9;
    Line.Column = 1;
    
    while( Line.Row <= 15 )
    {
       PEraseLineToEnd(Line);
       Line.Row += 1;
    }
    
}/* PClearRectangle */

/*#######################################################################
#									#
# Function Name:  PSaveCursorPosition					#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  Function saves the current PLocation of cursor in	#
#		  anticipation of a PRestoreCursorPosition() call.	#
#									#
#######################################################################*/

void PSaveCursorPosition( void )
{

    if ( CharacterSize == EIGHT_BIT )
	fprintf( stdout, "\033j" );
    else
	fprintf( stdout, "%d", 0x206A );

}/* PSaveCursorPosition */

/*#######################################################################
#									#
# Function Name:  PRestoreCursorPosition()				#
#									#
#  Return Value:  None							#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  Function restores the cursor to the PLocation of the	#
#		  most recent PSaveCursorPosition() call.		#
#									#
#######################################################################*/

void PRestoreCursorPosition( void )
{

    if ( CharacterSize == EIGHT_BIT )
	fprintf( stdout, "\033k" );
    else
	fprintf( stdout, "%d", 0x206B );

}/* PRestoreCursorPosition */

/*#######################################################################
#									#
# Function Name:  PInitScreenMenu.					#
#									#
#  Return Value:  SUCCESS or FAILURE					#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  This function sets the values of the global variables #
#		  PMaxColumn and PMaxRow to the size of the console.	#
#		  It also assures that escape sequence decoding and	#
#		  CR/LF input translation is enabled.  In the process,	#
#		  the current console settings are saved for restoration#
#		  by PUnInitScreenMenu.					#
#									#
#######################################################################*/

ULONG PInitScreenMenu( void )
{

    if( s_get( T_CON, 1L, &Console, (long)(sizeof Console) ) != SUCCESS )
	return( FAILURE );

    PMaxRow = Console.nrows;
    PMaxColumn = Console.ncols;

    /*
	Save the current console settings for later restoring.
    */

    OldSMode = Console.smode;
    OldKMode = Console.kmode;

    /*
	Now set up our correct environment.
    */

    Console.smode &= ~F_SMODE_ESC;

    Console.kmode &= ~F_KMODE_CRLF;

    if( s_set( T_CON, 1L, &Console, 6L ) != SUCCESS )
	return( FAILURE );

    if( OldSMode & F_SMODE_16BIT )
	CharacterSize = SIXTEEN_BIT;
    else
	CharacterSize = EIGHT_BIT;

    return( SUCCESS );

}/* PInitScreenMenu */

/*#######################################################################
#									#
# Function Name:  PUnInitScreenMenu.					#
#									#
#  Return Value:  SUCCESS or FAILURE					#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  This function restores the SMode and KMode values of	#
#		  the console to the state saved by PInitScreenMenu.	#
#									#
#######################################################################*/

ULONG PUnInitScreenMenu( void )
{

    fflush( stdout );

    if( s_get( T_CON, 1L, &Console, 6L ) != SUCCESS )
	return( FAILURE );

    Console.smode = OldSMode;
    Console.kmode = OldKMode;

    return( s_set( T_CON, 1L, &Console, 6L ) );

}/* PUnInitScreenMenu */

/*#######################################################################
#									#
# Function Name:  PGetTimeDate()					#
#									#
#  Return Value:  SUCCESS						#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  A pointer to a PTime structure and a pointer to a	#
#		  PDate structure					#
#									#
#   Description:  This function returns the current time and date in	#
#		  the format required for a directory entry.		#
#									#
#######################################################################*/

ULONG PGetTimeDate( PTime *TimePtr, PDate *DatePtr )
{
    DOUBLE     td;
    struct tm  *Ptr;

    time( &td );
    Ptr = localtime( &td );

    TimePtr->hour = Ptr->tm_hour;
    TimePtr->minute = Ptr->tm_min;
    TimePtr->second = Ptr->tm_sec;

    /*
	The Date Bias is 1980 and the Months are numbered starting with zero.
    */

    DatePtr->year = Ptr->tm_year - 80;
    DatePtr->month = Ptr->tm_mon + 1;
    DatePtr->day = Ptr->tm_mday;

    return( SUCCESS );

}/* PGetTimeDate */

/*#######################################################################
#									#
# Function Name:  PGetString						#
#									#
#  Return Value:  SUCCESS or ESC_CHARACTER				#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  A pointer to the buffer in which to read characters	#
#		  from stdin, a maximum character count and a mode flag:#
#		  STR_DEFAULT_VALUE	-- buffer is the default value	#
#		  STR_DIGITS_ONLY	-- accept only digits 0 - 9	#
#									#
#   Description:  Save the current cursor location, display the default #
#		  value (if specified) and then begin to accept input	#
#		  from the user.  If Enter is pressed as the first	#
#		  character and a default value was specified, return	#
#		  that, otherwise beep at the user.  If the user begins	#
#		  to make an entry, clear the default value if present	#
#		  and allow the user to enter up to MaxCount characters.#
#		  Invalid characters are simply ignored without comment.#
#		  If Esc is pressed anywhere in the command line, return#
#		  ESC_CHARACTER instead of SUCCESS.			#
#									#
#######################################################################*/

int PGetString(char *StrPtr, int MaxCount, int Mode)
{
    char    Chr;
    int     Count = 0;
    WORD    OurKMode;

    if( s_get( T_CON, 1L, &Console, 6L ) == SUCCESS )
    {
	OurKMode = Console.kmode;
	Console.kmode |= F_KMODE_INPUT;

	s_set( T_CON, 1L, &Console, 6L );
    }
    else
	OurKMode = -1;

    PSaveCursorPosition();


    if( Mode & STR_DEFAULT_VALUE )
    {
	printf( "\033r%s", StrPtr );
	Count = strlen( StrPtr );
    }
    else printf("\033r");

    while( Count++ <= MaxCount )
	putchar( ' ' );

    PRestoreCursorPosition();
    PSaveCursorPosition();

    Count = 0;

    for( ;; )
    {
	if( s_read( 1, (long)stdin->_fd, &Chr, 1L, 0L ) != 1
	 || Chr == ESCAPE_CHARACTER || Chr == ('C'-'@') )
	{
	    Chr = ESCAPE_CHARACTER;
	    break;
	}

	if( MaxCount )
	{
	    if( Chr == '\n' || Chr == '\r' )
	    {
		if( !(Mode & STR_DEFAULT_VALUE) )
		{
		    if( !Count )
		    {
			putchar( BELL_CHARACTER );
			continue;
		    }
		    *StrPtr = '\0';
		}
		Chr = SUCCESS;
		break;
	    }
	    if( Count && Chr == '\010' )
	    {
		--Count;
		--StrPtr;
		printf( "\010 \010" );
	    }
	    else
	    if( Count < MaxCount && Chr >= ' ' && Chr <= '~' )
	    {
		if( Mode & STR_DIGITS_ONLY
		 && (Chr < '0' || Chr > '9') )
		    continue;

		*StrPtr++ = Chr;
		if( Mode & STR_DEFAULT_VALUE )
		{
		    while( Count++ < MaxCount )
			putchar( ' ' );
		    Mode &= ~STR_DEFAULT_VALUE;
		    PRestoreCursorPosition();
		    PSaveCursorPosition();
		    Count = 0;
		}
		++Count;
		putchar( Chr );
	    }
	}
    }/* for */

    if( OurKMode != -1 && s_get( T_CON, 1L, &Console, 6L ) == SUCCESS )
    {
	Console.kmode = OurKMode;

	s_set( T_CON, 1L, &Console, 6L );
    }

    printf( "\033u" );
    return( Chr );

}/* PGetString */

/*#######################################################################
#									#
# Function Name:  PDisplay						#
#									#
#  Return Value:  SUCCESS, FAILURE, or the Escape-Character		#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  A pointer to a PDisplayObject structure		#
#									#
#   Description:  If the PScreenObject is targeted for output then it	#
#		  is display; otherwise, the object gets a value from	#
#		  stdin and verified using the Valid field.		#
#									#
#######################################################################*/

int PDisplay(PDisplayObject *ObjPtr, ...)
{
    /*
	Test the coordinates of the object against the screen boundaries
	and ignore the coordinates if they are out of bounds.
    */

    if( ObjPtr->Position.Row >= 1
     && ObjPtr->Position.Row <= PMaxRow
     && ObjPtr->Position.Column >= 1
     && ObjPtr->Position.Column <= PMaxColumn )
	PGoToXY( ObjPtr->Position );

    /*
	Output the string and up to three arguments.
    */

    printf( ObjPtr->Object, *( (ULONG *)&ObjPtr + 1 ),
			    *( (ULONG *)&ObjPtr + 2 ),
			    *( (ULONG *)&ObjPtr + 3 ) );
    return( SUCCESS );

}/* PDisplay */

/*#######################################################################
#									#
# Function Name:  PGetNum						#
#									#
#  Return Value:  The entered numeric value or -1 for the ESC character #
#									#
#   Environment:  Process						#
#									#
#     Arguments:  A pointer to the buffer in which to read characters	#
#		  from stdin, a maximum character count and a mode flag:#
#									#
#   Description:  Position the cursor at the desired location, display	#
#		  the default value if specified and begin to accept	#
#		  input from the user.  If Enter is pressed as the first#
#		  character and a default value was specified, return	#
#		  that, otherwise beep at the user.  If the user begins	#
#		  to make an entry, clear the default value if present	#
#		  and allow the user to enter a value, rejecting any	#
#		  non-numeric characters.  If the value is at least Min #
#		  but not greater than Max, it will be returned.  Else, #
#		  the user is beeped at and reprompted for a new value. #
#		  If Esc is pressed anywhere in the command line, a	#
#		  value of -1 is returned instead.			#
#									#
#######################################################################*/

int PGetNum(int Min, int Max, int Default)
{
    char   Response[ 16 ];
    int    MaxLen, Mode, Num;

    sprintf( Response, "%d", Max );
    MaxLen = strlen( Response );

    for( ;; )
    {
	if( Default < 0 )
	    Mode = STR_DIGITS_ONLY;
	else
	{
	    Mode = STR_DIGITS_ONLY | STR_DEFAULT_VALUE;
	    sprintf( Response, "%d", Default );
	}

	if( PGetString( Response, MaxLen, Mode ) == ESCAPE_CHARACTER )
	    return( -1 );
	if( (Num = atoi( Response )) >= Min && Num <= Max )
	    return( Num );

	putchar( BELL_CHARACTER );

	PRestoreCursorPosition();
    }

}/* PGetNum */

/*#######################################################################
#									#
# Function Name:  PConfirm						#
#									#
#  Return Value:  Yes == TRUE,  No == FALSE, or ESCAPE_CHARACTER	#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  a pointer to a PDisplayObject.			#
#									#
#   Description:  Function is used in cases where verification by a	#
#		  user is necessary before continuing further.		#
#									#
#######################################################################*/

int PConfirm(PDisplayObject *ObjPtr, int Default)
{
    char    Reply[ 4 ];
    extern  char   YesNoChar[];
    int     Mode;

    PDisplay( ObjPtr );

    if( Default == TRUE )
    {
	Default = YesNoChar[0];
	Mode    = STR_DEFAULT_VALUE;
    }
    else
    if( Default == FALSE )
    {
	Default = YesNoChar[2];
	Mode    = STR_DEFAULT_VALUE;
    }
    else
    {
	Default = '\0';
	Mode    = STR_NORMAL;
    }

    for( ;; )
    {
	*(int *)Reply = Default;
	if( PGetString( Reply, 1, Mode ) == ESCAPE_CHARACTER )
	    return( ESCAPE_CHARACTER );

	if( Reply[ 0 ] == YesNoChar[ 0 ]
	 || Reply[ 0 ] == YesNoChar[ 1 ] )
	    return( TRUE );

	if( Reply[ 0 ] == YesNoChar[ 2 ]
	 || Reply[ 0 ] == YesNoChar[ 3 ] )
	    return( FALSE );

	putchar( BELL_CHARACTER );

	PRestoreCursorPosition();
    }

}/* PConfirm */

/*#######################################################################
#									#
# Function Name:  PGetSwitchChar					#
#									#
#  Return Value:  The switchar from the PROCDEF Table.			#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  None							#
#									#
#   Description:  Function searches the PROCDEF Table for the value of	#
#		  the switchar and returns it to the caller.		#
#									#
#######################################################################*/

static int PGetSwitchChar( void )
{
    char  Prefix[ 8 ];

    /*
	Search the PROCDEF table and the SYSDEF table for a
	definition for "switchar".  If it is undefined, return
	the dash "-", consistent with FlexOS documentation, otherwise
	return its definition.
    */

    if ( ( s_define( GET_LIT_PROCDEF, "switchar", &Prefix, 8L ) == SUCCESS ) ||
	 ( s_define( GET_LIT_SYSDEF,  "switchar", &Prefix, 8L ) == SUCCESS ) )
	return( Prefix[ 0 ] );
    else
	return( '-' );

}/* PGetSwitchChar */

/*#######################################################################
#									#
# Function Name:  PParseOptions						#
#									#
#  Return Value:  SUCCESS or FAILURE					#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  argc - number of command line argument strings,	#
#		  argv - array of pointers to the command line strings, #
#		  argp - a pointer to a PParseOptions structure		#
#									#
#   Description:  Each command line argument is compared with the valid #
#		  options in the PParseOptions structure and the Found	#
#		  and Result fields are set accordingly.		#
#									#
#######################################################################*/

ULONG PParseOptions(int argc, char **argv, PCommandLineOptions *argp)
{
    char      SwitchChar;	/* Option Introducer Character	  */
    int       Index;		/* Index into Legal Options Array */
    POption   *ArgPtr;		/* Pointer a PParseOption element */

    SwitchChar = PGetSwitchChar();

    while ( --argc > 0 )
    {
	/*
	    Increment argument pointer to next command line argument.
	*/

	argv++;

	if( **argv != SwitchChar )
	{
	    /*
		The command line argument is a drive specification
		string.  Return a pointer to it in the result field.
	    */

	    ArgPtr = argp->ArgpPtr[ 0 ];

	    /*
		Only one drive specification is permitted.
	    */

	    if ( ArgPtr->Found )
		return( FAILURE );

	    ArgPtr->Found = TRUE;
	    ArgPtr->Result = (LONG) *argv;

	    continue;
	}

	while ( **argv == SwitchChar )
	{
	    /*
		Argument is introduced by the system defined switchchar.
		Increment to the option character.
	    */

	    (*argv)++;

	    for ( Index = 0; Index < argp->Total; Index++ )
	    {
		/*
		    Look through the list of legal option characters.
		*/

		ArgPtr = argp->ArgpPtr[ Index ];

		if ( **argv == ArgPtr->Option
		  || (**argv >= 'a' && **argv <= 'z'
		   && (**argv & ~0x20) == ArgPtr->Option) )
		{
		    /*
			If the option character has already been processed,
			then its second appearance is an error.
		    */

		    if ( ArgPtr->Found )
			return( FAILURE );

		    /*
			Mark it as having been processed, and set its
			return value.
		    */

		    ArgPtr->Found = TRUE;

		    (*argv)++;

		    /*
			If an string follows the option letter, attempt to
			convert it into long with 'strtol()'.  Base 0 is
			specified to allow hex (0x[0-9]...) octal (0[0-9]...)
			and decimal ([1-9][0-9]...) to be input. The point
			where the number parsing ended in the string will be
			returned in our argv pointer.
		    */

		    if ( **argv != '\0' && **argv != '-' )
			ArgPtr->Result = strtol( *argv, argv, 0 );
		    break;

		}/* if */

	    }/* for */

	    /*
		If the option character wasn't matched, return FAILURE.
	    */

	    if ( Index == argp->Total )
		return( FAILURE );

	}/* while an option character introducer is present */

	if( **argv != '\0' )
	{
	    return( FAILURE );
	}

    }/* while there are more args */

    return( SUCCESS );

}/* PParseOptions */

/*#######################################################################
#									#
#				NOTICE:					#
#									#
#    The source code contained in this listing is a proprietary trade	#
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and	#
#    is also subject to copyright protection as an unpublished work	#
#    pursuant to Section 104(a) of Title 17 of the United States Code.	#
#    Unauthorized copying, adaptation, distribution, use or display is	#
#    prohitibed by law and may be subject to civil and criminal		#
#    penalties.								#
#									#
#######################################################################*/
