/********************************************************************/
/*  btest ['control device file' 'frame buffer device file']        */
/*                                                                  */
/*  Program for directly reading or writing to the Catseye display  */
/*  boards.                                                         */
/*                                                                  */
/********************************************************************/

/*    #mkdev file for Catseye in DIO I space
      set -x
      cd /dev
      mknod = /etc/mknod
      # build device file for Catseye control space
      dev = /dev/catseye
      $mknod $dev c 10 0x005601
      # build device file for Catseye frame buffer space
      dev = /dev/cat_fb
      $mknod $dev c 10 0x002020
*/

#include <sys/iomap.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>

#define reg unsigned short
#define bool unsigned int
#define VERSION "6.00"

int bus_error();
int interrupt();
reg getword();

int fdes_cntrl,fdes_frm_bfr;
unsigned char *cntrl, *frm_bfr;

/* local global variables */
bool done = 0;
bool roop = 0;
bool bus_err = 0;
char line[80];
char *carg1, *carg2, *carg3, *carg4, *targ;
int i, temp;
unsigned int laddress; 
reg address;
jmp_buf env;
reg data;

main(argc,argv)
int argc;
char **argv;
{
    /*  open the catseye board        */
    cat_open(argc,argv);

    /*  *cntrl        maps in the register space on catseye                */
    /*  *frm_bfr      maps the catseye frame buffer on word boundaries        */
    /*  *byte_frm_bfr maps the catseye frame buffer on byte boundaries        */

    signal( SIGINT, interrupt); 
    signal( SIGBUS, bus_error); 

    printf("Board Test program -- version: %s\n\n",VERSION);

    /* main command loop */
    while (!done) {

        /* prompt for command */
        printf("-> ");
        fflush(stdout);
        fgets(line, 80, stdin);
        interpret(line,0,0);
    }

}


cat_open(n,argv)
int n;
char *argv[];
{
     switch (n) {
	 case 2: perror("usage: btest ['control device file'  'frame buffer device file'"),exit(1);
                 break;
	 case 3: if ((fdes_cntrl = open (argv[1],O_RDWR))<0)
                     {
                         perror("unable to open catseye control space"),exit(1);
                     }
                 if ((fdes_frm_bfr = open (argv[2],O_RDWR))<0)
                     {
                         perror("unable to open catseye frame buffer"),exit(1);
                     }
                 break;
	 case 1:
	 default: if ((fdes_cntrl = open ("/dev/catseye",O_RDWR))<0)
                     {
                         perror("unable to open catseye control space"),exit(1);
                     }
                 if ((fdes_frm_bfr = open ("/dev/cat_fb",O_RDWR))<0)
                     {
                         perror("unable to open catseye frame buffer"),exit(1);
                     }
                 break;
     }

     /* get the control space pointer */
     cntrl = (unsigned char *) 0x00;
     ioctl (fdes_cntrl,IOMAPMAP,&cntrl);

     /* get the frame buffer pointer */
     frm_bfr = (unsigned char *) 0x00;
     ioctl (fdes_frm_bfr,IOMAPMAP,&frm_bfr);

}

/****************************************************************/
/* Procedure for closing the Catseye display board              */
/****************************************************************/

cat_close()
{
     ioctl(fdes_cntrl,IOMAPUNMAP,&cntrl);
     close(fdes_cntrl);
     ioctl(fdes_frm_bfr,IOMAPUNMAP,&frm_bfr);
     close(fdes_frm_bfr);
}


/*****************************************************************************/
/*                                                                           */
/* interpret : interpret the first character on the line and do the          */
/*             appropriate action based on that command.  This routine may   */
/*             be called recursively when a 's' command is used.             */
/*                                                                           */
/* input : line : a pointer to a line of text read in from the input.        */
/*         sub_call : set to 1 if this has been called from itself.  This    */
/*                    is used to prevent certain actions (like quit and      */
/*                    shell escape) from working.                            */
/*         old_stream : set to 1 if invoked with a 's' command to allow      */
/*                      operation with stream files generated for the        */
/*                      software simulator. (adds 0x4000 to all register     */
/*                      values)                                              */
/*                                                                           */
/* output: none.                                                             */
/*                                                                           */
/* Note : this routine uses global variables declared for this module.       */
/*                                                                           */
/*****************************************************************************/
interpret(line,sub_call,old_stream)
char line[];
int sub_call;
int old_stream;
{

    FILE *tfp;


    /* interpret command */
    if ((line[1] != ' ') && (line[1] != '\n') && (line[0] != '\n'))
        printf("\nPardon me?\n");
    else 
        switch (line[0]) {

            case 'b': /* execute display ROM */
                if ((carg1 = strtok(&line[2]," \n")) == NULL)
                    temp = 0;
                else {
                    temp = (int) strtol(carg1,&targ,10);
                    if (strcmp(carg1,targ) == 0) {
                        printf("\nn must be a number\n");
                        break;
                    }
                    else if ((temp != 0) && (temp != 1)) {
                        printf("\nn must be a 0 or 1\n");
                        break;
                    }
                }
                execute_display_rom(temp);
                break;


            case 'f': /* frame buffer read/write */
                carg1 = strtok(&line[2]," \n");
                carg2 = strtok(NULL," \n");
                carg3 = strtok(NULL," \n");
                carg4 = strtok(NULL," \n");
                if ((!strcmp(carg1,"r") && ((carg2 == NULL) || (carg3 == NULL)))
                   || (!strcmp(carg1,"w") && ((carg2 == NULL) || 
                       (carg3 == NULL) || (carg4 == NULL)))
                   || (strcmp(carg1,"r") && strcmp(carg1,"w"))) {
                    printf("\nIncorrect usage - you silly human!!\n");
                    break;
                }
                else {
                    laddress = strtol(carg2,&targ,0);
                    if (strcmp(carg2,targ) == 0) {
                        printf("\nThe address must be a number\n");
                        break;
                    }
                    if (strcmp(carg3,"u") && strcmp(carg3,"l") &&
                      strcmp(carg3,"b")) {
                        printf("\nYou must specify 'u', 'l' or 'b'\n");
                        break;
                    }
                    /* generate glad part of address */
                    if (!strcmp(carg1,"w")) {
                        data = (reg) strtol(carg4,&targ,0);
                        if (strcmp(carg4,targ) == 0) {
                            printf("\nThe data must be a number\n");
                            break;
                        }
                        bus_err = 0;
                        if (!strcmp(carg3,"u"))
                            do
                                if (setjmp(env))  signal( SIGBUS, bus_error); 
                                else 
                                  *(frm_bfr + laddress) = (unsigned char) data;
                            while ( roop == 1);
                        else if (!strcmp(carg3,"l"))
                            do
                                if (setjmp(env))  signal( SIGBUS, bus_error); 
                                else 
                                  *(frm_bfr + laddress + 1) = (unsigned char) data;
                            while ( roop == 1);
                        else
                            do
                                if (setjmp(env))  signal( SIGBUS, bus_error); 
                                else 
                                  *((reg *)(frm_bfr + laddress)) = data;
                            while ( roop == 1);
                        if (bus_err) {
                            printf("\nBus Error at ");
                            print_hex(stdout,laddress,8);
                            printf("\n");
                        }
                    }
                    else {
                        bus_err = 0;
                        do
                            if (setjmp(env))  signal( SIGBUS, bus_error); 
                            else 
                                data = *((reg *)(frm_bfr + laddress));
                        while ( roop == 1);
                        if (!bus_err) {
                            printf("\nFrame Buffer data = ");
                            print_hex(stdout,data,4);
                            printf("\n");
                        }
                        else {
                            printf("\nBus Error at ");
                            print_hex(stdout,laddress,8);
                            printf("\n");
                        }
                    }
                }
                break;

            case 'h': /* print help screen */
                printf("\n\t\tAvailable commands:\n");
                printf("\tb [0] [1]       - execute display ROM\n");
                printf("\tf r address <u><l><b> - frame buffer read\n");
                printf("\tf w address <u><l><b> data - frame buffer write\n");
                printf("\th               - help\n");
                printf("\tk               - clear graphics device\n");
                printf("\tl               - set loop mode for next command\n");
                printf("\tq               - quit\n");
                printf("\tr address       - read register\n");
                printf("\ts file          - run old style stream file (add 0x4000)\n");
                printf("\tS file          - run new style stream file\n");
                printf("\tw address data  - write register\n");
                printf("\tx               - wait for RUG to go non-busy\n");
                printf("\tX               - wait for IRIS to go non-busy\n");
                printf("\t! [command]     - shell escape\n\n");
                break;

            case 'l': /* set loop mode for next command */
                signal( SIGINT, interrupt); 
                roop = 1;
                break;

            case 'q': /* quit */
                if (!sub_call) done = 1;
                break;

            case 'r': /* read register */
                if ((carg1 = strtok(&line[2]," \n")) == NULL)
                    printf("\nUsage: r address\n");
                else {
                    address = (int) strtol(carg1,&targ,0);
                    if (strcmp(carg1,targ) == 0)
                        printf("\nThe address must be a number\n");
                    else {
                        bus_err = 0;
                        if (old_stream) address += 0x4000;
                        do 
                            if (setjmp(env)) signal( SIGBUS, bus_error); 
                            else            data = *((reg *)(cntrl + address)); 
                        while (roop == 1); 
                        if (bus_err) {
                            printf("\nBus Error at ");
                            print_hex(stdout,address,4);
                            printf("\n");
                        }
                        else {
                            printf("\nRegister ");
                            print_hex(stdout,address,4);
                            printf(" = ");
                            print_hex(stdout,data,4);
                            printf("\n");
                        }
                    }
                }
                break;

            case 's': /* run old style stream file */
                if ((carg1 = strtok(&line[2]," \n")) == NULL)
                    printf("\nUsage: s filename\n");
                else {
                    if ((tfp = fopen(carg1,"r")) == NULL)
                        printf("\nCannot open file %s\n",carg1);
                    else {
                        printf("\nRunning old style stream file %s ",carg1);
                        fflush(stdout);
                        while (fgets(line, 80, tfp) != NULL) {
                            interpret(line,1,1);
                            printf(".");
                            fflush(stdout);
                        }
                        printf("\n");
                        fclose(tfp);
                    }
                }
                break;

            case 'S': /* run new style stream file */
                if ((carg1 = strtok(&line[2]," \n")) == NULL)
                    printf("\nUsage: s filename\n");
                else {
                    if ((tfp = fopen(carg1,"r")) == NULL)
                        printf("\nCannot open file %s\n",carg1);
                    else {
                        printf("\nRunning stream file %s ",carg1);
                        fflush(stdout);
                        while (fgets(line, 80, tfp) != NULL) {
                            interpret(line,1,0);
                            printf(".");
                            fflush(stdout);
                        }
                        printf("\n");
                        fclose(tfp);
                    }
                }
                break;

            case 'w': /* write register */
                if (((carg1 = strtok(&line[2]," \n")) == NULL) ||
                    ((carg2 = strtok(NULL," \n")) == NULL)) 
                    printf("\nUsage: w address data\n");
                else {
                    address = (reg) strtol(carg1,&targ,0);
                    if (strcmp(carg1,targ) == 0)
                        printf("\nThe address must be a number\n");
                    else {
                        data = (reg) strtol(carg2,&targ,0);
                        if (strcmp(carg2,targ) == 0)
                            printf("\nThe data must be a number\n");
                        else {
                            bus_err = 0;
                            if (old_stream) address += 0x4000;
                            do
                                if (setjmp(env))  signal( SIGBUS, bus_error); 
                                else *((reg *)(cntrl + address)) = data;
                            while ( roop == 1);
                            if (bus_err) {
                                printf("\nBus Error at ");
                                print_hex(stdout,address,4);
                                printf("\n");
                            }
                        }
                    }
                }
                break;


            case 'x': /* wait for RUG to go non-busy */
                wait_for_rug(); 
                break;


	    case 'X': /* wait for IRIS to go non-busy */
		wait_for_iris();
		break;


            case '!': /* spawn shell */
                if (!sub_call) {
                    if (strlen(line) == 2)
                        system("csh");
                    else
                        system(line + 1);
                    printf("\n");
                }
                break;

            case 'a': /* activate vector printing mode */
            case 'c': /* reset rug */
            case 'd': /* set max acknowledge delay */
            case 'k': /* clear graphics device */
            case 'm': /* load monitor file */
            case 'p': /* display parameters */
            case 't': /* set ntse bit */
            case 'v': /* run vector file */
            case '#' :  /* copy comment to output file */
            case '\n': break;

            default: /* unknown command */
                printf("\nSay what?\n");
                break;
        } 

}






/*****************************************************************************/
/*                                                                           */
/* print_hex : print a hexadecimal number to the named file with any leading */
/*             zeros and pre-pended with a '0x'.                             */
/*                                                                           */
/* input : fp : file descriptor.                                             */
/*         num : the hexadecimal number (of size reg)                        */
/*         just : justification - how many spaces to fit the number in.      */
/*                                                                           */
/* output: the number is printed to the named file.                          */
/*                                                                           */
/* Note: this routine will work for numbers of type up to a 32 bit integer.  */
/*                                                                           */
/*****************************************************************************/
print_hex(fp,num,just)
FILE *fp;
unsigned int num;
int just;
{
    char string[11];
    int i;
    int num_size;

    if (just > 8) just = 8;

    /* determine the size of num in digits */
    if (num < 0) num_size = 8;
    else if (num < 0x10) num_size = 1;
    else if (num < 0x100) num_size = 2;
    else if (num < 0x1000) num_size = 3;
    else if (num < 0x10000) num_size = 4;
    else if (num < 0x100000) num_size = 5;
    else if (num < 0x1000000) num_size = 6;
    else if (num < 0x10000000) num_size = 7;
    else num_size = 8;

    /* initialize string */
    string[2 + num_size] = '\0';

    /* place the number in string - right justified */
    sprintf((string + 2 + (just - num_size)),"%x",num);

    /* finish building the string */
    string[0] = '0';
    string[1] = 'x';

    for (i=0; i < just - num_size; i++)
        string[i+2] = '0';

    /* print the number */
    fprintf(fp,"%s",string);
}

bus_error(sig)
int sig;
{
    bus_err = 1;
    longjmp(env,1);
}

interrupt()
{
  roop = 0;
  fflush(stdin);
}


wait_for_rug()
{
    while (*((reg *)(cntrl + 0x4800)) & 0x0001) {};
}


wait_for_iris()
{
    while (*((reg *)(cntrl + 0x6002)) & 0x0004) {};
}


execute_display_rom(stop_flag)
int stop_flag;
{
    reg address;
    int i;
    static int regions[8] = {0x0023, 0x0027, 0x002b, 0x002f,
        0x003f, 0x0043, 0x0047, 0x004b};

    /* initialize regions */
    for (i=0; i<8; i++) {
        address = getword(regions[i]);
        if (address != 0) 
            init_region(address,stop_flag);
    }

}


reg getword(address)
reg address;
{
    unsigned char byte0, byte1;

    byte1 = *((unsigned char *)(cntrl + address));
    byte0 = *((unsigned char *)(cntrl + address + 2));
    return((byte1 << 8) + byte0);
}


int init_region(address,stop_flag)
reg address;
int stop_flag;
{
    char ch;
    unsigned char op;
    reg adr;
    unsigned char byte, byte2;
    reg word;
    unsigned char count;
    int i;
    int bit_test;
    int copy_data;
    int done;
    static reg mask[16] = {
        0x0001, 0x0002, 0x0004, 0x0008,
        0x0010, 0x0020, 0x0040, 0x0080,
        0x0100, 0x0200, 0x0400, 0x0800,
        0x1000, 0x2000, 0x4000, 0x8000 };

    /* initialize */
    copy_data = done = 0;

    while (!done) {

        /* get opcode */
        op = *((unsigned char *)(cntrl + address));
        address += 2;

        /* determine what it says to do */
        done = (op & 0x80);
        copy_data = (op & 0x40);
        bit_test = (op & 0x01);

        /* get remainder of block */
        count = *((unsigned char *)(cntrl + address));
        address += 2;
        adr = getword(address);
        address += 4;

        if (stop_flag) {
            printf("Hit <RETURN> to continue-BLOCK address : ");
            print_hex(stdout,address - 8, 4);
            printf("\n");
            fflush(stdin);
            scanf("%c",&ch);
        }

        /* do correct action */
        if (bit_test) {
            byte = *((unsigned char *)(cntrl + address));
            address += 2;
            byte2 = *((unsigned char *)(cntrl + address));
            address += 2;
            if (stop_flag) {
                printf("Bit test of address ");
                print_hex(stdout,adr,4);
                printf(", mask ");
                print_hex(stdout,mask[byte2],4);
                printf(", value ");
                print_hex(stdout,byte,2);
                printf("\n\n");
            }
            while ((((*((reg *)(cntrl + adr))) >> byte2) & 0x01) != byte) {};
        }
        else {
            for (i=0; i<count+1; i++) {
                if (!copy_data) {
                    word = getword(address);
                    address += 4;
                }
                if (stop_flag) {
                    printf("Writing ");
                    print_hex(stdout,word,4);
                    printf(" to ");
                    print_hex(stdout,adr,4);
                    printf("\n\n");
                }
                *((reg *)(cntrl + adr)) = word;
                adr += 2;
            }
        }
    }
}
