/*	
	Synopsis: catfiles  [files ...]
	
	Author: Scott J. Warren
	Date:   2/16/89

	Description:
	    Catfiles emulates cat(1) in the sense that the contents of the 
	    specified files are printed to stdout.  However, catfiles has the 
	    added ability to automatically extract and print specific versions 
	    from RCSR and BSCCS source trees.

	    Input to catfiles consists of an arbitrary number of files 
	    containing lines of the form:

	    pathname     [revision]

	    Pathname is the path name of the file to be printed and revision
	    specifies which version of the file is to be retrieved from source 
	    code control.  Catfiles can also read input from stdin.

	    Catfiles automatically retrieves a specified version from an RCS 
	    or SCCS file by calling co(1) or get(1) respectively.  A file is
	    considered to be an RCS file if the base name of the specified path
	    ends with the suffix, ",v".  Similarly, a file is considered to be
	    an SCCS file if the base name begins with the prefix, "s.".

	    Catfiles was originally designed to operate in a source line 
	    counting tool package.  Other tools in this package include:  
	    linecount(1), ssize(1), chkfiles(1), cstrip(1) and pstrip(1).

	    A return code of 0 is returned if the program exits normally, 
	    otherwise, 1 is returned.

	    Catfiles issues a warning message and continues if it cannot parse 
	    an input line.  Catfiles issues an error message and terminates if 
	    the call to co, get, or cat fails.  These failures are normally 
	    caused by specifying a file that does not exist or a revision 
	    number that is not valid.  The program chkfiles(1), can be used as 
	    a front end to filter out these types of errors.
*/



static char *version = "@(#) $Revision: 2.4 $   $Date: 89/02/28 15:30:06 $";

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <varargs.h>


/*
**************************************************************************
**			Constant Definitions				**
**************************************************************************
*/

#define	PROC				/* null macro useful for searching. */
#define	FALSE		0
#define	TRUE		1
#define	CHNULL	  ('\0')
#define	CHNEWLINE ('\n')
#define	CPNULL	  ((char *) NULL)
#define	FILENULL  ((FILE *) NULL)
#define RCSSUFF	  ",v"
#define DEFREL	  "0"
#define	CMD  0				/* for calling RunCommand()	*/
#define	PIPE 1
#define TOKENSEP  " \t"
#define	CHSIZE	  (sizeof(char))


/*
**************************************************************************
**			Global Variables    				**
**************************************************************************
*/

	char	*calloc();
	char	*myname;			/* how program was invoked	*/
	char	*defargv[] = { "-" };		/* default argument list	*/
	char	*tempfile = "/tmp/catfXXXXXX";  /* temporary file */

	int	exitnicely();


/*
**************************************************************************
**			Macro Definitions				**
**************************************************************************
* Basename is either the full string or past last '/', if any.
* File is SCCS if basename starts with "s.".
*/

char	*cpBNTemp;			/* temporary */

#define	BASENAME(cp)	\
	(((cpBNTemp = strrchr (cp, '/')) == CPNULL) ? cp : (cpBNTemp + 1))

#define	SCCS(filename)	(strncmp (BASENAME (filename), "s.", 2) == 0)





/*
**************************************************************************
**				Error					**
**************************************************************************

Print an error message to stderr and exit nonzero.  Message is preceded
by "<myname>: " using global char *myname, and followed by a newline.

Input:	message, arg1, arg2, arg3, arg4  - components of the error message.
Output: None.
Return: None.
Globs: 	myname, tempfile
*/

/* VARARGS */
PROC Error (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

	exitnicely (1, tempfile);

} /* Error */

/*
**************************************************************************
**				warn					**
**************************************************************************

Print a warning message to stderr.  Message is preceded by "<myname>: " 
using global myname, and followed by a newline.

Input:	message, arg1, arg2, arg3, arg4  - components of the error message.
Output: None.
Return: None.
Globs: 	myname
*/

/* VARARGS */
PROC warn (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: warning: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

} /* warn */


/*
**************************************************************************
**			    exitnicely					**
**************************************************************************

Remove the temporary file, if any (ignore errors), and exit with a given
value (maybe a signal number if called due to a signal).

Input:	exitval		The value returned to the caller.
	tempfile	The temporary file to unlink.
Output: None.
Return: None.
Globs: 	No globals.
*/

PROC exitnicely (exitval, tempfile)
	int	exitval;
	char	*tempfile;
{
	unlink (tempfile);

	exit (exitval);

} /* exitnicely */


/*
**************************************************************************
**				RCS					**
**************************************************************************

Determine if the given path name represents a file under rcs(1) control.

Input:	fname		The path name in question.
Output: None.
Return: TRUE or FALSE.
Globs: 	No globals.
*/


PROC int RCS(fname)
	char	*fname;
{
	int	rc = FALSE;

	if (strlen(fname) > 1)
	    if (strcmp(fname + strlen(fname) - 2, RCSSUFF) == 0)
		rc = TRUE;

	return(rc);
}


/*
**************************************************************************
**				get_sccs_file				**
**************************************************************************

Extract the correct version of the specified file from sccs via get(1).  The
extracted code is stored into the temporary file, tempfilename.  If the user
has not provided any release number, the top of the SCCS file is retrieved.
If an error occurs during the check out, the program terminates.

Input:	tempfilename	Where to store the extracted file.
	sccsfilename	The SCCS file in question.
	release		The release to extract.
Output: None.
Return: None.
Globs: 	No globals.
*/

PROC get_sccs_file (tempfilename, sccsfilename, release)
   	char	*tempfilename;	
   	char	*sccsfilename;
   	char	*release;
		
{
   	char 	*relopt; 	/* option string for get(1)  */
	char	*cmd;
   	int	result;		/* of get command	    */

	/*
 	** Set up the release option string.
 	*/

	if (release != CPNULL)			/* release given */
	{
	    relopt = calloc(strlen("-t -r") + strlen(release) + 1, CHSIZE);
	    sprintf (relopt, "-t -r%.*s%c", strlen("-t -r") + strlen(release) + 1, release, CHNULL);
	}
	else					/* use top level, i.e relopt is blank */
	{
	    relopt = calloc(strlen(" ") + 1, CHSIZE);
	    sprintf (relopt, "%.*s%c", strlen(" ") + 1, " ", CHNULL);
	}

	/*
 	*  Execute the get(1) command.
 	*/

	cmd = calloc(strlen("get -k -s -p ") + strlen(relopt) + strlen(sccsfilename) +
		     strlen(" > ") + strlen(tempfilename) + 1, CHSIZE);

	sprintf(cmd, "get -k -s -p %s %s > %s%c", relopt, sccsfilename, tempfilename, CHNULL);

	result = system(cmd);

	if (result) 
	    Error("get %s of file \"%s\" failed", sccsfilename, tempfilename);
	
	free(cmd);

} /* get_sccs_file */


/*
**************************************************************************
**			    get_rcs_file				**
**************************************************************************

Extract the correct version of the specified file from rcs via co(1).  The
extracted code is stored into the temporary file, tempfile.  If the user
has not provided any release number, the top of the RCS file is retrieved.
If an error occurs during the check out, the program terminates.

Input:	tempfilename	Where to store the extracted file.
	rcsfilename	The RCS file in question.
	release		The release to extract.
Output: None.
Return: None.
Globs: 	No globals.
*/

PROC get_rcs_file (tempfilename, rcsfilename, release)
   	char	*tempfilename;	
   	char	*rcsfilename;
   	char	*release;
		
{
   	char 	*relopt; 	/* option string for co(1)  */
	char	*cmd;
   	int	result;		/* of co command	    */

	/*
 	** Set up the release option string.
 	*/

	if (release != CPNULL)			/* release given */
	{
	    relopt = calloc(strlen("-r") + strlen(release) + 1, CHSIZE);
	    sprintf (relopt, "-r%.*s%c", strlen("-r") + strlen(release) + 1, release, CHNULL);
	}
	else					/* use top level, i.e relopt is blank */
	{
	    relopt = calloc(strlen(" ") + 1, CHSIZE);
	    sprintf (relopt, "%.*s%c", strlen(" ") + 1, " ", CHNULL);
	}

	/*
 	*  Execute the co(1) command.
 	*/

	cmd = calloc(strlen("co -q -p ") + strlen(relopt) + strlen(rcsfilename) +
		     strlen(" > ") + strlen(tempfilename) + 1, CHSIZE);

	sprintf(cmd, "co -q -p %s %s > %s%c", relopt, rcsfilename, tempfilename, CHNULL);

	result = system(cmd);

	if (result) 
	    Error("co %s of file \"%s\" failed", rcsfilename, tempfilename);

	free(cmd);

} /* GetRcsFile */



/*
**************************************************************************
**			      read_file					**
**************************************************************************

Given a filename (for error messages) and an open stream pointer, read
data lines from the stream until exhausted, dumping the contents of each
file to stdout.  Files are extracted from source control if necessary.

char line[] is static so it's only allocated once.

Input:	filename	The name of the input file.
	filep		The input file stream.
	tempfile	The name of the temporary file where code will
			be stored if extracted from source control.
Output: None.
Return: None.
Globs: 	No globals.
*/

PROC read_file (filename, filep, tempfile)
	char	*filename;			/* for error messages 	*/
   	FILE	*filep;				/* open stream	      	*/
	char	*tempfile;
{
static	char	line [BUFSIZ];			/* input line		*/
   	char	*cp;				/* pos in line		*/
	char	*dofilename;			/* the file to dump	*/
	char	*rev;				/* revision number	*/
	char	*cmd;				/* cat command string	*/
	int	cmdlength=200;			/* length of cat cmd	*/
   	int	linenum = 0;			/* in file		*/
	int	rc;


	cmd = calloc(cmdlength, CHSIZE);

	while (fgets (line, BUFSIZ, filep) != CPNULL)
	{
	    linenum++;
	    if ((cp = strchr (line, CHNEWLINE)) != CPNULL)
		*cp = CHNULL;			/* remove newline char */
	    
	    /*
	    ** Parse the line into tokens, i.e. the file to dump and an 
	    ** optional revision number. 
	    */

	    if ((dofilename = strtok(line, TOKENSEP)) == CPNULL)
		warn("invalid input in file \"%s\", line %d", filename, linenum);

	    else {
		rev = strtok(CPNULL, TOKENSEP);

		if (RCS(dofilename)) {
		    get_rcs_file(tempfile, dofilename, rev);
		    dofilename=tempfile;
		    }
		else if (SCCS(dofilename)) {
		    get_sccs_file(tempfile, dofilename, rev); 
		    dofilename=tempfile;
		    }

		if ((strlen(dofilename) + 4) > cmdlength) {		/* allocate more space */
		    cmdlength = strlen("cat ") + strlen(dofilename) + 1;
		    cmd = calloc(cmdlength, CHSIZE);
		    }

		sprintf(cmd, "cat %s%c", dofilename, CHNULL);
		rc = system(cmd);

		if (rc != 0)
		    Error("cat of file \"%s\" failed", dofilename);
		}
	}

	if (ferror (filep))
	    warn("read failed from \"%s\"", filename);

} /* ReadFile */


/*
**************************************************************************
**				Main					**
**************************************************************************
*/



PROC main (argc, argv)
   	int	argc;
   	char	**argv;
{
extern	int	optind;			/* from getopt()	*/
extern	char	*optarg;		/* from getopt()	*/
   	FILE	*filep;			/* open input file	*/
   	char	*filename;		/* name to use		*/

	myname = *argv;
	argv ++;			/* Skip past prog name. */
	argc --;

	/*
	**  Prepare the temporary file and signals.
	**
	**  The temporary file is initialized here, but the file itself is created
	**  when first needed.  Set signals if they're not already ignored.
	*/

	mktemp (tempfile);

	if (signal (SIGHUP,  SIG_IGN) != SIG_IGN)
	    signal (SIGHUP, exitnicely, tempfile);

	if (signal (SIGINT,  SIG_IGN) != SIG_IGN)
	    signal (SIGINT, exitnicely, tempfile);

	if (signal (SIGQUIT,  SIG_IGN) != SIG_IGN)
	    signal (SIGQUIT, exitnicely, tempfile);

	signal (SIGTERM, exitnicely, tempfile);		/* unconditional */

	/*
	**  Read from list of files or stdin.
	*/

	if (argc < 1) {				/* no file names */
	    argc = 1;
	    argv = defargv;
	    }

	while (argc-- > 0) {
	    if (strcmp (*argv, "-") == 0) {
		filename = "<stdin>";
		filep	 = stdin;
		}
	    else if ((filep = fopen ((filename = *argv), "r")) == FILENULL)
		Error("can't open \"%s\" to read it", filename);

	    read_file(filename, filep, tempfile);

	    if (filep != stdin)
		fclose (filep);			/* assume it works */

	    argv++;
	    }	/* while */

	exitnicely (0, tempfile);

} /* main */
