static char *version = "@(#) 1.1 851118";
/*
 * Print object code size metrics report.
 * See the manual entry for details.
 */

#include <stdio.h>
#include <string.h>

char	*malloc();
char	*realloc();

#define	PROC				/* null; easy to find procs */
#define	FALSE	  0
#define	TRUE	  1
#define	CHNULL	  ('\0')
#define	CHNEWLINE ('\n')
#define	CPNULL	  ((char *) NULL)
#define	FILENULL  ((FILE *) NULL)
#define	REG	  register

char	*myname;			/* how program was invoked	*/

char	*defargv[] = { "-" };		/* default argument list	*/


/************************************************************************
 * DATA COLLECTION VARIABLES:
 */

#define	NEW	0			/* for counting new files	*/
#define	RENEW	1			/* reused files, new versions	*/
#define	REOLD	2			/* reused files, old versions	*/
#define	REUNK	3			/* reused files, no old		*/
#define	CLASSES	(REUNK + 1)		/* total number of classes	*/

#define	COMMAND "size"			/* start of command lines	*/
#define	CMDLEN  5			/* initial string length	*/

char	* command [CLASSES];		/* commands and file args	*/

int	cmdlen [CLASSES] = { CMDLEN, CMDLEN, CMDLEN, CMDLEN };
int	files  [CLASSES] = { 0, 0, 0, 0 };	/* files in each class	*/
long	bytes  [CLASSES] = { 0, 0, 0, 0 };	/* bytes in each class	*/


/************************************************************************
 * M A I N
 *
 * Initialize, read data, get sizes, and print report.
 */

PROC main (argc, argv)
REG	int	argc;
REG	char	**argv;
{
REG	int	class;			/* current class	*/
REG	FILE	*filep;			/* open input file	*/
REG	char	*filename;		/* name to use		*/

/*
 * INITIALIZE:
 *
 * command[] values must reside in malloc'd space so they can be realloc'd.
 * cmdlen[] and files[] are pre-initialized.
 */

	myname = *argv++;

	for (class = 0; class < CLASSES; class++)
	{
	    if ((command [class] = malloc (CMDLEN)) == CPNULL)
		Error ("malloc failed");

	    strcpy (command [class], COMMAND);
	}

/*
 * READ FROM LIST OF FILES OR STDIN:
 */

	if (--argc < 1)				/* no file names */
	{
	    argc = 1;
	    argv = defargv;
	}

	while (argc-- > 0)
	{
	    if (strcmp (*argv, "-") == 0)	/* read stdin */
	    {
		filename = "<stdin>";
		filep	 = stdin;
	    }
	    else if ((filep = fopen ((filename = *argv), "r")) == FILENULL)
		Error ("can't open file \"%s\" to read it", filename);

	    ReadFile (filename, filep);

	    if (filep != stdin)
		fclose (filep);			/* assume it works */

	    argv++;
	}

/*
 * GET SIZES, PRINT REPORT:
 */

	for (class = 0; class < CLASSES; class++)
	    GetSizes (class);

	PrintReport();
	exit (0);

} /* main */


/************************************************************************
 * E R R O R
 *
 * Print an error message to stderr and exit nonzero.  Message is preceded
 * by "<myname>: " using global char *myname, and followed by a newline.
 */

/* VARARGS */
PROC Error (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

	exit (1);

} /* Error */


/************************************************************************
 * R E A D   F I L E
 *
 * Given a filename (for error messages) and an open stream pointer, read
 * data lines from the stream until exhausted, accumulating filenames from
 * each one, then check for error.
 *
 * char line[] is static so it's only allocated once.
 */

PROC ReadFile (filename, filep)
	char	*filename;	/* for errors	*/
REG	FILE	*filep;		/* open stream	*/
{
static	char	line [BUFSIZ];	/* input line	*/
REG	char	*cp;		/* pos in line	*/
REG	int	linenum = 0;	/* in file	*/

	while (fgets (line, BUFSIZ, filep) != CPNULL)
	{
	    if ((cp = strchr (line, CHNEWLINE)) != CPNULL)
		*cp = CHNULL;			/* terminate at newline */

	    DoInputLine (filename, ++linenum, line);
	}

	if (ferror (filep))
	    Error ("read failed from \"%s\"", filename);

} /* ReadFile */


/************************************************************************
 * D O   I N P U T   L I N E
 *
 * Given a filename and linenumber (for error messages) and an input line,
 * parse the line and accumulate the filename(s) from that line in global
 * char *command[] for the class, also updating globals cmdlen[] and files[].
 * Error out if anything is noticeably wrong with the line.
 */

PROC DoInputLine (filename, linenum, line)
	char	*filename;		/* for errors	*/
	int	linenum;		/* for errors	*/
	char	*line;			/* to parse	*/
{
REG	char	*tokensep = " \t";	/* token separator	*/
	char	*ltp;			/* linetype pointer	*/
REG	char	linetype;		/* 'n' | 'r' | 'u'	*/
	char	*dofilename;		/* file to size		*/
	char	*oldfilename;		/* from third field	*/

/*
 * PARSE LINE INTO TOKENS:
 */

	if (((ltp = strtok (line, tokensep)) == CPNULL)
	 || (((linetype = *ltp) != 'n')
	  && (linetype != 'r') && (linetype != 'u')))
	{
	    Error ("missing or invalid type-letter in file \"%s\", line %d",
		   filename, linenum);
	}

	if ((dofilename = strtok (CPNULL, tokensep)) == CPNULL)
	    Error ("missing filename in file \"%s\", line %d",
		filename, linenum);

	if (((oldfilename = strtok (CPNULL, tokensep)) != CPNULL)
	 && ((linetype != 'r') || (strtok (CPNULL, tokensep) != CPNULL)))
	{
	    Error ("too many tokens in file \"%s\", line %d",
		filename, linenum);
	}

/*
 * HANDLE DIFFERENT LINE TYPES:
 */

	switch (linetype)
	{
	    case 'n':	AppendFilename (dofilename, NEW);	break;
	    case 'u':	AppendFilename (dofilename, REUNK);	break;

	    case 'r':
		if (oldfilename == CPNULL)
		{
		    Error ("old filename is required in file \"%s\", line %d",
			filename, linenum);
		}

		AppendFilename (dofilename,  RENEW);
		AppendFilename (oldfilename, REOLD);
		break;
	}

} /* DoInputLine */


/************************************************************************
 * A P P E N D   F I L E N A M E
 *
 * Given a filename and a class to save it under, append it to the command
 * line for that class (global char *command[]) and update globals cmdlen[]
 * and files[].
 */

PROC AppendFilename (filename, class)
	char	*filename;	/* to append */
REG	int	class;		/* which one */
{
	int	oldlen = cmdlen [class];
	int	newlen = oldlen + 1 + strlen (filename);

	if ((command [class] = realloc (command [class], newlen)) == CPNULL)
	    Error ("realloc failed");

	command [class] [oldlen - 1] = ' ';		/* append separator */
	strcpy (command [class] + oldlen, filename);	/* append filename  */

	cmdlen [class] = newlen;
	files  [class] ++;

} /* AppendFilename */


/************************************************************************
 * G E T   S I Z E S
 *
 * Given a class and globals char *command[] and int files[], run the
 * command for that class, add up sizes from the output, and save the
 * total in global long bytes[].
 */

PROC GetSizes (class)
REG	int	class;			/* which one		*/
{
REG	FILE	*pipep;			/* to read pipe		*/
static	char	line [BUFSIZ];		/* only allocate once	*/
REG	int	field2;			/* use output field 2?	*/
REG	char	*tokensep = " \t";	/* token separator	*/
REG	char	*cp;			/* place in output line	*/
REG	long	total = 0;		/* to accumulate	*/

/*
 * RUN COMMAND, ADD SIZES:
 */

	if (files [class] < 1)		/* no files in class */
	    return;			/* no action needed  */

	field2 = (files [class] > 1);

	if ((pipep = popen (command [class], "r")) == FILENULL)
	    Error ("can't run %s command", COMMAND);

	while (fgets (line, BUFSIZ, pipep) != CPNULL)
	{
	    cp = strtok (line, tokensep);

	    if (field2)
		cp = strtok (CPNULL, tokensep);

	    total += atol (cp);		/* assume field exists and is usable */
	}

/*
 * CHECK ERRORS, FINISH UP:
 */

	if (ferror (pipep))
	    Error ("read failed from %s command", COMMAND);

	if (pclose (pipep))			/* any non-zero return */
	    Error ("%s command failed", COMMAND);

	bytes [class] = total;

} /* GetSizes */


/************************************************************************
 * P R I N T   R E P O R T
 *
 * Given global totals, print a report.
 */

char	*format1 = "%-16s%11ld\n";		/* to print values */
char	*format2 = "%-16s%11ld   (delta %+ld)\n";

PROC PrintReport()
{
	puts ("OBJECT BYTES          TOTAL");

	printf (format1, "New code",	     bytes [NEW]);

	printf (format2, "Reused, have old", bytes [RENEW],
					     bytes [RENEW] - bytes [REOLD]);

	printf (format1, "Reused, no old",   bytes [REUNK]);

	printf (format1, "TOTAL", bytes [NEW] + bytes [RENEW] + bytes [REUNK]);

} /* PrintReport */
