static char *version = "@(#) 2 860911";
/*
 * Strip comments and blank lines from shell source.
 * See the manual entry for details.
 */

#include <stdio.h>

#define	PROC				/* null; easy to find procs */
#define	FALSE	  0
#define	TRUE	  1
#define	CHNULL	  ('\0')
#define	CHNEWLINE ('\n')
#define	CPNULL	  ((char *) NULL)
#define	FILENULL  ((FILE *) NULL)
#define	REG	  register

char	*myname;			/* how program was invoked	*/

char	*defargv[] = { "-" };		/* default argument list	*/


/************************************************************************
 * STATE DEFINITIONS:
 *
 * Each input character drives a state machine with these states (starting
 * each file at WHITESP).  The state machine itself is embedded in code.
 * The shell allows complex nested quoting, but this program doesn't try
 * to handle it.
 *
 * All states except the QUOTE and [DB]BSLASH states reset to WHITESP at
 * end of line.  In other words, they don't continue across a newline.
 */

#define	WHITESP		0	/* at whitespace or file start	*/
#define	TEXT		1	/* in text, can't start comment	*/
#define	COMMENT		2	/* now in comment		*/
#define	BSLASH		3	/* naked backslash		*/
#define	SQUOTE		4	/* in single quotes		*/
#define	DQUOTE		5	/* in double quotes		*/
#define	BQUOTE		6	/* in backwards quotes		*/
#define	DBSLASH		7	/* backslash in double quotes	*/
#define	BBSLASH		8	/* backslash in back quotes	*/


/************************************************************************
 * M A I N
 *
 * Open each file, read it, check for error, close it.
 */

PROC main (argc, argv)
REG	int	argc;
REG	char	**argv;
{
REG	FILE	*filep;			/* open input file	*/
REG	char	*filename;		/* name to use		*/

	myname = *argv++;

	if (--argc < 1)				/* no file names */
	{
	    argc = 1;
	    argv = defargv;			/* use default */
	}

	while (argc-- > 0)
	{
	    if (strcmp (*argv, "-") == 0)	/* read stdin */
	    {
		filename = "<stdin>";
		filep	 = stdin;
	    }
	    else if ((filep = fopen ((filename = *argv), "r")) == FILENULL)
		Error ("can't open file \"%s\" to read it", filename);

	    ReadFile (filep);

	    if (ferror (filep))
		Error ("read failed from file \"%s\"", filename);

	    if (filep != stdin)
		fclose (filep);			/* assume it works */

	    argv++;
	}

	exit (0);

} /* main */


/************************************************************************
 * E R R O R
 *
 * Print an error message to stderr and exit nonzero.  Message is preceded
 * by "<myname>: " using global char *myname, and followed by a newline.
 */

/* VARARGS */
PROC Error (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

	exit (1);

} /* Error */


/************************************************************************
 * R E A D   F I L E
 *
 * Given an open stream pointer, read characters from the stream until
 * exhausted, driving the state machine and emitting only non-comment
 * characters on non-blank lines to stdout.  The caller must check for
 * read errors.
 *
 * This is one large procedure (gag) for efficiency.
 */

PROC ReadFile (filep)
REG	FILE	*filep;			/* open stream to read	 */
{
REG	int	state = WHITESP;	/* current machine state */
REG	int	stuff = FALSE;		/* non-blank input line? */
REG	int	ch;			/* current input "char"  */

static	char	outline [BUFSIZ];	/* allocate only once	 */
REG	char	*cp = outline;		/* place in outline	 */

#define	NONBLANK(ch)	((ch != ' ') && (ch != '\t'))

#define	PUTCHAR(ch)	{ *cp++ = ch; if (NONBLANK (ch)) stuff = TRUE; }


	while (TRUE)			/* until return */
	{

/*
 * HANDLE END OF FILE OR INPUT LINE:
 *
 * There's one weird special case.  If we get a newline and the state is
 * DBSLASH or BBSLASH, don't even recognize the newline as special; it's
 * just another character, escaped by the preceding slash.  It will go into
 * outline just fine and we can continue after the appropriate state change.
 */

	    if (((ch = getc (filep)) == EOF)
	     || ((ch == CHNEWLINE) && (state != DBSLASH) && (state != BBSLASH)))
	    {
		if ((state != SQUOTE) && (state != DQUOTE) && (state != BQUOTE))
		    state = WHITESP;		/* reset to default */

		if ((stuff)			/* non-blank char(s) on line */
		 || (state != WHITESP))		/* in quotes, so do print it */
		{
		    stuff = FALSE;
		    *cp   = CHNULL;		/* mark end of line	   */
		    puts (outline);		/* adds newline if missing */
		}

		if (ch == EOF)
		    return;

		cp = outline;			/* reset line	*/
		continue;			/* do next char	*/
	    }

/*
 * IN WHITESPACE, LOOK FOR COMMENT CHAR:
 */

	    switch (state)
	    {
		case WHITESP:
		    if (ch == '#')
		    {
			state = COMMENT;
			break;			/* do next char */
		    }

		    state = TEXT;		/* assume for now */

		    /* then fall through! */

/*
 * IN WHITESPACE OR TEXT, LOOK AT NEXT CHAR:
 */

		case TEXT:
		    switch (ch)
		    {
			case ' ':   /* fall through */
			case '\t':  state = WHITESP;	break;
			case '\\':  state = BSLASH;	break;
			case '\'':  state = SQUOTE;	break;
			case '"':   state = DQUOTE;	break;
			case '`':   state = BQUOTE;	break;
			/* default: do nothing */
		    }

		    PUTCHAR (ch);
		    break;

/*
 * IN COMMENT:
 */

	     /* case COMMENT: do nothing, ignore char */

/*
 * AFTER BACKSLASH, TREAT NEXT CHAR AS TEXT:
 */

		case BSLASH:
		    state = TEXT;
		    PUTCHAR (ch);
		    break;

/*
 * IN SINGLE QUOTES, LOOK FOR CLOSE QUOTE:
 */

		case SQUOTE:
		    if (ch == '\'')
			state = TEXT;

		    PUTCHAR (ch);
		    break;

/*
 * IN DOUBLE OR BACK QUOTES, LOOK FOR CLOSE QUOTE OR BACKSLASH:
 */

		case DQUOTE:
		    if (ch == '"')
			state = TEXT;

		    else if (ch == '\\')
			state = DBSLASH;

		    PUTCHAR (ch);
		    break;

		case BQUOTE:
		    if (ch == '`')
			state = TEXT;

		    else if (ch == '\\')
			state = BBSLASH;

		    PUTCHAR (ch);
		    break;

/*
 * IN DOUBLE OR BACK QUOTES AFTER BACKSLASH, "IGNORE" NEXT CHAR:
 */

		case DBSLASH:
		    state = DQUOTE;
		    PUTCHAR (ch);
		    break;

		case BBSLASH:
		    state = BQUOTE;
		    PUTCHAR (ch);
		    break;

	    } /* switch */
	} /* while */

} /* ReadFile */
