static char *version = "@(#) $Revision: 3.3 $   $Date: 89/02/23 16:01:06 $";
/*
 * Strip comments and blank lines from C [Pascal] source.
 * See the manual entry for details.
 *
 * Compile with -DPASCAL to get a Pascal stripper instead, in which case
 * some comments in "[]" apply, and see the other manual entry for details.
 */

#include <stdio.h>

#ifdef PASCAL
#undef C
#else
#define C 1
#endif

#define	PROC				/* null; easy to find procs */
#define	FALSE	  0
#define	TRUE	  1
#define	CHNULL	  ('\0')
#define	CHNEWLINE ('\n')
#define	CPNULL	  ((char *) NULL)
#define	FILENULL  ((FILE *) NULL)
#define	REG	  register

char	*myname;			/* how program was invoked */
char	*defargv[] = { "-" };		/* default argument list   */
int	rflag = FALSE;			/* -r (report) option	   */

#ifdef C
#define	CHOPEN  '/'			/* part of "/*"			   */
#define	CHCLOSE '/'			/* part of <can't mention it here> */

#else /* PASCAL */
#define	CHOPEN  '('			/* part of "(*" */
#define	CHCLOSE ')'			/* part of "*)" */

int	mflag = FALSE;			/* -m (MODCAL) option */
#endif


/************************************************************************
 * STATE DEFINITIONS:
 *
 * Each input character drives a state machine with these states (starting
 * each file at NORMAL).  The state machine itself is embedded in code.
 *
 * Assumes TRANSOUT resets to COMMENT at end of line; all other states except
 * COMMENT and DBSLASH reset to NORMAL.  In other words, transitions and
 * quoting don't continue across a newline (not allowed in C [or Pascal]).
 *
 * Naked backslashes are not legal in C *except* in macro definitions,
 * where they can convert a "comment" to a non-comment, e.g.:
 *
 *	#define X abc\/*test
 *
 * It's obscure, but must be supported.
 */

#define	NORMAL		0	/* now in typical text			*/
#define	TRANSIN		1	/* found CHOPEN, looking for '*'	*/
#define	COMMENT		2	/* now in comment			*/
#define	TRANSOUT	3	/* found '*', looking for CHCLOSE	*/
#define	BSLASH		4	/* naked backslash [C only]		*/
#define	SQUOTE		5	/* in single quotes			*/
#define	DQUOTE		6	/* in double quotes [C only]		*/
#define	SBSLASH		7	/* backslash in single quotes [C only]	*/
#define	DBSLASH		8	/* backslash in double quotes [C only]	*/


/************************************************************************
 * M A I N
 *
 * Open each file, read it, check for error, close it.
 */

PROC main (argc, argv)
REG	int	argc;
REG	char	**argv;
{
REG	FILE	*filep;			/* open input file	        */
REG	char	*filename;		/* name to use		        */
	int	lcount  = 0;		/* total number of lines        */
	int	blcount = 0;		/* number of blank lines        */
	int	comcount= 0;		/* number of whole comment lines*/

/*
 * CHECK ARGUMENTS:
 */

	myname = *argv++;

#ifdef C
	if ((argc >= 2) && (strcmp (*argv, "-r") == 0))
	{
	    rflag = TRUE;
	    argc--;
	    argv++;
	}

#else	/* PASCAL */

	if ((argc >= 2) && ((strcmp (*argv, "-r") == 0) || (strcmp(*argv, "-m") == 0)))
	{
	    if (strcmp(*argv, "-r") == 0)
		rflag = TRUE;
	    else
		mflag = TRUE;
	    argc--;
	    argv++;
	}

	if ((argc >= 2) && ((strcmp (*argv, "-r") == 0) || (strcmp(*argv, "-m") == 0)))
	{
	    if (strcmp(*argv, "-r") == 0)
		rflag = TRUE;
	    else
		mflag = TRUE;
	    argc--;
	    argv++;
	}
#endif

	if (--argc < 1)				/* no file names */
	{
	    argc = 1;
	    argv = defargv;			/* use default */
	}

/*
 * OPEN, READ, AND CLOSE EACH FILE:
 */

	while (argc-- > 0)
	{
	    if (strcmp (*argv, "-") == 0)	/* read stdin */
	    {
		filename = "<stdin>";
		filep	 = stdin;
	    }
	    else if ((filep = fopen ((filename = *argv), "r")) == FILENULL)
		Warn ("can't open file \"%s\" to read it", filename);

	    if (filep != FILENULL) 
	    {
		if (ReadFile (filep, &lcount, &blcount, &comcount))
		    Warn("file \"%s\" ends within a comment\n", filename);

		if (ferror (filep))
		    Warn ("read failed from file \"%s\"", filename);

		else if (filep != stdin)
		    fclose (filep);			/* assume it works */
	    }

	    argv++;
	}

	if (rflag) 
	{
	    printf("Total lines:           %6d\n", lcount);
	    printf("Whole comment lines:   %6d\n", comcount);
	    printf("Blank lines:           %6d\n", blcount);
	}

	exit (0);

} /* main */


/************************************************************************
 * W A R N
 *
 * Print a warning message to stdout.  Message is preceded
 * by "<myname>: " using global char *myname, and followed by a newline.
 */

/* VARARGS */
PROC Warn (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

} /* Error */


/************************************************************************
 * R E A D   F I L E
 *
 * Given an open stream pointer, read characters from the stream until
 * exhausted, driving the state machine and emitting only non-comment
 * characters on non-blank lines to stdout.  The caller must check for
 * read errors.  Return non-zero if the file ends in the middle of a
 * comment.
 *
 * This is one large procedure (gag) for efficiency.
 */

PROC int ReadFile (filep, lcount, blcount, comcount)
REG	FILE	*filep;			/* open stream to read	  */
	int	*lcount;
	int	*blcount;
	int	*comcount;
{
REG	int	state = NORMAL;		/* current machine state  */
REG	int	stuff = FALSE;		/* "stuff" on input line? */
REG	int	ch = ' ';		/* current input "char"   */
	int	lastcom = 0;
	int	comend = 0;

static	char	outline [BUFSIZ];	/* allocate only once	  */
REG	char	*cp = outline;		/* place in outline	  */

#ifdef PASCAL
REG	int	brace;			/* comment type is brace? */
#endif

#define	NONBLANK(ch)	((ch != ' ') && (ch != '\t'))

#define	PUTCHAR(ch)	{ *cp++ = ch; if (NONBLANK (ch)) stuff = TRUE; }
#define	PUTCONS(ch)	{ *cp++ = ch; stuff = TRUE; }


	while (ch != EOF)
	{

/**************************************
 * HANDLE END OF FILE OR INPUT LINE:  *
 **************************************/

	    if (((ch = getc (filep)) == EOF) || (ch == CHNEWLINE))
	    {
		if (state == TRANSIN)		/* were in transition, but */
		    PUTCONS (CHOPEN);		/* wasn't a comment start  */

		if (state == TRANSOUT)		/* reset state if needed */
		    state = COMMENT;
#ifdef C
		else if (state == DBSLASH)	/* special case, stay quoted */
		    state = DQUOTE;
#endif
		else if (state != COMMENT)
		    state = NORMAL;

/*
 * PRINT LINE IF NEEDED:
 */


		if (stuff)			/* non-blank char(s) on line */
		{
		    stuff = FALSE;
		    *cp   = CHNULL;		/* mark end of line	   */
		    comend = lastcom = *lcount;
		    if (! rflag)
		        puts (outline);		/* adds newline if missing */
		}
		else
		    if (ch != EOF)
		    {
			if (state == COMMENT)
			{
			    (*comcount)++;
			    lastcom = *lcount;
			}
			else if (comend > lastcom)	/* comment ended, nothing else on the line. */
			{
			    (*comcount)++;
			    comend = lastcom = *lcount;
			}
			else
			    (*blcount)++;
		    }

		cp = outline;
		(*lcount)++;
	    }
	    else
	    {


/**************************************
 * HANDLE CHARACTERS ON THE LINE      *
 **************************************/

/*
 * TRANSITION TO COMMENT, LOOK FOR SECOND CHAR:
 */

		switch (state)
		{
		    case TRANSIN:
			if (ch == '*')
			{
			    state = COMMENT;
#ifdef PASCAL
			    brace = FALSE;
#endif
			    break;			/* do next char */
			}

			PUTCONS (CHOPEN);		/* previous char */
			state = NORMAL;

			/* fall through! */

/*
 * IN NORMAL STATE, LOOK FOR COMMENT OR QUOTE:
 */

		    case NORMAL:
			if (ch == CHOPEN)
			{
			    state = TRANSIN;
			    break;			/* skip this char */
			}
			else if (ch == '\'')
			    state = SQUOTE;
#ifdef C
			else if (ch == '"')
			    state = DQUOTE;
			else if (ch == '\\')
			    state = BSLASH;
#else /* PASCAL */
			else if (ch == '{')
			{
			    state = COMMENT;
			    brace = TRUE;
			    break;			/* skip this char */
			}
#endif
		    /* else nothing special */

			PUTCHAR (ch);
			break;

/*
 * IN COMMENT, LOOK FOR END:
 */

		    case COMMENT:
#ifdef C
			if (ch == '*')
			    state = TRANSOUT;
#else /* PASCAL */
			if ((ch == '*') && ((! mflag) || (! brace)))
			    state = TRANSOUT;

			else if ((ch == '}') && ((! mflag) || brace))
			{
			    state = NORMAL;
			    comend = *lcount;
			}
#endif
			break;

/*
 * TRANSITION OUT, LOOK FOR SECOND CHAR:
 */

		    case TRANSOUT:
			if (ch == CHCLOSE)
			{
			    state = NORMAL;
			    comend = *lcount;
			}

			else if (ch != '*')
			    state = COMMENT;

			break;

#ifdef C
/*
 * AFTER BACKSLASH, "IGNORE" NEXT CHAR:
 */

		    case BSLASH:
			state = NORMAL;
			PUTCHAR (ch);
			break;
#endif

/*
 * IN QUOTES, LOOK FOR CLOSE QUOTE:
 */

		    case SQUOTE:
			if (ch == '\'')
			    state = NORMAL;
#ifdef C
			else if (ch == '\\')
			    state = SBSLASH;
#endif
			PUTCHAR (ch);
			break;

#ifdef C
		    case DQUOTE:
			if (ch == '"')
			    state = NORMAL;
			else if (ch == '\\')
			    state = DBSLASH;
			PUTCHAR (ch);
			break;

/*
 * IN QUOTES AFTER BACKSLASH, "IGNORE" NEXT CHAR:
 */

		    case SBSLASH:
			state = SQUOTE;
			PUTCHAR (ch);
			break;

		    case DBSLASH:
			state = DQUOTE;
			PUTCHAR (ch);
			break;
#endif
		} /* switch */
	    }
	} /* while */

	(*lcount)--;
	return (state == COMMENT);

} /* ReadFile */
