/* HPUX_ID: @(#)cs80.h	27.2     85/06/19  */

#include <sys/ioctl.h>


typedef long signed32;
typedef unsigned long unsigned32;
typedef short signed16;
typedef unsigned short unsigned16;
typedef char signed8;
typedef unsigned char unsigned8;


struct sva_type {
	unsigned16 utb;		/* upper two bytes */
	unsigned32 lfb;		/* lower four bytes */
};


/*
**  describe's controller type field masks
*/
#define CT_MULTI_UNIT	0x01
#define CT_MULTI_PORT	0x02
#define CT_SUBSET_80	0x04

/*
**  describe field byte counts
*/
#define	DESCRIBE_BYTES	37	/* total bytes returned by the describe cmd */
#define	CNTRL_BYTES	 5	/* controller description byte count */
#define	UNIT_BYTES	19	/* unit description byte count */
#define	VOLUME_BYTES	13	/* volume description byte count */

struct cntrl_desc {
	unsigned16	iv;	/* installed unit word, bits */
	unsigned16	mitr;	/* max inst. xfr rate (Kbytes) */
	unsigned8	ct;	/* controller type */
};

struct unit_desc {
	unsigned8	dt;	/* generic device type */
	unsigned8	dn[3];	/* device number (6 BCD digits) */
	unsigned16	nbpb;	/* # bytes per block */
	unsigned8	nbb;	/* # of blocks which can be buffered */
	unsigned8	rbs;	/* reccomendcd burst size */
	unsigned16	blocktime; /* block time in micorseconds */
	unsigned16	catr;	/* continuous avg xfr rate (Kbytes) */
	unsigned16	ort;	/* optimal retry time in centiseconds */
	unsigned16	atp;	/* access time parameter in centiseconds */
	unsigned8	mif;	/* maximum interleave factor */
	unsigned8	fvb;	/* fixed volume byte: 1/volume */
	unsigned8	rvb;	/* removeable volume byte: 1/volume */
};

struct volume_desc {
	unsigned	maxcadd:24; /* maximum cylinder address */
	unsigned8	maxhadd;    /* maximum head address */
	unsigned16	maxsadd;    /* maximum sector address */
	struct sva_type	maxsvadd;   /* maximum single vector address */
	unsigned8	currentif;  /* current interleave factor */
};

struct describe_type {
		/* controller description field */
	union {
		unsigned8 cntrl_bytes[CNTRL_BYTES];
		struct cntrl_desc controller;
	} controller_tag;
		/* unit description field */
	union {
		unsigned8 unit_bytes[UNIT_BYTES];
		struct unit_desc unit;
	} unit_tag;
		/* volume description field */
	union {
		unsigned8 volume_bytes[VOLUME_BYTES];
		struct volume_desc volume;
	} volume_tag;
};


/* reject errors field */
#define channel_parity_error		2
#define illegal_opcode			5
#define module_addressing		6
#define address_bounds			7
#define parameter_bounds		8
#define illegal_parameter		9
#define message_sequence		10
#define message_length			12

/* fault errors field */
#define cross_unit			17
#define controller_fault		19
#define unit_fault			22
#define diagnostic_result		24
#define operator_release_required 	26
#define diagnostic_release_required 	27
#define internal_maint_required		28
#define power_fail			30
#define retransmit			31

/* access errors field */
#define illegal_parallel_operation 	32
#define uninitialized_media		33
#define no_spares_available		34
#define not_ready			35
#define write_protect			36
#define no_data_found			37
#define unrecovb_data_overflow		40
#define unrecovb_data			41
#define end_of_file			43
#define end_of_volume			44

/* information errors field */
#define operator_request		48
#define diagnostic_request		49
#define intnl_maint_reqst		50
#define media_wear			51
#define latency_induced			52
#define error_bit_53			53
#define error_bit_54			54
#define auto_sparing_invoked		55
#define error_bit_56			56
#define recovb_data_overflow		57
#define marginal_data			58
#define recovb_data			59
#define error_bit_60			60
#define maint_track_overflow		61
#define error_bit_62			62
#define error_bit_63			63

/* pseudo error bit */
#define NO_ERRORS			64


struct status_mask_type {
	unsigned16 s_m[4];
	};

struct cs_status_type {
	unsigned8  vvvvuuuu;
	signed8	requesting_unit;
	struct status_mask_type errorbits;
	union {
		unsigned8 info[10];		/* generic way to index */
		struct { 
			struct sva_type nta;	/* new target address */
			int faulting;		/* fault log */
		} nta_fault;
		struct { 
			struct sva_type aaa;	/* affected area address */
			int afl;		/* affected field length */
		} aaa_fault;
		signed8 uee[6];			/* units experiencing errors */
		unsigned8 dor[6];		/* diagnostic results */
		struct sva_type ta;		/* target address */
		struct sva_type bba;		/* bad block address */
		signed8 urr[6];			/* units requesting release */
		struct sva_type btbs;		/* block to be spared */
		struct sva_type rba;		/* recovb block address */
	} st_u;
};

typedef enum { allow_release_timeout, suppress_release_timeout } t_type;

typedef enum { disable_auto_release, enable_auto_release} z_type;

#define CMDlocate_and_read 	0x00
#define CMDlocate_and_write	0x02
#define CMDlocate_and_verify	0x04
#define CMDspare_block		0x06
#define CMDcopy_data		0x08
#define CMDcold_load_read	0x0A
#define CMDrequest_status	0x0D
#define CMDrelease		0x0E
#define CMDrelease_denied	0x0F
#define CMDset_address_1V	0x10
#define CMDset_address_3V	0x11
#define CMDset_block_disp	0x12
#define CMDset_length		0x18
#define UNIT_BASE		0x20
#define CMDinit_util_NEM	0x30
#define CMDinit_util_REM	0x31
#define CMDinit_util_SEM	0x32
#define CMDinit_diagnostic	0x33
#define CMDno_op		0x34
#define CMDdescribe		0x35
#define CMDinit_media		0x37
#define CMDset_options		0x38
#define CMDset_rps		0x39
#define CMDset_retry_time	0x3A
#define CMDset_release		0x3B
#define CMDset_burst_LBO	0x3C
#define CMDset_burst_ABT	0x3D
#define CMDset_status_mask	0x3E
#define VOLUME_BASE		0x40
#define CMDset_retadd_mode	0x48
#define CMDwrite_file_mark	0x49
#define CMDunload		0x4A

#define XCMDchan_indep_clear	0x08
#define XCMDcancel		0x09

typedef unsigned8 CMD_type;

#define TRANSPARENT_SEC SCG_BASE + 18
#define COMMAND_SEC	SCG_BASE +  5
#define EXECUTION_SEC	SCG_BASE + 14
#define REPORTING_SEC	SCG_BASE + 16


/* status masks */

#define B(error) (0x8000>>(error % 16))

/* reject errors field */
#define status_mask_0 0

/* fault errors field */
#define status_mask_1 0

/* access errors field */
#define status_mask_2 0

/* information errors field */
#define status_mask_3	B(media_wear) |\
			B(latency_induced) |\
			B(error_bit_53) |\
			B(error_bit_54) |\
			B(auto_sparing_invoked) |\
			B(error_bit_56) |\
			B(recovb_data_overflow) |\
			B(marginal_data) |\
			B(recovb_data) |\
			B(error_bit_60) |\
			B(maint_track_overflow) |\
			B(error_bit_62) |\
			B(error_bit_63)

/*
 *  CIOC_VERIFY parameter structure
 */

struct verify_parms {
	long	start;		/* start address in bytes */
	long	length;		/* length in bytes */
};


/*
 *  CIOC_MARK parameter structure
 */

struct mark_parms {
	long	start;		/* start address in bytes */
};


/*
 *  CIOC_SET_OPTIONS bit definitions  -  allowed only with cartridge tapes
 */

#define	SO_CHARACTER_COUNT	0x01	/* enable character count */
#define	SO_SKIP_SPARING		0x02	/* skip vs jump auto sparing */
#define	SO_AUTO_SPARING		0x04	/* enable auto sparing */
#define	SO_IMMEDIATE_REPORT	0x08	/* enable immediate report (Buf/Mer) */
#define	SO_MEDIA_UNLOAD		0x80	/* media vs cartridge unload (Mer) */

/*
 *  CIOC_SET_CMD parameter structure
 */

#define	CMD_LEN	20	/* maximum # data bytes in the cmd message */

struct cmd_parms {
	long	cmd_ticks;		/* timeout for C-phase */
	long	exec_ticks;		/* timeout for E-phase if appl. */
	char	cmd_length;		/* actual command message length */
	char	cmd_message[CMD_LEN];	/* comand message byte string */
};


/*
**  currently undocumented CS80 ioctl functions used by tcio and mediainit
*/
#define	CIOC_DESCRIBE		_IOR(C, 1, struct describe_type)
#define	CIOC_UNLOAD		_IO (C, 2)
#define	CIOC_VERIFY		_IOW(C, 3, struct verify_parms)
#define	CIOC_MARK		_IOW(C, 4, struct mark_parms)
#define	CIOC_SET_OPTIONS	_IOW(C, 5, char)
#define	CIOC_CMD_MODE		_IOW(C, 6, int)
#define	CIOC_SET_CMD		_IOW(C, 7, struct cmd_parms)
#define	CIOC_CMD_REPORT		_IOW(C, 8, struct cmd_parms)
#define	CIOC_LOAD		_IOW(C, 9, char)
