;; epoch-version: 3.2

;; This file is part of Epoch, a modified version of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;; [cjl]
;;; uglify.el -- this file defines the function uglify-current-buffer and
;;;              supporting code, to reformat lisp code to match some
;;;              "standards".
;;; reworked by AMC
;;;
(defun uglify-current-buffer ()
  "Reformat lisp code in the current buffer to 'standard'"
  (interactive)
  (munge-parens)
  (goto-char (point-min))
  (indent-sexps))
;;
;; munge-parens -- collect any stray )'s separated from code by white-space
;;                 and/or newlines.
;;
(defun munge-parens ()
  (interactive)
  ;; gather up the close parens
  (goto-char (point-min))
  (while (re-search-forward close-srch nil t)
    (delete-region (+ (match-beginning 0) 1) (- (match-end 0) 1))
    (previous-line 1)
  )
  (goto-char (point-min))
  (while (re-search-forward close2-srch nil t)
    (delete-region (match-beginning 1) (match-end 1))
    (previous-line 1)
  )
  (goto-char (point-min))
  (while (re-search-forward open-srch nil t)
    (delete-region (+ (match-beginning 0) 1) (- (match-end 0) 1))
    (previous-line 1)
  )
  (goto-char (point-min))
  (while (re-search-forward open2-srch nil t)
    (delete-region (match-beginning 2) (match-end 2)) ;toast newline
    (if (/= (match-beginning 1) (match-end 1)) ;open followed by non-whitespace
      (progn
	(goto-char (match-end 1))
	(insert " ")
      )
    )
    (goto-char (match-end 1))
  )
)
;;
;; regular expression used as search pattern.
;;
;(setq close-srch  ")[ \t]*\n[\n\t ]*)" )
;(setq open-srch "([ \t]*\n[ \t\n]*(" )
(setq close-srch  ")[\n\t ]+)" )
(setq close2-srch "^[^;]*\\(\n[ \t]*\\))" )
(setq open-srch "([ \t\n]+(" )
(setq open2-srch "^[ \t]*(\\([^ \t)]*\\)\\([ \t]*\n[ \t]*\\)(")
;;
;; indent-sexps -- cycle through s-expressions in buffer, and reformat using
;;                 indent-region.
;;
(defun indent-sexps ()
  (interactive)
  (lisp-mode)
  (goto-char (point-min))
  (let
    (
      here
      (comment-indent-hook 'standard-lisp-comment-indent)
    )
    (while (not (eobp))
      (standard-indent-sexp)
      (message "At %d" (point))
      (setq here (point))
      ;; (sit-for 1)
      (cond
	((looking-at "\\s *;") (forward-line 1))
	((not (= 0 (current-column)))	;not at begining of line
	  (forward-line 1)
	)
	(t
	  (forward-sexp 1)
	  (if (not (eobp)) (backward-sexp 1))
	  (if (= (point) here) (forward-sexp 1))
	)
      )
    )
  )
)
;;;
;;; copies of standard versions
;;;
(defun standard-indent-sexp ()
  "Indent each line of the list starting just after point."
  (interactive)
  (let ((indent-stack (list nil)) (next-depth 0) bol
	outer-loop-done inner-loop-done state this-indent)
    ;; Get error now if we don't have a complete sexp after point.
    (save-excursion (forward-sexp 1))
    (save-excursion
      (setq outer-loop-done nil)
      (while (not outer-loop-done)
	(setq last-depth next-depth
	      innerloop-done nil)
	;; Parse this line so we can learn the state
	;; to indent the next line.
	;; This inner loop goes through only once
	;; unless a line ends inside a string.
	(while (and (not innerloop-done)
		    (not (setq outer-loop-done (eobp))))
	  (setq state (parse-partial-sexp (point) (progn (end-of-line) (point))
					  nil nil state))
	  (setq next-depth (car state))
	  ;; If the line contains a comment other than the sort
	  ;; that is indented like code,
	  ;; indent it now with indent-for-comment.
	  ;; Comments indented like code are right already.
	  ;; In any case clear the in-comment flag in the state
	  ;; because parse-partial-sexp never sees the newlines.
	  (if (car (nthcdr 4 state))
	      (progn (indent-for-comment)
		     (end-of-line)
		     (setcar (nthcdr 4 state) nil)))
	  ;; If this line ends inside a string,
	  ;; go straight to next line, remaining within the inner loop,
	  ;; and turn off the \-flag.
	  (if (car (nthcdr 3 state))
	      (progn
		(forward-line 1)
		(setcar (nthcdr 5 state) nil))
	    (setq innerloop-done t)))
	(if (setq outer-loop-done (<= next-depth 0))
	    nil
	  (while (> last-depth next-depth)
	    (setq indent-stack (cdr indent-stack)
		  last-depth (1- last-depth)))
	  (while (< last-depth next-depth)
	    (setq indent-stack (cons nil indent-stack)
		  last-depth (1+ last-depth)))
	  ;; Now go to the next line and indent it according
	  ;; to what we learned from parsing the previous one.
	  (forward-line 1)
	  (setq bol (point))
	  (skip-chars-forward " \t")
	  ;; But not if the line is blank, or just a comment
	  ;; (except for double-semi comments; indent them as usual).
	  (if (or (eobp) (looking-at "[;\n]"))
	      nil
	    (if (and (car indent-stack)
		     (>= (car indent-stack) 0))
		(setq this-indent (car indent-stack))
	      (let ((val (standard-calculate-lisp-indent
			  (if (car indent-stack) (- (car indent-stack))))))
		(if (integerp val)
		    (setcar indent-stack
			    (setq this-indent val))
		  (setcar indent-stack (- (car (cdr val))))
		  (setq this-indent (car val)))))
	    (if (/= (current-column) this-indent)
		(progn (delete-region bol (point))
		       (indent-to this-indent)))))))))

(defun standard-calculate-lisp-indent (&optional parse-start)
  "Return appropriate indentation for current line as Lisp code.
In usual case returns an integer: the column to indent to.
Can instead return a list, whose car is the column to indent to.
This means that following lines at the same level of indentation
should not necessarily be indented the same way.
The second element of the list is the buffer position
of the start of the containing expression."
  (save-excursion
    (beginning-of-line)
    (let ((indent-point (point))
          state paren-depth
          ;; setting this to a number inhibits calling hook
          (desired-indent nil)
          (retry t)
          last-sexp containing-sexp)
      (if parse-start
          (goto-char parse-start)
          (beginning-of-defun))
      ;; Find outermost containing sexp
      (while (< (point) indent-point)
        (setq state (parse-partial-sexp (point) indent-point 0)))
      ;; Find innermost containing sexp
      (while (and retry
		  state
                  (> (setq paren-depth (elt state 0)) 0))
        (setq retry nil)
        (setq last-sexp (elt state 2))
        (setq containing-sexp (elt state 1))
        ;; Position following last unclosed open.
        (goto-char (1+ containing-sexp))
        ;; Is there a complete sexp since then?
        (if (and last-sexp (> last-sexp (point)))
            ;; Yes, but is there a containing sexp after that?
            (let ((peek (parse-partial-sexp last-sexp indent-point 0)))
              (if (setq retry (car (cdr peek))) (setq state peek)))))
      (if retry
          nil
        ;; Innermost containing sexp found
        (goto-char (1+ containing-sexp))
        (if (not last-sexp)
	    ;; indent-point immediately follows open paren.
	    ;; Don't call hook.
            (setq desired-indent (current-column))
	  ;; Find the start of first element of containing sexp.
	  (parse-partial-sexp (point) last-sexp 0 t)
	  (cond ((looking-at "\\s(")
		 ;; First element of containing sexp is a list.
		 ;; Indent under that list.
		 )
		((> (save-excursion (forward-line 1) (point))
		    last-sexp)
		 ;; This is the first line to start within the containing sexp.
		 ;; It's almost certainly a function call.
		 (if (= (point) last-sexp)
		     ;; Containing sexp has nothing before this line
		     ;; except the first element.  Indent under that element.
		     nil
		   ;; Skip the first element, find start of second (the first
		   ;; argument of the function call) and indent under.
		   (progn (forward-sexp 1)
			  (parse-partial-sexp (point) last-sexp 0 t)))
		 (backward-prefix-chars))
		(t
		 ;; Indent beneath first sexp on same line as last-sexp.
		 ;; Again, it's almost certainly a function call.
		 (goto-char last-sexp)
		 (beginning-of-line)
		 (parse-partial-sexp (point) last-sexp 0 t)
		 (backward-prefix-chars)))))
      ;; Point is at the point to indent under unless we are inside a string.
      ;; Call indentation hook except when overriden by lisp-indent-offset
      ;; or if the desired indentation has already been computed.
      (let ((normal-indent (current-column)))
        (cond ((elt state 3)
               ;; Inside a string, don't change indentation.
               (goto-char indent-point)
               (skip-chars-forward " \t")
               (current-column))
              ((and (integerp lisp-indent-offset) containing-sexp)
               ;; Indent by constant offset
               (goto-char containing-sexp)
               (+ normal-indent lisp-indent-offset))
              (desired-indent)
              ((and (boundp 'lisp-indent-hook)
                    lisp-indent-hook
                    (not retry))
               (or (funcall lisp-indent-hook indent-point state)
                   normal-indent))
              (t
               normal-indent))))))

(defun standard-lisp-comment-indent ()
  (if (looking-at ";;;")
      (current-column)
    (if (looking-at ";;")
	(let ((tem (standard-calculate-lisp-indent)))
	  (if (listp tem) (car tem) tem))
      (skip-chars-backward " \t")
      (max (if (bolp) 0 (1+ (current-column)))
	   comment-column))))
