# Filename: devinfo.awk
#
# Input file format
# 	All entries for each type of device must be present
# 	and should appear in the positions shown below.
# 
# <DeviceName> <CBS> <Device_specific_info>
#
# Sample entries for various devices:
#
# 	DISK 00D03 ID 5 MAXTOR
# 	RAID 00A00 ID 2 SW 02 LV 3 DC 5 SID 7 RAID3
# 	RAID 00D03 ID 0 SW 04 LV 5 DC 5 SID 7 RAID5
# 	RAID 00B00 ID 3 SW 02 LV 3 DC 5 SID 7 RAID3 NO_PAGER
# 	RAID 00C00 ID 4 SW 02 LV 3 DC 5 SID 7 RAID3 PAGE_TO rz0b
# 	RAID 00D03 ID 0 SW 04 LV 5 DC 5 SID 7 RAID5
# 	TAPE 00D03 ID 2 DAT
# 	MIO  01D02 I04
#	ENET 02B15
#
# Output file format
#   CABINET <cabNum> BP <BpNum> S <SlotNum> [<Device/Controller names>]
#
# Modification History:
#
# $Log: devinfo.awk,v $
# Revision 1.7  1994/02/03  00:22:55  mouli
# Updated the documentation for NO_PAGER and PAGE_TO keywords.
#
# Revision 1.6  1993/12/20  19:41:49  mouli
# Added support for NOPAGER and PAGE_TO keywords.
#
# Revision 1.5  1993/12/16  19:01:02  mouli
# Added NOPAGER and PAGE_TO keywords for OS debugging.
#
# Revision 1.4  1993/05/13  20:52:11  mouli
# Added HIPPI device support.
#
# Revision 1.3  1993/03/22  01:38:09  mouli
# 1. took off the requirement to have block count in user-created
#    device config file. Standard devices (disk,raid) will have
#    an entry in a file with two entries (device_name, block_count)
#    The awk script uses this file for outputing block count info.
#
# Revision 1.2  1993/03/18  21:25:08  mouli
# 1. Added ENET & MIO processing
# 2. printdev() now handles generic additional info such as brand
#    name, device name, fab-rev.
#
# Revision 1.1  1993/03/12  01:32:30  mouli
# Initial revision
#
# 	03/09/93	mouli	Initial version
#
# Set up Table for Cable to Backplane mapping
# Initialize keyword and entry count table
BEGIN {
	self = "devinfo.awk"
	if (errfile == "")
	{
		printf "%s:no errfile\n", self
		exit 2
	}
	bp["A"] = 0	# Cable A
	bp["B"] = 1	# Cable B
	bp["C"] = 2	# Cable C
	bp["D"] = 3	# Cable D
	bp["E"] = 4	# Cable E (Power control)
	TAPE_KWD  = "TAPE"
	DISK_KWD = "DISK"
	RAID_KWD = "RAID"
	ENET_KWD = "ENET"
	MIO_KWD  = "MIO"
	HIPPI_KWD  = "HIPPI"
	DISK_REC_MIN = 5
	DISK_REC_MAX = 7
	RAID_REC_MIN = 13
	RAID_REC_MAX = 15
	ecount[TAPE_KWD] = 5
	ecount[DISK_KWD] = DISK_REC_MIN
	ecount[RAID_KWD] = RAID_REC_MIN
	ecount[ENET_KWD] = 2
	ecount[MIO_KWD] =  3
	ecount[HIPPI_KWD] =  3
	ecount[NOPAGER_KWD] = 2
	ecount[PAGETO_KWD] = 3
	
}
# Separate out the Cabinet, Backplane, Slot info 
# use the manufacturer's name while outputing
# Note that the backplanes are output as Alpha (A,B,C,D)
function printdev(name, cbs, addinfo)
{
	printf "CABINET %s ", substr(cbs, 1, 2)
	printf "BP %s ", substr(cbs, 3, 1)
	printf "S %s %s %s\n", substr(cbs, 4, 2), name, addinfo
}
#
# Process the device configuration
{
	if ($1 == TAPE_KWD || $1 == DISK_KWD || 
	    $1 == RAID_KWD || $1 == ENET_KWD || 
	    $1 == MIO_KWD || $1 == HIPPI_KWD )
	{
#		Check sanity of record 
		if ($1 == RAID_KWD || $1 == DISK_KWD)
		{
			if ($1 == RAID_KWD)
			{
				minval = RAID_REC_MIN;
				maxval = RAID_REC_MAX;
			}
			else
			{
				minval =  DISK_REC_MIN;
				maxval =  DISK_REC_MAX;
			}
			if ((NF < minval) || (NF > maxval))
			{
				printf "%d: Incorrect %s entry\n",NR,$1 >errfile
				next
			}
		}
		else
		{
			if (NF != ecount[$1]) 
			{
				printf "%d: Incorrect %s entry\n",NR,$1 >errfile
				next
			}
		}
		if ($1 == TAPE_KWD)
		{
			printdev($1, $2, $5)
		}
		else if ($1 == DISK_KWD)
		{
			diskinfo = ""
			for (f = 5; f <= NF; f++)
			{
				diskinfo = diskinfo " " $f
			}
			printdev($1, $2, diskinfo)
		}
		else if ($1 == RAID_KWD)
		{
			diskinfo = ""
			for (f = RAID_REC_MIN+1; f <= NF; f++)
			{
				diskinfo = diskinfo " " $f
			}
			printdev($13, $2, diskinfo)
		}
		else if ($1 == ENET_KWD)
		{
			printdev($1, $2, "")
		}
		else if ($1 == MIO_KWD || $1 == HIPPI_KWD)
		{
			printdev($1, $2, $3)
		}
	}
	next
}
/^$/
{
}
