/*
 *	$Source: /u1/Xr/usr/contrib/RB/RCS/file.c,v $
 *	$Header: file.c,v 1.1 86/12/17 08:53:07 swick Exp $
 */

#ifndef lint
static char *rcsid_file_c = "$Header: file.c,v 1.1 86/12/17 08:53:07 swick Exp $";
#endif	lint

static char rcsid[] = "$Header: file.c,v 1.1 86/12/17 08:53:07 swick Exp $";
/*************************************<+>*************************************
 *****************************************************************************
 **
 **   File:        file.c
 **
 **   Project:     X-ray Toolbox
 **
 **   Description: file.c handles the write, read, and get info functions
 **                for accessing a raster file.
 **
 **   *******************************************************************
 **   * (c)  Copyright Hewlett-Packard Company, 1986.  All rights are   *
 **   * reserved.  Copying or other reproduction of this program except *
 **   * for archival purposes is prohibited without the prior written   *
 **   * consent of Hewlett-Packard Company.                             *
 **   *******************************************************************
 **
 **
 **   ------------------------ MODIFICATION RECORD   ------------------------
 *
 * $Log:	file.c,v $
 * Revision 1.1  86/12/17  08:53:07  swick
 * Initial revision
 * 
 *
 *****************************************************************************
 *************************************<+>*************************************/



#include <X/Xlib.h>
#include <Xr/defs.h>
#include <Xr/types.h>



/*************************************<->*************************************
 *
 *  WriteFile (name, image)
 *  INT8     * name;
 *  xrPixmap * image;
 *
 *
 *   Description: 
 *   ------------
 *     Write out the image contained in the xrPixmap structure image.
 *
 *
 *   Inputs:
 *   -------
 *     name   =  The name of the file to write to.
 *     image  =  The image to be written.
 *
 *
 *   Outputs:
 *   --------
 *     TRUE   =  returned if the write is successful.
 *     FALSE  =  returned if the write fails.
 *
 *
 *   Procedures Called
 *   -----------------
 *     BZPixmapSize  -  Xlib
 *     WZPixmapSize  -  Xlib
 *     creat()
 *     write()
 *     close()
 *
 *
 *************************************<->*************************************/




WriteFile (name, image)
INT8     * name;
xrPixmap * image;

{
   register INT32 count;
   register INT32 fd;
   register INT32 i;
   INT16   csum;
   INT8  * rasterPtr;


   /*
    *  Create the file, write out the height, width, and depth,
    *  calculate and write out the check sum, calculate the byte
    *  size of the image and write out the image.
    */

   if ((fd = creat (name, 0666)) == -1)
      return (FALSE);

   ValueWrite (fd, image -> height);
   ValueWrite (fd, image -> width);
   write (fd, &(image -> depth), 1);

   csum = image -> height + image -> width + image -> depth;
   ValueWrite (fd, csum);

   if (image -> depth == XrBIT1)
      count = BitmapSize (image -> width, image -> height);
   else if (image -> depth <= XrBYTE1)
      count = BZPixmapSize (image -> width, image -> height);
   else
      count = WZPixmapSize (image -> width, image -> height);

   rasterPtr = (INT8 *) (image -> raster);
   for (i = 0; i < count; i++)
      write (fd, rasterPtr + i, 1);

   close (fd);
}



/*************************************<->*************************************
 *
 *  ReadFile (name, image)
 *  INT8     * name;
 *  xrPixmap * image;
 *
 *
 *   Description: 
 *   ------------
 *     Read the image from the file identified by name into image.
 *
 *
 *   Inputs:
 *   -------
 *     name   =  The name of the file to read.
 *     image  =  The image structure to be set.
 *
 *
 *   Outputs:
 *   --------
 *     TRUE   =  Returned if the read is successful.
 *     FALSE  =  Returned if the read fails.
 *     image  =  The members of the image structure will be filled
 *               with the image information when the read is successful.
 *
 *
 *   Procedures Called
 *   -----------------
 *     BZPixmapSize  -  Xlib
 *     WZPixmapSize  -  Xlib
 *     open()
 *     close()
 *     read()
 *
 *
 *************************************<->*************************************/


ReadFile (name, image)
INT8     * name;
xrPixmap * image;

{
   INT32   fd;
   INT32   count;
   INT16   csum;
   INT32   i;
   INT8  * rasterPtr;


   /*
    *  Open the file, read the height, width, depth, and check sum,
    *  verify that the check sum is correct, calculate the byte size
    *  of the image and read the image into the pixmap.
    */

   if ((fd = open (name, 0)) == -1)
      return (FALSE);

   ValueRead (fd, &(image -> height));
   ValueRead (fd, &(image -> width));
   read (fd, &(image -> depth), 1);
   ValueRead (fd, &csum);

   if (csum != (image -> height + image -> width + image -> depth))
   {
      close (fd);
      return (FALSE);
   }

   if (image -> depth == XrBIT1)
      count = BitmapSize (image -> width, image -> height);
   else if (image -> depth <= XrBYTE1)
      count = BZPixmapSize (image -> width, image -> height);
   else
      count = WZPixmapSize (image -> width, image -> height);

   rasterPtr = (INT8 *) (image -> raster);
   for (i = 0; i < count; i++)
      read (fd, rasterPtr + i, 1);

   close (fd);
   return (TRUE);
}




/*************************************<->*************************************
 *
 *  GetFileInfo (name, image)
 *  INT8     * name;
 *  xrPixmap * image;
 *
 *
 *   Description: 
 *   ------------
 *     Get the information about a image from the file.
 *
 *
 *   Inputs:
 *   -------
 *     name   =  The name of the file that contains the image.
 *     image  =  The structure which will hold the information.
 *
 *
 *   Outputs:
 *   --------
 *     TRUE   =  Returned if the write is successful.
 *     FALSE  =  Returned if the write fails.
 *     image  =  The width, height, and depth fields of the image
 *               structure are set if successful.
 *
 *
 *   Procedures Called
 *   -----------------
 *     open()
 *     read()
 *     close()
 *
 *
 *************************************<->*************************************/


GetFileInfo (name, image)
INT8     * name;
xrPixmap * image;

{
   INT32 fd;
   INT16 csum;


   /*
    *  Open the file, get the height, width, depth, and check sum,
    *  verify that the check sum is correct and return.
    */

   if ((fd = open (name, 0)) == -1)
      return (FALSE);

   ValueRead (fd, &(image -> height));
   ValueRead (fd, &(image -> width));
   read (fd, &(image -> depth), 1);
   ValueRead (fd, &csum);

   close (fd);

   if (csum != (image -> height + image -> width + image -> depth))
      return (FALSE);

   return (TRUE);
}




/*************************************<->*************************************
 *
 *  ValueRead (fd, value)
 *  INT32   fd;
 *  INT16 * value;
 *
 *  ValueWrite (fd, value)
 *  INT32 fd;
 *  INT16 value;
 *
 *
 *   Description: 
 *   ------------
 *     Read or write a value into or out of a string.
 *
 *
 *   Procedures Called
 *   -----------------
 *     read()
 *     write()
 *
 *
 *************************************<->*************************************/


ValueRead (fd, value)
INT32   fd;
INT16 * value;

{
   INT8 string [4];

   read (fd, string, 4);
   *value = atoi (string);
}



ValueWrite (fd, value)
INT32 fd;
INT16 value;

{
   INT8 string [5];

   sprintf (string, "%4d", value);
   write (fd, (char *) string, 4);
}
