/* -----------------------------------------------------------------

	These are the Mindset bios call routines for interrupt 0efh.

----------------------------------------------------------------- */

#include <cuser.inc>

/* --------------------- INTERRUPT EF ------------------------ */

/* ---------------------- set_screen_mode ------------------------------*/
set_screen_mode(mode)
char mode;							/* 0 = 320 x 200, 2 colors */
									/* 1 = 320 x 200, 4 colors */
									/* 2 = 320 x 200, 16 colors */
									/* 3 = 640 x 200, 2 colors */
									/* 4 = 640 x 200, 4 colors */
									/* 5 = 320 x 400, 4 colors */
									/* 6 = 320 x 400, 2 colors */
{
	struct regs myregs;

	myregs.ax = mode;
	callbios(ms_v_int, &myregs);
	return;
}

/* -------------------------- get_screen_mode -----------------------------*/
char get_screen_mode(flags, bitspix)
char *bitspix;			/* number of bits per pixel */
int *flags;				/* system video flags */
{
	struct regs myregs;

	myregs.ax = 0x100;
	callbios(ms_v_int, &myregs);
	*flags = myregs.bx;
	*bitspix = myregs.ax;
	return(myregs.ax);
}

/* ---------------------------- set_transfer_mode ----------------------------*/
set_transfer_mode(mode)
int	mode;   				/* bit 8 = 1 for transparency  + other bits */
							/* lower byte = 0	straight replace
											1	and source into dest
											2	or source into dest
											3	xor source into dest
											4	not source and replace dest
											5	and (not source) into dest
											6	or (not source) into dest
											7	xor (not source into dest
							*/
{
	struct regs myregs;

	myregs.ax = 0x200;
	myregs.bx = mode;
	callbios(ms_v_int, &myregs);
	return;
}


/* ------------------------ get_transfer_mode ------------------------------*/
int get_transfer_mode()
{
	struct regs myregs;

	myregs.ax = 0x300;
	callbios(ms_v_int, &myregs);
	return(myregs.bx);					/* return what transfer mode was set */
}

/* ------------------------- set_dest_buffer ---------------------------------*/
set_dest_buffer(offset, segment, width, lines)
int	offset, segment;				/* offset and seg of dest buffer */
int	width;  						/* width in bytes of one scan line */
int	lines;   						/* number of scan lines */
{
	struct regs myregs;

	myregs.ax = 0x400;
	myregs.cx = width;
	myregs.dx = lines;
	myregs.es = segment;
	myregs.bx = offset;
	callbios(ms_v_int, &myregs);
	return;
}

/* ------------------------ get_dest_buffer ------------------------------*/
get_dest_buffer(offset, segment, width, lines)
int	*offset, *segment;				/* offset and seg of dest buffer */
int	*width; 						/* width in bytes of one scan line */
int	*lines;	        				/* number of scan lines in buffer */
{
	struct regs myregs;

	myregs.ax = 0x500;
	callbios(ms_v_int, &myregs);
	*offset  = myregs.bx;
	*segment = myregs.es;
	*lines   = myregs.dx;
	*width   = myregs.cx;
	return;
}

/* -------------------- set_write_mask --------------------------------*/
set_write_mask(mask)
int mask;							/* 16 bit write mask */
{
	struct regs myregs;

	myregs.ax = 0x600;
	myregs.bx = mask;
	callbios(ms_v_int, &myregs);
	return;
}

/* ------------------------- get_write_mask ------------------------------*/
int get_write_mask()
{
	struct regs myregs;

	myregs.ax = 0x700;
	callbios(ms_v_int, &myregs);
	return(myregs.bx);				/* write mask returned */
}

/* ---------------------- blt_copy ----------------------------------------*/
blt_copy(id, count, mode, xorg, yorg, offset, segment)
char id;					/* id of blt */
int count;					/* number of blts to do */
int mode;     				/* 0 = top to bottom, source and dest */
							/* bit 0 = 0 -> read from top to bottom */
							/* bit 0 = 1 -> read from bottom to top */
							/* bit 1 = 0 -> write from top to bottom */
							/* bit 1 = 1 -> write from bottom to top */
							/* bit 2 and 3 (same) */
							/* bits = 0 -> read and write from left to right */
							/* bits = 1 -> read and write from right to left */
							/* bit 4 = 1 -> fast blt (conditions to meet) */
int xorg, yorg;             /* x and y origin for destination */
int offset;					/* offset of blt descriptor */
int segment;                /* segment of blt descriptor */
{
	struct regs myregs;

	myregs.ax = 0x800 + id;
	myregs.bx = offset;
	myregs.cx = count;
	myregs.dx = mode;        
	myregs.es = segment;     
	myregs.si = xorg;
	myregs.di = yorg;
	callbios(ms_v_int, &myregs);
	return;
}

/* ------------------------- blt_copy_word -------------------------------*/
blt_copy_word(id, count, xorg, yorg, offset, segment)
char id;					/* id of blt */
int count;					/* number of blts to do */
int xorg, yorg;     		/* x and y origin for destination */
int offset;					/* offset of blt descriptor */
int segment;                /* segment of blt descriptor */
{
	struct regs myregs;

	myregs.ax = 0x900 + id;
	myregs.bx = offset;
	myregs.cx = count;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.es = segment;    
	callbios(ms_v_int, &myregs);
	return;
}

/* -------------------------- set_palette --------------------------------*/
set_palette(border, index, count, offset, segment)
char border;					/* which table color is border color */
int	index;						/* first of colors to set */
int	count;						/* number of colors to set */
int offset;						/* offset of pointer to palette table */
int segment;                    /* segment of pointer to palette table */
{
	struct regs myregs;

	myregs.ax = 0xa00 + border;
	myregs.bx = offset;
	myregs.cx = count;
	myregs.dx = index;
	myregs.es = segment;        
	callbios(ms_v_int, &myregs);
	return;
}

/* ------------------------ get_palette ---------------------------------*/
get_palette(border, index, count, offset, segment)
char *border;					/* border color */
int	index;						/* first of colors to get */
int	count;						/* number of colors to get */
int offset;						/* offset of pointer to palette table */
int segment;                    /* segment of pointer to palette table */
{
	struct regs myregs;

	myregs.ax = 0xb00;
	myregs.cx = count;
	myregs.dx = index;
	myregs.es = segment;        
	myregs.bx = offset;
	callbios(ms_v_int, &myregs);
	*border = myregs.ax;
	return;
}

/* --------------------------- blt_polypoint -----------------------------*/
blt_polypoint(id, count, color, xorg, yorg, offset, segment)
char id;						/* id number for blts */
int count;						/* number of points */
char color;						/* color for these points */
int xorg, yorg;     			/* x and y origin for destination */
int offset;						/* offset of pointer to coordinates table */
int segment;                    /* segment of pointer to coordinates table */
{
	struct regs myregs;

	myregs.ax = 0xc00 + id;
	myregs.cx = count;
	myregs.dx = color << 8;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.bx = offset;
	myregs.es = segment;    
	callbios(ms_v_int, &myregs);
	return;
}

/* --------------------------- blt_polyline -------------------------------*/
blt_polyline(id, mode, count, color, xorg, yorg, offset, segment)
char id;						/* id number for blts */
char mode;						/* 0 = chained, 1 = seperated */
int count;						/* number of points */
char color;						/* color for these points */
int xorg, yorg;     			/* x and y origin for destination */
int offset;						/* offset of pointer to coordinates table */
int segment;                    /* segment of pointer to coordinates table */
{
	struct regs myregs;

	myregs.ax = 0xd00 + id;
	myregs.cx = count;
	myregs.dx = (color << 8) + mode;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.bx = offset;
	myregs.es = segment;       
	callbios(ms_v_int, &myregs);
	return;
}


/* ----------------------- get_buffer_info ---------------------------------*/
get_buffer_info(fb1_segment, fb2_segment, size)
int *fb1_segment, *fb2_segment;			/* paragraphs of each half */
int *size;    							/* number of bytes in each half */
{
	struct regs myregs;

	myregs.ax = 0xe00;
	callbios(ms_v_int, &myregs);
	*fb1_segment = myregs.bx;
	*fb2_segment = myregs.dx;
	*size        = myregs.cx;
	return;
}

/* ------------------------- set_disp_int_addr ------------------------------*/
set_disp_int_addr(offset, segment)
int	offset;                             /* offset of pointer to routine */
int segment;                            /* segment of pointer to routine */
{
	struct regs myregs;

	myregs.ax = 0xf00;
	myregs.bx = offset;
	myregs.es = segment;        
	callbios(ms_v_int,&myregs);
	return;
}

/* ------------------------- get_disp_int_addr ----------------------------*/
get_disp_int_addr(offset, segment)
int	*offset;                            /* offset of pointer to routine */
int	*segment;                           /* segment of pointer to routine */
{
	struct regs myregs;

	myregs.ax = 0x1000;
	callbios(ms_v_int,&myregs);
	*segment = myregs.es;
	*offset  = myregs.bx;
	return;
}

/* --------------------------- switch_active_buffer ----------------------*/
switch_active_buffer()	    			/* no params, just switch */
{
	struct regs myregs;

	myregs.ax = 0x1100;
	callbios(ms_v_int, &myregs);
	return;
}

/* --------------------------- set_collision_pattern ----------------------*/
set_collision_pattern(polarity, bit_pattern, mask)
int	polarity;  				/* 0 - disable all_except mode, 1 - enable mode */
int	bit_pattern;            /* bit pattern for collision detection */
int mask;                   /* define bitmask of pixel don't care bits */
{
	struct regs myregs;

	myregs.ax = 0x1200 + polarity;
	myregs.bx = (mask << 8) + bit_pattern;
	callbios(ms_v_int,&myregs);
	return;
}

/* ----------------------- get_collision_pattern ----------------------------*/
get_collision_pattern(polarity, pattern, mask)
int *polarity;          	/* 0 - all_except mode disabled, 1 - mode enabled*/
int *pattern;               /* bit pattern for collision detection */
int *mask;                  /* define bitmask of pixel don't care bits */
{
	struct regs myregs;

	myregs.ax = 0x1300;
	callbios(ms_v_int,&myregs);
	*polarity       = myregs.ax && 0x00ff;
	*pattern        = myregs.bx && 0x00ff;
	*mask           = myregs.bx && 0xff00;
	return;
}

/* -------------------------- set_clip_rectangle --------------------------*/
set_clip_rectangle(x_left, x_right, y_top, y_bottom)
int x_left, x_right;				/* left and right x values */
int y_top, y_bottom;				/* top and bottom y values */
{
	struct regs myregs;

	myregs.ax = 0x1400;
	myregs.si = x_left;
	myregs.di = x_right;
	myregs.cx = y_top;
	myregs.dx = y_bottom;
	callbios(ms_v_int, &myregs);
	return;
}

/* ----------------------------- get_clip_rectangle -----------------------*/
get_clip_rectangle(x_left, x_right, y_top, y_bottom)
int *x_left, *x_right;				/* left and right x values */
int *y_top, *y_bottom;				/* top and bottom y values */
{
	struct regs myregs;

	myregs.ax = 0x1500;
	callbios(ms_v_int, &myregs);
	*x_left   = myregs.si;
	*x_right  = myregs.di;
	*y_top    = myregs.cx;
	*y_bottom = myregs.dx;
	return;
}

/* -------------------------- set_collclip_detect -------------------------*/
set_collclip_detect(mode, offset, segment)
int mode;   			/* bit #0 (clipping): 0 - disable, 1 - enable */
                        /* bit #1 (collision): 0 - disable, 1 - enable */
                        /* bit #2 (task done int): 0 - disable, 1 -enable */
int offset;             /* offset of service routine */
int segment;            /* segment of service routine */
{
	struct regs myregs;

	myregs.ax = 0x1600 + mode;                                              
	myregs.bx = offset;
	myregs.es = segment;          
	callbios(ms_v_int,&myregs);
	return;
}

/* ------------------------ get_collclip_detect ---------------------------*/
get_collclip_detect(mode, offset, segment)
int	*mode;              /* bit #0 (clipping):  0 - disable, 1 - enable */
                        /* bit #1 (collision):  0 - disable, 1 - enable */
                        /* bit #2 (task done int): 0 - disabled, 1 - enabled */
int *offset;            /* offset of service routine */
int *segment;           /* segment of service routine */
{
	struct regs myregs;

	myregs.ax = 0x1700;
	callbios(ms_v_int,&myregs);
	*mode   	  = myregs.ax && 0x0007;
	*offset       = myregs.bx;	
    *segment      = myregs.es;
	return;
}

/* -------------------------- blt_wait -----------------------------------*/
blt_wait(id, coll_status, clip_status)
char *id;						/* id of blt that finished */
int *coll_status;				/* collision status */
int *clip_status;				/* clip status */
{
	struct regs myregs;

	myregs.ax = 0x1800;
	callbios(ms_v_int, &myregs);
	*id          = myregs.ax;
	*coll_status = myregs.bx;
	*clip_status = myregs.cx;
	return;
}

/* ------------------------ blt_polygon ----------------------------------*/
blt_polygon(id, count, even_color, odd_color, xorg, yorg, offset, segment)
char id;						/* id number for blts */
int count;						/* number of points */
char even_color, odd_color;		/* color for even and odd pixels */
int xorg, yorg;     			/* x and y origin for destination */
int offset;						/* offset of pointer to coordinates table */
int segment;                    /* segment of pointer to coordinates table */
{
	struct regs myregs;

	myregs.ax = 0x1900 + id;
	myregs.cx = count;
	myregs.dx = (even_color << 8) + odd_color;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.bx = offset;
	myregs.es = segment;
	callbios(ms_v_int, &myregs);
	return;
}

/* -------------------------- blt_fellipse ------------------------------*/
blt_fellipse(id, count, even_color, odd_color, xorg, yorg, offset, segment)
char id;						/* id number for blts */
int count;						/* number of points */
char even_color, odd_color;		/* color for even and odd pixels */
int xorg, yorg;	        		/* x and y origin for destination */
int offset;						/* offset of pointer to coordinates table */
int segment;                    /* segment of pointer to coordinates table */
{
	struct regs myregs;

	myregs.ax = 0x1a00 + id;
	myregs.cx = count;
	myregs.dx = (even_color << 8) + odd_color;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.bx = offset;
	myregs.es = segment;
	callbios(ms_v_int, &myregs);
	return;
}

/* ----------------------- blt_hellipse ------------------------------------*/
blt_hellipse(id, count, color, xorg, yorg, offset, segment)
char id;						/* id number for blts */
int count;						/* number of points */
char color;                     /* color for even and odd pixels */
int xorg, yorg;     			/* x and y origin for destination */
int offset;						/* offset of pointer to coordinates table */
int segment;                    /* segment of pointer to coordinates table */
{
	struct regs myregs;

	myregs.ax = 0x1b00 + id;
	myregs.cx = count;
	myregs.dx = color << 8;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.bx = offset;
	myregs.es = segment;        
	callbios(ms_v_int, &myregs);
	return;
}

/* ---------------------------- save_GCP --------------------------------*/
int save_GCP(mode, offset, segment, data_size)
char mode;                        /* 0 = no save, just size; 1 = save GCP */
int offset;                       /* offset of memory to store GCP status */
int segment;                      /* segment of memory to store GCP status*/
{
	struct regs myregs;

	myregs.ax = 0x1c00 + mode;  
	myregs.bx = offset;
	myregs.es = segment;     
	callbios(ms_v_int, &myregs);
	return(myregs.ax);			/* return number of bytes needed to save */
}

/* ---------------------------- restore_GCP ----------------------------*/
restore_GCP(offset, segment)
int offset;                     /* offset of memory containing GCP status */
int segment;                    /* segment of memory containing GCP status */
{
	struct regs myregs;

	myregs.ax = 0x1d00;
	myregs.bx = offset;
	myregs.es = segment;          
	callbios(ms_v_int, &myregs);
	return;
}

/* ------------------------ fill_dest_buffer -------------------------------*/
fill_dest_buffer(pattern)
int pattern;
{
	struct regs myregs;

	myregs.ax = 0x1e00;
	myregs.bx = pattern;
	callbios(ms_v_int, &myregs);
	return;
}

/* ---------------------- set_font_pointer -------------------------------*/
set_font_pointer(offset, segment)
int offset;                    /* offset of font description */
int segment;                   /* segment of font description */ 
{
	struct regs myregs;

	myregs.ax = 0x1f00;
	myregs.bx = offset;
	myregs.es = segment;        
	callbios(ms_v_int, &myregs);
	return;
}

/* ----------------------- get_font_pointer --------------------------------*/
get_font_pointer(offset, segment)
int *offset;				/* offset of font descriptor storage */
int *segment;               /* segment of font descriptor storage */
{
	struct regs myregs;

	myregs.ax = 0x2000;
	callbios(ms_v_int, &myregs);
	*offset  = myregs.bx;
	*segment = myregs.es;
	return;
}

/* ---------------------------- blt_string --------------------------------*/
blt_string(id, count, ignore, direction, color, xorg, yorg, offset, segment)
char id;						/* id for the blt */
char count;						/* number of strings */
char ignore;					/* number of characters to ignore */
char direction;					/* 0=l to r, 1=r to l, 2=t to b, 3=b to t */
char color;						/* one byte color mask */
int xorg, yorg;     			/* x and y origin in destination */
int offset;      				/* offset of pointer to descriptor */
int segment;                    /* segment of pointer to descriptor */
{
	struct regs myregs;

	myregs.ax = 0x2100 + id;
	myregs.bx = offset;
	myregs.cx = (count << 8) + ignore;
	myregs.dx = (direction << 8) + color;
	myregs.si = xorg;
	myregs.di = yorg;
	myregs.es = segment;    
	callbios(ms_v_int, &myregs);
	return;
}

/* -------------------------- set_link_mode --------------------------------*/
set_link_mode(mode)
char mode;						/* 0 for contiguous lists, 1 for linked */
{
	struct regs myregs;

	myregs.ax = 0x2200 + mode;
	callbios(ms_v_int, &myregs);
	return;
}

/* -------------------------- get_link_mode -------------------------------*/
char get_link_mode()
{
	struct regs myregs;

	myregs.ax = 0x2300;
	callbios(ms_v_int, &myregs);
	return(myregs.ax);			/* return value set with set_link_mode */
}

/* --------------------------- get_GCP_status -------------------------------*/
get_GCP_status(gcp_idle, system_fly, collision_detected)
int *gcp_idle;					/* task is done */
int *system_fly;
int *collision_detected;		/* collision was detected on last task */
{
	struct regs myregs;

	myregs.ax = 0x2400;
	callbios(ms_v_int,&myregs);
	*gcp_idle	    	= myregs.bx && 0x0001;
	*system_fly	    	= myregs.bx && 0x0002;
	*collision_detected = myregs.bx && 0x0004;
	return;
}

/* ----------------------- get_char_bitmap --------------------------------*/
get_char_bitmap(normal, extras, segment)/*standard bios font, non-blt format */
int *normal, *extras;			/* offset for normal char font, extra font */
int *segment;					/* segment for both fonts */
{
	struct regs myregs;

	myregs.ax = 0x2500;
	callbios(ms_v_int, &myregs);
	*normal = myregs.si;
	*extras = myregs.di;
	*segment = myregs.es;
	return;
}

/* --------------------------- get_GCP_memory ------------------------------*/
get_GCP_memory(memory_flag, first_segment, last_segment)
int	*memory_flag;			/* 0 - all memory avail. to blter, 1 - part only */
int *first_segment;			/* segment address of first 64KB avail. to blter */
int *last_segment;			/* segment address of last 64KB avail. to blter  */
{
	struct regs myregs;

	myregs.ax = 0x2600;
	callbios(ms_v_int,&myregs);
	*memory_flag	= myregs.ax && 0x0001;
	*first_segment	= myregs.si;
	*last_segment	= myregs.di;
	return;
}
