/*	$NetBSD: if_esh_pci.c,v 1.6 2000/01/21 23:39:59 thorpej Exp $	*/

/*
 * Copyright (c) 1997, 1998 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code contributed to The NetBSD Foundation by Kevin M. Lahey
 * of the Numerical Aerospace Simulation Facility, NASA Ames Research 
 * Center.
 *
 * Partially based on a HIPPI driver written by Essential Communications 
 * Corporation.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the NetBSD
 *      Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include "opt_inet.h"
#include "opt_ns.h"
#include "bpfilter.h" 
 
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h> 
#include <sys/socket.h> 
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/select.h>
#include <sys/device.h>
#include <sys/buf.h>

#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_hippi.h>
#include <net/if_media.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h> 
#include <netinet/if_inarp.h>
#endif
 
#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif
  
#if NBPFILTER > 0
#include <net/bpf.h>
#include <net/bpfdesc.h>
#endif

#include <machine/cpu.h>
#include <machine/bus.h>
#include <machine/intr.h>

#include <dev/ic/rrunnerreg.h>
#include <dev/ic/rrunnervar.h>

#include <dev/pci/pcivar.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pcidevs.h>

/*
 * PCI constants.
 * XXX These should be in a common file!
 */
#define PCI_CONN		0x48    /* Connector type */
#define PCI_CBIO		0x10    /* Configuration Base IO Address */

#define MEM_MAP_REG	0x10

int esh_pci_match __P((struct device *, struct cfdata *, void *));
void esh_pci_attach __P((struct device *, struct device *, void *));
static u_int8_t esh_pci_bist_read __P((struct esh_softc *));
static void esh_pci_bist_write __P((struct esh_softc *, u_int8_t));


struct cfattach esh_pci_ca = {
	sizeof(struct esh_softc), esh_pci_match, esh_pci_attach
};

int
esh_pci_match(parent, match, aux)
	struct device *parent;
	struct cfdata *match;
	void *aux;
{
	struct pci_attach_args *pa = (struct pci_attach_args *) aux;

	if (PCI_VENDOR(pa->pa_id) != PCI_VENDOR_ESSENTIAL)
		return 0;

	switch (PCI_PRODUCT(pa->pa_id)) {
	case PCI_PRODUCT_ESSENTIAL_RR_HIPPI:
	case PCI_PRODUCT_ESSENTIAL_RR_GIGE:
		break;
	default:
		return 0;
	}
	return 1;
}

void
esh_pci_attach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct esh_softc *sc = (void *)self;
	struct pci_attach_args *pa = aux;
	pci_chipset_tag_t pc = pa->pa_pc;
	pci_intr_handle_t ih;
	char *model;
	const char *intrstr = NULL;

	if (pci_mapreg_map(pa, MEM_MAP_REG,
			   PCI_MAPREG_TYPE_MEM | PCI_MAPREG_MEM_TYPE_32BIT, 0,
			   &sc->sc_iot, &sc->sc_ioh, NULL, NULL) != 0) {
	    printf(": unable to map memory device registers\n");
	    return;
	}

	sc->sc_dmat = pa->pa_dmat;

	switch (PCI_PRODUCT(pa->pa_id)) {
	case PCI_PRODUCT_ESSENTIAL_RR_HIPPI:
		model = "RoadRunner HIPPI";
		break;
	case PCI_PRODUCT_ESSENTIAL_RR_GIGE:
		model = "RoadRunner Gig-E";
		break;
	default:
		model = "unknown model";
		break;
	}

	printf(": %s\n", model);

	sc->sc_bist_read = esh_pci_bist_read;
	sc->sc_bist_write = esh_pci_bist_write;

	eshconfig(sc);

	/* Enable the card. */
	pci_conf_write(pc, pa->pa_tag, PCI_COMMAND_STATUS_REG,
	    pci_conf_read(pc, pa->pa_tag, PCI_COMMAND_STATUS_REG) |
	    PCI_COMMAND_MASTER_ENABLE);

	/* Map and establish the interrupt. */
	if (pci_intr_map(pc, pa->pa_intrtag, pa->pa_intrpin,
	    pa->pa_intrline, &ih)) {
		printf("%s: couldn't map interrupt\n", sc->sc_dev.dv_xname);
		return;
	}
	intrstr = pci_intr_string(pc, ih);
	sc->sc_ih = pci_intr_establish(pc, ih, IPL_NET, eshintr, sc);
	if (sc->sc_ih == NULL) {
		printf("%s: couldn't establish interrupt",
		    sc->sc_dev.dv_xname);
		if (intrstr != NULL)
			printf(" at %s", intrstr);
		printf("\n");
		return;
	}
	printf("%s: interrupting at %s\n", sc->sc_dev.dv_xname, intrstr);
}

u_int8_t
esh_pci_bist_read(sc)
    struct esh_softc *sc;
{
    bus_space_tag_t iot = sc->sc_iot;
    bus_space_handle_t ioh = sc->sc_ioh;
    u_int32_t pci_bist;

    pci_bist = bus_space_read_4(iot, ioh, RR_PCI_BIST);

    return ((u_int8_t) (pci_bist >> 24));
}

void
esh_pci_bist_write(sc, value)
    struct esh_softc *sc;
    u_int8_t value;
{
    bus_space_tag_t iot = sc->sc_iot;
    bus_space_handle_t ioh = sc->sc_ioh;
    u_int32_t pci_bist;
    u_int32_t new_bist;

    pci_bist = bus_space_read_4(iot, ioh, RR_PCI_BIST);
    new_bist = ((u_int32_t) value << 24) | (pci_bist & 0x00ffffff);

    bus_space_write_4(iot, ioh, RR_PCI_BIST, new_bist);
}

