using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace PERQDisk
{
    class Program
    {
        static void Main(string[] args)
        {
            Program p = new Program();
            p.Run(args);
        }

        public Program()
        {
            // For now we always assume a Shugart 4002 14" hard disk, at some point this can be enhanced...

            // The Shugart has 202 cylinders, 8 tracks per sector, and 30 sectors per track.
            _physicalDisk = new PhysicalDisk(202, 8, 30);
        }

        public void Run(string[] args)
        {
            Console.WriteLine("PERQDisk v0.0; J. Dersch 8/16/2006");

            if (args.Length != 1)
            {
                Console.WriteLine(" -- usage : PERQDisk imagefile -- ");
                return;
            }

            LoadImage(args[0]);

            BuildVolume();

            RunPrompt();
            
        }

        private void LoadImage(string path)
        {
            Console.WriteLine("Reading image from {0}...", path);
            FileStream fs = new FileStream(path, FileMode.Open);
            _physicalDisk.Load(fs);
            fs.Close();           
        }

        private void BuildVolume()
        {
            Console.WriteLine("Reading volume info...");
            _volume = new Volume(_physicalDisk);
            _volume.PrintDIBInfo();
        }

        private void RunPrompt()
        {
            bool exit = false;

            _path = new Stack<File>(16);

            while (!exit)
            {
                Console.WriteLine();
                Console.Write("{0}>", PathToString());
                string cmd = Console.ReadLine().ToLower();

                string[] cmds = cmd.Split(new char[] { ' ' }, 100, StringSplitOptions.RemoveEmptyEntries);

                if (cmds.Length == 0)
                    continue;

                if (cmds[0] == "dir")
                {
                    PrintDirectory();
                }
                else if (cmds[0] == "cd")
                {
                    if (cmds.Length != 2)
                    {
                        Console.WriteLine("cd requires a directory argument.");                        
                    }
                    else
                    {
                    ChangeDirectory(cmds[1]);
                    }
                }
                else if (cmds[0] == "type")
                {
                    if (cmds.Length != 2)
                    {
                        Console.WriteLine("type requires a file argument.");                     
                    }
                    else
                    {   
                        TypeFile(cmds[1]);
                    }
                }
                else if (cmds[0] == "info")
                {
                    if (cmds.Length != 2)
                    {
                        Console.WriteLine("info requires a file or directory argument.");                     
                    }
                    else
                    {
                        PrintInfo(cmds[1]);
                    }

                }
                else if (cmds[0] == "copy")
                {
                    if (cmds.Length != 3)
                    {
                        Console.WriteLine("copy requires a source and destination argument.");
                    }
                    else
                    {
                        Copy(cmds[1], cmds[2]);
                    }
                }
                else if (cmds[0] == "exit")
                {
                    exit = true;                    
                }
                else
                {
                    Console.WriteLine("Unknown command.");
                }
            }
        }

        /// <summary>
        /// Prints the contents of the current directory.
        /// </summary>
        private void PrintDirectory()
        {
            if (_path.Count == 0)
            {
                // At root level we show the available partitions
                foreach (Partition p in _volume.Partitions)
                {
                    Console.Write("{0}\t", p.Name);
                }
            }
            else
            {
                int i = 0;
                int cols = Math.Max(Console.WindowWidth / 25,1);                

                foreach (File f in _path.Peek().Children)
                {
                    // Show directories in a different color
                    if (f.IsDirectory)
                    {
                        Console.ForegroundColor = ConsoleColor.Red;
                    }

                    Console.Write("{0,-26:G}", f.SimpleName);

                    Console.ForegroundColor = ConsoleColor.Gray;
                    
                    i++;

                    if ((i % cols) == 0)
                    {
                        Console.WriteLine();
                    }
                }
            }
        }

        /// <summary>
        /// Moves to a new directory in the filesystem tree.
        /// </summary>
        /// <param name="newDir"></param>
        private void ChangeDirectory(string newDir)
        {
            //Special case ".." for parent dir.
            if (newDir == "..")
            {
                if (_path.Count == 0)
                {
                    //We're at root, do nothing.
                    return;
                }
                else
                {
                    _path.Pop();
                }

                return;
            }

            if (_path.Count == 0)
            {                
                // At top level we select partitions
                Partition p = GetPartitionForName( newDir );
                if( p != null )
                {
                    _path.Push(p.Root);
                    return;
                }                             
            }
            else
            {
                File f = GetDirectoryForName( newDir );
                if( f != null )
                {
                    _path.Push(f);                        
                    return;
                }                 
            }
            
            Console.WriteLine("Directory '{0}' does not exist.", newDir);
        }

        /// <summary>
        /// Dumps the contents of the given file to screen.
        /// </summary>
        /// <param name="file"></param>
        private void TypeFile(string file)
        {
            if (_path.Count == 0)
            {
                // At root level we can't type files
                Console.WriteLine("Invalid argument to type");
            }
            else
            {
                File f = GetFileForName(file);

                if( f != null )
                {
                    f.Print();
                    return;
                }                
            }

            Console.WriteLine("File '{0}' does not exist.", file);
        }

        /// <summary>
        /// Displays info about the given file or directory.
        /// </summary>
        /// <param name="file"></param>
        private void PrintInfo(string file)
        {
            // Special case for "." (current directory)
            if (file == ".")
            {
                if (_path.Count == 0)
                {
                    _volume.PrintDIBInfo();
                }
                else
                {
                    _path.Peek().PrintFileInfo();
                }
                return;
            }

            if (_path.Count == 0)
            {
                Partition p = GetPartitionForName( file );

                if( p != null )
                {
                    p.PrintPIBInfo();
                    return;
                }                
            }
            else
            {
                File f = GetFileForName( file );
                if( f != null )
                {
                    f.PrintFileInfo();
                    return;
                }                
            }

            Console.WriteLine("File or directory '{0}' does not exist.", file);
        }

        /// <summary>
        /// Recursively copies files or directories from the disk image to a local filesystem.
        /// </summary>
        /// <param name="source"></param>
        /// <param name="dest"></param>
        private void Copy(string source, string dest)
        {
            File sourceFile = null;

            if (_path.Count == 0)
            {
                //Root, see if user selected a whole partition
                Partition p = GetPartitionForName(source);

                if (p != null)
                {
                    sourceFile = p.Root;
                }
                else
                {
                    Console.WriteLine("The specified partition '{0}' does not exist.", source);
                    return;
                }
            }
            else
            {
                sourceFile = GetFileForName(source);
            

                if (sourceFile == null)
                {
                    Console.WriteLine("File or Directory '{0}' does not exist.", source);
                    return;
                }
            }
            
            // Commence copying!
            CopyRecursive(sourceFile, dest);
            
        }

        private void CopyRecursive(File sourceFile, string destDir)
        {
            //If the current source file is a directory, we create a new local directory
            //and recursively copy the files in it to the destination directory.
            
            //Otherwise, it's just a file and we copy it to the destination directory.
            if (sourceFile.IsDirectory)
            {
                // Create a new directory and recurse into it.
                string newPath = destDir + "\\" + sourceFile.SimpleName;
                newPath.Replace('$', '#');

                System.IO.Directory.CreateDirectory(newPath);

                foreach (File f in sourceFile.Children)
                {
                    CopyRecursive(f, newPath);
                }
            }
            else
            {
                string newFile = destDir + "\\" + sourceFile.SimpleName;

                // TODO: I don't know if POS can have filenames that Windows doesn't allow.
                //       So this may crash -- probably should try/catch here and prompt the user.
                FileStream fs = new FileStream(newFile, FileMode.Create);
                
                //Write the file!
                fs.Write(sourceFile.Data, 0, sourceFile.Data.Length);
                fs.Close();
            }
        }

        /// <summary>
        /// Given a filename, finds the File in the current path that has that name, if any.
        /// Returns null if no file is found.
        /// </summary>
        /// <param name="fileName"></param>
        /// <returns></returns>
        private File GetFileForName(string fileName)
        {
            foreach (File f in _path.Peek().Children)
            {
                if (f.SimpleName.Trim().ToLower() == fileName.Trim())
                {                    
                    return f;
                }
            }

            return null;
        }

        /// <summary>
        /// Given a directory name (sans ".DR" extension), finds the directory in the current path that has that name, if any.
        /// NOTE: Assumes (possibly incorrectly) that directory names always end in ".dr" and as such strips off the ending ".dr".  
        /// Returns null if no directory with that name is found.
        /// </summary>
        /// <param name="dirName"></param>
        /// <returns></returns>
        private File GetDirectoryForName( string dirName )
        {
            foreach (File f in _path.Peek().Children)
            {
                if (f.IsDirectory && f.SimpleName.Substring(0, f.SimpleName.Length - 3).ToLower() == dirName)
                {
                    return f;                                        
                }
            }

            return null;
        }

        /// <summary>
        /// Given a partition name, finds a partition that matches, if any.
        /// Returns null if no such partition exists.
        /// </summary>
        /// <param name="partName"></param>
        /// <returns></returns>
        private Partition GetPartitionForName( string partName )
        {
            foreach (Partition p in _volume.Partitions)
            {
                if (p.Name.Trim().ToLower() == partName )
                {
                    return p;                    
                }
            }

            return null;
        }

        /// <summary>
        /// Converts the current path to a string.
        /// </summary>
        /// <returns></returns>
        private string PathToString()
        {
            StringBuilder sb = new StringBuilder();

            foreach (File f in _path)
            {
                sb.Insert(0, String.Format(">{0}", f.SimpleName.Trim()));
            }

            return sb.ToString();
        }

        private Stack<File> _path;                

        private PhysicalDisk _physicalDisk;
        private Volume _volume;
    }
}
