/* ---------------------------------------------------------------------- */
/*                  Floating Point Functions Package                      */
/*                 (c) 1982 Knowledge Engineering, Inc.                   */
/* ---------------------------------------------------------------------- */

/* Precision: >= 13 decimal digits.
   Each floating point number requires 8 bytes of storage.

   Note: routines starting with capital 'F' are local and should not
   be accessed outside this package.  All others are public.

   Source for the transcendental functions: "Computer Approximations,"
   by Hart, Cheney, et al., John Wiley & Sons, Inc., NY, 1968.  Part of
   the SIAM series in Applied Mathematics.  Available from UT PMA library,
   call number QA 297 C64.

   Average times in msec:
     FUNCTION          BCD     DOUBLE
        add             5        .2
        sub             5        .2
        mul             9        2
        div             12       2
        intgr           7        0
        itof            5        0
        ftoi            7        0
        abs             2        .1
        neg             2        .1
        compares        5        0
        sqrt            280      45
        10^x            x integer = 5; x real = 180
        exp             <= 200
        log             x a power of 10 = 5; x real = 200
        ln              <= 250                                  45
        pow             <= 470, depending on x in y^x
        sin             350                                     65
        cos             360                                     70
        atan            300                                     60
*/

double ln10 = 2.302585092994;
double pi2 = 1.570796326794;
double abs(), Fpoly(), ln(), log(), Frece(), sin(), cos(), atan();
double sqrt(), exp(), pow10(), pow();

Ferror(err_code)
int err_code;  {rterr();}
/*
{       if(err_code==0) return(0);
        if(err_code & 1) cprintf("Arithmetic overflow.\r\n"); 
        if(err_code & 2) cprintf("Divison by zero.\r\n"); 
        if(err_code & 32) cprintf("log(): Not defined.\r\n");
        if(err_code & 64) cprintf("pow(): Not defined.\r\n");
        if(err_code & 128) cprintf("sqrt(): Not defined.\r\n");
        exit(1); 
}
*/

double abs(f)           /* absolute value of 'f' */
double f;
{   return((f<0.0)? -f : f);   }

double Fpoly(x, p, n)  /* evaluate nth deg. polynomial */
double x, p[]; int n;  /* n >= 1 */
{   double r; 
    int i;
    r = x*p[n];
    for (i=n-1; i>0; i--) r = x*(r+p[i]);
    return(r+p[0]);
}

double ln(f)            /* natural log of 'f' */
double f;
/*      From Eq. 6.3.21 */
{   return(ln10 * log(f));   }

double log(f)   /* log (base 10) of 'f' */
double f;
/*      From Table 2325 & Eq. 6.3.28 */
{   static double p[4] =
    {   -8.625170319686,
        11.70031513942,
        -3.932918863942,
        .1960631750250
    };
    static double q[4] =
    {   -9.930094301066,
        16.78051701465,
        -8.135425915213,
        1.0
    };
    static double sqrt10 = 3.162277660168;
    double z, zsq, num;  int n;

    if ((f<0.0) || f == 0) {Ferror(32);  return(f);}
    n = 0;  while (abs(f) > 1.0) {f /= 10.0;  ++n;}
    while (abs(f) < 0.1) {f *= 10.0;  --n;}
    num = f*sqrt10;  z = (num-1)/(num+1);  zsq = z*z;
    return(Fpoly(zsq, p, 3)/Fpoly(zsq, q, 3)*z + n - 0.5);
}

double Freduce(f)   /* trig support -- returns f in range -pi/2 ... pi/2 */
double f;
{   static double c1 = 6.283185307179586;
    static double c2 = 3.141592653589793;
    static double c3 = .159154943091895;
    double f0, w, t;  long li;
    if (abs(f) <= pi2) return(f); /*this test for speed, not essential*/
    if (abs(f) >= 10E9) {
	rterr();  
	return(0.0);
	} /*reduce range to +/- 2*PI */
    li = (long) (w = abs(f)*c3);  
    f0 = (w-(double)li)*c1;   
    w=f0*c3;
    if (w <= 0.25) return((f<0.0)? -f0: f0);
    if (w <= 0.75) f0=c2-f0;
       else f0= f0 - c1;
    return((f<0.0)? -f0: f0); 
}

double sin(f)
double f;
{    static double c[9] =
    {   .002923976608187135,
	.003676470588235294,
        .004716904761904762,
        .006410256410256410,
        .009090909090909091,
        .013888888888888889,
        .02380952380952381,
        .05,
        .166666666666666667
    };
    double f0, t1;  
    int i;
    f0 = Freduce(f);  
    t1 = f0*f0;  
    f = 1;
    for (i=0; i<9; i++) f = 1.0 - (f*t1*c[i]);
    return(f*f0);
}

double cos(f)
double f;  {return(sin(abs(f)+pi2));}

double atan(f)  /* arctan(f) */
double f;
/*      Eqns. 6.5.21, & 6.5.22  & Table 5097 */
{   static double p[6] =
    {   33.05861847399,
        58.655751569,
        32.3909748562,
        5.853195211263,
        .1952374193623,
        -.002434603300441
    };
    static double q[5] =
    {   33.05861847399,
        69.67529105952,
        49.00434821822,
        12.97557886271,
        1.0
    };
    int neg, recip;  double fsq;
    if (f == 0) return(f);
    if (neg=(f<0.0)) f = -f;
    if (recip=(f > 1.0)) f = 1/f;
    fsq = f*f;  
    f *= Fpoly(fsq, p, 5)/Fpoly(fsq, q, 4);
    if (recip) f = pi2 - f;  
    if (neg) f = -f;  
    return(f);
}

double sqrt(f)
double f;
/*      Eqns. 6.1.3 & 6.1.7;   Initial guess from table 0231 */
{   static double p[3] = {.58812E-2, .5267875, .5881229};
    static double q0 = .999998E-1;
    /* q1 == flone */
    double f0, temp, scale_exp;  
    int t;
    if (f<0.0) {
	Ferror(128);  
	goto rt;}
    if (f == 0) return(0.0);
    scale_exp = 1.0;  f0 = f;
    while (f0 > 1.0) {
	scale_exp *= 10.0;  
	f0 *= 0.01;}
    while (f0 < (double)0.01) {
	scale_exp /= 10.0;  
	f0 *= 100.0;}
    f = Fpoly(f0, p, 2)/(q0+f0);  
    t = 10;
    while (t--) {   
	temp = (f+f0/f)*0.5;  
	if (f == temp) break; 
	   else f = temp;   }
    f *= scale_exp;
 rt:return(f);
}

double exp(f)           /* returns e^f */
double f;
{   return(pow10(f/ln10));   }

double pow10(f) /* return 10^f */
double f;
/*      Eq. 6.2.34 & Table 1403 */
{   static double r[7] =
    {   1.0,
        .2302585092158E1,
        .2650949191501E1,
        .2034670312104E1,
        .1171493469046E1,
        .5358845480519,
        .2321569786604
    };
    static double pw10[9] =
    {   .1258925411794E1,
        .1584893192461E1,
        .1995262314969E1,
        .2511886431510E1,
        .3162277660168E1,
        .3981071705535E1,
        .5011872336273E1,
        .6309573444802E1,
        .7943282347243E1
    };
    int n1, n2, recip;  
    double temp;

    if (recip=(f<0.0)) f = -f;
    if (f > 300.0) {
	Ferror(1);  
	goto rt;}
    n1 = n2 = 0;  
    temp = 0.0;
    if (f >= 0.01) {   
	n1 = f; 
	n2 = (f-n1)*10.0; 
	temp = n1;
        if (n2) 
	    temp = n2*0.1 + temp;
    }
    if (temp=f-temp) 
	f = Fpoly(temp,r,6);
    if (n2) 
	f *= pw10[n2-1];  
    while (n1--) 
	f *= 10.0;
 rt:return((recip)? 1.0/f : f);
}
    
double pow(f1,f2)       /* returns  f1 ^ f2 */
double f1, f2;
{   double f, temp2;  
    int int_pow;  
    long lt1, lt2;

    if (f2 == 0) 
	return(1.0);  
    f = f1;
    temp2 = abs(f2);
    lt2 = temp2;  
    int_pow = (lt2 == temp2);
    if (!int_pow && ( f1 < 0.0 )) {
	Ferror(64); 
	return(f);}
    if (int_pow && temp2 < 1.0E9)  {     /* repeated mul. */
        lt1 = 1;
        while ((lt1 <<= 1) <= lt2) 
	    f *= f;
        lt1 >>= 1;  
	lt1 = lt2 - lt1;
        while (lt1-- > 0) 
	    f *= f1;
        }
	else f = pow10(temp2*log(f1));
return( (f2 < 0.0)? 1.0/f : f);
}

f_lt(f1, f2)
double f1, f2;
{   double f;  
    f = f1 - f2;  
    return(f<0.0);   
}

f_gt(f1, f2)
double f1, f2;
{   double f;  
    f = f2 - f1;  
    return(f<0.0);   
}

