;
; I/O System for 86-DOS version 1.10 and later. Revised 11/03/81.
;
DOSSEG:	EQU	0090H		; Segment for 86-DOS.
VIDSEG:	EQU	0FE00H		; Video board base segment.
DSKSEG:	EQU	0FE80H		; Disk controller base segment.
HRZ:	EQU	10H		; Horizon motherboard base port.
PIO:	EQU	08H		; IMSAI PIO base port.
SLOW:	EQU	1CH		; Slow speed output character.
FAST:	EQU	1DH		; High speed output character.
UCLOCK:	EQU	1EH		; Upper-case lock character.
UNLOCK:	EQU	1FH		; Unlock character.

	ORG	0
	PUT	100H

	JMP	INIT
	JMP	CONSTAT
	JMP	CONIN
	JMP	CONOUT
	JMP	LIST
	JMP	AUXIN
	JMP	AUXOUT
	JMP	READ
	JMP	WRITE
	JMP	DSKCHG
	JMP	SETDATE
	JMP	SETTIME
	JMP	GETTIME

INIT:
	XOR	AX,AX
	MOV	SS,AX
	MOV	SP,400H		; Set stack just below I/O system.
;
; Initialize the character I/O devices here.
;
	PUSH	CS		; DS = CS (so we won't have to SEG CS
	POP	DS		;  all the time).
	MOV	CH,0		; High bytes = 0.
	MOV	DH,CH
	MOV	SI,PORTINIT	; Address of port initialization table.
	UP			; Increment SI after LODB.
INITPORT:
	MOV	DL,[SI]		; Get port number.
	CMP	DL,-1		; Check for end-of-table flag.
	JZ	CLEARSCREEN	; Jump if no more ports.
	INC	SI
	MOV	CL,[SI]		; Get byte count.
	INC	SI
BYTELOOP:
	LODB			; Get a byte.
	OUTB	DX		; Output it.
	LOOP	BYTELOOP
	JP	INITPORT
CLEARSCREEN:
	INB	PIO+0		; Clear console ready.
	MOV	AX,DS		; Save the DS register.
	MOV	BX,VIDSEG	; Set up the data segment for the video board.
	MOV	DS,BX
	MOV	W,[7D2H],0	; Initialize the video board.
	MOV	DS,AX		; Get the original DS register back.
	MOV	AL,12		; Clear screen.
	CALL	9,40H		; Call the output routine.
;
; Tell 86-DOS about the disk system, then load COMMAND.COM
;
	MOV	SI,INITTAB
	CALL	0,DOSSEG
	MOV	DX,100H
	MOV	AH,26		;Set DMA address
	INT	21H
	MOV	CX,[6]		;Get size of segment
	MOV	BX,DS		;Save segment for later
;DS must be set to CS so we can point to the FCB
	MOV	AX,CS
	MOV	DS,AX
	MOV	DX,FCB		;File Control Block for COMMAND.COM
	MOV	AH,15
	INT	21H		;Open COMMAND.COM
	OR	AL,AL
	JNZ	COMERR		;Error if file not found
	XOR	AX,AX
	MOV	[FCB+33],AX	;Set 4-byte Random Record field to
	MOV	[FCB+35],AX	;   beginning of file
	INC	AX
	MOV	[FCB+14],AX	;Set record length field
	MOV	AH,39		;Block read (CX already set)
	INT	21H
	JCXZ	COMERR		;Error if no records read
	TEST	AL,1
	JZ	COMERR		;Error if not end-of-file
;Make all segment registers the same
	MOV	DS,BX
	MOV	ES,BX
	MOV	SS,BX
	MOV	SP,5CH		;Set stack to standard value
	XOR	AX,AX
	PUSH	AX		;Put zero on top of stack for return
	MOV	DX,80H
	MOV	AH,26
	INT	21H		;Set default transfer address (DS:0080)
	PUSH	BX		;Put segment on stack
	MOV	AX,100H
	PUSH	AX		;Put address to execute within segment on stack
	RET	L		;Jump to COMMAND

COMERR:
	MOV	DX,BADCOM
	MOV	AH,9		;Print string
	INT	21H
	EI
STALL:	JP	STALL

;
; Console input  - IMSAI PIO input port 0 connected to Pat's weird keyboard.
; Console output - Pat's video board.
; List output    - IMSAI PIO output port 0.
; Auxillary in   - Horizor standard serial I/O port.
; Auxillary out  - Horizon standard serial I/O port.
;
; Console status & console input routines for Pat's funny keyboard.
;
CONSTAT:
	INB	PIO+1		; Get status.
	AND	AL,02H		; Check if anything there.
	JZ	RETCS		; Return if nothing.
	INB	PIO+0		; Get character.
	CALL	LOOKUP		; Convert to ASCII.
	JNZ	SAVECHAR	; Jump if it's a real character.
	XOR	AL,AL		; Not a real character, AL = zero.
	RET	L
SAVECHAR:
	SEG	CS		; The character was real, save it for CONIN.
	MOV	[CHAR],AL
	AND	AL,AL		; NUL = no character.
RETCS:
	RET	L

CONIN:
	SEG	CS		; See if anything from CONSTAT.
	MOV	AL,[CHAR]
	AND	AL,AL
	JZ	INPUTLOOP	; Nothing from CONSTAT, wait for a character.
	SEG	CS		; Clear CHAR.
	MOV	B,[CHAR],0
	JP	CHECKCONT	; Check for `control characters'.
INPUTLOOP:
	INB	PIO+1		; Loop till character ready.
	TEST	AL,02H
	JZ	INPUTLOOP
	INB	PIO+0		; Get character.
	CALL	LOOKUP		; Convert the data from the keyboard to ASCII.
	JZ	INPUTLOOP	; If the character wasn't real, get another.
CHECKCONT:
	AND	AL,AL		; Check bit seven.
	JNS	RETCI		; Jump if bit seven is zero.
	AND	AL,7FH		; Set bit seven to zero.
	CMP	AL,'a'		; Check for lower-case.
	JB	UPPERCASE
	CMP	AL,'z'+1
	JAE	UPPERCASE
	AND	AL,0DFH		; Convert to upper-case.
UPPERCASE:
	CMP	AL,'@'		; See if `controllable'.
	JB	RETCI		; Jump if too small.
	CMP	AL,'_'+1	; See if too big.
	JAE	RETCI
	SUB	AL,'@'		; We've got a contender.
RETCI:
	RET	L		; Otherwise, we're done.

LOOKUP:
	PUSH	DX
	NOT	AL		; Correct polarity.
	MOV	DL,AL		; Save data.
	AND	AL,7FH		; Ignore bit 7.
	CMP	AL,'A'		; See if less than `A'.
	JB	LOOK		; Characters less than `A' need looking up.
	CMP	AL,'Z'		; See if greater than `Z'.
	JBE	ALPHA		; `A' - `Z' are OK.
	CMP	AL,'a'		; See if less than `a'.
	JAE	CONTIN		; Continue to decode if not between `Z' & `a'.
	SUB	AL,26		; Subtract alphabet length to skip `A' - `Z'.
	JP	LOOK
CONTIN:
	CMP	AL,'z'		; See if greater than `z'.
	JBE	ALPHA		; `a' - `z' are OK.
	SUB	AL,52		; Skip upper & lower case alphabets.
LOOK:
	PUSH	BX
	MOV	BX,KEYTAB	; Address of look-up table.
	SEG	CS		; Look-up correct character.
	XLAT
	AND	DL,80H		; Keep only bit seven of the original data.
	OR	AL,DL		; Merge bit seven with looked-up data.
	POP	BX
	JP	CHKCHAR		; See if we have any special characters.
ALPHA:
	MOV	AL,DL		; Get original character back.
	SEG	CS		; Check for upper-case lock.
	TEST	B,[UCFLAG],0FFH
	JZ	CHKCHAR		; Jump if not.
	AND	AL,0DFH		; Force bit 5 off.
CHKCHAR:
	POP	DX
	CMP	AL,SLOW		; Check for special characters.
	JNE	CHKFAST
	SEG	CS		; Clear HISPD flag.
	MOV	B,[HISPD],00H
	JP	RET		; Return with Z flag set.
CHKFAST:
	CMP	AL,FAST
	JNE	CHKLOCK
	SEG	CS		; Set HISPD flag.
	MOV	B,[HISPD],0FFH
	JP	RET		; Return with Z flag set.
CHKLOCK:
	CMP	AL,UCLOCK
	JNE	CHKUNLOCK
	SEG	CS		; Set UCFLAG.
	MOV	B,[UCFLAG],0FFH
	JP	RET		; Return with Z flag set.
CHKUNLOCK:
	CMP	AL,UNLOCK
	JNE	RET		; Return with Z flag clear if not
	SEG	CS		;  a special character.
	MOV	B,[UCFLAG],00H	; Clear UCFLAG.
	RET			; Return with Z flag set.
;
; 8086 output driver for Pat's fabulous video board.  AL = character
; to print on input.  Only the flags are changed on return.
;
CONOUT:
	PUSH	AX		; Save registers.
	PUSH	DI
	PUSH	DS
	PUSH	ES
	MOV	DI,VIDSEG	; Set up the data segments
	MOV	DS,DI		;  to use the video board.
	MOV	ES,DI
	SEG	CS
	MOV	AH,[HISPD]	; Fast: HISPD=FF, slow: HISPD=00.
WAITLOOP:
	ADD	AH,1		; Increment AH till CY flag is set.
	JNC	WAITLOOP
	MOV	DI,[7D0H]	; Get cursor position.
	CMP	AL,0DH		; Carriage return?
	JE	CRET
	CMP	AL,0AH		; Line feed?
	JE	LINEF
	CMP	AL,08H		; Back space?
	JE	BKSP
	CMP	AL,0BH		; Erase to End-Of-Line.
	JE	CLEOL
	CMP	AL,0CH		; Form feed (clear screen).
	JE	CLEAR
	CMP	AL,20H		; Ignore other control characters.
	JB	ZIP
	CMP	AL,7FH		; Ignore rubouts also.
	JE	ZIP

	MOV	[DI],AL		; Put the character on the screen.
	INC	DI		; Next location.
	CMP	DI,780H		; Check if off screen.
	JB	RESTR		; Jump if on-screen.
	MOV	DI,730H		; If off screen, set cursor to start of last
	JP	SCROLL		;  line and scroll screen up.
CRET:
	AND	B,[DI],7FH	; Turn off cursor.
	CALL	BOL		; Find beginning of line.
	MOV	DI,AX		; Point to beginning-of-line.
	JP	RESTR
LINEF:
	AND	B,[DI],7FH	; Turn off cursor.
	CMP	DI,730H		; See if we're on the bottom line.
	JAE	SCROLL		; Jump if on bottom line.
	ADD	DI,80		; Move down one line.
	JP	RESTR
BKSP:
	AND	DI,DI		; See if we're at the beginning of the first
	JZ	ZIP		;  line.  If so, we can't back up any more.
	AND	B,[DI],7FH	; Turn off cursor.
	DEC	DI		; Back up the pointer.
	JP	RESTR
CLEOL:
	PUSH	CX
	MOV	CX,80		; Eighty characters/line.
	CALL	BOL
	SUB	AX,DI		; Characters between BOL & current position.
	ADD	CX,AX		; Add to characters/line for blank count.
	ADD	DI,CX		; Find the end-of-line.
	JP	CLEARBYTES	; Use part of clear-screen funtion.
CLEAR:
	PUSH	CX
	MOV	CX,780H		; Number of characters on the screen.
	MOV	DI,CX
CLEARBYTES:
	DEC	DI		; Compensate for post-decrement.
	MOV	AL,20H		; Blanks.
	DOWN			; Decrement DI during REP-STOB.
	REP
	STOB			; Fill the screen with blanks.
	INC	DI		; Compensate for post-decrement.
	POP	CX
	JP	RESTR

SCROLL:
	PUSH	CX
	PUSH	DI		; Save pointer.
	MOV	AL,20H		; Blank the invisible 25th line to prevent
	MOV	CX,80		;  glitching when it's moved up onto the
	MOV	DI,780H		;  screen.
	UP			; Increment DI during REP-STOB.
	REP
	STOB
	POP	DI		; Restore pointer.
	POP	CX
	MOV	B,[7D3H],21H	; Scroll command to video board.
RESTR:
	OR	B,[DI],80H	; Turn the cursor back on.
	MOV	[7D0H],DI	; Store the cursor pointer.
ZIP:
	POP	ES		; Restore registers.
	POP	DS
	POP	DI
	POP	AX
	RET	L
;
; Subroutine to find Beginning-Of-Line.
;
BOL:
	PUSH	DX
	MOV	AX,DI
	MOV	DL,80		; 80 characters/line.
	DIV	AL,DL		; AL = line number, AH = position in line.
	MUL	AL,DL		; AX = line number * 80 characters/line.
	POP	DX
	RET
;
; List output, auxillary input, auxillary output routines.
;
LIST:
	PUSH	AX		; Save character.
LISTLOOP:
	INB	PIO+1		; Loop till list status ready.
	TEST	AL,01H
	JZ	LISTLOOP
	POP	AX		; Get character back.
	AND	AL,7FH		; Turn off bit seven.
	OUTB	PIO+0		; Send data out.
	RET	L

AUXIN:
	INB	HRZ+3		; Loop till serial input ready.
	TEST	AL,02H
	JZ	AUXIN
	INB	HRZ+2		; Get data.
	AND	AL,7FH		; Turn off bit seven.
	RET	L

AUXOUT:
	PUSH	AX		; Save character.
AUXOUTLOOP:
	INB	HRZ+3		; Loop till serial output ready.
	TEST	AL,01H
	JZ	AUXOUTLOOP
	POP	AX		; Get character back.
	AND	AL,7FH		; Turn off bit seven.
	OUTB	HRZ+2		; Send out data.
	RET	L
;
; Disk I/O drivers for North Star single-density disk controller.
;
WRTADR:	EQU	200H
CMMND:	EQU	300H
DRVSEL:	EQU	1H
WRTSEC:	EQU	4H
STPOFF:	EQU	8H
STPON:	EQU	9H
NOP:	EQU	10H
RSETSF:	EQU	14H
STPOUT:	EQU	1CH
STPIN:	EQU	1DH
BSTAT:	EQU	20H
RDBYTE:	EQU	40H
MOTOR:	EQU	80H
;
; Status bits.
;
TK0:	EQU	01H		; Track 0 bit.
WP:	EQU	02H		; Write protect bit.
BDY:	EQU	04H		; Data body (sync byte) found.
WRT:	EQU	08H		; Write bytes status flag.
MO:	EQU	10H		; Motor on.
SF:	EQU	80H		; Indicates sector hole was detected.
;
; Delay times in sectors for various disk functions.
;
MOTORD:	EQU	31		; Motor up-to-speed time (1 second).
HEADD:	EQU	14		; Head-load settle time.  Actually, the head
				;  doesn't require this much time to settle,
				;  but this much time is required to
				;  synchronize the sector counter.
STEPD:	EQU	1		; Step time.  One or two only.
				;  1 -> 20mS, 2 -> 40mS.
;
; Various numbers of things.
;
NSECT:	EQU	10		; 10 North Star sectors per track.
NTRACK:	EQU	35		; 35 tracks on standard SA-400 drive.
ERRLIM:	EQU	10		; Number of soft errors.
;
; READ and WRITE functions.
; AL = drive number.
; CX = Number of sectors to transfer.
; DX = Logical record number.
; DS:BX = Transfer address.
;
READ:	
	MOV	AH,1		; AH = 1 to read.
	JP	READWRITE
WRITE:
	MOV	AH,0		; AH = 0 to write.
READWRITE:
	CMP	DX,350		; See if too large a sector number is requested
	JB	SECTOROK	; Jump if OK.
	MOV	AL,0CH		; Error type C, "data error".
	STC			; Set CY flag to indicate error.
	RET	L		; Quit immediatly.
SECTOROK:
	MOV	SI,BX		; Transfer address to SI & DI.
	MOV	DI,BX
	UP			; Set direction flag for autoincrement.
	PUSH	ES		; Store extra segment.
	MOV	BX,DS		; Put data segment in extra segment.
	MOV	ES,BX
	PUSH	DS		; Save data segment.
	MOV	BX,DSKSEG	; DS is North Star controller segment.
	MOV	DS,BX
	PUSH	AX		; Store read/write flag.
	CBW			; Drive number is sixteen bits.
	MOV	BX,AX		; Put in BX.
	MOV	AX,DX		; Compute track & sector.
	MOV	DL,NSECT	; Ten sectors/track.
	DIV	AL,DL		; AL = track number, AH = sector number.
	MOV	CH,AH		; Sector number to CH.
	PUSH	CX		; Save sector number & number of sectors.
	MOV	DH,AL		; Put track number in DH.
	SEG	CS		; TRACKTAB is in the code segment.
	MOV	AH,[BX+TRACKTAB]	; Find out what the current track is.
	SEG	CS
	MOV	[BX+TRACKTAB],DH	; Update TRACKTAB.
	MOV	BP,CMMND+MOTOR+STPIN	; Assume step direction is in.
	MOV	CL,DH		; Put track number in CL.
	SUB	CL,AH		; Calculate how many steps required.
	JAE	DIRECTION	; Direction is correct if >= 0.
	DEC	BP		; Direction is out (STPOUT = STPIN-1).
	NEG	CL		; Make number of steps positive.
DIRECTION:
	IF	STEPD-1		; Multiply number of steps by two if step delay
	SAL	CL		;  is 40mS per step.
	ENDIF
	TEST	B,[CMMND+MOTOR+NOP],MO	; Turn motors on & check MO status.
	JZ	MOTORS		; If motors were off, wait for them to start.
	SEG	CS		; OLDDRIVE is in the code segment.
	CMP	BL,[OLDDRIVE]	; See if the correct drive is selected.
	JNZ	SELECT		; If wrong drive is selected, select right one.
	JP	SEEK		; Motors on, drive selected, go and step.
MOTORS:
	MOV	DL,MOTORD	; Wait for motors to come up to speed.
	CALL	WSECTOR
SELECT:
	CALL	ONESECT		; Wait for write gate to go off.
	MOV	AL,[BX+CMMND+MOTOR+DRVSEL]	; Select new drive.
	SEG	CS		; OLDDRIVE is in code segment.
	MOV	[OLDDRIVE],BL	; Update OLDDRIVE.
	MOV	DL,HEADD-1	; Full head load delay (-1 because waiting for
				;  the correct sector delays at least one more)
	MOV	AL,CL		; See if we've ever used the drive before.
	IF	STEPD-1		; Compute the actual number of steps if 40mS
	SAR	AL		;  step delay is used.
	ENDIF
	CMP	AL,NTRACK	; If the number of steps is >= NTRACK, we can't
	JAE	HEADDELAY	;  count on step time for head load delay.
	SUB	DL,CL		; Subtract stepping time.
	JB	SEEK		; Don't wait if we'll step long enough for the
				;  head to settle & the sector counter to sync.
HEADDELAY:
	CALL	WSECTOR
SEEK:
	IF	STEPD-1		; Convert back to the actual number of steps
	SAR	CL		;  rather than step time if the step time
	ENDIF			;  is 40mS per step.
	XOR	CH,CH		; CX = CL.
	JCXZ	SEEKCOMPLETE	; Jump if we're already there.
	SEG	DS		; BP normally uses stack segment.
	MOV	AL,[BP]		; Set the step direction.
	CALL	ONESECT		; Wait for the write gate to turn off.
;
; Step routine.  Step direction has already been given to the disk
; controller.  DH has destination track number.
; CX has number of sectors to step, >= 1.
; If track zero is ever reached, the head position is recalibrated using DH.
;
STEP:
	MOV	AL,[CMMND+MOTOR+NOP]	; Get `A' status.
	ROR	AL		; Track 0 bit to CF.
	JNC	STEPOK		; Recalibrate if track zero.
	MOV	CL,DH		; Track # to step count.
	JCXZ	SEEKCOMPLETE	; If destinination = 0, we're there.
	MOV	AL,[CMMND+MOTOR+STPIN]	; Set direction.
STEPOK:
	MOV	AL,[CMMND+MOTOR+STPON]
	AAM			; Waste time for > 10 uS.
	MOV	AL,[CMMND+MOTOR+STPOFF]
	MOV	DL,STEPD	; Step time (sectors).
	CALL	WSECTOR
	LOOP	STEP		; Loop till we get there.
SEEKCOMPLETE:
	POP	CX		; Restore sector number & number of sectors.
	MOV	BP,BX		; Put drive number in BP.
SECTORLOOP:
	MOV	DH,ERRLIM	; Soft error limit.
ERRORRETRY:
	DI			; Interrupts illegal till after read, write,
WAITSECTOR:			;  or error.
	CALL	ONESECT		; Wait for next sector to come by.
	MOV	AL,[CMMND+MOTOR+BSTAT+NOP]	; Get `B' status.
	AND	AL,0FH		; Mask to sector number.
	CMP	AL,CH
	JNE	WAITSECTOR	; Wait till the one we want comes by.
	POP	AX		; Get function.
	PUSH	AX		; Back on the stack for next time.
	AND	AH,AH		; AH = 1 -> read, AH = 0 -> write.
	JZ	WRITESECTOR	; Jump if write.
;
READSECTOR:
	MOV	SI,CMMND+MOTOR+RDBYTE+NOP
	PUSH	CX		; Save sector number and number of sectors.
	MOV	CX,352		; Time limit for sync byte.  352 passes through
				;  the loop @ 35 clocks/pass = 24 byte times.
RSYNCLP:
	TEST	B,[CMMND+MOTOR+NOP],BDY
	LOOPZ	RSYNCLP		; Test for sync byte. Loop till sync or timeout
	JNZ	READSECT	; Found sync byte, read data bytes.
	MOV	AL,8		; Error number 8, "Record Not Found".
	JP	ERROR
READSECT:
	MOV	CX,256		; Byte count.
	MOV	DL,CL		; CRC = 0.
READLOOP:
	AAD			; Waste time >= 7.5 uS.
	MOV	AL,[SI]		; Read a byte.
	STOB			; Store byte.
	XOR	DL,AL		; Compute CRC.
	ROL	DL
	LOOP	READLOOP	; Loop for 256 bytes.
	AAD			; Waste time >= 7.5 uS.
	MOV	AL,[SI]		; Get CRC from disk.
	CMP	AL,DL		; Same as computed?
	JE	NEXTSECTOR	; Jump if sucessful read.
	SUB	DI,256		; Back-up the index for retry.
	MOV	AL,4		; Error number 4, "CRC Error".
ERROR:
	EI			; Interrupts OK now.
	POP	CX		; Get sector number & number of sectors.
	DEC	DH		; Decrement error count.
	JNZ	ERRORRETRY	; Wait for the sector to come by again.
	POP	BX		; Pop junk off the stack.
	POP	DS		; Pop segment registers.
	POP	ES
	XOR	CH,CH		; CX is number of sectors left to read.
	STC			; Set CY flag to indicate error.
	RET	L		; Return.
;
WRITESECTOR:
	TEST	B,[CMMND+MOTOR+NOP],WP
	JZ	NOTPROT		; Jump if not protected.
	EI			; Interrupts OK now.
	POP	AX		; Pop junk off the stack.
	POP	DS
	POP	ES
	XOR	CH,CH		; CX = number of sectors left to write.
	MOV	AL,CH		; AL = 0 to indicate write protect.
	STC			; Set CY flag to indicate error.
	RET	L
NOTPROT:
	PUSH	CX		; Save sector number and number of sectors.
	MOV	AL,[CMMND+MOTOR+WRTSEC]
WWRT:
	TEST	B,[CMMND+MOTOR+NOP],WRT
	JZ	WWRT		; Loop till WRT bit goes high.
	MOV	CX,15		; Number of zeros to write.
	MOV	BX,WRTADR	; Address to write zeros.
WRTZERO:
	MOV	AL,[BX]		; Write a zero.
	AAD			; Waste time for >= 7.5 uS.
	LOOP	WRTZERO		; Write 15 of them.
	MOV	BL,0FBH		; Sync byte.
	MOV	AL,[BX]		; Write sync byte.
	AAD			; Waste time for >= 7.5 uS.
	MOV	CX,256		; Byte count.
	MOV	DL,CL		; CRC = 0.
WRTBYTE:
	SEG	ES		; Data is in extra segment.
	LODB			; Get write data.
	MOV	BL,AL		; Data to BL to write.
	MOV	AL,[BX]		; Write it.
	AAD			; Waste time for >= 7.5 uS.
	XOR	DL,BL		; Compute CRC.
	ROL	DL
	LOOP	WRTBYTE		; Write 256 bytes.
	MOV	BL,DL		; Write CRC byte.
	MOV	AL,[BX]
;
NEXTSECTOR:
	EI			; Interrupts OK now.
	POP	CX		; Get sector count.
	DEC	CL		; Decrement sector count.
	JZ	OKRETURN	; Return if done.
	INC	CH		; Increment sector number.
	CMP	CH,10		; Compare with number of sectors on track.
	JAE	NEEDSTEP
	JMP	SECTORLOOP	; Read another sector from same track.
NEEDSTEP:
	MOV	CH,0		; Reset sector number.
	CALL	ONESECT		; Wait for write gate to go off.
	MOV	AL,[CMMND+MOTOR+STPIN]
	MOV	AL,[CMMND+MOTOR+STPON]
	AAM			; Wait > 10 uS for step pulse width.
	MOV	AL,[CMMND+MOTOR+STPOFF]
	SEG	CS		; BP normally uses stack segment.
	INC	B,[BP+TRACKTAB]	; Increment the track table.
				; We don't have to wait for STEPD because
				;  waiting for the write gate to go off caused
				;  us to blow the sector and we have to wait
				;  a whole revolution anyway.
	JMP	SECTORLOOP	; Read a sector from the new track.
OKRETURN:
	POP	AX		; Get function, AH=0 -> write, AH=1 -> read.
	POP	DS		; Get original data & extra segments.
	POP	ES
	CLC			; No errors.
	RET	L
;
; Wait for sector routine.  ONESECT waits for the next sector.
; WSECTOR waits the number of sectors given by DL.
;
ONESECT:
	MOV	DL,1		; Wait for next sector.
WSECTOR:
	MOV	AL,[CMMND+MOTOR+RSETSF]
SECTLOOP:
	MOV	AL,[CMMND+MOTOR+NOP]
	TEST	AL,SF		; Check sector flag.
	JZ	SECTLOOP	; Loop till new sector.
	DEC	DL		; Decrement sector count.
	JNZ	WSECTOR		; Loop till zero.
	RET
;
DSKCHG:
	MOV	AH,0		; AH = 0 in case we don't know.
	SEG	CS
	CMP	AL,[OLDDRIVE]	; See if that's the last drive used.
	JNE	RETL		; Return if not.
	PUSH	DS		; See if the motors are still on.
	PUSH	BX
	MOV	BX,DSKSEG
	MOV	DS,BX
	TEST	B,[CMMND+NOP],MO	
	POP	BX
	POP	DS
	JZ	RETL		; Motors off, disk could be changed.
	MOV	AH,1		; If motors on, assume disk not changed.
RETL:
	RET	L
;
; Time-Of-Day and date stuff.  Time is always 00:00:00.00, date
; returned by GETTIME is simply what was last set by SETDATE.
; Date is initially zero (January 1, 1980).
;
SETDATE:
	SEG	CS
	MOV	[DATE],AX	; Save new date.
SETTIME:
	RET	L

GETTIME:
	XOR	CX,CX		; Hours and minutes are zero.
	MOV	DX,CX		; So are seconds and 1/100 seconds.
	SEG	CS
	MOV	AX,[DATE]	; Find out what the date is.
	RET	L

PORTINIT:
	DB	HRZ+6,9,10H,12H,14H,16H
	DB	18H,1AH,1CH,1EH,40H
	DB	HRZ+3,4,0B7H,77H,0EAH,37H
	DB	PIO+2,1,0FFH
	DB	PIO+3,1,0FFH
	DB	-1		; End of table.

BADCOM:	DB	13,10,"Error in loading Command Interpreter",13,10,"$"
FCB:	DB	1,"COMMAND COM"
	DS	25

KEYTAB:
	DB	60H,12H,01H,06H,03H,13H,05H,03H
	DB	08H,09H,1AH,7BH,7DH,0DH,1BH,18H
	DB	60H,05H,13H,04H,18H,12H,15H,03H
	DB	0AH,1FH,1AH,7BH,7DH,1EH,1BH,18H
	DB	')!@#$%^&*(',00H,':<+>?0123456789'
	DB	00H,';,=./"]_',1DH,00H,00H,'''[-',1CH,' ',00H

CHAR:	DB	0		; Storage for characters read by CONST.
UCFLAG:	DB	0FFH		; Upper-case lock flag.
HISPD:	DB	0FFH		; High speed output flag.
;
; Storage locations for the disk drivers.
;
OLDDRIVE:
	DB	0		; Old drive will be number 0 after boot.
TRACKTAB:
	DB	2		; Drive 0 will be on track 2 after boot.
	DB	NTRACK-1+NTRACK-1+24	; Number of steps to restore the head
	DB	NTRACK-1+NTRACK-1+24	;  if never used before.
;
; Disk initialization tables.
;
INITTAB:
	DB	2		; Number of drives.
	DW	DPT,DPT		; Address of disk parameter tables.
	DW	0		; Minimum buffer space
	DW	30		; Stack space

DPT:
	DW	256		; Sector size.
	DB	1		; One sector per allocation unit.
	DW	30		; Number of sectors allocated to system.
	DB	2		; Two allocation tables.
	DW	64		; Number of directory entries.
	DW	350		; Number of sectors on the disk.

DATE:	DW	0		; Initial date.

	END
