/*****************************************************\
*                                                     *
*                 AVIGO GUI Library                   *
*                    Version 1.01                     *
*    Designed for Gerard Vermeulen's Freeware SDK     *
*        Copyright (C)1999 Charles "Randy" Gill       *
*           Released as licenced freeware             *
*                                                     *
\*****************************************************/

/**************************************************************************\

This library contains the data structures, function prototypes, and
macros to utilize the objects in TI Avigo's Graphical User Interface
(GUI).  It also defines the macros and functions needed for event-driven
program flow.

The Avigo's built-in GUI is somewhat like Microsoft Windows, in that it
somewhat simulates an object-oriented design.  Each type of graphical
object from "Deskboxes" (windows) to pushbuttons has its own "class".
These classes are not C++ classes, but really just structs.  They contain
data members and also pointers to handler functions that are used on the
class, such as "handleEvent", "draw", and "setState".  The internal Avigo
API set contains functions to create these classes (allocate them from
"resource memory"), to destroy them and free their memory, as well as 
default member functions (handlers).  In Avigo GUI programming, a large
portion of the code is replacing these default handlers with your own (to 
respond to commands, paint the screen, etc.)  Most of your own handlers
will call the default handler at some point.

The most significant function in the library is probably ExecView().  In
the main() function, after you have created your classes and assigned
replacement handler functions to them (the default functions are already
assigned, so only assign your modified versions), you call ExecView()
and the Avigo GUI takes over.  It draws your Deskbox, then processes
input via buttons, menus, etc.  It calls the functions you set up to
handle the events.  You don't normally need to worry about the mechanics
of getting and dispatching events.  At the end you normally destroy the
deskbox.

There are many GUI elements (mostly controls) in the library, and the UI
Guidelines.pdf file from TI (attached) shows what they look like. The 16
classes are as follows:

    * PVIEW        - an 'abstract' visual element conceptually
                     similar to the 'window' class in Microsoft Windows.
                     This is the fundamental class of the library.  All 
                     other classes build on to this type. 
    * PGROUP       - a PVIEW with the added ability to 'contain' objects
                     of other classes.
    * PDESKBOX     - based on PGROUP, it is a 'main' window or 'picker' 
                     (dialog box) which can have a title, command menu,
                     and can contain other classes in it.
    * PBUTTON      - a pushbutton object of several visual styles
    * PSTACKEDLIST - a group of two to five vertically stacked 'option
                     buttons'.  These take the place of a group of radio
                     buttons.  The object itself somewhat resembles an
                     oversized listbox`.
    * PPAGEARROW   - a set of small arrow-shaped buttons.  These are used
                     to page up and down, as one would with a scroll bar.
    * PNUMBERFIELD - an input field that will pop-up a calculator-style
                     numeric keyboard picker to enter a number.
    * PNUMBERSET   - a 'spinner' style numeric control, which has buttons
                     to increment and decrement the value in it.
    * PBOOLFIELD   - a visual 'checkbox' control.
    * PLISTER      - a listbox control.
    * PMENUFIELD   - a field which pop-up a menu of choices.  Functional
                     equivalent of a drop-down listbox.

The following classes are present in the Avigo OS, but are not yet
implemented in this library, due to some apparent bugs:

    * PCONTROLMENU - a drop-down menu at the top of some deskboxes.  There
                     are two types of menus commonly at the top of desk-
                     boxes.  The upper 'command menu' with options like 
                     'show private' and 'preferences', and the lower
                     'control menu' commonly used to 'choose a category'.
                     The command menu doen't have a class, as its func-
                     tionality is built into the PDESKBOX class.
    * PTEXTFIELD   - an input field that will pop-up an onscreen keyboard
                     picker to enter any alphanumeric data.
                     Until this is implemented, you may want to use the
                     PTEXTBOX custom control included in a seperate 
                     header in the distribution.
    * PDATEFIELD   - an input field that will pop-up a calendar-style
                     picker to enter a date.
    * PTIMEFIELD   - an input field that will pop-up a picker to allow
                     entry of a time. 
    * PLABEL       - a plain text visual element.  Until this class is
                     implemented, you can use the WriteString() function
                     within the DeskBox's draw() handler.

Some differences between this library and the Official SDK:

Native GUI code implements far pointers as "longs".  Since FWSDK longs
are only 16 bit, they are implemented here as structs, called 'long32',
which are for most purposes compatible.

MK_FAR_PTR is implemented as a function (the Official SDK defines it
as a macro) that takes a 16 bit address and returns a long32 with the
'high' member generated by GetRomBank(), and the 'low' member containing
the address.

The arguments declared as unsigned short by the Official SDK are
declared as unsigned int here.  As these are both 16 bit types, this
shouldn't be much of a problem.

The Official SDK uses one macro, BankedAssign, to assign handlers to
classes.  This library uses two: BankedAssign1 (if the function has
only one parameter) or BankedAssign2 (if it has two or more parameters).
These insert some stub code before and after the call to take care of
stack differences.

None of the functions are declared "BANKED" (although all FWSDK
functions are implicitly banked, in a different way).

There is no implementation of 'LoadBank', and there is no simple way
at this time to call a handler attached to a class object, except those
that respond to the SendMessage() function.

The macros that call the handlers attached to 'thisView' are not
implemented.

\**************************************************************************/

#ifndef _AVGUI_H_
#define _AVGUI_H_

#ifndef AVSYS_H_
#include <avsys.h>
#endif

/*********************************/
/* Library Constant Declarations */
/*********************************/

/* Event Types 

evNothing      This a null event. This condition exists if nothing
               has happened or after an event has been cleared.

evPenDown      This event occurs when the pen is down on the screen.
               It is usually the most important event to be
               processed by event handlers.

evPenDownMove  This event occurs when the pen is down on the screen and is
               being moved. If the pen is moved between 2 calls of
               GetEvent, evPenDownMove occurs.

evPenAuto      This event occurs when the pen is down on an object in the
               screen and is not lifted up. It is useful
               for situations where you might want to rapidly increment a
               number counter without having to tap
               repeatedly on the counter. If the pen is not moved after 100
               calls of GetEvent, evPenAuto occurs.
               After that, if the pen is not moved after 16 calls of
               GetEvent another evPenAuto occurs.

evPenUp        This event occurs when the pen is lifted up from the screen
               after a EvPenDown.

evPen          This event occurs when either a evPenDown, evPenDownMove,
               evPenAuto or evPenUp event occurs.

evKeyDown      This event occurs when a hardkey on the Avigo organizer is
               pressed. Also, when the PC starts a
               synchronization an evKeyDown event occurs.

evTimer        This is a timer event which is generated every second by the
               Avigo system.

evUnknown      This value might be unused.

evCommand      This event occurs after a pen event takes place. For
               example, when a button is created, a command number is
               associated with the button. When a user taps on the button,
               the button handler will generate an event command with the
               command number that you associate with it. Your
               application code would understand what action to take after
               checking the command number.
*/

#define evNothing	0x00  
#define evPenDown	0x01
#define evPenDownMove	0x02
#define evPenAuto	0x04
#define evPenUp		0x08
#define evPen		0x0F
#define evKeyDown	0x10
#define evTimer		0x20
#define evUnknown	0x40
#define evCommand	0x80
 
/* Standard Command Values */
#define cmUnknown1	0x0000
#define cmUnknown2	0x0001
#define cmQuit		0x0002
#define cmOK		0x0003
#define cmCancel	0x0004
#define cmYes		0x0005
#define cmNo		0x0006
#define cmSave		0x0007
#define cmRedraw	0x0008
#define cmUnknown3	0x0009
#define cmUnknown4	0x000A
#define cmUnknown5	0x000B
#define cmUnknown6	0x000C
#define cmUnknown7	0x000D
#define cmUnknown8	0x000E
#define cmUnknown9	0x000F
#define cmApplication	0x0010
#define cmTable		0x0011
#define cmCalculator	0x0012
#define cmExpenses	0x0013
#define cmMemo		0x0014
#define cmTask		0x0015
#define cmSchedule	0x0016
#define cmAddress	0x0017
#define cmCalendar	0x0018
#define cmFind		0x0019
#define cmTime		0x001A
#define cmSync		0x001B
#define cmMemManager	0x001C
#define cmSecurity	0x001D
#define cmSetup		0x001E
#define cmSketch	0x001F
#define cmRunApp	0x0020
#define cmRunCalc	0x0021
#define cmNew		0x1000
#define cmEdit		0x1001
#define cmDelete	0x1002
#define cmView		0x1003
#define cmPgUp		0x1008
#define cmPgDn		0x1009
#define cmPaste		0x100A
#define cmCopy		0x100B
#define cmCut		0x100C
#define cmUnknown10	0x100D
#define cmUnknown11	0x100E
#define cmUnknown12	0x100F
#define cmDuplicate	0x1010
#define cmClear		0x1012

/* Option Flags */
#define ofFocusable	0x01 /* The object is capable of receiving input
                                focus */
#define ofHardKeyable	0x04 /* The app can be interrupted by one of the
                                'hard keys' (eg. ADDRESS, TODO) */
#define ofFindable	0x08 /* The silk-screened FIND button is enabled */
#define ofSyncable	0x10 /* The app can be interrupted to go into sync
                                mode */ 
/* Note: ofHardKeyable, ofFindable, and ofSyncable only apply to PDESKBOX */

/* State Flags */
#define sfFocused	0x01 /* The object currently has the input focus;
                                not used by all classes */
#define sfDisplay	0x02 /* The object is displayed; not used by all
                                classes */
#define sfDisable	0x04 /* The object is disabled */

/* MENUITEM Flags */
#define lstNormal	0x00 /* Display menu item normally */
#define lstUnderLine	0x01 /* Display menu item with a line below */
#define lstBoldFont	0x02 /* Display menu item in boldface */
#define lstDisplay	0x80 /* Not used */

/* PBUTTON Types */
#define bttBasic	0x00 /* Usual shape, long with round ends */
#define bttGoTo		0x01 /* Like 'Go To' in the SCHEDULE app, has
                                pointed ends */
#define bttToDo		0x02 /* Like 'To Do' in the CALENDAR app, oval
                                with a slight shadow */
#define bttRotate	0x03 /* Round button with a curved arrow.  Fixed
                                size */
#define bttSingle	0x04 /* Button has a single boundary */
#define bttSinBold	0x05 /* Button has a single boundary and bold font */

/* PDESKBOX Box Attributes */
#define bafClose	0x0001 /* Has a close (x) button at top right */
#define bafCheck	0x0002 /* Has a checkmark button at top right */
#define bafComMenu	0x0004 /* Has a command menu */
#define bafComLine	0x0008 /* Has a dotted horizontal line near bottom */
#define bafDotTitle	0x0010 /* Has a line of dots after the title */
#define bafPlace	0x0100 /* The DeskBox is an 'activity place' type */
#define bafPoint	0x0200 /* The DeskBox is a 'navigation point' type */
#define bafSmall	0x0400 /* The DeskBox is a 'small picker' type */
#define bafDialog	0x0800 /* The DeskBox is a dialog box */

/* PPAGEARROW type flags */
#define csfUpCom	0x01 /* Enable up arrow */
#define csfDnCom	0x02 /* Enable down arrow */

/* Data Type Numbers */
#define DTNOTYPE        0x00 /* No type */
#define DTTEXT16        0x01 /* Text field - max 16 chars */
#define DTTEXT128       0x02 /* Text field - max 128 chars */
#define DTTEXT256       0x03 /* Text field - max 256 chars */
#define DTTEXT512       0x04 /* Text field - max 512 chars */
#define DTTEXT4K        0x05 /* Text field - max about 4K chars */
#define DTBOOLEAN       0x06 /* Boolean (0=unchecked,1=checked) */
#define DTIINTEGER      0x07 /* Integer */
#define DTREAL          0x08 /* Floating Point */
#define DTCURRENCY      0X09 /* Currency */
#define DTDATE          0X0A /* Date */
#define DTTIME          0X0B /* Time */
#define DTDBTIME        0X0C /* Time range <???> */
#define DTALARM         0X0D /* Uses the alarm lookup list */
#define DTDRAWING       0X0E /* Sketch drawing */
#define DTCATEGORY      0X0F /* Category */
#define DTTASKPRIORITY  0X10 /* Uses the priority lookup list */
#define DTREPEAT        0X11 /* Schedule repeat frequency */
#define DTLIST1         0X12 /* Uses list 1 for lookup */
#define DTLIST2         0X13 /* Uses list 2 for lookup */
#define DTLIST3         0X14 /* Uses list 3 for lookup */
#define DTLIST4         0X15 /* Uses list 4 for lookup */
#define DTLIST5         0X16 /* Uses list 5 for lookup */
#define DTLIST6         0X17 /* Uses list 6 for lookup */
#define DTLIST7         0X18 /* Uses list 7 for lookup */
#define DTLIST8         0X19 /* Uses list 8 for lookup */
#define DTLIST9         0X1A /* Uses list 9 for lookup */
#define DTLIST10        0X1B /* Uses list 10 for lookup */
#define DTLIST11        0X1C /* Uses list 11 for lookup */
#define DTLIST12        0X1D /* Uses list 12 for lookup */
#define DTLIST13        0X1E /* Uses list 13 for lookup */
#define DTLIST14        0X1F /* Uses list 14 for lookup */
#define DTLIST15        0X20 /* Uses list 15 for lookup */
#define DTLIST16        0X21 /* Uses list 16 for lookup */

/******************************************************************/
/* 32 Bit Pointer Workaround (freeware compiler uses 16 bit longs)*/
/******************************************************************/

typedef struct {
    unsigned int addr;
    unsigned int bank;
} FAR_PTR;

const FAR_PTR NULL_FP = {0,0};

/**********************/
/* Conversion Macros  */
/**********************/

#define TOWORD(hi, lo) ((unsigned int)(hi)<<8)+(lo)
/* Parameters:
      hi	High byte
      lo	Low byte
   Return Value:
      Word derived from hi and lo
   Description:
      This is often used to derive the command number from the x and
      y parameters of an evCommand event.
*/

#define LOWBYTE(x) (unsigned char)((unsigned int)(x)&0x00FF)
/* Parameters:
      x		word
   Return Value:
      Low byte of word x
*/

#define HIGBYTE(x) (unsigned char)((unsigned int)(x)>>8)
/* Parameters:
      x		word
   Return Value:
      High byte of word x
*/

/*******************************/
/* Pointer Macros and Funtions */
/*******************************/

#define BANK_NO unsigned int
#define CONST_FAR_PTR FAR_PTR
typedef char *VOID_PTR;
#define PVIEW_PTR VOID_PTR
#define PGROUP_PTR VOID_PTR
#define NULL ((VOID_PTR) 0x0000);

typedef struct
{
    void (*addr)();
    BANK_NO bank; 
} HANDLER_PTR;

extern unsigned int GetRomBank();
/* Parameters:
      None
   Return Value:
      Bank number
   Description:
      Returns the number of the currently active ROM bank.
*/

/* The following two declarations, _handleraddr and _hndptr are intended
for the internal use of the library, only.  */

extern void (*_handleraddr)();

HANDLER_PTR *_hndptr;

#define BankedAssign1(dest,src)  _asm{\
 jr .+7::\
 call __bkpush1::\
 .dw _##src::\
 call __GetTargetAddr::\
 ld (__handleraddr),hl::\
 pop hl}\
 _hndptr=(HANDLER_PTR *)(&(dest));\
 _hndptr->addr=_handleraddr;\
 _hndptr->bank=GetRomBank();

/* Parameters:
      dest	destination variable
      src	address of handler function
   Return Value:
      None
   Description:
      Used to assign a handler function, that has only one parameter, to
      a class object.
*/

#define BankedAssign2(dest,src)  _asm{\
 jr .+7::\
 call __bkpush2::\
 .dw _##src::\
 call __GetTargetAddr::\
 ld (__handleraddr),hl::\
 pop hl}\
 _hndptr=(HANDLER_PTR *)(&(dest));\
 _hndptr->addr=_handleraddr;\
 _hndptr->bank=GetRomBank();

/* Parameters:
      dest	destination variable
      src	address of handler function
   Return Value:
      None
   Description:
      Used to assign a handler function, that has two or more parameters,
      to a class object.
*/

FAR_PTR MK_FAR_PTR(void *addr) 
{
   FAR_PTR retval;
   retval.addr = (unsigned int) addr;
   retval.bank = GetRomBank();
   return retval;
}
/* Parameters:
      addr	Address to convert to a far pointer, or something that
                converts to an address, eg. array with no subscript,
                string literal, function name (no parenthesis).
   Return Value:
      Far pointer equivalent of addr.
   Description:
      This function creates a far pointer (having address and bank number)
      from a given address.  Note: MK_FAR_PTR must be performed in the same
      source unit (hence the same bank) where the function or variable
      indicated by 'addr' is defined. 'addr' indicates
*/

/******************************/
/* General Resource Functions */
/******************************/

extern unsigned char *AllocBlock(int size, unsigned char ID, char mode);
/* Parameters:
      size	Size of memory block
      id	Id of block
      mode      0=random; 1=stack mode
   Return Value:
      Pointer to created memory block, or 0 if error
   Description:
      This will dynamically allocate a block from 'resource memory'.
      It is normally used only when creating a custom class from scratch.
      The 'Create...' functions all allocate their own memory.  There is
      about 5.7K of resource memory available, which is enough for 150 
      average sized objects at any given time.  ID numbers can be reused
      but the new object using the ID must not be larger than the original.
*/
    
extern void FreeBlock(VOID_PTR block);
/* Parameters:
      block	Pointer to the memory block to be freed
   Return Value:
      None
   Description:
      Frees a memory block allocated with AllocBlock.  Destroy() will
      also free the block, and will work on custom classes as well, so
      this isn't often needed.
      If the block was allocated in stack mode, it will be released,
      along with any 'child' classes.  If it was created in random mode,
      it is just marked as 'available'.  Stack mode blocks must be freed
      in the reverse order of which they were created.  Random blocks
      can be freed in any order.
*/

extern void Destroy(VOID_PTR object);
/* Parameters:
      object	Pointer to the object to be destroyed.
   Return Value:
      None
   Description:
      Destroys the object pointed to by ptr.  If the object is a PGROUP
      derived type, for example PDESKBOX, it will also destroy any objects
      that were inserted it (but not any that were inserted into those).
*/

extern unsigned int ExecPicker(char *str, unsigned int len); 
/* Parameters:
      str	String to be edited
      len	Maximum length of string
   Return Value:
      Uncertain
   Description:
      This displays the standard/T9 keyboard picker and allows editing
      of a string.
*/

void SetPickerTitle(CONST_FAR_PTR NewTitle)
{
   FAR_PTR *PickerTitle;
   PickerTitle = (FAR_PTR *) 0xF154;
   *PickerTitle = NewTitle;
}
/* Parameters:
      NewTitle	Far pointer to the new title
   Return Value:
      None
   Description:
      Sets the default title of the system pickers.
*/

extern void ResourceRotate(unsigned char dir);
/* Parameters:
      dir	Direction: 0 = portrait, 1 = landscape 
   Return Value:
      None
   Description:
      Sets the display direction for the GUI objects.  Note: the system
      pickers will not be rotated.
*/

unsigned int GetScreenRight()
{
   return 159;
}
/* Parameters:
      None
   Return Value:
      Right-most pixel coordinate of screen (159)
*/

unsigned int GetScreenBottom()
{
   return 239;
}
/* Parameters:
      None
   Return Value:
      Bottom-most pixel coordinate of screen (239)
*/


/*******************/
/* Event Functions */
/*******************/

extern void PutEvent(unsigned int evType, unsigned char x,
                     unsigned char y);
/* Parameters:
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Puts an event message into (the front of) the event queue.  If this
      function is used a second time without the first event being
      processed, the second event will overwrite the first.
*/

extern unsigned int TestPutEvent(unsigned char *x, unsigned char *y);
/* Parameters:
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      The event type
   Description:
      Checks the event at the front of the queue.  Useful to test
      if a event previously inserted using PutEvent() has been
      processed, before using PutEvent() again.
*/

extern char SendMessage(PVIEW_PTR receiver, unsigned int evType,
                        unsigned char x, unsigned char y);
/* Parameters:
      receiver	Pointer to object to receive event message
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      1 if the sent event has been handled
      0 if not
   Description:
      Sends an event message directly to a specific object.
*/

extern unsigned int GetEvent(unsigned char *x, unsigned char *y);
/* Parameters:
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      The event type
   Description:
      Gets the next event in the event queue.
*/

extern void ClearEvent(unsigned int *evType);
/* Parameters:
      evType	The event type to clear
   Return Value:
      None
   Description:
      Clears an event (changes it to evNothing).  Used after a control
      processes an event, so that other controls do not try to process it.
*/

extern void SetEventMask(unsigned int evType);
/* Parameters:
      evType	The event types to allow
   Return Value:
      None
   Description:
      Specifies the event masking.  All events not specified in evType will 
      be filtered out.
*/

extern unsigned int GetEventMask();
/* Parameters:
      None
   Return Value:
      The event types allowed
   Description:
      Reads the current status of the event masking.
*/

/*******************************************/
/* PVIEW Definition, Functions             */
/*******************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
} PVIEW;

extern void PVIEW_setState(VOID_PTR thisView, unsigned char state,
                           char enable);
/* Parameters:
      thisView	the object the handler applies to
      state	new state
      enable	<???>
   Return Value:
      None
   Description:
      Default setState handler for PVIEW class
*/

extern void PVIEW_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PVIEW class.  Draws the object.
*/

extern void PVIEW_handleEvent(VOID_PTR thisView, unsigned int *evType,
                              unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PVIEW.  Processes the events the object
      receives.
*/

/********************************************/
/* PGROUP Definition, Functions             */
/********************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    PVIEW_PTR current;		/* Pointer to object that has input focus */
    unsigned int endState;	/* Flag to finish execution if non-zero */
    unsigned int (*execute)();	/* Address of execute handler */ 
    BANK_NO executeBank;	/* Bank of execute handler */ 
    void (*redraw)();		/* Address of redraw handler */
    BANK_NO redrawBank;		/* Bank of redraw handler */
    void (*preView)();		/* Address of preView handler */
    BANK_NO preViewBank;	/* Bank of preView handler */    
    void (*setCurrent)();	/* Address of setCurrent handler */
    BANK_NO setCurrentBank;	/* Bank of setCurrent handler */
    void (*insert)();		/* Address of insert handler */
    BANK_NO insertBank;		/* Bank of insert handler */
} PGROUP;

extern void PGROUP_setState(VOID_PTR thisView, unsigned char state,
                            char enable);
/* Parameters:
      thisView	the object the handler applies to
      state	new state
      enable	<???>
   Return Value:
      None
   Description:
      Default setState handler for PGROUP class
*/

extern void PGROUP_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PGROUP class.  Draws the object.
*/

extern void PGROUP_handleEvent(VOID_PTR thisView, unsigned int *evType,
                               unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PGROUP.  Processes the events the object
      receives.
*/

extern unsigned int PGROUP_execute(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      Exit command code (cmOK, cmQuit, etc...)
   Description:
      Default execute handler for PGROUP class.  See ExecView.
*/

extern void PGROUP_redraw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default redraw handler for PGROUP class.  Checks to see if any events
      are pending, then calls the draw handler.
*/

extern void PGROUP_setCurrent(VOID_PTR thisView, PVIEW *target);
/* Parameters:
      thisView	the object the handler applies to
      target	the object to give input focus
   Return Value:
      None
   Description:
      Default setCurrent handler for PGROUP class.  Assigns the input
      focus to the child object indicated by target.
*/

extern void PGROUP_insert(VOID_PTR thisView, PVIEW *target);
/* Parameters:
      thisView	the object the handler applies to
      target	the object to insert
   Return Value:
      None
   Description:
      Default insert handler for PGROUP class.  Inserts target object into
      the Group or DeskBox object.  Target object must be 'created' prior
      to inserting it.  When the Group/DeskBox is destroyed, any inserted
      objects will also be destroyed.
*/

extern void Delete(PGROUP *group, PVIEW *target);
/* Parameters:
      thisView	the Group or DeskBox object
      target	the object to delete
   Return Value:
      None
   Description:
      Removes target object that was previously inserted into the Group or
      DeskBox object.  Does not destroy the target object (use Destroy() 
      afterward).
*/

/**********************************************/
/* PDESKBOX Definitions, Functions            */
/**********************************************/

typedef struct {
    unsigned char item[17];	/* Text of the menu item */
    unsigned int command;	/* Command number */
    unsigned char flag;		/* Menu item flag */
} MENUITEM;

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    PVIEW_PTR current;		/* Pointer to object that has input focus */
    unsigned int endState;	/* Flag to finish execution if non-zero */
    unsigned int (*execute)();	/* Address of execute handler */ 
    BANK_NO executeBank;	/* Bank of execute handler */ 
    void (*redraw)();		/* Address of redraw handler */
    BANK_NO redrawBank;		/* Bank of redraw handler */
    void (*preView)();		/* Address of preView handler */
    BANK_NO preViewBank;	/* Bank of preView handler */    
    void (*setCurrent)();	/* Address of setCurrent handler */
    BANK_NO setCurrentBank;	/* Bank of setCurrent handler */
    void (*insert)();		/* Address of insert handler */
    BANK_NO insertBank;		/* Bank of insert handler */
    CONST_FAR_PTR title;	/* Far pointer to the title text */
    void (*drawTitle)();	/* Address of drawTitle handler */
    BANK_NO drawTitleBank;	/* Bank of drawTitle handler */
    unsigned int boxAttrib;	/* Box attributes */
    CONST_FAR_PTR commandMenu;  /* Far pointer to array of menuitems to
                                   be used as the command menu.  Array can
                                   have between 1 and 32 entries */
    unsigned int menuDisableMask; /* Mask used to disable menu items */ 
    CONST_FAR_PTR menuPromptStr;  /* Far pointer to string to use as the
                                     menu prompt */
} PDESKBOX;

extern void PDESKBOX_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PDESKBOX class.  Draws the object.
*/

extern void PDESKBOX_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                 unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PDESKBOX.  Processes the events the object
      receives.
*/

extern void PDESKBOX_redraw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default redraw handler for PDESKBOX class.  Checks to see if any events
      are pending, then calls the draw handler.
*/

extern void PDESKBOX_drawTitle(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default drawTitle handler for PDESKBOX class.
*/

extern VOID_PTR CreateDeskBox(unsigned char ID, unsigned char left,
                              unsigned char top, unsigned char right,
                              unsigned char bottom, CONST_FAR_PTR title,
                              CONST_FAR_PTR menu, CONST_FAR_PTR prompt,
                              unsigned int attr);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      right	Right screen coordinate
      bottom	Bottom screen coordinate
      title	Far pointer to the title text
      menu	Far pointer to array of menuitems to use as the command menu
      prompt	Far pointer to string to use as the menu prompt
      attr	Box attributes
   Return Value:
      Pointer to the newly-created deskbox
   Description:
      Creates a DeskBox object
*/

extern VOID_PTR InsertButton(PDESKBOX *dsk, unsigned char ID,
                             CONST_FAR_PTR caption, unsigned int command,
                             unsigned char type, unsigned char alignment);
/* Parameters:
      dsk	Pointer to the DeskBox to insert the button into
      ID	Unique ID for the button
      caption	Far pointer to the caption text
      command	Command associated with the button
      type	Button type
      alignment Use 0 for the first button left-aligned
          	Use 1 for subsequent left-aligned buttons
                Use 2 for a right-aligned button
   Return Value:
      Pointer to the newly-created button
   Description:
      This function both creates a button object, and inserts it into the
      DeskBox, at the bottom (under the dotted line if present).
*/

extern VOID_PTR InsertPageArrow(PDESKBOX *dsk, unsigned char ID,
                                PVIEW *reciever);
/* Parameters:
      dsk	Pointer to the DeskBox to insert the PageArrow into
      ID	Unique ID for the PageArrow
      Receiver	Pointer to the object to receive messages from the PageArrow
   Return Value:
      Pointer to the newly-created PageArrow
   Description:
      This function both creates a PageArrow object, and inserts it into the
      DeskBox, at the bottom right (under the dotted line if present).
*/

extern unsigned int ExecView(PGROUP_PTR group);
/*
   Parameters:
      Pointer to the DeskBox (or Group) to execute
   Return Value:
      Exit command code (cmOK, cmQuit, etc...)
   Description:
      This will draw, and redraw when needed, our DeskBox and all our
      controls.  It will continuously poll for pen and button input, then
      send messages to the controls and the DeskBox itself, and invoke their
      handlers.  It will exit when endState is non-zero.
*/
      
extern void ChangeCommandMenu(VOID_PTR thisView, CONST_FAR_PTR newMenu);
/*
   Parameters:
      thisView	Pointer to the DeskBox
      newMenu	Pointer to the new menu
   Return Value:
      None
   Description:
      Assigns a new command menu to a DeskBox
*/

extern void DisableCommand(unsigned int command);
/*
   Parameters:
      command	Command to be disabled
   Return Value:
      None
   Description:
      Disables a command (associated with a Button, Command Menu, or
      PageArrow) within the currently executing DeskBox.
*/

extern void EnableCommand(unsigned int command);
/*
   Parameters:
      command	Command to be enabled
   Return Value:
      None
   Description:
      Re-enables a command (associated with a Button, Command Menu, or
      PageArrow) within the currently executing DeskBox.
*/

/*********************************************/
/* PBUTTON Definition, Functions             */
/*********************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    CONST_FAR_PTR text;		/* Far pointer to the text on the button */
    unsigned int command;	/* Command associated with the button */
    void (*drawStatus)();	/* Address of drawStatus handler */
    BANK_NO drawStatusBank;	/* Bank of drawStatus handler */
    unsigned char buttonType;	/* Button type */
} PBUTTON;

extern void PBUTTON_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PBUTTON class.  Draws the object.
*/

extern void PBUTTON_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PBUTTON.  Processes the events the object
      receives.
*/

extern void PBUTTON_drawStatus(VOID_PTR thisView, unsigned char status);
/* Parameters:
      thisView	the object the handler applies to
      status	0 = unpressed; 1 = pressed
   Return Value:
      None
   Description:
      Default drawStatus handler for PBUTTON class.  Changes the state of
      the button to pressed or unpressed.
*/

extern VOID_PTR CreateButton(unsigned char ID, unsigned char left,
                             unsigned char top, unsigned char right,
                             unsigned char bottom,
                             CONST_FAR_PTR caption, unsigned int command,
                             unsigned char type);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      right	Right screen coordinate
      bottom	Bottom screen coordinate
      caption	Far pointer to the caption text
      command	Command associated with the button
      type	Button type
   Return Value:
      Pointer to the newly-created button
   Description:
      Creates a Button object.  Note: button height must be certain fixed
      values.  Calculate 'bottom' based on button type as follows:
         'top' + 16: for bttBasic, bttGoto, bttToDo, bttRotate
         'top' + 14: for bttSingle, bttSinBold
*/

extern void ChangeButton(PBUTTON *bt, CONST_FAR_PTR caption,
                         unsigned int command);
/* Parameters:
      thisView	the object the handler applies to
      caption	Far pointer to the new caption text
      command	New command to associate with the button
   Return Value:
      None
   Description:
      Changes the caption and command associated with the button
*/

/***************************************************/
/* PSTACKEDLIST Definitions, Functions             */
/***************************************************/

typedef struct
{
    unsigned char item[37];
} STACKEDITEM;

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    void (*drawExample)();	/* Address of drawExample handler */
    BANK_NO drawExampleBank;	/* Bank of drawExample handler */
    CONST_FAR_PTR stackedItems;	/* Far pointer to array of STACKEDITEM (can
                                   have 2 to 5 elements) */
    unsigned char stackedSum;	/* Number of items in this list */
    unsigned char *stackedFocused; /* Pointer to variable that holds the
                                      number of the selected item */
} PSTACKEDLIST;

extern void PSTACKEDLIST_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PSTACKEDLIST class.  Draws the object.
*/

extern void PSTACKEDLIST_handleEvent(VOID_PTR thisView, unsigned int *evType, unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PSTACKEDLIST.  Processes the events the object
      receives.
*/

extern void PSTACKEDLIST_drawExample(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default drawExample handler for PSTACKEDLIST class.
*/

extern VOID_PTR CreateStackedList(unsigned char ID, unsigned char left,
                                  unsigned char top, unsigned char width,
                                  CONST_FAR_PTR items, unsigned char *focused);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      width	Width of StackedList
      items	Far pointer to the array of StackedItems
      focused	Pointer to variable that holds the number of selected item
   Return Value:
      Pointer to the newly-created StackedList
   Description:
      Creates a StackedList object
*/

/************************************************/
/* PPAGEARROW Definition, Functions             */
/************************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    unsigned int upCommand;	/* Command to associate with up arrow */
    unsigned int dnCommand;	/* Command to associate with down arrow */
    unsigned char comStatus;	/* PageArrow type flag */
    PVIEW *comReceiver;		/* Object to send commands to */ 
} PPAGEARROW;

extern void PPAGEARROW_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PPAGEARROW class.  Draws the object.
*/

extern void PPAGEARROW_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                   unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PPAGEARROW.  Processes the events the object
      receives.
*/

extern VOID_PTR CreatePageArrow(unsigned char ID, unsigned char left,
                                unsigned char top, PVIEW *reciever,
                                unsigned int upCmd, unsigned int dnCmd);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      receiver	Object to send commands to
      upCmd	Command to associate with up arrow
      dnCmd	Command to associate with down arrow
   Return Value:
      Pointer to the newly-created PageArrow
   Description:
      Creates a PageArrow object
*/

/**************************************************/
/* PNUMBERFIELD Definition, Functions             */
/**************************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    int totalRows;		/* Number of rows */
    double *numeric;		/* Variable to hold field value */
    unsigned char numType;	/* DTINTEGER, DTREAL, or DTCURRENCY */
    CONST_FAR_PTR pickerTitle;	/* Far pointer to string to use as the
                                   picker title */
} PNUMBERFIELD;

extern void PNUMBERFIELD_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PNUMBERFIELD class.  Draws the object.
*/

extern void PNUMBERFIELD_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                     unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PNUMBERFIELD.  Processes the events the object
      receives.
*/

extern VOID_PTR CreateNumberField(unsigned char ID, unsigned char left, 
                                  unsigned char top, unsigned char *numeric,
                                  unsigned char numType, CONST_FAR_PTR pTitle);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      numeric	Variable to hold field value
      numType	DTINTEGER, DTREAL, or DTCURRENCY
      pTitle	Far pointer to string to use as the picker title
   Return Value:
      Pointer to the newly-created NumberField
   Description:
      Creates a NumberField object
*/

/************************************************/
/* PNUMBERSET Definition, Functions             */
/************************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    void (*drawNumber)();	/* Address of drawNumber handler */
    BANK_NO drawNumberBank;	/* bank of drawNumber handler */
    unsigned char *numAddr;	/* Variable to hold NumberSet value */
    unsigned char numChange;	/* Increment qty for each pen tap */
    unsigned char minNumber;	/* Minimum NumberSet value */
    unsigned char maxNumber;	/* Maximum NumberSet value (99 is highest) */
} PNUMBERSET;

extern void PNUMBERSET_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PNUMBERSET class.  Draws the object.
*/

extern void PNUMBERSET_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                   unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PNUMBERSET.  Processes the events the object
      receives.
*/

extern void PNUMBERSET_drawNumber(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default drawNumber handler for PNUMBERSET class.
*/

extern VOID_PTR CreateNumberSet(unsigned char ID, unsigned char left,
                                unsigned char top, unsigned char min,
                                unsigned char max, unsigned char change,
                                unsigned char *NumAddr);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      min	Minimum NumberSet value
      max	Maximum NumberSet value
      change	Increment qty for each pen tap
      NumAddr	Address of variable to hold the NumberSet value
   Return Value:
      Pointer to the newly-created NumberSet
   Description:
      Creates a NumberSet object
*/

/************************************************/
/* PBOOLFIELD Definition, Functions             */
/************************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    int totalRows;		/* Number of rows */
    unsigned char *data;	/* Variable to hold BoolField value */
    unsigned char bit;		/* Bit to toggle in data */
} PBOOLFIELD;

extern void PBOOLFIELD_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PBOOLFIELD class.  Draws the object.
*/

extern void PBOOLFIELD_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                   unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PBOOLFIELD.  Processes the events the object
      receives.
*/

extern VOID_PTR CreateBoolField(unsigned char ID, unsigned char left, 
                                unsigned char top, unsigned char *data,
                                unsigned char mask);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      data	Address of variable to hold the BoolField value
      mask	Mask indicating bit to be toggled in data
   Return Value:
      Pointer to the newly-created BoolField
   Description:
      Creates a BoolField object
*/

/*********************************************/
/* PLISTER Definition, Functions             */
/*********************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    unsigned int itemSum;	/* Number of list items */
    unsigned int topItem;	/* List item currently at top of display */
    unsigned int focused;	/* The number of the currently selected
                                   list item */ 
    void (*writeItem)();	/* Address of writeItem handler */
    BANK_NO writeItemBank;	/* Bank of writeItem handler */
    void (*penDownAct)();	/* Address of penDownAct handler */
    BANK_NO penDownActBank;	/* Bank of penDownAct handler */
    unsigned int lastItem;	/* List item at bottom of display? */
    unsigned int topItemCounter; /* <???> */
} PLISTER;

extern void PLISTER_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PLISTER class.  Draws the object.
*/

extern void PLISTER_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PLISTER.  Processes the events the object
      receives.
*/

extern void PLISTER_writeItem(VOID_PTR thisView, unsigned char left, 
                              unsigned char top, unsigned int item);
/* Parameters:
      thisView	the object the handler applies to
      left	Left screen coordinate
      top	Top screen coordinate
      item	Number of item to be drawn
   Return Value:
      None
   Description:
      Default writeItem handler for PLISTER.  Called for each list item
      displayed.  This must be overriden as the program is responsible for
      drawing its own items.  Note: the row height is always 12 pixels.
*/

extern void PLISTER_penDownAct(VOID_PTR thisView, unsigned char x,
                               unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      x		x coordinate of pen
      y		y coordinate of pen
   Return Value:
      None
   Description:
      Default penDownAct handler for PLISTER.  Called when the Lister
      receives an evPenDown event message.
*/

extern VOID_PTR CreateLister(unsigned char ID, unsigned char left,
                             unsigned char top, unsigned char right,
                             unsigned char bottom, unsigned int itemSum);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      right	Right screen coordinate
      bottom	Bottom screen coordinate
      itemSum	Number of items in list
   Return Value:
      Pointer to the newly-created Lister
   Description:
      Creates a Lister object.
*/

/************************************************/
/* PMENUFIELD Definition, Functions             */
/************************************************/

typedef struct {
    PVIEW_PTR next;		/* Pointer to the next object in the same
                                   group as this one */
    PGROUP_PTR owner;		/* Pointer to the group that owns this */
    unsigned char state;	/* State flags */
    unsigned char options;	/* Option flags */
    unsigned char left;         /* Left screen coordinate */
    unsigned char top;		/* Top  screen coordinate */
    unsigned char right;	/* Right screen coordinate */
    unsigned char bottom;	/* Bottom screen coordinate */
    void (*setState)();		/* Address of setState handler */
    BANK_NO setStateBank;	/* Bank of setState handler */
    void (*draw)();		/* Address of draw handler */
    BANK_NO drawBank;		/* Bank of draw handler */ 
    void (*handleEvent)();	/* Address of handleEvent handler */
    BANK_NO handleEventBank;	/* Bank of handleEvent handler */
    int totalRows;		/* Number of rows */
    unsigned char menuType;	/* <Reserved> */
    CONST_FAR_PTR menuFieldItems; /* Far pointer to array of menu items.  
                                     The array can have 1 to 32 elements */
    unsigned char *selectedItem;  /* Pointer to variable that holds the
                                     number of the selected item */
} PMENUFIELD;

extern void PMENUFIELD_draw(VOID_PTR thisView);
/* Parameters:
      thisView	the object the handler applies to
   Return Value:
      None
   Description:
      Default draw handler for PMENUFIELD class.  Draws the object.
*/

extern void PMENUFIELD_handleEvent(VOID_PTR thisView, unsigned int *evType,
                                   unsigned char x, unsigned char y);
/* Parameters:
      thisView	the object the handler applies to
      evType	The event type
      x		For pen-related events, x coordinate of pen
		For command events, high byte of command
      y		For pen-related events, y coordinate of pen
		For command events, low byte of command
   Return Value:
      None
   Description:
      Default event handler for PMENUFIELD.  Processes the events the object
      receives.
*/

extern VOID_PTR CreateMenuField(unsigned char ID, unsigned char left,
                                unsigned char top, unsigned char *selected,
                                CONST_FAR_PTR items, unsigned char type);
/* Parameters:
      ID	Unique ID for the object
      left	Left screen coordinate
      top	Top screen coordinate
      selected	Pointer to variable to hold currently selected item
      items	
      type	Currently unused
   Return Value:
      Pointer to the newly-created Lister
   Description:
      Creates a Lister object.
*/

#endif
