/*
 * WinampRC.c - Winamp Remote Control
 * Copyright (C) 2001 Jean-michel Bunouf (jeanmichel.bunouf@oreka.com)
 * 
 * The program is freely distributable.  
 * This means that you have permission to use, copy or distribute 
 * this software and its documentation.  You do not have permission 
 * to charge anything other than a duplication fee for distributing 
 * copies of this software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * */


#include <avsys.h>
#include <avdata.h>
#include <avgui.h>
#include "comm.h"


#define cmAbout  0x2020
#define cmPort   0x2021
#define cmLearn  0x2022

#define cmPREV    0x2001
#define cmPLAY    0x2002
#define cmNEXT    0x2003

#define cmREW     0x2004
#define cmSTOP    0x2005
#define cmFFW     0x2006

#define cmPAUSE   0x2007

#define cmREPEAT  0x2008
#define cmEQU     0x2009
#define cmVOLUP   0x2010
#define cmVOLDN   0x2011
#define cmRANDOM  0x2012

#define cmLIST1   0x2013
#define cmLIST2   0x2014
#define cmLIST3   0x2015
#define cmLIST4   0x2016
#define cmLIST5   0x2017
#define cmLIST6   0x2018


#pragma codeseg
static char WindowTitle[]="WinampRC";
static char AboutMessage[]="Winamp Remote Control\n\n Copyright (c) 2001\nJean-michel Bunouf";
#pragma defaultseg


MENUITEM MainMenu[]=
{
   {"IR port",   cmPort,  lstDisplay},
   {"Learn mode ",cmLearn, lstDisplay}, 
   {"About...",	 cmAbout, lstDisplay | lstUnderLine},
   {"Exit",      cmQuit,  lstDisplay},
   {"\0", 0, 0} 
};


int id=100;
PDESKBOX *dsk;
PMENUFIELD *MenuFld;
int port=PORT_RS232;
unsigned char Refresh=0;
unsigned char Learn=(1==0);

void UpdateMenu()
{
	if (port==PORT_RS232)
		strcpy(MainMenu[0].item,"IR port");
	else
		strcpy(MainMenu[0].item,"Serial port");

	if (Learn)
		strcpy(MainMenu[1].item,"Normal mode");
	else
		strcpy(MainMenu[1].item,"Learn mode");
}


void LoadPreferences()
{
	/* load application preferences */
	if (GetPreferenceLen(CurrentAppID))
		ReadPreference(CurrentAppID, 0, &port, sizeof(port));
	else
	{		
		port=PORT_RS232;
	};
	UpdateMenu();
}


void SavePreferences()
{
	/* write out the application preferences */
	SavePreference(CurrentAppID, &port, sizeof(port));
}


void SendCommand(char *command)
{
	InitComm( port, BAUD_9600);
	
	do
	{ 		
		SendOneByte(*command);
	} 
	while (*command++ != 0);

	Delay_10ms();
	Delay_10ms();
	Delay_10ms();
	Delay_10ms();
	Delay_10ms();

	CloseComm();
}


void SendCommandAndReceive(char *command)
{
	int ch=0;
	int loop=0;
	unsigned char PositionX=10;
	unsigned char PositionY=41;

	InitComm( port, BAUD_9600);
	DisableCommAutoWakeup();

	SetFontType(PRPFONT11B);
	ClearLCDArea(9,40,151,94,0);
	
	do
	{ 		
		SendOneByte(*command);
	} 
	while (*command++ != 0);
	
	do
	{
		if((ch=ReceiveOneByte())>0)
        {
			PositionX = WriteChar( PositionX,PositionY,ch,0);
			if (PositionX>141) 
			{
				PositionX=10;
				PositionY+=11;	
			}

        }
    }
	while (ch!=0 & loop++<=400);
	
	Delay_10ms();
	Delay_10ms();
	Delay_10ms();
	Delay_10ms();
	Delay_10ms();

	CloseComm();	
	EnableCommAutoWakeup();
    
}



void dsk_draw(VOID_PTR view)
{ 
	PDESKBOX_draw(view); 
	
	SetLineSize(1);
	DrawLine(8,27,151,27,DRAW_BLACK);
	DrawLine(8,30,151,30,DRAW_BLACK);
	DrawLine(8,33,151,33,DRAW_BLACK);
	SetFontType(PRPFONT11B);
	WriteString(40,25,(char *)" WINAMP INFO ",0);

	SetLineSize(0);
	DrawLine(1,104,158,104,DRAW_GRAY);
	DrawLine(1,160,158,160,DRAW_GRAY);
	DrawLine(1,198,158,198,DRAW_GRAY);

	DrawRect (8,38,152,95,DRAW_BLACK);	

}	


void dsk_handleEvent(VOID_PTR view, unsigned int *evType, unsigned char x, unsigned char y)
{  
	PDESKBOX_handleEvent(view,evType,x,y); 
	
	if ((*evType==evTimer) & (!Learn))
	{
		if (Refresh++ == 5)
		{
			Refresh=0;
			SendCommandAndReceive((char *)"SONG INFO");
		}
	}

	if (*evType == evKeyDown)
	{	
		switch(x)
   		{
		case PGUP :
			SendCommand((char *)"PREV");	
			break;
		case PGDN :
			SendCommand((char *)"NEXT");
			break;
		}
	}
	
	if (*evType == evCommand)
	{
   		switch(TOWORD(x,y))
   		{
		case cmPort :
			if (port==PORT_RS232) 
				port=PORT_IR;
			else
				port=PORT_RS232;
			UpdateMenu();
			break;
		case cmLearn :
			Learn=!Learn;
			UpdateMenu();
			break;
		case cmAbout :
		        MessageBox(AboutMessage, mfInformation | mfOKButton);
		       	break;
		case cmPREV :		
			SendCommand((char *)"PREV");			
			break;
		case cmPLAY :		
			SendCommand((char *)"PLAY");			
			break;
		case cmNEXT :		
			SendCommand((char *)"NEXT");			
			break;
		case cmREW :		
			SendCommand((char *)"REW");			
			break;
		case cmSTOP :		
			SendCommand((char *)"STOP");			
			break;
		case cmFFW :		
			SendCommand((char *)"FFW");			
			break;
		case cmPAUSE :		
			SendCommand((char *)"PAUSE");			
			break;
		case cmREPEAT :		
			SendCommand((char *)"REPEAT ON/OFF");			
			break;
		case cmVOLUP :		
			SendCommand((char *)"VOLUME UP");			
			break;
		case cmVOLDN :		
			SendCommand((char *)"VOLUME DOWN");			
			break;
		case cmEQU :		
			SendCommand((char *)"EQUALIZER ON/OFF");			
			break;
		case cmRANDOM :		
			SendCommand((char *)"RANDOM ON/OFF");			
			break;
		case cmLIST1 :		
			SendCommand((char *)"LIST 1");			
			break;
		case cmLIST2 :		
			SendCommand((char *)"LIST 2");			
			break;
		case cmLIST3 :		
			SendCommand((char *)"LIST 3");			
			break;
		case cmLIST4 :		
			SendCommand((char *)"LIST 4");			
			break;
		case cmLIST5 :		
			SendCommand((char *)"LIST 5");			
			break;
		case cmLIST6 :		
			SendCommand((char *)"LIST 6");			
			break;
		default :
			break;
		}
   	}
}


int main()
{

	LoadPreferences();

	dsk = (PDESKBOX *)CreateDeskBox(id++, 0, 0, 159, 239, 
	      	MK_FAR_PTR(WindowTitle),MK_FAR_PTR(MainMenu), NULL_FP,
      		bafClose | bafDotTitle); 

	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 3, 106, 52, 122, MK_FAR_PTR((char *)"|\003"), cmPREV, bttBasic));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 55, 106, 104, 122, MK_FAR_PTR((char *)"\004"), cmPLAY, bttBasic));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 107, 106, 156, 122, MK_FAR_PTR((char *)"\004|"), cmNEXT, bttBasic));	

	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 3, 124, 52, 140, MK_FAR_PTR((char *)"\003\003"), cmREW, bttBasic));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 55, 124, 104, 140, MK_FAR_PTR((char *)"\016"), cmSTOP, bttBasic));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 107, 124, 156, 140, MK_FAR_PTR((char *)"\004\004"), cmFFW, bttBasic));	

	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 55, 142, 104, 158, MK_FAR_PTR((char *)"| | "), cmPAUSE, bttBasic));	

	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 8, 162, 68, 178, MK_FAR_PTR((char *)"Volume \005"), cmVOLUP, bttBasic));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 71, 162, 131, 178, MK_FAR_PTR((char *)"Volume \006"), cmVOLDN, bttBasic));	


	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 8, 180, 25, 196, MK_FAR_PTR((char *)""), cmREPEAT, bttRotate));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 27, 180, 87, 196, MK_FAR_PTR((char *)"Equalizer"), cmEQU, bttBasic));
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 90, 180, 150, 196, MK_FAR_PTR((char *)"Random"), cmRANDOM, bttBasic));	

	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 3, 200, 52, 216, MK_FAR_PTR((char *)"List 1"), cmLIST1, bttGoTo));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 55, 200, 104, 216, MK_FAR_PTR((char *)"List 2"), cmLIST2, bttGoTo));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 107, 200, 156, 216, MK_FAR_PTR((char *)"List 3"), cmLIST3, bttGoTo));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 3, 218, 52, 234, MK_FAR_PTR((char *)"List 4"), cmLIST4, bttGoTo));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 55, 218, 104, 234, MK_FAR_PTR((char *)"List 5"), cmLIST5, bttGoTo));	
	PGROUP_insert((VOID_PTR) dsk, (PVIEW *) 
		CreateButton( id++, 107, 218, 156, 234, MK_FAR_PTR((char *)"List 6"), cmLIST6, bttGoTo));	

	BankedAssign1(dsk->draw,dsk_draw); 
	BankedAssign2(dsk->handleEvent,dsk_handleEvent);
	ExecView((PGROUP_PTR) dsk);
	Destroy((VOID_PTR) dsk);
	
	SavePreferences();	
	
	return APPLICATION;
}
