/* .\avsdk\lib\avcgui.c

    Designed for Gerard Vermeulen's Freeware SDK
        Copyright (C)1999 Charles "Randy" Gill
           Released as licenced freeware

  Revised and extended by Kai Nothdurft (KNO) 01/2000.
       Extensions and changes are Public Domain.

  C-funktions for the AVGUI-library. Taken from AVGUI.H to separate
    prototypes from implementation, otherwise each include would
         reimplement the functions and occupy memory.

*/

#include <avgui.h>

HANDLER_PTR *_hndptr;

const FAR_PTR NULL_FP = {0,0};
const FAR_PTR *_PickerTitle = (FAR_PTR *) 0xF154;  /* KNO: now public. */

FAR_PTR _MakeFarPtr(unsigned int bank, void *addr)
{
   FAR_PTR retval;

   retval.addr = (unsigned int)addr;
   retval.bank = bank;
   return retval;
}

void SetPickerTitle(CONST_FAR_PTR NewTitle)
/* Parameters:
      NewTitle  Far pointer to the new title
   Return Value:
      None
   Description:
      Sets the default title of the system pickers.
*/
{
   *_PickerTitle = NewTitle;
}

/************************************************/
/* ExecPicker wrapper by KNO 02/2000            */
/************************************************/
extern unsigned int ExecPickerOS(char *str, unsigned int len); 
unsigned int ExecPicker(char *str, unsigned int maxLen)
/* Parameters:
      str   String to be edited
      len   Maximum length of string
   Return Value:
      Actual length of string.
   Description:
      This displays the standard/T9 keyboard picker and allows editing
      of a string.
*/
{
  unsigned int actLen;
  char *workstr;

  /* Reserve enough space for str+0x0a+0x00  */
  workstr=(char *)AllocBlock(maxLen+2,255,1);
  strncpy(workstr, str, maxLen);

  /* Add trailing 0x0a for Picker. Otherwise it behaves strange */
  actLen=strlen(workstr);
  workstr[actLen++]=0x0a;
  workstr[actLen]=0;
  ExecPickerOS(workstr, ++maxLen);

  /* Copy workstring without 0x0a back to str */
  strncpy(str, workstr, strlen(workstr)-1);

  /* free memory of workstr */
  FreeBlock((VOID_PTR)workstr);
}

/*************************************************/
/* PTEXTFIELD_handleEvent wrapper by KNO 02/2000 */
/*************************************************/
extern void PTEXTFIELD_handleEventOS(VOID_PTR thisView, unsigned short *evType, unsigned char x, unsigned char y);

void PTEXTFIELD_handleEvent(VOID_PTR thisView, unsigned short *evType, unsigned char x, unsigned char y)
{
  FAR_PTR oldPickerTitle;

  if (*evType == evPenDown)
  {
    oldPickerTitle = *_PickerTitle;
    *_PickerTitle = PTXFPickerTitle;

    PTXFText[strlen(PTXFText)+1]=0;
    PTXFText[strlen(PTXFText)]=0x0a;

    PTEXTFIELD_handleEventOS(thisView, evType, x, y);

    PTXFText[strlen(PTXFText)-1]=0;
    *_PickerTitle = oldPickerTitle;
  }
  else
    PTEXTFIELD_handleEventOS(thisView, evType, x, y);
}

/*************************************************/
/* CreateTextFieldOS wrapper by KNO 02/2000      */
/*************************************************/
extern VOID_PTR CreateTextFieldOS(unsigned char ID,
                                  unsigned char left, unsigned char top,
                                  unsigned char right, unsigned char bottom,
                                  unsigned char *text, unsigned char fontType,
                                  unsigned char rowHig, short lengthLimit,
                                  CONST_FAR_PTR pickerTitle);
VOID_PTR CreateTextField(unsigned char ID,
                         unsigned char left, unsigned char top,
                         unsigned char right, unsigned char bottom,
                         unsigned char *text, unsigned char fontType,
                         unsigned char rowHig, short lengthLimit,
                         CONST_FAR_PTR pickerTitle)
{
  PTEXTFIELD *textfield;
  textfield=(PTEXTFIELD *)CreateTextFieldOS(ID, left, top, right, bottom, text,
                                            fontType, rowHig, lengthLimit, pickerTitle);

/* The following is quick and dirty, but working. CreateTextFieldOS doesn't allocate
   memory for pickerTitle. A subsequent call to AllocBlock will reserve memory directly
   behind the block allocated within CreateTextfieldOS. I give the block an ID of 255,
   which it lost. A call to FreeBlock for '(VOID_PTR) textfield' won't free the memory
   for the PickerTitle, but a call to FreeBlock for the block, which is allocated with
   CreateDeskBox, since CreateDeskBox uses stacked mode, instead of random mode.
*/
  AllocBlock(255,sizeof(CONST_FAR_PTR),0);
  textfield->pickerTitle.bank=pickerTitle.bank;
  textfield->pickerTitle.addr=pickerTitle.addr;
  BankedAssign2(textfield->handleEvent, PTEXTFIELD_handleEvent);
  return textfield;
}

/***************************************************/
/* PNUMBERFIELD_handleEvent wrapper by KNO 02/2000 */
/***************************************************/
extern void PNUMBERFIELD_handleEventOS(VOID_PTR thisView, unsigned int *evType,
                                       unsigned char x, unsigned char y);

void PNUMBERFIELD_handleEvent(VOID_PTR thisView, unsigned int *evType,
                              unsigned char x, unsigned char y)
{
  FAR_PTR oldPickerTitle;

  if (*evType == evPenDown)
  {
    oldPickerTitle = *_PickerTitle;
    *_PickerTitle = PNMFPickerTitle;

    PNUMBERFIELD_handleEventOS(thisView, evType, x, y);

    *_PickerTitle = oldPickerTitle;
  }
  else
    PNUMBERFIELD_handleEventOS(thisView, evType, x, y);
}

/***************************************************/
/* CreateNumberFieldOS wrapper by KNO 02/2000      */
/***************************************************/
extern VOID_PTR CreateNumberFieldOS(unsigned char ID, unsigned char left, 
                                    unsigned char top, unsigned char *numeric,
                                    unsigned char numType, CONST_FAR_PTR pickerTitle);
VOID_PTR CreateNumberField(unsigned char ID, unsigned char left, 
                           unsigned char top, unsigned char *numeric,
                           unsigned char numType, CONST_FAR_PTR pickerTitle)
{
  PNUMBERFIELD *numberfield;
  numberfield=(PNUMBERFIELD *)CreateNumberFieldOS(ID, left, top, numeric, numType, pickerTitle);

/* The following is quick and dirty, but working. CreateTextFieldOS doesn't allocate
   memory for pickerTitle. A subsequent call to AllocBlock will reserve memory directly
   behind the block allocated within CreateTextfieldOS. I give the block an ID of 255,
   which it lost. A call to FreeBlock for '(VOID_PTR) numberfield' won't free the memory
   for the PickerTitle, but a call to FreeBlock for the block, which is allocated with
   CreateDeskBox, since CreateDeskBox uses stacked mode, instead of random mode.
*/
  AllocBlock(255,sizeof(CONST_FAR_PTR),0);
  numberfield->pickerTitle.bank=pickerTitle.bank;
  numberfield->pickerTitle.addr=pickerTitle.addr;
  BankedAssign2(numberfield->handleEvent, PNUMBERFIELD_handleEvent);
  return numberfield;
}


/***************************************************/
/* PDATEFIELD_handleEvent wrapper by KNO 02/2000   */
/***************************************************/
extern void PDATEFIELD_handleEventOS(VOID_PTR thisView, unsigned int *evType,
                                     unsigned char x, unsigned char y);

void PDATEFIELD_handleEvent(VOID_PTR thisView, unsigned int *evType,
                              unsigned char x, unsigned char y)
{
  FAR_PTR oldPickerTitle;

  if (*evType == evPenDown)
  {
    oldPickerTitle = *_PickerTitle;
    *_PickerTitle = PDTFPickerTitle;

    PDATEFIELD_handleEventOS(thisView, evType, x, y);

    *_PickerTitle = oldPickerTitle;
  }
  else
    PDATEFIELD_handleEventOS(thisView, evType, x, y);
}

/***************************************************/
/* CreateDateFieldOS wrapper by KNO 02/2000        */
/***************************************************/
extern VOID_PTR CreateDateFieldOS(unsigned char ID, unsigned char left,
                                    unsigned char top, unsigned char *selectedDate,
                                    CONST_FAR_PTR pickerTitle);
VOID_PTR CreateDateField(unsigned char ID, unsigned char left,
                           unsigned char top, unsigned char *selectedDate,
                           CONST_FAR_PTR pickerTitle)
{
  PDATEFIELD *datefield;
  datefield=(PDATEFIELD *)CreateDateFieldOS(ID, left, top, selectedDate, pickerTitle);

/* The following is quick and dirty, but working. CreateTextFieldOS doesn't allocate
   memory for pickerTitle. A subsequent call to AllocBlock will reserve memory directly
   behind the block allocated within CreateTextfieldOS. I give the block an ID of 255,
   which it lost. A call to FreeBlock for '(VOID_PTR) datefield' won't free the memory
   for the PickerTitle, but a call to FreeBlock for the block, which is allocated with
   CreateDeskBox, since CreateDeskBox uses stacked mode, instead of random mode.
*/
  AllocBlock(255,sizeof(CONST_FAR_PTR),0);
  datefield->pickerTitle.bank=pickerTitle.bank;
  datefield->pickerTitle.addr=pickerTitle.addr;
  BankedAssign2(datefield->handleEvent, PDATEFIELD_handleEvent);
  return datefield;
}

/***************************************************/
/* PTIMEFIELD_handleEvent wrapper by KNO 02/2000   */
/***************************************************/
extern void PTIMEFIELD_handleEventOS(VOID_PTR thisView, unsigned int *evType,
                                     unsigned char x, unsigned char y);

void PTIMEFIELD_handleEvent(VOID_PTR thisView, unsigned int *evType,
                              unsigned char x, unsigned char y)
{
  FAR_PTR oldPickerTitle;

  if (*evType == evPenDown)
  {
    oldPickerTitle = *_PickerTitle;
    *_PickerTitle = PTMFPickerTitle;

    PTIMEFIELD_handleEventOS(thisView, evType, x, y);

    *_PickerTitle = oldPickerTitle;
  }
  else
    PTIMEFIELD_handleEventOS(thisView, evType, x, y);
}

/***************************************************/
/* CreateTimeFieldOS wrapper by KNO 02/2000        */
/***************************************************/
extern VOID_PTR CreateTimeFieldOS(unsigned char ID, unsigned char left,
                                    unsigned char top, unsigned char *selectedTime,
                                    CONST_FAR_PTR pickerTitle);
VOID_PTR CreateTimeField(unsigned char ID, unsigned char left,
                           unsigned char top, unsigned char *selectedTime,
                           CONST_FAR_PTR pickerTitle)
{
  PTIMEFIELD *timefield;
  timefield=(PTIMEFIELD *)CreateTimeFieldOS(ID, left, top, selectedTime, pickerTitle);

/* The following is quick and dirty, but working. CreateTextFieldOS doesn't allocate
   memory for pickerTitle. A subsequent call to AllocBlock will reserve memory directly
   behind the block allocated within CreateTextfieldOS. I give the block an ID of 255,
   which it lost. A call to FreeBlock for '(VOID_PTR) timefield' won't free the memory
   for the PickerTitle, but a call to FreeBlock for the block, which is allocated with
   CreateDeskBox, since CreateDeskBox uses stacked mode, instead of random mode.
*/
  AllocBlock(255,sizeof(CONST_FAR_PTR),0);
  timefield->pickerTitle.bank=pickerTitle.bank;
  timefield->pickerTitle.addr=pickerTitle.addr;
  BankedAssign2(timefield->handleEvent, PTIMEFIELD_handleEvent);
  return timefield;
}

/************************************************/
/* PLABEL extensions by KNO 01/2000             */
/************************************************/

/*
   IsValidRange. Used by PLABEL_draw, to check if coordinates make sense.

   Parameters:
      left      upper left corner x
      top       upper left corner y
      right     lower right corner x
      bottom    lower right corner y
   Return Value:
      unsigned char (bool). 1=boundaries OK, 0=boundaries out of range.
   Description:
      IsValidRange checks if boundaries are on screen

 */
unsigned char IsValidRange(unsigned char left, unsigned char top, unsigned char right, unsigned char bottom)
{
    if (left > right || top > bottom)
        return 0;
    if (GetDisplayDirection())
    {
        if (right>=LCD_HEIGHT || bottom>=LCD_WIDTH)
            return 0;
    }
    else
    {
        if (right>=LCD_WIDTH || bottom>=LCD_HEIGHT)
            return 0;
    }
    return 1;
}

void PLABEL_setState(VOID_PTR thisView, unsigned char state, char enable)
{
  PVIEW_setState(thisView, state, enable);
}

void PLABEL_draw(VOID_PTR thisView)
{
  PLABEL *label;
  char *caption;
  unsigned char oldFont,
                stringLength,
                maxLength,
                x,
                align;


  PVIEW_draw(thisView);
  label = (PLABEL *)thisView;

  caption = AllocBlock(lbMaxLen+1,255,1);
  fstrcpy( caption, label->labelText );
  if ((label->labelOptions) & lbofDefineFont)
  {
    oldFont = GetFontType();
    SetFontType(label->labelFont);
    stringLength = GetStringLength(caption, label->labelFont);
  }
  else
    stringLength = GetStringLength(caption, GetFontType());

  align = ALIGN_LEFT;
  maxLength = (label->view).right - (label->view).left;

  if ((label->labelOptions) & lbofAlignCenter)
  {
    x = (label->view).left + (maxLength-stringLength)/2;
  }
  else
  {
    if ((label->labelOptions) & lbofAlignRight)
    {
      align = ALIGN_RIGHT;
      x = (label->view).right;
    }
    else
      x = (label->view).left;
  }
  
  WriteAlignString(x, (label->view).top, caption, maxLength, align, 0);
  FreeBlock((VOID_PTR)caption);
  if (!(label->labelOptions) & lbofDefineFont)
  {
    SetFontType(oldFont);
  }
}

void PLABEL_handleEvent(VOID_PTR thisView, unsigned int *evType, unsigned char x, unsigned char y)
{
  PVIEW_handleEvent(thisView, evType, x, y);
}

VOID_PTR CreateLabel(unsigned char ID,
                     unsigned char left, unsigned char top,
                     unsigned char right, unsigned char bottom,
                     CONST_FAR_PTR caption,
                     unsigned char options, unsigned char font)
{
  PLABEL *label;
  
  if (!IsValidRange(left, top, right, bottom))
      return (VOID_PTR)0;

  /* Allocate memory block for the new object */
  label = (PLABEL *)AllocBlock(sizeof(PLABEL), ID, 0); /* 0=Random mode memory allocation. */

  if (!label)
      return (VOID_PTR)0;

  /* initialize the members of PVIEW */
  (label->view).next    = (PVIEW_PTR)0;   /* Pointer to the next object in the same
                                             group as this one */
  (label->view).owner   = (PGROUP_PTR)0;  /* Pointer to the group that owns this */
  (label->view).state   = 0;              /* State flags */
  (label->view).options = 0;              /* Option flags */
  (label->view).left    = left;           /* Left screen coordinate */
  (label->view).top     = top;            /* Top  screen coordinate */
  (label->view).right   = right;          /* Right screen coordinate */
  (label->view).bottom  = bottom;         /* Bottom screen coordinate */

  BankedAssign2((label->view).setState, PLABEL_setState);       /* Address of setState handler */
  BankedAssign1((label->view).draw, PLABEL_draw);               /* Address of draw handler */
  BankedAssign2((label->view).handleEvent, PLABEL_handleEvent); /* Address of handleEvent handler */


  /* initialize the members of PLABEL */
  label->labelOptions = options;
  label->labelFont = font;
  label->labelText = caption;

  return (VOID_PTR) label;
}
