/* $Header: access.c,v 6.1 86/08/18 13:14:26 peter Exp $ */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

#include <errno.h>
#include "cdb.h"

/*
 * This file contains routine to read and write information
 * from the child process.  As it currently stands, this is done
 * by calling `Ptrace'.  This will change when we get farther on
 * in the remote debuging/multi-process stuff.
 */

#define cbCacheMax	2048
char		vsbCommCache[cbCacheMax];
ADRT		vadrCacheMin, vadrCacheMax; /* the address range in the cache */
export FLAGT	vfCacheValid = false;


/* G E T   U S E R */

export REGT GetUser(adr)
ADRT    adr;
{
    int4	ret;
    REGT	val;

    /* Read information from the `user' structure */

    if (adr == adrNil)
	return(0);

    if (v->pid != pidNil) {

	ret = Ptrace(ptReadUser, v->pid, (char *)adr, 0);
	if ((ret == -1) AND (errno == EIO))
	    SysPanic("Failed on ptReadUser - %s.", SbFAdr(adr, true));
	val = ret;

    } else if (v->fnCore != fnNil) {

	if (lseek(v->fnCore, adr, seekBegin) < 0L)
	    UError("Address (%s) not found.", SbFAdr(adr, true));
	if (read(v->fnCore, &val, v->cpu->cbPointer) < 1)
	    UError("Cannot read location (%s).", SbFAdr(adr, true));

    } else {

	UError("No child process AND no core file.");

    } /* if */

    return(val);
} /* GetUser */


/* P U T   U S E R */

export void PutUser(adr, val)
ADRT    adr;
REGT    val;
{
    int2	i;

    /* Write information to the `user' structure */

    if (adr == adrNil)
	return;

    if (v->pid != pidNil) {

	i = Ptrace(ptWriteUser, v->pid, (char *)adr, val);
	if ((i == -1) AND (errno == EIO))
	    SysPanic("Failed on ptWriteUser - %s, %d.", SbFAdr(adr, true), val);

    } else if (v->fnCore != fnNil) {

	if (lseek(v->fnCore, adr, seekBegin) < 0L)
	    UError("Address (%s) not found.", SbFAdr(adr, true));
	if (write(v->fnCore, &val, v->cpu->cbPointer) < 1 )
	    UError("Cannot write location (%s).", SbFAdr(adr, true));

    } else {

	UError("No child process AND no core file.");

    } /* if */
} /* PutUser */


/* G E T   W O R D */

export int GetWord(adr, space)
ADRT	adr;
int2	space;
{
    /* Read an `int' from the child */

    return(Access(accRead, adr, space, 0));
} /* GetWord */


/* P U T   W O R D */

export void PutWord(adr, space, val)
ADRT	adr;
int2	space;
int	val;
{
    /* Write an `int' to the child */

    Access(accWrite, adr, space, val);
} /* PutWord */


/* S E T   C O M M   C A C H E */

export void SetCommCache(adrSrc, cb)
ADRT	adrSrc;
int	cb;
{
    int		cbCache, cbSlop, cbMove;
    ADRT	adrMax;
    
    if (!vfUseCache)
	return;

    /* This cache is used for READING large blocks from the
     * child process.  This can substantially reduce the communications
     * overhead during stack traces and other large read operations.
     * The cache also has write-through capabilities, but this
     * is simply to help with the read operations.
     */

    if (cb > cbCacheMax)
	cb = cbCacheMax;

    adrMax = adrSrc + cb;

    if (!vfCacheValid)
	vadrCacheMin = vadrCacheMax = 0;

    cbCache = vadrCacheMax - vadrCacheMin;

    /* see if we already have part of this in memory */

    if ((adrSrc >= vadrCacheMin) AND (adrSrc < vadrCacheMax)) {
	/* we have the beginning of it */
	if (adrMax <= vadrCacheMax)
	    return;	/* it's ALL in the cache already */

	cbSlop = adrMax - vadrCacheMax; /* amount NOT in cache */
	cbMove = cbSlop + (cbCache - cbCacheMax); /* cache motion needed */

	if (cbMove > 0) {
	    /* fix it so there is room to read cbSlop bytes */
	    MoveBytes(vsbCommCache, vsbCommCache+cbMove, cbCache-cbMove);
	    vadrCacheMin += cbMove;
	} /* if */

	/* fill in end of buffer */
	if (v->fnComm == fnNil)
	    GetBlock(vadrCacheMax, spaceData,
			(ADRT)(vsbCommCache+cbCache), cbSlop);
	else
	    XBlockTransfer(v->pid, (ADRT)(vsbCommCache + cbCache),
			vadrCacheMax, cbSlop, false);
	vadrCacheMax += cbSlop;

    } else if ((adrMax > vadrCacheMin) AND (adrMax <= vadrCacheMax)) {

	/* we have the end of it */
	if (adrSrc >= vadrCacheMin)
	    return;	/* it's ALL in the cache already */

	cbSlop = vadrCacheMin - adrSrc; /* amount NOT in cache */
	cbMove = cbSlop + (cbCache - cbCacheMax); /* cache motion needed */

	if (cbMove > 0) {
	    /* fix it so there is room to read cbSlop bytes */
	    MoveBytes(vsbCommCache+cbMove, vsbCommCache, cbCache-cbMove);
	    vadrCacheMax -= cbMove;
	} /* if */

	vadrCacheMin = adrSrc;
	/* fill in head of buffer */
	if (v->fnComm == fnNil)
	    GetBlock(adrSrc, spaceData, (ADRT)vsbCommCache, cbSlop);
	else
	    XBlockTransfer(v->pid, (ADRT)vsbCommCache, adrSrc, cbSlop, false);

    } else {

	/* total miss - read it in */
	if (v->fnComm == fnNil)
	    GetBlock(adrSrc, spaceData, (ADRT)vsbCommCache, cb);
	else
	    XBlockTransfer(v->pid, (ADRT)vsbCommCache, adrSrc, cb, false);
	vadrCacheMin = adrSrc;
	vadrCacheMax = adrSrc + cb;
    } /* if */

    vfCacheValid = true;
} /* SetCommCache */


/* R E A D   D A T A */

export void ReadData(adrDest, adrSrc, cb)
ADRT	adrDest, adrSrc;
int2	cb;
{
    /* Block read from data space */

    GetBlock(adrSrc, spaceData, adrDest, cb);
} /* ReadData */


/* W R I T E   D A T A */

export void WriteData(adrDest, adrSrc, cb)
ADRT	adrDest, adrSrc;
int2	cb;
{
    /* Block write to data space */

    PutBlock(adrDest, spaceData, adrSrc, cb);
} /* WriteData */


/* B L O C K   T R A N S F E R */

export void BlockTransfer(adrDest, adrSrc, space, cb)
ADRT	adrDest, adrSrc;
int2	space, cb;
{
#define cbMax        1024
    int2		cbMove;
    char	buf[cbMax], *sbT = buf;

    /* Move an arbitrary block ->within<- the child */

    while (cb > 0) {
	cbMove = Min(cb, cbMax);
	GetBlock(adrSrc, space, (ADRT)sbT, cbMove);
	PutBlock(adrDest, space, (ADRT)sbT, cbMove);
	cb -= cbMove;
	sbT += cbMove;
    } /* while */
} /* BlockTransfer */


/* A C C E S S */

export int Access(accRequest, adr, space, value)
int2	accRequest, space;
int	value;
ADRT    adr;
{
    int		ret, val, pt, fn;
    int2	*pshort;
    int4	*plong;
    ADRT	adrTemp;
    FLAGT       fRead;
    pMAPR	map;

    /* Read or Write to the Data or Text space of the child */

    fRead = (accRequest == accRead);
    if ((v->pid != pidNil)		/* live process */
       AND ((space != spaceText)	/* data area OR */
	  OR (! v->fPureText)) ) {	/* corruptible text */

	/* we get the info from the process */

	if (space == spaceData) {
	    pt = (fRead) ? ptReadD : ptWriteD;  /* data space */
	} else {
	    pt = (fRead) ? ptReadI : ptWriteI;  /* instruction space */
	    adr += v->adrText;	/* add in offset of text space */
	} /* if */

	/* try to avoid going over comm link */
	if (vfUseCache			/* They want us to use cache */
	   AND vfCacheValid		/* and it is correct */
	   AND (space == spaceData)	/* We only cache data space */
	   AND (adr >= vadrCacheMin)
	   AND ((adr+v->cpu->cbInt) <= vadrCacheMax)) {

	    /* it is in the cache */
	    if (v->cpu->cbInt == 2) {
		pshort = (int2 *) (vsbCommCache + (adr - vadrCacheMin));
		if (fRead)
		    return(*pshort);
		else
		    *pshort = value;
	    } else {
		plong = (int4 *) (vsbCommCache + (adr - vadrCacheMin));
		if (fRead)
		    return(*plong);
		else
		    *plong = value;
	    } /* if */
	} /* if in cache */

	errno = 0;
	val = Ptrace(pt, v->pid, (char *)adr, value);

	if ((val == -1) AND (errno == EIO)) {
	    if (v->ps == psRunning)
	       UError("Cannot read data from a running process: %d.",
			IprFPr(v));

	    if (vfInDisplay) {
		vfBadAddress = true;
		UError(sbNil);	/* let DisplaySe do it's own thing */
	    } /* if */
	    UError("Cannot %s location (%s).",
		((fRead) ? "read" : "write"), SbFAdr(adr, true));
	} /* if */
	return(val);
    } /* if */

    /* we do not (or cannot) get it from the process.
     * Get it from a file.
     */

    val = 0;

    if ((space == spaceData)
       AND (v->fnCore != fnNil)
       AND ((adrTemp = AdrFPurify(adr, false)) != adrNil) ) {
	fn = v->fnCore;
	adr = adrTemp;
    } else {
	adr = AdrFPurify(adr, true);
	fn = v->fnExec;
    } /* if */

    if (adr == adrNil)
	UError("Address not found (%s).", SbFAdr(adr, true));

    if (lseek(fn, adr, seekBegin) < 0L)
	UError("Address not found (%s).", SbFAdr(adr, true));

    ret = (fRead) ? read(fn, &val, v->cpu->cbInt) 
		  : write(fn, &value, v->cpu->cbInt);

    if (ret != v->cpu->cbInt)
	UError("Cannot %s location (%s): %s.",
		((fRead) ? "read" : "write"),
		SbFAdr(adr, true), SbFError(errno));
    return(val);
} /* Access */


/* A D R   F   P U R I F Y */

export ADRT AdrFPurify(adr, fExec)
ADRT	adr;
FLAGT	fExec;
{
#define FInBounds(x, low, high) ((x>=low) && (x<high))
    pMAPR	map;

    /* We first make sure that the appropriate file is open,
     * then we use the map to move the virutal address to the
     * correct file offset.
     */

    if (fExec) {
	if (v->fnExec == fnNil) {
	    v->fnExec = open(v->sbExec, O_RDONLY);
	    if (v->fnExec == fnNil)
		Panic("Cannot open %s: %s.", v->sbExec, SbFError(errno));
	} /* if */
	map = &v->mapExec;
    } else {
	if (v->fnCore == fnNil) {
	    if (v->sbCore == sbNil)
		return(adrNil);
	    v->fnCore = open(v->sbExec, O_RDONLY);
	    if (v->fnCore == fnNil) {
		printf("Cannot open %s: %s\n", v->sbCore, SbFError(errno));
		free(v->sbCore);
		v->sbCore = sbNil;
		UError(sbNil);
	    } /* if */
	} /* if */
	map = &v->mapCore;
    } /* if */

    if (FInBounds(adr, map->b1, map->e1)) {
	adr += map->f1 - map->b1;
    } else if (FInBounds(adr, map->b2, map->e2)) {
	adr += map->f2 - map->b2;
    } else {
	adr = adrNil;
    } /* if */
    return(adr);
} /* AdrFPurify */


/* D O   M A P */

export void DoMap()
{
    int		i;
    char        ch;
    ADRT        *padr, adrT;
    TKE         tk;
    SER		ase;

    tk = TkNext();
    if (tk != tkNil) {
	/* they want to modify the map in some way */
	ch = vsbTok[0];
	if ( (tk != tkStr)
	   OR (ch != 'E' AND ch != 'C') ) {
	    UError("usage: M (E|C) [exp] ; [exp] ; .. (up to 6 expressions).");
	} else {
	    padr = (ch == 'E') ? &(v->mapExec.b1) : &(v->mapCore.b1);
	    TkNext();
	} /* if */

	for (i=0; i<6; i++, padr++) {
	    if (vtk == tkSemi) {
		TkNext();
		continue;
	    } /* if */
	    adrT = LongFExpr(&ase, vtk);
	    if (ase.asym.st == stNil)
		break;
	    *padr = adrT;
	    if (vtk == tkSemi)
		TkNext();
	} /* for */
    } /* if */

    if (v->sbExec != sbNil) {
	printf("Executable file (%s):\n", v->sbExec);
	printf(" %10s", SbFAdr(v->mapExec.b1, false));
	printf(" %10s", SbFAdr(v->mapExec.e1, false));
	printf(" %10s\n", SbFAdr(v->mapExec.f1, false));
	printf(" %10s", SbFAdr(v->mapExec.b2, false));
	printf(" %10s", SbFAdr(v->mapExec.e2, false));
	printf(" %10s\n", SbFAdr(v->mapExec.f2, false));
    } /* if */

    if (v->sbCore != sbNil) {
	printf("\nCore file (%s):\n", v->sbCore);
	printf(" %10s", SbFAdr(v->mapCore.b1, false));
	printf(" %10s", SbFAdr(v->mapCore.e1, false));
	printf(" %10s\n", SbFAdr(v->mapCore.f1, false));
	printf(" %10s", SbFAdr(v->mapCore.b2, false));
	printf(" %10s", SbFAdr(v->mapCore.e2, false));
	printf(" %10s\n", SbFAdr(v->mapCore.f2, false));
    } /* if */
} /* DoMap */
