/* $Header */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

/* This module is the REMOTE end of the CDB Standard Remote Debugging package.
 * It is set up to run as a normal UNIX user process.  To modify it for
 * use in other environments, simply implement the functionality without
 * changing the interface.  A full description of the protocol may be
 * found in the file "rptrace.h".
 */

#include <signal.h>
#include <errno.h>
#include <stdio.h>
#include <termio.h>
#include "rptrace.h"

#define O_RDONLY	0
#define chNull '\0'
typedef char *SBT;
#define sbNil ((SBT)0)
#define true 1
#define false 0
#define FLAGT unsigned short
#define FAST register
#define AND &&
#define OR ||

extern int errno;
extern void SendStatus();

#define DEBUG 0

#if DEBUG
#define dprint(x, y) {if (x <= vlevel) fprintf y; fflush(stderr);}
#else
#define dprint(x, y)
#endif

int	vpidChild = 0;	/* KLUDGE - this limits us to one sub-process */
int	vfCommDebug;	/* if true, causes echoing of all messages */
int	vlevel = 0;		/* debug print level for dprint */


#define cArgsMax	100	/* max number of args to child */
#define cbBufMax 1024	/* pick a number suited to the size of your system */
char	vsbBuf[cbBufMax];	/* buffer where we format messages */
SBT	vsbTx = vsbBuf;

int	vfnTx, vfnRx;
char	vsbTxLast[cbMsgMax]; /* copy of last transmisson */
char	vsbRxLast[cbMsgMax]; /* copy of last recieve data */
FLAGT	vfRecieveDone;	/* true if remote sent new line rather than ack */
char	vTxSeq;		/* next sequence number to transmit */
char	vRxChk;		/* last valid check character receieved */
struct termio	vTtyOrig,vTtyTmp;

/* T I M E U P */
timeup()
{
	/* just catch an alarm and let i/o routine get EINTR */
	return;
}


/* Q U I T */
quit(stat)
int stat;
{
	/* restore tty attributes */
	ioctl(0,TCSETAF,&vTtyOrig);

	/* terminate the process */
	exit(stat);
}

/* M A I N */

void main(argc, argv)
int	argc;
char	**argv;
{
	/* tell user about special feature */
	printf("Type \"quit\\n\" to force exit\n");

	/* get current tty characteristics for restoring later */
	ioctl(0, TCGETA, &vTtyOrig);

	/* catch all signals so that tty gets restored */
	signal(SIGHUP, quit);
	signal(SIGINT, SIG_IGN);
	signal(SIGQUIT, quit);
	signal(SIGILL, quit);
	signal(SIGIOT, quit);
	signal(SIGEMT, quit);
	signal(SIGFPE, quit);
	signal(SIGBUS, quit);
	signal(SIGSEGV, quit);
	signal(SIGSYS, quit);
	signal(SIGPIPE, quit);
	signal(SIGTERM, quit);
	signal(SIGUSR1, quit);
	signal(SIGUSR2, quit);

	/* change tty to not echo */
	vTtyTmp = vTtyOrig;
	vTtyTmp.c_oflag &= ~OPOST;
	vTtyTmp.c_lflag = ICANON;
	vTtyTmp.c_cc[VERASE] = 0;
	vTtyTmp.c_cc[VKILL] = 0;
	vTtyTmp.c_cc[VEOF] = 0;
	vTtyTmp.c_cc[VEOL] = 0;
	ioctl(0,TCSETAF,&vTtyTmp);

	/* do the processing loop */
	vfnRx = 0;	/* standard in */
	vfnTx = 1;	/* standard out */
	DebugServer();

	/* all done */
	quit(0);
} /* main */

/* R E A D   S B */

ReadSb(sb, cb)
int	cb;
SBT	sb;
{
    FAST int	i, ret;

	/* if WriteSb found our message return it now */
	if (vfRecieveDone) {
		vfRecieveDone = false;
		strncpy(sb,vsbRxLast+2,cb-1);
		sb[cb-1] = 0;
		return(strlen(sb));
	}

	/* loop reading until a valid string is received */
	for(;;) {
		ret = read(vfnRx, vsbRxLast, sizeof(vsbRxLast));
		if (ret <= 0)
	    		RemotePanic("Read from Remote on %d.", vfnRx);
		
		/* if this is a proper request, ack and return results */
		if (fFsbValidMsg(vsbRxLast,vRxChk)) {
			char sbAck[2];
			vRxChk = sbAck[0] = vsbRxLast[0];
			sbAck[1] = '\n';
			write(vfnTx, sbAck, 2);
			strncpy(sb,vsbRxLast+2,cb-1);
			sb[cb-1] = 0;
			return(strlen(sb));
		}

		/* if this is a resync request then ignore sequence bit */
		if (strncmp(vsbRxLast,"~~~",3) == 0) {
			vRxChk = 0;
			continue;
		}

		/* no good, so send a NAK and try again */
		write(vfnTx,"~\n",2);

		/* if this is a quit request then exit */
		if(strncmp(vsbRxLast,"quit\n",4) == 0) quit(1);
	}

} /* ReadSb */


/* W R I T E   S B */

WriteSb(sb)
FAST SBT	sb;
{
    FAST int	i, ret;
    FAST char ch;
    FAST SBT sbTx;
    int AlarmTime = 2;
    int sum = 0;
    int cnt = 0;
    char sbAck[sizeof(vsbRxLast)];

	/* save the string in vsbTxLast and compute the check byte */
	sbTx = vsbTxLast+2;
	while((ch = *sb++) != '\n') {
		if (ch < ' ' || ch > '~')
			RemotePanic("WriteSb: Attempt to send non-printable char");
		sum += ch;
		*sbTx++ = ch;
		cnt++;
	}
	*sbTx++ = '\n';
	*sbTx = 0;
	vsbTxLast[0] = sum%47 + vTxSeq + ' ';
	vsbTxLast[1] = cnt + ' ';
	cnt += 3;

	/* advance the sequence number for the next character */
	vTxSeq = vTxSeq ? 0 : 47;

	/* loop transmitting until a valid ack is recieved */
	i = 0;
	for(;;) {
		
		/* after too many errors try to resync the sequence bit */
		if (++i > 10) {
			write(vfnTx,"~~~\n",4);
			i = 0;
		}
		
		/* flush out any extraneous garbage in input buffer */
		ioctl(vfnRx, TCFLSH, 0);

		/* send the message */
		ret = write(vfnTx, vsbTxLast, cnt);
		if (cnt != ret)
			RemotePanic("WriteSb: %d bytes to go, %d went.", cnt, ret);

		/* get an ack or timeout */
		signal(SIGALRM, timeup);
		alarm(AlarmTime);
		ret = read(vfnRx, sbAck, sizeof(sbAck));
		alarm(0);

		/* check for error */
		if (ret < 0) {
			if (errno != EINTR)
				RemotePanic("WriteSb: error reading ack");

			/* try again but give a longer timeout */
			AlarmTime += 2;
			continue;
		}

		/* if we got a valid ack then we are done */
		if (ret == 2 && sbAck[0] == vsbTxLast[0]) return;

		/* retransmit if we get a nak */
		if (sbAck[0] == '~') {
			/* if this is a resync request then do so */
			if (sbAck[1] == '~' && sbAck[2] == '~')
				vRxChk = 0;
			continue;
		}

		/* if this is another copy of the last message he sent,
		 * then he probably lost my last ack and is still trying to
		 * send a message that I already processed. So we just send our
		 * new message again. It will have the side effect of being an
		 * ack to the last message as well.
		 */
		if (sbAck[0] == vsbRxLast[0] && ret > 2) continue;

		/* if this is a valid new message rather than an ack, then we
		 * must have lost the ack from the last message and this is 
		 * his next transmission. We will acknowledge the reciept of
		 * a valid message and save this line for ReadSb.
		 */
		if (fFsbValidMsg(sbAck,vRxChk)) {
			vfRecieveDone = true;
			strcpy(vsbRxLast,sbAck);
			sbAck[1] = '\n';
			vRxChk = sbAck[0];
			write(vfnTx, sbAck, 2);
			return;
		}

		/* if this is a quit request then exit */
		if(strncmp(sbAck,"quit\n",4) == 0) quit(1);

		/* I have no idea what this line is so we will just try again */
	}
} /* WriteSb */


/* F   F   S B   V A L I D   M S G */

fFsbValidMsg(sbMsg, chk)
SBT		sbMsg;		/* message to validate */
char		chk;		/* check character from previous message */
{
	FAST SBT sb;
	FAST char ch;
	int sum = 0;
	int cnt = ' ';

	/* calculate expected check byte */
	sb = sbMsg+2;
	while((ch = *sb++) != '\n') {
		if (ch < ' ' || ch > '~')
			return(false);
		sum += ch;
		cnt++;
	}
	sum = sum%47 + ' ';

	/* calculate sequence bit */
	if (chk) {
		/* must be opposite of last transmission */
		if (chk < ' '+47) sum += 47;
	} else {
		/* resync with whatever sequence bit this has */
		if (sbMsg[0] >= ' '+47) sum += 47;
	}

	return(sum == sbMsg[0] && cnt == sbMsg[1]);
} /* fFsbValidMsg */


/* C D B   P A N I C */

int RemotePanic(sbFmt, arg1, arg2, arg3)
SBT	sbFmt;
int	arg1, arg2, arg3;
{
    fprintf(stderr, "Remote Panic - ");
    fprintf(stderr, sbFmt, arg1, arg2, arg3);
    fprintf(stderr, "\n");
    fflush(stderr);
} /* RemotePanic */


/* V A L   F   P S B   */

long ValFPsb(psb)
SBT	*psb;
{
    int		x = 0;
    char	ch, *sb = *psb;

    /* this routine eats a hex number from a string and advances
     * the string pointer to the next character AFTER the first
     * non-hex character.  E.g if we start with "1234|abcd",
     * we will return the integer value 0x1234 and leave the string
     * pointing to the 'a'.
     */

#define FHex(ch) (((ch >='0') AND (ch <= '9')) OR ((ch >= 'a') AND (ch <= 'f')))

    if ((sb == sbNil) OR (*sb == chNull))
	return(0);

    while (true) {
	ch = *sb++;
	if (! FHex(ch))
	    break;
	x = (x * 16) + ((ch <= '9') ? (ch - '0') : ((ch - 'a') + 10));
    } /* while */
    *psb = sb;
    return(x);
} /* ValFPsb */


/* R E A D   H E X   B Y T E S */

void ReadHexBytes(adr, count, sb)
int	count;
long	adr;
SBT	sb;
{
    char	ch1, ch2, *sbDest;
    short	val;

    /* used with the host-to-remote block transfer command */
    sbDest = (SBT) adr;
    while (count--) {
	ch1 = *sb++;
	ch2 = *sb++;
	val = ((ch1 - ((ch1 > '9') ? ('a'-10) : '0')) << 4)
	       + (ch2 - ((ch2 > '9') ? ('a'-10) : '0'));
	*sbDest++ = val;
    } /* while */
} /* ReadHexBytes */


/* W R I T E   H E X   B Y T E S */

void WriteHexBytes(adr, count)
int	count;
long	adr;
{
    SBT		sbTx, sbSrc;
    short	val;

    /* used with the remote-to-host block transfer command */

    sbSrc = (char *) adr;
    sbTx = vsbTx;
    sprintf(sbTx, "%c%x|", ' '+statusOk, count);
    sbTx = sbTx + strlen(sbTx);
    while (count--) {
	val = (*sbSrc & 0xf0) >> 4;
	*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
	val = *sbSrc++ & 0x0f;
	*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
    } /* while */
    *sbTx = '\n';
    WriteSb(vsbTx);
} /* WriteHexBytes */


/* L O C A L   P T R A C E */

void LocalPtrace(pt, pid, adr, value)
int	pt, pid, value;
long	adr;
{
    int		status;
    
    /* This routine does whatever is necessary to implement
     * the ptrace(2) system call.
     *
     * For UNIX systems, this means we just call ptrace itself.
     */

    /* On a majority of systems, Instruction and Data Space
     * are identical.  On systems where this is not true, you
     * will need to deal with them separatly.
     */

    dprint(10, (stderr, "pt: %d  adr: %x  value: %d\n", pt, adr, value));

    errno = 0;
    switch (pt) {
	default: errno = EINVAL;
		value = -1;

	case  ptReadI:
	case  ptReadD:
		/* Read memory */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptWriteI:
	case  ptWriteD:
		/* Write memory */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptReadUser:
		/* Read Registers */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptWriteUser:
		/* Write Registers */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptSingle:
		/* Single Step the child */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptResume:
		/* Continue an existing child */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptTerm:
		/* Terminate an existing child */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptMultiChild:
		/* we own you and your firstborn */
		value = ptrace(pt, adr, 0, 0);
		if (value == 0)
			vpidChild = adr;
		break;

	case  ptReleaseChild:
		/* go `way ya bother me, Kid.... */
		value = ptrace(pt, pid, adr, value);
		vpidChild = 0;
		break;

	case  ptPpid:
		/* give us pid of parent of this process */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptCmdName:
		/* return name of executable */
		{
		char name[256];

		memset(name,0,sizeof(name));
		value = ptrace(pt, pid, name, 0);
    		sprintf(vsbTx, "%c%s\n", ' '+errno, name);
    		WriteSb(vsbTx);
		return;
		}

    } /* switch */

    SendStatus(errno, value);
} /* LocalPtrace */


/* S E N D   S T A T U S */

void SendStatus(status, value)
int	status, value;
{
    sprintf(vsbTx, "%c%x\n", ' '+status, value);
    WriteSb(vsbTx);
} /* SendStatus */


/* N E X T   A R G */

char * NextArg(psbIn)
SBT	*psbIn;
{
    SBT		sbRet;

    /* skip white space */
    while (**psbIn == ' ' OR **psbIn == '\t')
	(*psbIn)++;

    if (**psbIn == chNull)
	return(sbNil);	/* that's all, folks! */

    sbRet = *psbIn;	/* set current arg pointer */
    while (**psbIn != chNull
       AND **psbIn != ' '
       AND **psbIn != '\t')
	(*psbIn)++;	/* skip all non-white space characters */

    **psbIn = chNull;	/* overwrite the tab, space, or 0 with a 0 */
    (*psbIn)++;
    return(sbRet);
} /* NextArg */


/* D O	 A R G S */

void DoArgs(sbProgram, sbArgs, argv, fReassign)
SBT	sbProgram, sbArgs;
char	**argv;
int	fReassign;
{
    int		i;
    SBT		sbFile, sbArgCur;
    char	sbBuffer[1024];

    for (i=0; i < cArgsMax; i++)
	argv[i] = sbNil;

    *argv++ = sbProgram; /* set the filename */

    if (sbArgs == sbNil)
	return;

    strcpy(sbBuffer, sbArgs);	/* copy it to a safe place to hack on it */
    sbArgs = sbBuffer;
    sbArgs[strlen(sbArgs)+1] = chNull; /* this puts a SECOND null at end */

    /* eat the arguments */
    while ((sbArgs != sbNil)
       AND (*sbArgs != chNull)) {
	sbArgCur = NextArg(&sbArgs);
	if (sbArgCur == sbNil)
	    break;

	/* check for indirection */
	if (!fReassign
	   OR ((*sbArgCur != '<') AND (*sbArgCur != '>')) ) {
	    *argv++ = sbArgCur;	/* we just pass it on */
	} else {
	    /* find that file name */
	    sbFile = sbArgCur + 1;  /* initially we assume `<foo' vs `< foo' */
	    if (*sbFile == chNull) {
		/* there was white space between < and sbFile */
		sbFile = NextArg(&sbArgs);
		if (sbFile == sbNil) {
		    printf("Illegal indirection\n");
		    exit(1);
		} /* if */
	    } /* if */

	    if (*sbArgCur == '<') {
		/* changing stdin */
		close(0);
		if (open(sbFile, O_RDONLY) < 0) {
		    perror(sbFile);
		    exit(1);
		} /* if */
	    } else if (*sbArgCur == '>') {
		/* changing stdout */
		close(1);
		if ( (open(sbFile, 2) < 0) 	/* O_WRONLY */
		   AND (creat(sbFile, 0666) < 0) ) {
		    perror(sbFile);
		    exit(1);
		} /* if */
	    } /* if */
	} /* if */
    } /* while */
    *argv = sbNil;
} /* DoArgs */


/* P I D   F   E X E C   C H I L D */

int PidFExecChild(sbExec, sbArgs)
SBT	sbExec, sbArgs;
{
    int		pid, fn, sig;
    char	*argvChild[cArgsMax];

    /* Parse the argument list passed into an argv array.
     * Create a child process, initialize it to a more or less virgin state,
     * (i.e. clean up the signals, close the files,....) and return the pid
     * to the caller.
     */

    if ((pid = fork()) == 0) {
	/* we are the CHILD process */

	for (sig=1; sig<NSIG; sig++)
	    signal(sig, SIG_DFL);	/* start out with default response */

	for (fn = 3; fn < _NFILE; fn++)
	    close(fn);

	DoArgs(sbExec, sbArgs, argvChild, true);
	ptrace(ptChild, 0, (char *)0, 0); /* Hi, Mom! */
	execv(sbExec, argvChild);

	/* if we get here, we are child BUT exec failed */
	perror(sbExec);
	RemotePanic("Exec failed.");
	exit(1);
    } /* if */

    return(pid);
} /* PidFExecChild */


/* D E B U G   S E R V E R */

int DebugServer(fpTx, fpRx)
FILE	*fpTx, *fpRx;
{
    int		result, status, msg, pt, value, count;
    int		sig;
    long	adr, adr2;
    char	*sb, sbBuf[cbBufMax], sbName[cbBufMax], sbArgs[cbBufMax];

    /* This routine is the main command loop.
     * It acts as the Parent process to the target process.
     */

    dprint(1, (stderr, "Entering debug server\n"));


    while (true) {
	ReadSb(sbBuf,sizeof(sbBuf));	/* get a message from the host */
	sb = sbBuf;
	msg = *sb++ - ' ';	/* get message number */
	dprint(1, (stderr, "msg = %d\n", msg));

	status = statusOk;	/* to start with */

	switch (msg) {
	    default:
		/* check for in the range of Ptrace messages */
		if (msg<msgPtrace || msg>msgPtrace+ptMax) {
			SendStatus(statusRetry, 0);
			break;
		}

	    	/* must be a ptrace request */
		pt = msg-msgPtrace;
		adr = ValFPsb(&sb);
		if (sb[-1] == '\n')
			value = 0;
		else
			value = ValFPsb(&sb);

		LocalPtrace(pt, vpidChild, adr, value);
		break;

	    case msgWait:
		waitagain:
		result = wait(&value);
		if (result < 0) {
			SendStatus(errno, value);
		} else if (result == vpidChild) {
			if ((value&0xff) != 0177) vpidChild = 0;
			SendStatus(statusOk, value);
		} else {
			/* our child must have forked so we will release this */
			ptrace(ptReleaseChild, result, 0, 0);
			goto waitagain;
		}
		break;

	    case msgQuit:
		quit(0);
		return;	/* all done */


	    case msgExec:
		if (vpidChild > 1) {
		    ptrace(ptTerm, vpidChild, (char *)0, 0);
		    vpidChild = 0;
		} /* if */

		/* sb now points at "program|args" */

		sscanf(sb, "%[^|]|%[^|]", sbName, sbArgs);
		sbArgs[strlen(sbArgs)-1] = '\0'; /* clear the <nl> */
		dprint(1, (stderr, "Execing: %s, args:%s\n", sbName, sbArgs));

		vpidChild = PidFExecChild(sbName, sbArgs);
		if (vpidChild == -1) {
		    vpidChild = 0;
		    status = errno;
		} /* if */
		SendStatus(status, vpidChild);
		break;

	    case msgHtoR:
		adr = ValFPsb(&sb);
		count = ValFPsb(&sb);
		ReadHexBytes(adr, count, sb);
		SendStatus(statusOk, 0);
		break;

	    case msgRtoH:
		adr = ValFPsb(&sb);
		count = ValFPsb(&sb);
		WriteHexBytes(adr, count);
		break;

	    case msgMove:
		adr = ValFPsb(&sb);
		adr2 = ValFPsb(&sb);
		count = ValFPsb(&sb);
		/* MoveBlock((char *)adr, (char *)adr2, count); */
		SendStatus(status, 0);
		break;

	    case msgSignal:
		sig = ValFPsb(&sb);
		result = kill(vpidChild, sig);
		SendStatus(result?errno:statusOk, result);
		break;

	    case msgLevel:
		vlevel = ValFPsb(&sb);
		SendStatus(statusOk, 0);
		break;

	} /* switch */
    } /* while */
} /* DebugServer */
