/* $Header: files.c,v 6.1 86/09/14 20:23:43 peter Exp $ */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

/* This module does the source file open/closes and the source line
 * printing, etc.
 */

#include "cdb.h"
#include <errno.h>

FILE	*vfpSrc;	/* currently open source file */
export char vsbFilename[256];	/* full name OR error message from SbFSearch */


/* S B   F   A L L O C */

export SBT SbFAlloc(sb)
SBT	sb;
{
    SBT		sbNew;
    
    /* Given a string, allocate new space and copy it into it */

    if (sb == sbNil)
	return(sbNil);

    sbNew = (SBT) malloc(strlen(sb) + 1);
    strcpy(sbNew, sb);
    return(sbNew);
} /* SbFAlloc */


/* A D D   D I R */

export void AddDir(pr, sbDir)
pPRR	pr;
SBT	sbDir;
{
    /* Add an alternate directory to the list */
#define increment	10

    if (pr->idirMac >= pr->idirMax) {
	pr->idirMax += increment;
	if (pr->idirMax == increment) {
	    pr->rgSbDir = (SBT *) malloc(pr->idirMax * sizeof(SBT));
	} else {
	    pr->rgSbDir = (SBT *)realloc(pr->rgSbDir, pr->idirMax*sizeof(SBT));
	} /* if */
	if (pr->rgSbDir == 0)
	    Panic("Ran out of memory.");
    } /* if */

    pr->rgSbDir[pr->idirMac] = SbFAlloc(sbDir);
    pr->idirMac++;
} /* AddDir */


/* L I S T   D I R */

export void ListDir()
{
    FAST int	i;

    /* Show alternate directories */

    printf("Source directories - search order\n");
    for (i = 0; i < v->idirMac; i++)
	printf("%2d  %s\n", i, v->rgSbDir[i]);
    printf("%2d  %s\n", i, ".");
} /* ListDir */


/* S B   F   B A S E */

export SBT SbFBase(sb)
SBT sb;
{
    FAST SBT	sbTemp;

    sbTemp = strrchr(sb, '/');
    if (sbTemp != sbNil)
	return(SbFAlloc(sbTemp + 1));
    return(sb);
} /* SbFBase */


/* E D I T   F I L E */

export void EditFile(ifd, iln)
int2	ifd, iln;
{
    int2	cb;
    SBT		sbFile;
    char	sbCopy[256], sbCmd[356];
    pFDR	fd;

    /* Edit the given file, starting at line iln */

    fd = v->rgFd + ifd;
    sbFile = SbFIss(fd->iss);	/* get the current name */

    /* see if they are allowed to do this */
    if (access(sbFile, 4+2) == -1)
	/* that didn't work - tell them why */
	UError("%s: %s.", SbFError(errno), sbFile);

    cb = strlen(sbFile);
    strcpy(sbCopy, sbFile);

    if (fd->fEdited) {

	/* file has already been edited, don't copy it again */
	sbCopy[cb-1] = chNull;	/* cut off the # */
	printf("%s has already been edited, going to NEW file\n", sbCopy);

    } else if (ifd != ifdTemp) {

	/* cp file to sbFile# */
	sprintf(sbCmd, "cp %s %s#", sbFile, sbFile);
	if (vfpSrc != NULL) {
	    fclose(vfpSrc); /* close source file */
	    vfpSrc = NULL;
	} /* if */
	system(sbCmd);	/* do the copy */
	strcat(sbCopy, "#");
	fd->iss = IssFSb(SbFAlloc(sbCopy));
	fd->issShort = IssFSb(SbFBase(sbCopy));
	fd->fEdited = true;
	if (fd->rgLn != 0) {
	    free(fd->rgLn);
	    fd->rgLn = 0;
	} /* if */

	v->ifd = ifdNil;
	OpenIfd(ifd);	/* force re-open to new file name */

	sbCopy[strlen(sbCopy)-1] = chNull;	/* cut off the # */
    } /* if */

    /* goto vi at correct line number */
    sprintf(sbCmd, "vi +%d %s", iln, sbCopy);
    CallShell(sbCmd);
    v->iln = iln;
} /* EditFile */


/* S B   F   S E A R C H */

export SBT SbFSearch(sbFile, permissions)
SBT	sbFile;
int2	permissions;
{
    int		i, result, errnoSave;
    FLAGT	fFullPath;
    char	sbTemp[256], sbError[256];;

    /* Find the named file. If there are alternate directories,
     * try them in order until we succeed.
     */

    errnoSave = 0;
    sbTemp[0] = chNull;
    result = -1;

    fFullPath = (sbFile[0] == '/');

    if (! fFullPath) {
	if (sbFile[0] == '.') {
	    /* it starts with a dot - try a pure access */
	    result = access(sbFile, permissions);
	    if (result != -1) {
		strcpy(vsbFilename, sbFile);
		return(vsbFilename);
	    } else if (errno != ENOENT) {
		errnoSave = errno;
		strcpy(sbError, sbFile);
	    } /* if */
	} /* if */

	for (i = 0; i < v->idirMac; i++) {
	    /* slap the directory name on the front */
	    if (FSbCmp(v->rgSbDir, ".")) {
		strcpy(sbTemp, sbFile);
	    } else {
		strcpy(sbTemp, v->rgSbDir[i]);
		strcat(sbTemp, "/");
		strcat(sbTemp, sbFile);
	    } /* if */

	    /* try to open it */
	    result = access(sbTemp, permissions);
	    if (result != -1) {
		/* we found something interesting */
		strcpy(vsbFilename, sbTemp);
		return(vsbFilename);
	    } /* if */
	    if ((errnoSave == 0) AND (errno != ENOENT)) {
		errnoSave = errno;
		strcpy(sbError, sbTemp);
	    } /* if */
	} /* for */
    } /* if */

    /* if we get here either a) it is a full path name
     * or b) we could not find it in one of the alternate directories.
     * So we try using exactly the name they gave us.
     */
    result = access(sbFile, permissions);
    if (result != -1) {
	strcpy(vsbFilename, sbFile);
	return(vsbFilename);
    } else if (errnoSave == 0) {
	errnoSave = errno;
	strcpy(sbError, sbFile);
    } /* if */

    sprintf(vsbFilename, "%s: %s", sbError, SbFError(errnoSave));
    return(sbNil);
} /* SbFSearch */


/* F P   F   S B */

FILE *FpFSb(sbFile, sbFlags)
SBT	sbFile, sbFlags;
{
    int		i;
    SBT		sb;
    FILE	*fp;

    /* Open the named file. If there are alternate directories,
     * try them in order until we succeed.
     */

    sb = SbFSearch(sbFile, 04);
    if (sb == sbNil)
	return(NULL);

    fp = fopen(sb, sbFlags);
    if (fp == NULL)
	sprintf(vsbFilename, "%s: %s", sb, SbFError(errno));
    else
	strcpy(sbFile, vsbFilename);
    return(fp);
} /* FpFSb */


/* S E T   R G   L N */

local void SetRgLn(ifdIn, fp)
int2	ifdIn;
FILE	*fp;
{
    int2	cbNew, ifd, iln, ilnMac;
    uint4	*rgLn;
    char	buf[cbLineMax];
    pFDR	fdT, fd = v->rgFd + ifdIn;

    /* Create the line index array for the given file.
     * We make two passes - the first to size the array
     * and the second to fill it.
     */

    if (fd->rgLn != 0)
	return;	/* it's already there */

    /* count the lines */
    iln = 0;
    rewind(fp);
    while (fgets(buf, cbLineMax, fp) != NULL)
	iln++;
    iln++;	/* this array is 1 based, not zero */

    cbNew = iln * sizeof(uint4);
    rgLn = (uint4 *) malloc(cbNew);  /* get enough space for line array */

    if (rgLn == 0) {	/* allocation failed */
	/* we just walk down the list, releasing space */
	fdT = v->rgFd;

	for (ifd = 0; ifd < v->hdr.ifdMax; fdT++, ifd++) {
	    if (fdT->rgLn != 0) {
		free(fdT->rgLn);	/* throw it away */
		fdT->rgLn = 0;
		rgLn = (uint4 *) malloc(cbNew);  /* try again */
		if (rgLn != 0)
		    break;	/* NOW it fits! */
	    } /* if */
	} /* for */

	if (ifd >= v->hdr.ifdMax)
	    Panic("Ran out of memory in SetRgLn.");
    } /* if */

    fd->ilnMac = iln;
    fd->rgLn = rgLn;

    rewind(fp);	/* once more, from the top */
    ilnMac = iln;
    for (iln = 1; iln < ilnMac; iln++) {
	rgLn[iln] = ftell(fp);
	if (fgets(buf, cbLineMax, fp) == NULL)
	    break;
    } /* for */
} /* SetRgLn */


/* I P D   F   N A M E */

export int IpdFName(sbProc)
FAST SBT	sbProc;
{
    FAST int2	ipd, ipdMax, ipdNextFile;

    /* Map a procedure name to an ipd */

    if ((v->ipd != ipdNil)
       AND (FSbCmp(SbFIpd(v->ipd), sbProc)) )
	return(v->ipd);

    ipdNextFile = (v->ifd == v->hdr.ifdMax - 1) ?
			v->hdr.ipdMax : v->rgFd[v->ifd+1].ipd;

    /* first we check for this name in CURRENT file - i.e. check for statics */
    ipdMax = ipdNextFile;
    for (ipd = v->rgFd[v->ifd].ipd; ipd < ipdMax; ipd++)
	if (FSbCmp(SbFIpd(ipd), sbProc))
	    return(ipd);

    /* check UP TO current file */
    ipdMax = v->rgFd[v->ifd].ipd;
    for (ipd = 0; ipd < ipdMax; ipd++)
	if (FSbCmp(SbFIpd(ipd), sbProc))
	    return(ipd);

    /* check AFTER current file */
    ipdMax = v->hdr.ipdMax;
    for (ipd = ipdNextFile; ipd < ipdMax; ipd++)
	if (FSbCmp(SbFIpd(ipd), sbProc))
	    return(ipd);

    return(ipdNil);
} /* IpdFName */


/* I L N   F I R S T   F   I P D */

export int IlnFirstFIpd(ipd)
int2	ipd;
{
    int2	iln, slop;
    int4	iline;

    /* Return the iln of the first executable line in ipd */
    iline = v->rgPd[ipd].iline;
    if (iline == 0)
	return(ilnNil);	/* no line info */

    return(LineFIline(iline)->iln);
} /* IlnFirstFIpd */


/* A D R   F I R S T   F   I P D */

export int AdrFirstFIpd(ipd)
int2	ipd;
{
    int2	iln, slop;
    int4	iline;

    /* Return the adr of the first executable line in ipd, based on line info.
     * return adrNil if no line info.
     */

    iline = v->rgPd[ipd].iline;
    if (iline == 0)
	return(adrNil);	/* no line info */

    /* First iline is now THE CORRECT first iline!!! */
    return(v->rgPd[ipd].adr + LineFIline(iline)->dadr);
} /* AdrFirstFIpd */


/* O P E N   I P D */

export void OpenIpd(ipd, fForce)
int2	ipd;
FLAGT	fForce;
{
    int2	ifd;
    pPDR	pd;

    /* 'Open' the procedure - this includes getting the right source file
     * setup and determining the first 'real' line in the proc.
     */

    if (ipd == ipdNil) {
	v->ipd = ipdNil;
	return;
    } /* if */

    pd = v->rgPd + ipd;
    ifd = IfdFAdr(pd->adr);	
    if (fForce OR (ipd != v->ipd))
	OpenIfd(ifd);	/* open parent file */
    v->ifd = ifd;
    v->iln = (pd->iline == 0) ? ilnNil : IlnFirstFIpd(ipd);
    v->ipd = ipd;
    v->slop = 0;
    v->adrInst = pd->adr;
    v->stackdepth = -1;
} /* OpenIpd */


/* O P E N   I F D */

export void OpenIfd(ifd)
int2	ifd;
{
    char	sbTemp[256];
    int2	cb;
    int4	mtimeSrc;
    pFDR	fd;
    FILE	*fp, *fpTemp;
    static pPRR	prLast = prNil;

    /* Open the source file connected with ifd */

    if (ifd == v->ifd)		/* same file */
	return;

    if (ifd < 0 OR ifd >= v->hdr.ifdMax)
	Panic("Ifd out of range: %d.", ifd);

    fd = v->rgFd + ifd;

    /* we use ifdTemp for `unrelated' files */
    if (ifd != ifdTemp) {
	/* a normal source file */
	if ((ifd == v->ifd) AND (fd->rgLn != 0))
	    return;

	if (fd->fNoSource OR (fd->iline == 0))
	    goto NoSource;
	strcpy(sbTemp, SbFIss(fd->iss));
    } else {
	strcpy(sbTemp, vsbFileTemp);
    } /* if */


    /* Now we try to open the source file */

     cb = strlen(sbTemp);
     if ((sbTemp[cb-1] == 'o') AND (sbTemp[cb-2] == '.'))
 	goto NoSource;
 
    fp = FpFSb(sbTemp, "r");

    if (fp == NULL) {
	/* well, that didn't work */
	if (ifd == ifdTemp)
	    UError(vsbFilename);

	cb = strlen(sbTemp);
	if ((sbTemp[cb-1] == 'o') OR (sbTemp[cb-1] == 's')) {
	    /* it's a ".o" file - try to find a ".c" somewhere */
	    sbTemp[cb-1] = 'c';
	    fp = FpFSb(sbTemp, "r");
	    if (fp == NULL)
		goto NoSource;
	} else {
	    if (fd->iline != 0)
		printf("%s\n", vsbFilename);
	    goto NoSource;
	} /* if */
    } /* if */

    /* We got a good one! */

    if (ifd == ifdTemp) {
	/* do some quick cleanup (kludges??) */
	if (fd->rgLn != 0) {
	    /* ie. there was already a file connected with ifdTemp */
	    free(fd->rgLn);
	    fd->rgLn = 0;
	    fd->ilnMac = 0;
	} /* if */
	fd->iline = -1;	/* so we think of it as printable */
	fd->iss = IssFSb(vsbFileTemp);
	fd->issShort = IssFSb(SbFBase(vsbFileTemp));
    } /* if */

    /* let's get rid of any currently open stuff */
    if (vfpSrc != NULL) {
	fclose(vfpSrc);	/* there is a currently open file */
	vfpSrc = NULL;
    } /* if */

    vfpSrc = fp;
    prLast = v;	/* remember which process we opened the file for */
    v->ipd = v->rgFd[ifd].ipd;
    v->ifd = ifd;
    v->stackdepth = -1;
    v->iln = 1;
    v->slop = 0;

    if (fd->rgLn == 0) {

	/* we need to see its age and build line array */
	mtimeSrc = AgeFFn(fileno(vfpSrc));

	/* Remember - the greater the time, the YOUNGER it is */
	if ((!fd->fWarned) AND (mtimeSrc > v->mtimeExec)) {
	    /* This file is younger than the executable.
	     * See if there is a # file around.
	     */
	    strcat(sbTemp, "#");;
	    fpTemp = fopen(sbTemp, "r");
	    if (fpTemp == NULL) {

		/* no # file exists - remove the # from the name */
		sbTemp[strlen(sbTemp)-1] = '\0';

	    } else {

		/* let's use the older (better?) temp */
		fclose(fp);
		fp = vfpSrc = fpTemp;
		printf("WARNING: Using \"%s\" instead of \"%s\"\n",
			    sbTemp, SbFIss(fd->iss));
		fd->fEdited = true;

	    } /* if */
	    if (!fd->fEdited)
		printf("WARNING: \"%s\" is younger than \"%s\"\n",
				sbTemp, v->sbExec);
	} /* if */
	SetRgLn(ifd, vfpSrc);	/* figure out line endings */
    } /* if */

    if (! FSbCmp(SbFIss(fd->iss), sbTemp)) {
	/* we modified name in some way */
	fd->iss = IssFSb(SbFAlloc(sbTemp));
	fd->issShort = IssFSb(SbFBase(sbTemp));
    } /* if */

    if (viwwSrc != iwwNil)
	EdOpenFile(ifd);
    
    fd->fWarned = true;
    return;	/* normal exit */

NoSource:
    /* there is no source file for us to open
     * could be either a .o OR a missing source.
     */
    fd->fNoSource = true;
    if (vfpSrc != NULL)
	fclose(vfpSrc);
    vfpSrc = NULL;
    v->ifd = ifd;
    v->stackdepth = -1;
    v->ipd = fd->ipd;
    v->slop = 0;
    v->iln = 0;
} /* OpenIfd */


/* S E T   V I E W   P O S I T I O N */

export void SetViewPosition(adr)
{
    int2	ipd, iln, slop;

    /* Set the position globals to be correct for 'adr */

    if (adr == adrNil) {

	/* determine adr FROM the current file/line/stack context */
	if (v->rgFd[v->ifd].iline == 0) {
	    /* no source line numbers */
	    return;	/* ?? */
	} /* if */

	v->slop = 0;
	adr = AdrFIfdIln(v->ifd, v->iln);
	if (adr == adrNil) {
	    v->ipd = ipdNil;
	    v->adrInst = adrNil;
	    return;
	} /* if */
	v->ipd = IpdFAdr(adr);
	v->adrInst = adr;

    } else {

	ipd = IpdIlnFAdr(&iln, &slop, adr);	/* map to line number */
	OpenIpd(ipd, true);	/* This will also open our file for us */
	v->adrInst = adr;
	v->iln = iln;
	v->slop = slop;

    } /* if */

} /* SetViewPosition */


/* I F D   F   N A M E */

export int IfdFName(sbFile)
FAST SBT	sbFile;
{
    FAST int2	ifd, ifdMax;
    FAST pFDR	fd;

    /* Map a name to an ifd */

    if ((v->ifd != ifdNil)
       AND FSbCmp(SbFIss(v->rgFd[v->ifd].iss), sbFile))
	return(v->ifd);

    ifdMax = v->hdr.ifdMax;
    for (ifd = 0, fd = v->rgFd; ifd < ifdMax; fd++, ifd++) {
	if (FSbCmp(SbFIss(fd->iss), sbFile))
	    return(ifd);
    } /* for */

    for (ifd = 0, fd = v->rgFd; ifd < ifdMax; fd++, ifd++) {
	if ( (fd->iss != fd->issShort)
	   AND (FSbCmp(SbFIss(fd->issShort), sbFile)) )
	    return(ifd);
    } /* for */

    return(ifdNil);
} /* IfdFName */


/* S B   F   P R E P A R E   L I N E */

export SBT SbFPrepareLine(sbIn)
FAST SBT	sbIn;
{
    FAST int	col, cnt;
    FAST SBT	sb;
    static char	sbBuf[1024];

    /* this routine expands any tabs and removes any trailing \n */

    col = 0;
    sb = sbBuf;
    while (*sbIn != '\n') {
	if (*sbIn != '\t') {
	    *sb++ = *sbIn++;
	    ++col;
	} else {
	    cnt = 8 - col % 8;
	    col += cnt;
	    while (cnt--)
		*sb++ = ' ';
	    ++sbIn;
	} /* if */
    } /* for */

    *sb = chNull;
    return(sbBuf);
} /* SbFExpandTabs */


/* P R I N T   L I N E */

export void PrintLine(iln, count, fPrintLn, fp)
int2	iln, count;
FLAGT	fPrintLn;
FILE	*fp;
{
    int2	ifd, col;
    int4	offset;
    char	linebuf[cbLineMax];
    pFDR	fd;

    /* Print the given line(s) in the currently open source file.  */
    col = (vfWindows) ? 71 : 100;

    if (v->ifd == ifdNil)
	UError("I have no source file for this address.");

    fd = v->rgFd + v->ifd;
    if ((fd->iline == 0) OR (fd->fNoSource))
	UError("I have no source file for this address.");

    if (iln == ilnNil)
	iln = 1;
    iln = Min(iln, fd->ilnMac);
    count = Min(fd->ilnMac-iln, count);

    offset = fd->rgLn[iln];
    if (fseek(vfpSrc, offset, 0))
	SysPanic("Bad seek in PrintLine.");

    while (count--) {
	if (fgets(linebuf, cbLineMax, vfpSrc) == NULL)
	    SysPanic("Bad read in PrintLine.");
	if (fPrintLn)
	    fprintf(fp, "%4d:", iln);
	if (linebuf[0] == '\014')
	    strcpy(linebuf, "\n"); /* line feeds casue the window to clear */
	fprintf(fp, "	%.*s", col, SbFPrepareLine(linebuf));
	if (count)
	    fprintf(fp, "\n");
	iln++;
    } /* while */

    if (fp != stdout)
	fflush(fp);
    else
	printf("\n");

    v->iln = iln - 1;	/* set to last line printed */
} /* PrintLine */


/* F I N D */

export void Find(sbTarget, fBack)
SBT	sbTarget;
FLAGT	fBack;
{
    int		i, lineinc, count, iln, ilnStart, ilnMac;
    uint4	*rgLn;
    char	linebuf[cbLineMax];
    pFDR	fd;

    /* Search the currently open source (forwards or backwards)
     * for an instance of the target.
     * Wrap around the end/beginning of the file.
     */

    fd = v->rgFd + v->ifd;
    if (fd->fNoSource OR (fd->iline == 0))
	UError("I have no source for file %s.", SbFIss(fd->iss));

    iln = v->iln;
    ilnMac = fd->ilnMac;
    rgLn = fd->rgLn;
    if (fBack) {
	lineinc = -1;
	iln--;  /* for backup */
	count = iln;
    } else {
	lineinc = 1;
	iln++;  /* start at next line */
	count = ilnMac - iln;
    } /* if */
    ilnStart = iln;

    if (fseek(vfpSrc, rgLn[iln], 0))
	SysPanic("Bad seek in Find - offset %d.",rgLn[iln]);

    InitPattern(sbTarget);
    while (count--) {
	if (fBack AND (fseek(vfpSrc, rgLn[iln], 0) != 0) )
	    SysPanic("Bad seek in Find - offset %d.",rgLn[iln]);
	if (fgets(linebuf, cbLineMax, vfpSrc) == NULL)
	    SysPanic("Bad read in Find.");
	if ((i = IFPattern(linebuf)) >= 0)
	    goto Winner;
	iln += lineinc;
    } /* while */

    /* that didn't work, try it from the top of file */
    if (fBack) {
	iln = ilnMac - 1;
	count = ilnMac - ilnStart - 1;
    } else {
	iln = 1;
	count = ilnStart;
    } /* if */

    if (fseek(vfpSrc, rgLn[iln], 0))
	SysPanic("Bad seek in Find.");

    while (count--) {
	if (fBack AND (fseek(vfpSrc, rgLn[iln], 0) != 0) )
	    SysPanic("Bad seek in Find.");
	if (fgets(linebuf, 200, vfpSrc) == NULL)
	    SysPanic("Bad read in Find.");
	if ((i = IFPattern(linebuf)) >= 0)
	    goto Winner;
	iln += lineinc;
    } /* while */
    UError("No Match - %s.", sbTarget);

Winner:
    v->iln = iln;
    ShowViewPosition(fmtLn+fmtLine+fmtEol);
} /* Find */


/* S H O W   V I E W   P O S I T I O N */

ShowViewPosition(fmt)
int2	fmt;
{
    if ((vfWindows)
       AND (v->iln != ilnNil)) {
	JumpScrollWw(v->iln);
	if (fmt & fmtSrcLineOnly)
	    fmt &= ~(fmtLine|fmtLn);
	PrintPosition(adrNil, fmt);
	MarkLine(v->iln);
    } else {
	if (fmt & fmtSrcLineOnly)
	    fmt |= fmtLine;
	PrintPosition(adrNil, fmt);
    } /* if */
} /* ShowViewPosition */



/* P R I N T   P O S I T I O N */

export void PrintPosition(adr, fmt)
ADRT	adr;
int2	fmt;
{
    int2	ifdSave, ipdSave, ilnSave, slopSave;
    int2	ifd, ipd, iln, slop;
    SBT		sbLine;
#define cbBuf	200
    char	sbBuf[cbBuf];
    FLAGT	fHaveLines;
    ADRT	adrSave;

    /* Do a formatted print of the given address.
     * fmt options are:
     *	fmtFile		print the file name
     *	fmtProc		print the procedure name
     *	fmtLn		print the line number
     *	fmtPrint	print the line (or instruction if no source)
     *  fmtInst		print instruction, even if we have source
     *
     *	These options can be added or |'d together, e.g. fmtFile+fmtProc.
     */

    /* clean out buffer */
    ZeroBlock(sbBuf, cbBuf);
    sbLine = sbBuf;

    if (adr != adrNil) {
	ifd = IfdFAdr(adr);
	ipd = IpdIlnFAdr(&iln, &slop, adr);
    } else {
	ifd = v->ifd;
	ipd = v->ipd;
	iln = v->iln;
	slop = v->slop;
	adr = v->adrInst;
    } /* if */

    fHaveLines = (ifd == ifdTemp) OR (v->rgFd[ifd].iline != 0);

    if (fmt & fmtFile) {
	strcat(sbLine, SbFIss(v->rgFd[ifd].issShort));
	sbLine += strlen(sbLine);
	*sbLine++ = ':';
    } /* if */

    if (slop < 0) {
	/* we are in the intro code of a proc with source info */
	strcat(sbLine, SbFNearest(adr, true, false, true));
	sbLine += strlen(sbLine);
	*sbLine++ = ':';
    } else {
	if ((fmt & fmtProc) AND (ifd != ifdTemp) AND (ipd != ipdNil)) {
	    if (fHaveLines) {
		strcat(sbLine, SbFIpd(ipd));
	    } else {
		strcat(sbLine, SbFNearest(adr, true, false, false));
	    } /* if */
	    sbLine += strlen(sbLine);
	    *sbLine++ = ':';
	} /* if */

	if (fHaveLines AND (fmt & fmtLn)) {
	    if (iln != ilnNil) {
		if (fmt & (fmtFile|fmtProc))
		    sprintf(sbLine, "%d", iln);
		else
		    sprintf(sbLine, "%4d", iln);
		sbLine += strlen(sbLine);
	    } /* if */
	    if (slop != 0) {
		*sbLine++ = '+';
		strcat(sbLine, SbFAdr(slop, true));
		sbLine += strlen(sbLine);
	    } /* if */
	    *sbLine++ = ':';
	} /* if */
    } /* if */

    if (fmt & (fmtLine | fmtInst)) {
	if (fHaveLines
	   AND ! v->rgFd[ifd].fNoSource
	   AND ((fmt & fmtInst) == 0)) {
	    printf("%s", sbBuf);
	    if (ifd != v->ifd) {
		ifdSave = v->ifd;
		ipdSave = v->ipd;
		ilnSave = v->iln;
		slopSave = v->slop;
		adrSave = v->adrInst;

		OpenIfd(ifd);
		PrintLine(iln, 1, false, stdout);

		v->ifd = ifdNil;
		if (ifdSave != ifdNil) 
		    OpenIfd(ifdSave);
		v->ipd = ipdSave;
		v->iln = ilnSave;
		v->slop = slopSave;
		v->adrInst = adrSave;
	    } else {
		PrintLine(iln, 1, false, stdout);
	    } /* if */
	 } else {
	    printf("%-20s", sbBuf);
	    PrintInstruction(adr);
	    if (fmt & fmtEol)
		printf("\n");
	 } /* if */
    } else {
	*--sbLine = chNull;	/* takes out last `:' */
	if (fmt & fmtEol)
	    *sbLine++ = '\n';
	printf("%s", sbBuf);
    } /* if */
} /* PrintPosition */
