/* $Header: rptrace.c,v 6.1 86/08/05 17:46:51 peter Exp $ */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

#include <errno.h>
#include <signal.h>
#include <termio.h>

#include "cdb.h"
#include "rptrace.h"

extern void Transmit();

int	vfnTx, vfnRx;
int	vlevelSave = 0;
FLAGT	vfTest = false;

int	vsignal, vpidServer;
char	vsbRemoteDevice[100];	/* name of device connected to remote */

char	_sbRx[cbMsgMax];	/* buffer for received messages */
SBT	vsbRx;		/* points to next character in receive buffer */
#define cbRx sizeof(_sbRx)
char	vsbTxLast[cbMsgMax]; /* copy of last transmisson */
char	vsbRxLast[cbMsgMax]; /* copy of last recieve data */
int	vpidRemote;		/* pid of the process being remotely debugged */
FLAGT	vfRecieveDone;	/* true if remote sent new line rather than ack */
char	vTxSeq;		/* next sequence number to transmit */
char	vRxChk;		/* last valid check character receieved */

struct termio vTtyarg;	/* initial tty settings */

/* V A L   F   P S B   */

long ValFPsb(psb)
SBT	*psb;
{
    int		x = 0;
    char	ch, *sb = *psb;

    /* this routine eats a hex number from a string and advances
     * the string pointer to the next character AFTER the first
     * non-hex character.  E.g if we start with "1234|abcd",
     * we will return the integer value 0x1234 and leave the string
     * pointing to the 'a'.
     */

#define FHex(ch) (((ch >='0') AND (ch <= '9')) OR ((ch >= 'a') AND (ch <= 'f')))

    if ((sb == sbNil) OR (*sb == chNull))
	return(0);

    while (true) {
	ch = *sb++;
	if (! FHex(ch))
	    break;
	x = (x * 16) + ((ch <= '9') ? (ch - '0') : ((ch - 'a') + 10));
    } /* while */
    *psb = sb;
    return(x);
} /* ValFPsb */


/* S B   F   T X   M S G */

local void SbFTxMsg(sbTx)
SBT	sbTx;
{
    int		msg, pt, value, count;
    long	adr, adr2;
    char	*sb, sbName[cbMsgMax], sbArgs[cbMsgMax];

    printf("TxHost: %s", sbTx);

    if (v->fCommDebug AND (v->debuglevel >= 100)) {
	sb = sbTx;
	msg = *sb++ - ' ';	/* get message number */

	switch (msg) {
	    default:
		/* check for in the range of Ptrace messages */
		if (msg<msgPtrace || msg>msgPtrace+ptMax) {
			printf("msg = %d\n", msg);
			break;
		}

	    	/* must be a ptrace request */
		pt = msg-msgPtrace;
		adr = ValFPsb(&sb);
		if (sb[-1] == '\n')
			value = 0;
		else
			value = ValFPsb(&sb);

		switch (pt) {
		    default:    printf("Unknown Ptrace request: %d", pt);
				break;

		    case  ptReadI:	printf("Read Text");	break;
		    case  ptReadD:	printf("Read Data");	break;
		    case  ptWriteI:	printf("Write Text");	break;
		    case  ptWriteD:	printf("Write Data");	break;
		    case  ptReadUser:	printf("Read User");	break;
		    case  ptWriteUser:	printf("Write User");	break;
		    case  ptSingle:	printf("Single Step");	break;
		    case  ptResume:	printf("****** Resume ******");	break;
		    case  ptTerm:	printf("Terminate");	break;
		} /* switch */
		printf("	adr: %#x	value: %d (%#x)\n",
			    adr, value, value);
		break;

	    case msgWait:
		    printf("******* Waiting for process ********\n");
		    break;

	    case msgExec:
		sscanf(sb, "%[^|]|%[^\n]", sbName, sbArgs);
		printf("Exec %s with args \"%s\"\n", sbName, sbArgs);
		break;


	    case msgRtoH:
		adr = ValFPsb(&sb);
		count = ValFPsb(&sb);
		printf("Remote to Host: from: %x  count %d\n",
				adr, count);
		break;

	    case msgHtoR:
		adr = ValFPsb(&sb);
		count = ValFPsb(&sb);
		printf("Host to Remote: start at: %x  count %d\n",
				adr, count);
		break;

	    case msgMove:
		adr = ValFPsb(&sb);
		adr2 = ValFPsb(&sb);
		count = ValFPsb(&sb);
		printf("Block Move: dest: %x  src: %#x, count %d\n",
				adr, adr2, count);
		break;

	} /* switch */
    } /*  if */
} /* SbFTxMsg */


/* F   F   S B   V A L I D   M S G */

local FLAGT fFsbValidMsg(sbMsg, chk)
SBT		sbMsg;		/* message to validate */
char		chk;		/* check character from previous message */
{
	FAST SBT sb;
	FAST char ch;
	int sum = 0;
	int cnt = ' ';

	/* calculate expected check byte */
	sb = sbMsg+2;
	while((ch = *sb++) != '\n') {
		if (ch < ' ' || ch > '~')
			return(false);
		sum += ch;
		cnt++;
	}
	sum = sum%47 + ' ';

	/* calculate sequence bit */
	if (chk) {
		/* must be opposite of last transmission */
		if (chk < ' '+47) sum += 47;
	} else {
		/* resync with whatever sequence bit this has */
		if (sbMsg[0] >= ' '+47) sum += 47;
	}


	return(sum == sbMsg[0] && cnt == sbMsg[1]);
} /* fFsbValidMsg */


/* R E A D   S B */

export int ReadSb(sb, cb)
int	cb;
SBT	sb;
{
    FAST int	i, ret;

	/* if WriteSb found our message return it now */
	if (vfRecieveDone) {
		vfRecieveDone = false;
		strncpy(sb,vsbRxLast+2,cb-1);
		sb[cb-1] = 0;
		return(strlen(sb));
	}

	/* loop reading until a valid string is received */
	for(;;) {
		ret = read(vfnRx, vsbRxLast, sizeof(vsbRxLast));
		if (ret <= 0) {
			if (errno == EINTR) continue;
	    		SysPanic("Read from Remote on %d.", vfnRx);
		}
		
		/* if this is a proper request, ack and return results */
		if (fFsbValidMsg(vsbRxLast,vRxChk)) {
			char sbAck[2];

			vRxChk = sbAck[0] = vsbRxLast[0];
			sbAck[1] = '\n';
			write(vfnTx, sbAck, 2);
			strncpy(sb,vsbRxLast+2,cb-1);
			sb[cb-1] = 0;
			return(strlen(sb));
		}

		/* if this is a resync request then ignore sequence bit */
		if (strncmp(vsbRxLast,"~~~",3) == 0) {
			vRxChk = 0;
			continue;
		}

		/* no good, so send a NAK and try again */
		write(vfnTx,"~\n",2);
	}

} /* ReadSb */


/* W R I T E   S B */

export int WriteSb(sb)
FAST SBT	sb;
{
    static int success = false;
    int AlarmTime = 2;
    int sum = 0;
    int cnt = 0;
    char sbAck[sizeof(vsbRxLast)];
    FAST int	i, ret;
    FAST char ch;
    FAST SBT sbTx;

	/* save the string in vsbTxLast and compute the check byte */
	sbTx = vsbTxLast+2;
	while((ch = *sb++) != '\n') {
		if (ch < ' ' || ch > '~')
			Panic("WriteSb: Attempt to send non-printable char");
		sum += ch;
		*sbTx++ = ch;
		cnt++;
	}
	*sbTx++ = '\n';
	*sbTx = 0;
	vsbTxLast[0] = sum%47 + vTxSeq + ' ';
	vsbTxLast[1] = cnt + ' ';
	cnt += 3;

	/* advance the sequence number for the next character */
	vTxSeq = vTxSeq ? 0 : 47;

	/* loop transmitting until a valid ack is recieved */
	i=0;
	for(;;) {

		/* after too many errors checkout whats happening */
		if (++i % 10 == 0) {
			/* if we have never communicated successfully then
			 * the line is probably not initialized correctly
			 */
			if (!success) {
				printf("Unable to establish contact with cdbremote\n");
				CloseComm();
				exit(1);
			}

			/* After too many tries to resync give up */
			if (i >= 40) {
				printf("Lost contact with cdbremote.\n");
				CloseComm();
				exit(1);
			}

			/* try to resync the sequence bit */
			write(vfnTx,"~~~\n",4);
		}
		
		/* flush out any extraneous garbage in input buffer */
		if (!vfTest) ioctl(vfnRx, TCFLSH, 0);

		/* send the message */
		ret = write(vfnTx, vsbTxLast, cnt);
		if (cnt != ret)
			SysPanic("WriteSb: %d bytes to go, %d went.", cnt, ret);

		/* get an ack or timeout */
		alarm(AlarmTime);
		ret = read(vfnRx, sbAck, sizeof(sbAck));
		alarm(0);

		/* check for error */
		if (ret < 0) {
			if (errno != EINTR)
				SysPanic("WriteSb: error reading ack");

			/* try again but give a longer timeout */
			AlarmTime += 2;
			continue;
		}

		/* if we got a valid ack then we are done */
		if (ret == 2 && sbAck[0] == vsbTxLast[0]) {
			success = true;
			return;
		}

		/* retransmit if we get a nak */
		if (sbAck[0] == '~') {
			/* if this is a resync request then do so */
			if (sbAck[1] == '~' && sbAck[2] == '~')
				vRxChk = 0;
			continue;
		}

		/* if this is another copy of the last message he sent,
		 * then he probably lost my last ack and is still trying to
		 * send a message that I already processed. So we just send our
		 * new message again. It will have the side effect of being an
		 * ack to the last message as well.
		 */
		if (sbAck[0] == vsbRxLast[0] && ret > 2) continue;

		/* if this is a valid new message rather than an ack, then we
		 * must have lost the ack from the last message and this is 
		 * his next transmission. We will acknowledge the reciept of
		 * a valid message and save this line for ReadSb.
		 */
		if (fFsbValidMsg(sbAck,vRxChk)) {
			/* if cdbremote is not running the echo of our command
			 * could look valid so check for that */
			if (!success && strncmp(sbAck,vsbTxLast,ret) == 0) {
				printf("You forgot to start cdbremote\n");
				CloseComm();
				exit(1);
			}
			vfRecieveDone = true;
			strcpy(vsbRxLast,sbAck);
			sbAck[1] = '\n';
			vRxChk = sbAck[0];
			write(vfnTx, sbAck, 2);
			success = true;
			return;
		}

		/* I have no idea what this line is so we will just try again */
	}
} /* WriteSb */


/* S E T   R E M O T E   L E V E L */

local void SetRemoteLevel(level)
int	level;
{
    char	sbBuf[100];

    sprintf(sbBuf, "%c%x\n", ' '+msgLevel, level);
    vlevelSave = level;
    Transmit(sbBuf);
    RetFReply();
} /* SetRemoteLevel */


/* T R A N S M I T */

export void Transmit(sbTx)
SBT	sbTx;
{
    /* Transmit the string */
    if (vlevelSave != v->debuglevel)
	SetRemoteLevel(v->debuglevel);

    if (v->fCommDebug)
	SbFTxMsg(sbTx);

    if (vfnTx <= 2)
	Panic("Transmit - v->fnComm: %d.", v->fnComm);

    vcsRemote = csWriting;
    WriteSb(sbTx);
    vcsRemote = csWriteDone;
} /* Transmit */


/* R E C E I V E */

export void Receive()
{
    int		cb, cRetry, status;

    cRetry = 0;
    while (true) {
	vsbRx = _sbRx;

	vcsRemote = csReading;
	cb = ReadSb(vsbRx, cbRx);
	vcsRemote = csNil;
	if (vfInterruptPending) {
	    vfInterruptPending = false;
	    Fixer();
	}

	if (v->fCommDebug)
	    printf("RxHost: %s", vsbRx);

	status = *vsbRx++ - ' ';

	switch (status) {
	    default:
		errno = status;	/* "OK" or error for caller */
		return;

	    case statusRetry:
		cRetry++;
		if (cRetry >= cRetryMax)
		    Panic("Excessive retries, Last Transmit was:\n%s\n",
			vsbTxLast+2);
		Transmit(vsbTxLast+2);
		break;

	} /* switch */
    } /* while */
} /* Receive */


/* R E T   F   R E P L Y */

local int RetFReply()
{
    int		ret;

    /* Get response from remote.
     * ASSUME that there is a simple numerical reply.
     */

    Receive();	/* get a string from remote */
    if (sscanf(vsbRx, "%x", &ret) != 1)
	Panic("Bad reply from target: %s", _sbRx);
    return(ret);
} /* RetFReply */


/* P T R A C E */

export int Ptrace(pt, pid, adr, value)
int	pt, pid;
char	*adr;
int	value;
{
    int		ret;
    char	sbPt[100];

    if (v->fnComm == fnNil)
	return( ptrace(pt, pid, adr, value) );

	/* if this is for some other process then panic */
	if(pid != vpidRemote && pt != ptMultiChild)
		Panic("Ptrace: Remote pid is %d but request is for %d.",
			vpidRemote, pid);

    switch (pt) {
	case ptMultiChild:
		/* the remote needs the new pid so we cheat */
		adr = (char*)pid;
		vpidRemote = pid;
		/* and fall through */
	case ptReadI:
	case ptReadD:
	case ptReadUser:
		value = 0;
		/* and fall through */
	case ptWriteI:
	case ptWriteD:
	case ptWriteUser:
	case ptResume:
	case ptTerm:
	case ptSingle:
	case ptReleaseChild:
	case ptPpid:
	    if (value == 0)
	        sprintf(sbPt, "%c%x\n", ' '+msgPtrace+pt, adr);
	    else
	        sprintf(sbPt, "%c%x|%x\n", ' '+msgPtrace+pt, adr, value);
	    Transmit(sbPt);	/* send a message to junior */
	    return( RetFReply() );

	case ptCmdName:
	    sprintf(sbPt, "%c%x|%x\n", ' '+msgPtrace+pt, 0, 0);
	    Transmit(sbPt);	/* send a message to junior */
    	    Receive();	/* get a string from remote */
	    vsbRx[strlen(vsbRx)-1] = 0;	/* clear the \n from the string */
	    strcpy(adr,vsbRx);
	    return;
    } /* switch */

} /* Ptrace */


/* S E N D   S I G N A L */

export int SendSignal(sig)
int2	sig;
{
    char	sbPt[100];

    if (v->fnComm == fnNil)
	return(kill(v->pid, sig));

    sprintf(sbPt, "%c%x\n", ' '+msgSignal, sig);
    Transmit(sbPt);
    /* we DON'T wait for a reply since this is happening at
     * interrupt level.  The target should stop and return
     * status indictaing reciept of this signal.  This assumes
     * that the target has not stopped all by itself prior
     * to the signal.
     */

     /* I do not believe the above comment so I will get reply. billb */
     return(RetFReply());
} /* SendSignal */


/* W A I T */

export int Wait(pstatus)
int	*pstatus;
{
    int		ret;
    char	sbTemp[20];

    if (v->fnComm == fnNil)
	return(wait(pstatus));

    /* here we do what is necessary to *look like* "wait (2)" */

    sprintf(sbTemp, "%c\n", ' '+msgWait);
    Transmit(sbTemp);
    vfInWait = true;
    ret = RetFReply();
    vfInWait = false;

    *pstatus = ret;

    return(errno ? -1: vpidRemote);
} /* Wait */


/* C R E A T E   T A R G E T */

export void CreateTarget()
{
    int		ret;
    char	sbBuf[200];

    if (v->fnComm == fnNil) {
	OsCreateTarget();	/* cause a NATIVE process to be born */
	return;
    } /* if */

    if (vfTest) {
	close(v->fnExec);
	v->fnExec = fnNil;
    } /* if */

    sprintf(sbBuf, "%c%s|%s\n", ' '+msgExec, v->sbExec, v->sbArgsChild);
    Transmit(sbBuf);

    v->pid = RetFReply();
    vpidRemote = v->pid;

    if (errno)
	dprint(1, ("Bad exec in target.  Remote reports error: %d", errno));
} /* CreateTarget */


/* X   B L O C K   M O V E */

export void XBlockMove(pid, adrDest, adrSrc, cb)
int	pid;
ADRT	adrDest, adrSrc;
int	cb;
{
    /* transfer a block WITHIN the remote machine */
    char	sbBuf[cbMsgMax];

    sprintf(sbBuf, "%c%x|%x|%x\n", ' '+msgMove, adrDest, adrSrc, cb);
    Transmit(sbBuf);
    RetFReply();
} /* XBlockMove */


/* X   B L O C K   T R A N S F E R */

export void XBlockTransfer(pid, adrDest, adrSrc, cbIn, fTx)
int	pid;
ADRT	adrDest, adrSrc;
int	cbIn;
FLAGT	fTx;
{
    char	ch1, ch2, sbBuf[cbMsgMax];
    SBT		sbTx, sbRx, sbSrc, sbDest, sbT;
    short	val, cb, cbThisTime, cbSave;
    ADRT	adrDestIn;

    if (fTx) {
	/* write FROM host TO remote */
    } else {
	/* read FROM remote TO host */
	adrDestIn = adrDest;
	cbSave = cbIn;
    } /* if */
    sbSrc = (SBT) adrSrc;

    while (cbIn >= 0) {
	int rcb;
	cb = Min(cbIn, (cbMsgMax-24)/2);
	if (fTx) {
	    cbThisTime = cb;
	    cbIn -= cb;
	    /* write FROM host TO remote */
	    sprintf(sbBuf, "%c%x|%x|", ' '+msgHtoR, adrDest, cb);
	    sbTx = sbBuf + strlen(sbBuf);
	    while (cb--) {
		val = (*sbSrc & 0xf0) >> 4;
		*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
		val = *sbSrc++ & 0x0f;
		*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
	    } /* while */
	    *sbTx = '\n';
	    Transmit(sbBuf);
	    RetFReply();
	} else {
	    /* read FROM remote TO host */
	    sprintf(sbBuf, "%c%x|%x\n", ' '+msgRtoH, adrSrc, cb);
	    Transmit(sbBuf);
	    Receive();
	    if (sscanf(vsbRx, "%x", &rcb) != 1)
	    	Panic("bad target byte count %s", _sbRx);
	    if ((vsbRx = strchr(vsbRx, chDelim)) == sbNil) /* skip count */
	        Panic("missing separator 3 %s", _sbRx);
	    vsbRx++;
	    if ( rcb > cb )
		Panic("too many bytes transferred R to H");
	    cbThisTime = rcb;
	    cbIn -= rcb;
	    sbDest = (SBT) adrDest;
	    while (rcb--) {
		ch1 = *vsbRx++;
		ch2 = *vsbRx++;
		val = ((ch1 - ((ch1 > '9') ? ('a'-10) : '0')) << 4)
		    | (ch2 - ((ch2 > '9') ? ('a'-10) : '0'));
		if ((val & ~0x0ff) != 0 )
		    Panic("bad block transfer R to H");
		*sbDest++ = val;
	    } /* while */
	} /* if */
	adrSrc += cbThisTime;
	adrDest += cbThisTime;
    } /* while */

} /* XBlockTransfer */


/* C L O S E   C O M M */

export void CloseComm()
{
    int		ipr;
    char	sbBuf[cbMsgMax];

    if (v->fnComm != fnNil) {
	sprintf(sbBuf, "%c\n", ' '+msgQuit);
	Transmit(sbBuf);	/* send a message to junior */
    } /* if */

    if (vfTest) {
	close(vfnTx);
	close(vfnRx);
	kill(vpidServer, 9);
    } else {
	ioctl(v->fnComm, TCSETAF, &vTtyarg);
	close(v->fnComm);
    } /* if */
} /* CloseComm */


/* I N I T   C O M M */

export void InitComm(pr)
pPRR	pr;
{
    int		rgFnPtoC[2], rgFnCtoP[2];
    char	sbTx[10], sbRx[10], sbLevel[10];

    if (FSbCmp(pr->sbComm, "TEST")) {

	vfTest = true;
	if (pipe(rgFnPtoC) != 0)
	    Panic("Failed on pipe open: %s", SbFError(errno));

	if (pipe(rgFnCtoP) != 0)
	    Panic("Failed on pipe open: %s", SbFError(errno));

	/* pipes - write 1, read 0 */

	if (vpidServer = fork()) {

	    /* parent */
	    v->fnComm = vfnTx = rgFnPtoC[1];
	    vfnRx = rgFnCtoP[0];
	    close(rgFnCtoP[1]);
	    close(rgFnPtoC[0]);

	} else {

	    /* child */
	    if (v->fnExec != fnNil)
		close(v->fnExec);
	    if (v->fnCore != fnNil)
		close(v->fnCore);
	    close(rgFnCtoP[0]);
	    close(rgFnPtoC[1]);

	    close(0);
	    dup(rgFnPtoC[0]);
	    close(rgFnPtoC[0]);

	    close(1);
	    dup(rgFnCtoP[1]);
	    close(rgFnCtoP[1]);

	    execl("cdbremote", "cdbremote", 0);

	    printf("Execl of remote failed: %d %d\n", sbTx, sbRx);
	    exit(1);

	} /* if */

    } else {
	struct termio ttyarg;
	static int bauds[]={ 300, 600, 1200, 1800, 2400, 4800, 9600};
	static int bcode[]={B300,B600,B1200,B1800,B2400,B4800,B9600};
	int n;

	vfTest = false;

	/* check for startup from cu with channel already open */
	if (pr->sbComm[0] == ':' ) {
		/* cu passed us an open file descriptor */
		v->fnComm = strtol(&pr->sbComm[1],0,0);
	} else {
		v->fnComm = open(pr->sbComm, O_RDWR, 0);
		if (v->fnComm == -1)
		    UError("Cannot open %s: %s.", pr->sbComm, SbFError(errno));
	}

	/* set baud rates, etc */
	ioctl(v->fnComm, TCGETA, &ttyarg);
	vTtyarg = ttyarg;
	for (n = 0; n < sizeof(bauds)/sizeof(int); n++) {
		if (bauds[n] == v->iBaud) {
			ttyarg.c_cflag &= ~CBAUD;
	        	ttyarg.c_cflag |= bcode[n];
		}
	} /* for */

	ttyarg.c_cc[VERASE] = 0177;
	ttyarg.c_cc[VKILL] = 025;
	ttyarg.c_cc[VEOL] = 0;
	ttyarg.c_cc[VEOF] = 0;
	ttyarg.c_iflag |= ISTRIP|IGNCR;
	ttyarg.c_iflag &= ~(INLCR|ICRNL);
	ttyarg.c_oflag &= ~(OPOST);
	ttyarg.c_lflag = ICANON;
	ioctl(v->fnComm, TCSETAF, &ttyarg);
	vfnTx = vfnRx = v->fnComm;
	
    } /* if */

    /* Test the link and (as a side effect) set debuglevel */

    vlevelSave = v->debuglevel;
    sprintf(sbLevel, "%c%x\n", ' '+msgLevel, v->debuglevel);
    Transmit(sbLevel);
    Receive();
} /* InitComm */
