/* $Header: sym.c,v 6.1 86/08/05 17:45:06 peter Exp $ */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

#include <sys/types.h>
#include <sys/stat.h>

#include "cdb.h"

export FLAGT	vfCacheSymbols = true;	/* If TRUE, we are cacheing symbols */

#ifndef bitSb
#define bitSb 0x80000000	/* used to make memory strings into ISS's */
#endif


/*  L I N E   F   I L I N E */

export pLINER LineFIline(iline)
FAST int4 iline;
{
    int4	ilineRead, cRead, offset;

    /* This returns line symbol 'iline' */

    if ((iline < 0) OR (iline >= v->hdr.ilineMax))
	Panic("iline out of bounds (%d).", iline);

    if (! vfCacheSymbols)
	return(v->lineCache + iline);

    if ((iline >= v->ilineLo) AND (iline < v->ilineLim))
	return(v->lineCache + iline - v->ilineLo); /* already in memory */

    ilineRead = (iline / v->ilineCacheMax) * v->ilineCacheMax;
    offset = v->hdr.cbLineOffset + (ilineRead * cbLINER);
    if (lseek(v->fnCdb, offset, 0) < 0L)
	SysPanic("Bad seek in LineFIline.");
    cRead = read(v->fnCdb, v->lineCache, v->ilineCacheMax*cbLINER);

    if (cRead < 0)
	SysPanic("Bad read in LineFIline (cRead: %ld).", cRead);
    v->ilineLo = ilineRead;
    v->ilineLim = ilineRead + (cRead/cbLINER);
    if (v->ilineLim > v->hdr.ilineMax)
	v->ilineLim = v->hdr.ilineMax;
    return(v->lineCache + iline - v->ilineLo);
} /* LineFIline */


/*  A U X   F   I A U X */

export pAUXU AuxFIaux(iaux)
FAST int4 iaux;
{
    int4	iauxRead, cRead, offset;

    /* This guarantees that auxillary symbol 'iaux' is in the cache */

    if ((iaux < 0) OR (iaux >= v->hdr.iauxMax))
	Panic("iaux out of bounds (%d).", iaux);

    if (! vfCacheSymbols)
	return(v->auxCache + iaux);

    if ( (iaux >= v->iauxLo)
       AND ( ((iaux + cAuxMax) < v->iauxLim)
	  OR ((iaux < v->iauxLim) AND (v->iauxLim == v->hdr.iauxMax)) ) )
	return(v->auxCache + iaux - v->iauxLo); /* it is already in memory */

    iauxRead = (iaux / v->iauxCacheMax) * v->iauxCacheMax;
    offset = v->hdr.cbAuxOffset + (iauxRead * cbAUXU);
    if (lseek(v->fnCdb, offset, 0) < 0L)
	SysPanic("Bad seek in AuxFIaux.");
    cRead = read(v->fnCdb, v->auxCache, v->iauxCacheMax*cbAUXU);

    if (cRead < 0)
	SysPanic("Bad read in AuxFIaux (cRead: %ld).", cRead);
    v->iauxLo = iauxRead;
    v->iauxLim = iauxRead + (cRead/cbAUXU);
    if (v->iauxLim > v->hdr.iauxMax)
	v->iauxLim = v->hdr.iauxMax;
    return(v->auxCache + iaux - v->iauxLo);
} /* AuxFIaux */


/* I S Y M   F   S Y M */

export int4 IsymFSym(sym)
pSYMR	sym;
{
    return((sym - v->symCache) + v->isymLo);
} /* IsymFSym */


/* S Y M   F   I S Y M */

export pSYMR SymFIsym(isym)
int4    isym;
{
    int4	isymRead, cRead, offset;

    /* This sets the global "vsym" to point at symbol "isym" */
    if ((isym < 0) OR (isym >= v->hdr.isymMax))
	Panic("isym out of bounds (%d).", isym);

    v->isym = isym;
    if (! vfCacheSymbols) {
	v->sym = v->symCache + isym;
	return(v->sym);
    } /* if */

    if ((isym < v->isymLo) OR (isym >= v->isymLim)) {
	/* rats - cache fault */
	isymRead = (isym / v->isymCacheMax) * v->isymCacheMax;
	offset = v->hdr.cbSymOffset + (isymRead * cbSYMR);
	if (lseek(v->fnCdb, offset, 0) < 0L)
	    SysPanic("Bad seek in SymFIsym.");
	cRead = read(v->fnCdb, v->symCache, v->isymCacheMax*cbSYMR);

	if (cRead < 0)
	    SysPanic("Bad read in SymFIsym (cRead: %ld).", cRead);
	v->isymLo = isymRead;
	v->isymLim = isymRead + (cRead/cbSYMR);
	if (v->isymLim > v->hdr.isymMax)
	    v->isymLim = v->hdr.isymMax;
    } /* if */

    v->sym = v->symCache + isym - v->isymLo;
    return(v->sym);
} /* SymFIsym */


/* S E T   N E X T   S Y M */

export void SetNextSym(isym)
FAST int4    isym;
{
    if (isym < 0 OR isym >= v->hdr.isymMax)
	Panic("isym out of bounds (%d).", isym);

    v->isym = isym - 1;
    if (v->isym < 0)
	v->sym = symNil;
     else
	SymFIsym(v->isym);
} /* SetNextSym */


/* F   N E X T   S Y M */

export FLAGT FNextSym(stHit1, stHit2, stHit3, stStop, fEnterBlocks, sbHit, Cmp)
FAST int2	stHit1, stHit2, stHit3, stStop;
FLAGT	fEnterBlocks;
SBT	sbHit;
FLAGT   (* Cmp)();
{
    FAST int2	st;
    int4	isym;
    pSYMR	sym;

    /* This routine is the main symbol table scanner.  It looks for
     * a symbol of type stHit1, 2 or 3 (optionally checking the name),
     * or for something of type stStop, bailing out at that point.
     *
     * If fEnterBlocks is FALSE and a symbol of type stBlock, stProc,
     * stCommon or stFile is encountered, we skip to the symbol
     * following the matching stEnd.
     * Requesting stBlock as stHit1, 2, 3 or stStop will cause this 
     * setting to be moot.
     */

    isym = v->isym + 1;	/* always startwith the NEXT sym */
    while (isym < v->hdr.isymMax) {
	/* look through what is in the cache */
	
	for (sym = SymFIsym(isym); isym < v->isymLim; sym++, isym++) {
	    
	    st = sym->st;
	    if ( ((st == stHit1)
		   OR (st == stHit2) 
		   OR (st == stHit3)) 
	       AND ((sbHit == sbNil)
		     OR ((*Cmp)(sbHit, SbFSym(sym)))) ) {
		v->isym = isym;
		v->sym = sym;
		return(true);
	    } else if (st == stStop) {
		v->isym = isym;
		v->sym = sym;
		return(false);
	    } else if (!fEnterBlocks
		    AND ((st == stBlock)
		      OR (st == stCommon)
		      OR (st == stProc)
		      OR (st == stFile)) ) {
		isym = IsymNextFBlock(isym);
		if (isym == 0) {
		    v->isym = isymNil;
		    v->sym = symNil;
		    return(false);	/* KLUDGE - for text labels */
		} /* if */
		break;	/* to force cache adjustment, if any */
	    } /* if */
	} /* for */

    } /* while */

    v->isym = 0;
    v->sym = symNil;
    return(false); /* off the end of the symbol table */
} /* FNextSym */


/* S B   F   I S S */

export SBT SbFIss(iss)
FAST int4	iss;
{
    FAST int4	offset, cRead, issSave;
    SBT		sbTest;

    /* given an Index into String Space, return a pointer to the string */

    if (iss == issNil)
	return(sbNil);

    if (iss & bitSb)
	/* this is a memory resident string; remove bit and return */
	return((SBT)(iss & ~bitSb));

    if (v->hdr.issMax == 0)
	return(sbNil);

    if ((iss < 0) OR (iss >= v->hdr.issMax))
	Panic("Iss out of bounds (%d).", iss);

    if (! vfCacheSymbols)
	return(v->sbCache + iss );

    if ((iss >= v->issLo) AND (iss < v->issLim)) {
	/* start of string is in the cache */

	if (v->hdr.issMax == v->issCacheMax) {
	    /* entire string space is in memory */
	    return(v->sbCache + iss);
	} /* if */

	/* make sure the WHOLE string is in the cache */
	issSave = iss;
	sbTest = v->sbCache + (iss - v->issLo);
	while ((iss < v->issLim) AND (*sbTest != chNull)) {
	    sbTest++;
	    iss++;
	} /* while */
	iss = issSave;
	if (*sbTest == chNull)
	    return(v->sbCache + (iss - v->issLo));
    } /* if */

    /* not in (or not ALL in) the cache - make it the first thing */
    offset = v->hdr.cbSsOffset + iss;
    if (lseek(v->fnCdb, offset, 0) < 0L)
	SysPanic("Bad seek in SbFIss.");
    cRead = read(v->fnCdb, v->sbCache, v->issCacheMax);
    if (cRead < 0)
	SysPanic("Bad read in SbFIss (cRead: %d).", cRead);
    v->issLo = iss;
    v->issLim = iss + cRead;
    return(v->sbCache + (iss - v->issLo));
} /* SbFIss */


/* S B   F   I P D */

export SBT SbFIpd(ipd)
int2	ipd;
{
    if (ipd == ipdNil)
	return("");
    return( SbFIss( SymFIsym(v->rgPd[ipd].isym)->iss) );
} /* SbFIpd */


/* S B   F   S Y M */

export SBT SbFSym(sym)
pSYMR	sym;
{
    return(SbFIss(sym->iss));
} /* SbFSym */


/* I S S   F   S B */

export int4 IssFSb(sb)
SBT	sb;
{
    return( bitSb | ((uint4)(sb)) );
} /* IssFSb */


/* S B   S A F E   F   S B */

export SBT SbSafeFSb(sb)
FAST SBT  sb;
{
    SBT		sbSave, sbMax, sbRet;
#define cbSbSafe 2048
    static char	sbSafe[cbSbSafe];	/* where we stick names of things */
    static SBT	sbSafeNext = sbSafe;	/* next available place */

    /* this guy sticks names in a TEMPORARILY safe place */

    sbSave = sb;
    sbMax = sbSafe + cbSbSafe;
    sbRet = sbSafeNext;

    while ((sbSafeNext < sbMax) AND (*sbSafeNext++ = *sb++))
	;

    if (sbSafeNext >= sbMax) {
	/* start over at beginning of buffer */
	sbRet = sbSafeNext = sbSafe;	/* reuse the buffer */
	sb = sbSave;
	while ((sbSafeNext < sbMax) AND (*sbSafeNext++ = *sb++))
	    ;
	if (sbSafeNext >= sbMax)
	    UError("String is too long for internal buffer of %d bytes.",
			cbSbSafe);
    } /* if */

    return(sbRet);
} /* SbSafeFSb */


/* I S S   S A F E   F   S B */

export int4 IssSafeFSb(sb)
SBT	sb;
{
    return( IssFSb( SbSafeFSb(sb) ) );
} /* IssSafeFSb */


/* S B   S A F E   F   I S S */

export SBT SbSafeFIss(iss)
int4	iss;
{
    return( SbSafeFSb( SbFIss(iss) ) );
} /* SbSafeFIss */


/* F   S B   C M P */

export FLAGT FSbCmp(sb1, sb2)
SBT	sb1, sb2;
{
    if ((sb1 == sbNil) OR (sb2 == sbNil))
	return(sb1 == sb2);

    while ((*sb1 | vcaseMod) == (*sb2 | vcaseMod)) {
	if (*sb1 == chNull)
	    return(true);
	sb1++;
	sb2++;
    } /* while */

    return(false);
} /* FSbCmp */


/* F   H D R   C M P */

export FLAGT FHdrCmp(sbHdr, sbCheck)
SBT	sbHdr, sbCheck;
{
    while ((*sbHdr | vcaseMod) == (*sbCheck | vcaseMod)) {
	if (*sbHdr == chNull)
	    return(true);
	sbHdr++;
	sbCheck++;
    } /* while */

    return(*sbHdr == chNull);
} /* FHdrCmp */


/* F   T A I L   C M P */

export FLAGT FTailCmp(sbTail, sbIn)
SBT	sbTail, sbIn;
{
    int		cbIn, cbTail;

    cbIn = strlen(sbIn);
    cbTail = strlen(sbTail);
    if (cbIn < cbTail)
	return(false);
    return(FSbCmp(sbTail, sbIn+(cbIn-cbTail)));
} /* FTailCmp */


/* F   N E X T   L O C A L */

export FLAGT FNextLocal(fLocal, fParam, isym)
FLAGT   fLocal, fParam;
int4	isym;
{
    int2	st;
    static int2	ifd;
    static pPRR	prSave;
    static int4	isymMin, isymMac;

    if ((v != prSave) OR (isym < isymMin) OR (isym >= isymMac)) {
	/* refresh our hint */
	prSave = v;
	ifd = IfdFIsym(isym);
	isymMin = v->rgFd[ifd].isym;
	isymMac = v->rgFd[ifd+1].isym;
    } /* if */

    if (v->rgFd[ifd].lc == lcC) {
	SetNextSym(isym);
	while (FNextSym(stParam, stLocal, stStatic, stProc, true, sbNil, nil)) {
	    if (v->isym >= isymMac)
		return(false);
	    /* we have a param OR a local */
	    st = v->sym->st;
	    if ( ((st == stParam) AND !fParam)
	       OR ((st == stLocal OR st == stStatic) AND !fLocal) ) {
		/* not what we were looking for */
		if (fParam AND !fLocal)
		    return(false); /* NEVER any params after locals */
		continue;
	    } /* if */
	    return(true);
	} /* while */
    } else if (v->rgFd[ifd].lc == lcFortran) {
	SetNextSym(isym);
	while (FNextSym(stParam,stCommon, stStatic, stProc,false, sbNil, nil)) {
	    if (v->isym >= isymMac)
		return(false);
	    /* we have a param OR a local */
	    st = v->sym->st;
	    if ( ((st == stParam) AND !fParam)
	       OR ((st == stStatic OR st == stCommon) AND !fLocal) ) {
		/* not what we were looking for */
		continue;
	    } /* if */
	    return(true);
	} /* while */
    } /* if */
    return(false);
} /* FNextLocal */


/* I P D   F   A D R */

export int IpdFAdr(adr)
FAST ADRT	adr;
{
    FAST int2 	ipd, ipdHi, ipdLo;

    if (adr >= v->rgPd[v->hdr.ipdMax-1].adr)
	return(v->hdr.ipdMax-1);	/* check for past last entry */

    if ((v->ipd != ipdNil)
       AND (v->rgPd[v->ipd].adr <= adr)
       AND (v->rgPd[v->ipd+1].adr > adr))
	return(v->ipd);

    /* binary search the proc table */
    ipdLo = 0;
    ipdHi = v->hdr.ipdMax - 1;
    while (true) {
	ipd = (ipdLo+ipdHi) / 2;
	if (adr < v->rgPd[ipd].adr) {

	    ipdHi = ipd - 1;

	} else if (adr >= v->rgPd[ipd+1].adr) {

	    ipdLo = ipd + 1;

	} else {

	    /* walk backwards to first proc with this adr */
	    while (ipd && v->rgPd[ipd].adr == v->rgPd[ipd-1].adr)
		ipd--;
	    return(ipd);	/* gotchya! */

	} /* if */

	if (ipdLo > ipdHi)
	    UError("Address (%s) does not map to instruction space.",
			SbFAdr(adr, false));
    } /* while */
} /* IpdFAdr */


/* I F D   F   A D R */

export int IfdFAdr(adr)
FAST ADRT	adr;
{
    FAST int2	ifd, ifdHi, ifdLo;
    pFDR	fd;

    if (adr >= v->rgFd[v->hdr.ifdMax-1].adr)
	return(v->hdr.ifdMax-1);	/* check for past last entry */

    if ((v->ifd != ifdNil)
       AND (v->rgFd[v->ifd].adr <= adr)
       AND (v->rgFd[v->ifd+1].adr > adr))
	return(v->ifd);

    /* binary search the file table */
    ifdLo = (v->rgFd[0].adr == adrNil) ? 1 : 0; /* ifd0 is for temp files */
    ifdHi = v->hdr.ifdMax - 1;
    while (true) {
	ifd = (ifdLo+ifdHi) / 2;
	if (adr < v->rgFd[ifd].adr) {

	    ifdHi = ifd - 1;
		
	} else if (adr >= v->rgFd[ifd+1].adr) {

	    ifdLo = ifd + 1;

	} else {

	    /* gotchya! */
	    /* Now we walk forward to the last file i a row with this adr */
	    fd = v->rgFd + ifd;
	    while (fd->adr == fd[1].adr)
		fd++;
	    return(fd - v->rgFd);	/* give them the index */

	} /* if */

	if (ifdLo > ifdHi)
	    UError("Address (%s) does not map to a file.", SbFAdr(adr, false));
    } /* while */
} /* IfdFAdr */


/* I F D   F   I S Y M */

export int IfdFIsym(isym)
FAST int4	isym;
{
    FAST int2	ifd, ifdHi, ifdLo;
    pFDR	fd;

    if (isym >= v->rgFd[v->hdr.ifdMax-1].isym)
	return(v->hdr.ifdMax-1);	/* check for past last entry */

    if ((v->ifd != ifdNil)
       AND (v->rgFd[v->ifd].isym <= isym)
       AND (v->rgFd[v->ifd+1].isym > isym))
	return(v->ifd);

    /* binary search the file table */
    ifdLo = 1; /* ifd0 is for temp files */
    ifdHi = v->hdr.ifdMax - 1;
    while (true) {
	ifd = (ifdLo+ifdHi) / 2;
	if (isym < v->rgFd[ifd].isym) {

	    ifdHi = ifd - 1;
		
	} else if (isym >= v->rgFd[ifd+1].isym) {

	    ifdLo = ifd + 1;

	} else {

	    /* gotchya! */
	    return(ifd);	/* give them the index */

	} /* if */

	if (ifdLo > ifdHi)
	    UError("Isym (%d) does not map to a file.", isym);
    } /* while */
} /* IfdFIsym */


/* I L I N E   L I M   F   I P D */

local int4 IlineLimFIpd(ipd)
int2	ipd;
{
    int4	iline, ilineMax;
    ADRT	adr;
    pPDR	pd = v->rgPd + ipd;
    FAST int	ipdNext;

    /* given an ipd, find the next highest iline NOT in the proc */
    if (pd->iline == 0)
	return(0);	/* no line info */

    /* find the next procedure with symbols */
    if (ipd == v->hdr.ipdMax-1)  {
	return(v->hdr.ilineMax);	/* we are last procedure */
    } else {
	ipdNext = ipd + 1;
	while (ipdNext < v->hdr.ipdMax) {
	    if (v->rgPd[ipdNext].iline != 0)
		break;
	    ipdNext++;
	} /* while */
	if (ipdNext == v->hdr.ipdMax)
	    return(v->hdr.ilineMax);	/* we are last procedure with symbols */
    } /* if */

    return(v->rgPd[ipdNext].iline);
} /* IlineLimFIpd */


/* A D R   F   I P D   I L N */

export ADRT AdrFIpdIln(ipd, iln)
int2     ipd, iln;
{
    FAST int4	iline, ilineLim;
    ADRT	adr;
    pPDR	pd;

    if (ipd == ipdNil)
	return(adrNil);

    pd = v->rgPd + ipd;
    if (pd->iline == 0)
	return(pd->adr);

    iline = pd->iline;
    if (iln < LineFIline(iline)->iln)
	return(adrNil);		/* this iln is BEFORE this procedure */

    ilineLim = IlineLimFIpd(ipd);
    if (iln > LineFIline(ilineLim-1)->iln)
	return(adrNil);	/* this iln is AFTER this procedure */

    /* start at beginning of enclosing proc */
    while ((iline < ilineLim) AND (LineFIline(iline)->iln < iln))
	++iline;

    if (iline >= ilineLim)
	iline = ilineLim - 1;	/* off end of proc */

    adr = pd->adr + LineFIline(iline)->dadr;
    return(adr);
} /* AdrFIpdIln */


/* I P D   I L N   F   A D R */

export int IpdIlnFAdr(piln, pslop, adr)
int2     *piln, *pslop;
ADRT    adr;
{
    int2	ipd;
    FAST int4	iline, ilineLim;
    FAST ADRT	dadr;
    pPDR	pd;
    pLINER	line;

    *piln = ilnNil;
    *pslop = 0;

    if (adr == adrNil)
	return(ipdNil);

    ipd = IpdFAdr(adr);
    pd = v->rgPd + ipd;
    /* delta adr from start of proc */
    dadr = adr - pd->adr;

    if (pd->iline == 0) {  /* proc does not have line numbers */
	*pslop = dadr;
	return(ipd);
    } /* if */

    /* we now have the proc, try to get the line number */

    iline = pd->iline;
    ilineLim = IlineLimFIpd(ipd);

    /* find highest iln that maps to this dadr */
    while ((iline < ilineLim) AND LineFIline(iline)->dadr <= dadr)
	iline++;

    if ((iline > pd->iline)
       AND ((iline >= ilineLim)
	   OR (LineFIline(iline)->dadr > dadr)))
	iline--;	/* overshoot - backup one */

    line = LineFIline(iline);
    *piln = line ->iln;
    *pslop = dadr - line->dadr;
    return(ipd);
} /* IpdIlnFAdr */


/* I F D   I L N   F A D R */

export int2 IfdIlnFAdr(piln, adr)
int2	*piln;
ADRT	adr;
{
    int4	slop;

    IpdIlnFAdr(piln, &slop, adr);
    return(IfdFAdr(adr));
} /* IfdIlnFAdr */


/* A D R   F   I F D   I L N */

export ADRT AdrFIfdIln(ifd, iln)
int2     ifd, iln;
{
    FAST pPDR	pd, pdLim;

    if ((ifd == ifdNil) OR (ifd == ifdTemp))
	return(adrNil);

    if (v->rgFd[ifd].iline == 0)
	return(adrNil);	/* no symbols, so this is a meaningless map */

    /* first find ipd of enclosing procedure */

    pd = v->rgPd + v->rgFd[ifd].ipd;	/* first procedure in file */
    if (iln < LineFIline(pd->iline)->iln)
	return(adrNil);	/* line is BEFORE first procedure */

    if (ifd == v->hdr.ifdMax-1)
	pdLim = v->rgPd + v->hdr.ipdMax - 1;	/* last procedure in file */
    else
	pdLim = v->rgPd + v->rgFd[ifd+1].ipd;

    /* loop til we find one with greater iln's */
    while ((pd < pdLim) AND (LineFIline(pd->iline)->iln < iln))
	pd++;

    if (pd >= pdLim)
	pd = pdLim - 1;	/* off end of file */
    else if (LineFIline(pd->iline)->iln > iln)
	pd--;	/* overshoot */

    /* now we know proc, call main routine */
    return( AdrFIpdIln(pd-v->rgPd, iln) );
} /* AdrFIfdIln */


/* A D R   F   E N D   O F   P R O C */

export ADRT AdrFEndOfProc(ipd)
int2	ipd;
{
    int4	iline;

    iline = IlineLimFIpd(ipd);
    if (iline == 0)
	UError("No symbols - unable to determine end-of-procedure.");
    /* get address of last line */
    return(v->rgPd[ipd].adr + LineFIline(iline-1)->dadr);
} /* AdrFEndOfProc */


/* C B L   F   F N */

export long CblFFn(fn)
int2	fn;
{
    struct stat statbuf;

    fstat(fn, &statbuf);
    return(statbuf.st_size);
} /* CblFFn */


/* A G E   F   F N */

export long AgeFFn(fn)
int2	fn;
{
    struct stat statbuf;

    fstat(fn, &statbuf);
    return(statbuf.st_mtime);
} /* AgeFFn */
