/* $Header: var.c,v 6.2 86/09/14 20:24:02 peter Exp $ */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

#include "cdb.h"

/* In the following array, symbols of type stExpr point to a debugger global,
 * and can therefore have their address directly computed.  Those
 * with type stNil point into the CURRENT process structure (v).
 * These have the field offset of the desired field.  By default, the fields
 * are of type INT.  If they are going to be something else, you
 * have to do extra fiddling aroung in InitSpecials and FSpecial.
 * The first 2 are extra special in that thay are a dynamic calculation
 * of the current FILE and PROCEDURE names.
 */

export int	vfStrict = true;	/* prevents sloppy struct references */
export int	vfTypedefs = true;	/* allows typedefs in scanner */
export int	vfNotify = true;	/* notify me about stopped processes */
export int	vdadrMax = 1024; /* distance to nearest label to use offset */
export int	vfArnulf = false;	/* Arnulf's disassembly flag */

#if (CPU == PDN)
int	vfPdnExtStart = 1;
#endif /* (CPU == PDN) */

#define APR(x)		((uint4)(&(((pPRR)0)->x)))

static pSER vrgSpc;	/* array of System Special variables */
static int vispcMax;


/* I N I T   S P C */

export void InitSpc(se, sb, value, st, sc, tq0, bt, fConstant)
pSER	se;
SBT	sb;
int	st, sc, tq0, bt;
FLAGT	fConstant;
{
    *se = *vseInit;
    se->asym.iss = IssFSb(sb);;
    se->asym.value = value;
    se->asym.st = st;
    se->asym.sc = sc;
    se->ti.fConstant = fConstant;
    se->ti.tq0 = tq0;
    se->ti.bt = bt;
} /* InitSpc */


/* I N I T   S P E C I A L S */

export void InitSpecials()
{
    int		ispc;
    pSER	se;

    vispcMax = 20;	/* This number MUST BE >= number of system specials */
    vrgSpc = (pSER) malloc(vispcMax * cbSER);
    se = vrgSpc;

	/* ----- P R O C E S S   V A L U E S ------ */
#define ispcProcess	0
    InitSpc(se++, "_PROCESS", 0L, stNil, scCdbSystem, tqNil, btInt, true);
#define ispcFile	1
    InitSpc(se++, "_FILE", 0L, stNil, scCdbSystem, tqArray, btChar, true);
#define ispcProcedure	2
    InitSpc(se++, "_PROCEDURE", 0L, stNil, scCdbSystem, tqArray, btChar, true);
#define ispcLine	3
    InitSpc(se++, "_LINE", APR(iln), stNil, scCdbSystem, tqNil, btInt, true);
#define ispcBreak	4
    InitSpc(se++, "_BREAK", 0L, stNil, scCdbSystem, tqNil, btInt, true);
#define	ispcText	5
    InitSpc(se++,"_TEXT",APR(adrText), stNil, scCdbSystem, tqNil, btInt, false);
#define	ispcData	6
    InitSpc(se++,"_DATA",APR(adrData), stNil, scCdbSystem, tqNil, btInt, false);
#define ispcNormalMin	7

    InitSpc(se++, "_TRACE", APR(tracelevel), stNil,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_LEVEL", APR(debuglevel), stNil,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++,  "_LANGUAGE", APR(lc), stNil,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_COMM", APR(fCommDebug), stNil,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_CACHE", (uint4)&vfUseCache, stExpr,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_ARNULF", (uint4)&vfArnulf, stExpr,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_NOTIFY", (uint4)&vfNotify, stExpr,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_STRICT", (uint4)&vfStrict, stExpr,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "_TYPES", (uint4)&vfTypedefs, stExpr,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "SIGNAL", APR(sig), stNil,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "R_SIGNAL", APR(sigOrig), stNil,
				scCdbSystem, tqNil, btInt, false);
    InitSpc(se++, "DELTA", (uint4)&vdadrMax, stExpr,
				scCdbSystem, tqNil, btInt, false);
#if (CPU == PDN)
    InitSpc(se++, "_EXT", (Ulong)&vfExtStart, stExpr,
				scCdbSystem, tqNil, btInt, false);
#endif /* (CPU == PDN) */

    ispc = se - vrgSpc;
    if (ispc >= vispcMax)
       Panic("Under allocated systems specials array: %d > %d.",
		ispc, vispcMax);
    vispcMax = ispc;
} /* InitSpecials */


/* L I S T   S P E C I A L S */

export void ListSpecials(sbSpc)
SBT	sbSpc;
{
    int		i;
    SBT		sb;
    pSER	se;
    SER		ase;

    MoreOn();

    /* first check the System Specials */
    printf("System special variables\n");
    for (i=0; i < vispcMax; i++) {
	se = vrgSpc + i;
	sb = SbFIss(se->asym.iss);
	if (sbSpc != sbNil) {
	    if (!FHdrCmp(sbSpc, sb))
		continue;	/* not interesting */
	} else {
	    if (*sb == '_')
		continue;	/* they didn't ask for it */
	} /* if */
	ase = *se;	/* we do this on a COPY */
	FSpecial(&ase);
	DisplaySe(&ase, modeNil, true, true, true);
	printf("\n");
    } /* for */

    /* next do user defined stuff */
    if ( v->ilvMac > 0) {
	printf("\nUser special variables\n");
	for (i=0; i < v->ilvMac ; i++) {
	    se = v->rgLv + i;
	    if ((sbSpc != sbNil)
	       AND (!FHdrCmp(sbSpc, SbFIss(se->asym.iss))) )
		continue;	/* not interesting */
	    ase = *se;	/* we do this on a COPY */
	    FSpecial(&ase);
	    DisplaySe(&ase, modeNil, true, true, true);
	    printf("\n");
	} /* for */
    } /* if */

    MoreOff();
} /* ListSpecials */


/* F   S P E C I A L */

export FLAGT FSpecial(seRet)
pSER	seRet;
{
    int		i;
    SBT		sbSpc = SbSafeFIss(seRet->asym.iss);
    pSER	seT;
    FLAGT	fDollar = false;

    if (*sbSpc == '$') {
	sbSpc++;
	fDollar = true;
	if (*sbSpc == '_')
	    fDollar = false;	/* hidden system special */
    } /* if */

    /* first check the registers */
    for (i=0; i < v->cpu->iregMax; i++) {
	seT = v->cpu->rgReg + i;
	if (FSbCmp(SbFIss(seT->asym.iss), sbSpc)) {
	    *seRet = *seT;
	    return(true);
	} /* if */
    } /* for */

    /* next, check system specials */
    for (i=0; i < vispcMax; i++) {
	seT = vrgSpc + i;
	if (FSbCmp(SbFIss(seT->asym.iss), sbSpc)) {
	    *seRet = *seT;
	    if (seRet->asym.st == stNil) {
		seRet->asym.st = stExpr;
		switch (i) {
		    default:
			seRet->asym.value += (long)v; /* offset in cur PRR */
			if (seRet->ti.fConstant) {
			    seRet->ti.fConstant = false;
			    seRet->val = ValFSe(seRet);
			    seRet->ti.fConstant = true;
			} /* if */
			break;

		    case ispcProcedure:	/* magic hack for procedure name */
			seRet->asym.value =
				seRet->val.valAdr = (ADRT) SbFIpd(v->ipd);
			break;

		    case ispcFile:	/* magic hack for file name */
			seRet->asym.value =
			 seRet->val.valAdr = (ADRT) SbFIss(v->rgFd[v->ifd].iss);
			break;

		    case ispcProcess:	/* infer process number */
			seRet->asym.value = seRet->val.valInt = IprFPr(v);
			break;

		    case ispcBreak:	/* infer break point number */
			seRet->asym.value = seRet->val.valInt = 
				(v->bp == bpNil) ? -1 : v->bp - v->rgBp;
			break;

		} /* switch */
	    } /* if */
	    return(true);
	} /* if */
    } /* for */

    /* next check the user defined specials */
    for (i=0; i < v->ilvMac; i++) {
	seT = v->rgLv + i;
	if (FSbCmp(SbFIss(seT->asym.iss), sbSpc)) {
	    *seRet = *seT;
	    return(true);
	} /* if */
    } /* for */

    if (!fDollar)
	return(false);	/* keeps typo's from becoming special variables! */

    /* no luck = try to define a new special with this name */
    if (v->ilvMac >= v->ilvMax) {
	SER	ase, *se;

	if (v->ilvMax == 0) {
	    v->ilvMax = 10;
	    v->rgLv = (pSER) malloc(v->ilvMax * cbSER);
	} else {
	    v->ilvMax += 10;
	    v->rgLv = (pSER) realloc(v->rgLv, v->ilvMax * cbSER);
	    if (v->rgLv == 0)
		Panic("Ran out of memory.");
	} /* if */

	ase = *vseInt;
	ase.asym.sc = scCdbLocal;
	ase = *vseInt;
	ase.asym.sc = scCdbLocal;
	for (i = v->ilvMax-10, se = v->rgLv+i; i < v->ilvMax; se++, i++)
	    *se = ase;
    } /* if */

    i = v->ilvMac++;
    seT = v->rgLv + i;
    *seT = *vseInt;
    seT->asym.iss = IssFSb(SbFAlloc(sbSpc));
    seT->asym.sc = scCdbLocal;
    seT->asym.st = stExpr;
    seT->asym.value = (ADRT) seT;	/* points  at itself */
    *seRet = *seT;
    return(true);
} /* FSpecial */


/* S P E C I A L   S I D E   E F F E C T S */

export void SpecialSideEffects(seNew, seCur)
pSER	seNew, seCur;
{
    /* this is called just before ASSIGNing a new value to a special */
    if (FSbCmp("_TRACE", SbFIss(seCur->asym.iss))) {
	v->cCallNest =  -1;	/* force them to reinitialize this */
    } /* if */
} /* SpecialSideEffects */


/* A D J U S T   F I E L D   O F F S E T */

export void AdjustFieldOffset(seStruct, seField)
pSER	seStruct, seField;
{
    if (seField->asym.sc == scBits) {
	/* set address to base of struct */
	seField->asym.value = seStruct->asym.value;
    } else {
	/* set address to byte boundary */
	seField->asym.value = (seField->asym.value / 8) + seStruct->asym.value;
    } /* if */
    if ((seField->ti.tq0 == tqArray) OR (seField->ti.tq0 == tqProc))
	seField->val.valAdr = seField->asym.value;
} /* AdjustFieldOffset */


/* F   F I E L D */

export FLAGT FField(seStruct, seField)
pSER	seStruct, seField;
{
    FLAGT	fFoundIt = false;
    SBT		sbField;

    /* somewhere in this struct/union is the field info, stuff it in seField */

    sbField = SbSafeFIss(seField->asym.iss); /* get the name of the field */

    SetNextSym(seStruct->isymRef + 1);
    if (FNextSym(stMember, stNil, stNil, stEnd, false, sbField, FSbCmp))
	fFoundIt = true;

    if (! fFoundIt AND ! vfStrict AND (v->ifd != ifdNil)) {
	/* No luck! -  search elsewhere for this field name.  */
	SetNextSym(v->rgFd[v->ifd].isym+1);	/* start at beginning of file */

	if (FNextSym(stMember, stNil, stNil, stFile, true, sbField, FSbCmp))
	    fFoundIt = true;
    } /* if */

    if (! fFoundIt) {
	UError("No such field name \"%s\" for \"%s %s\".",
		    sbField,
		    (seStruct->ti.bt==btStruct) ? "struct" : "union",
		    SbFIss(seStruct->asym.iss) );
	return(false);
    } /* if */

    /* Success! - get the type goodies */
    SeFSym(seField, v->sym, true);
    AdjustFieldOffset(seStruct, seField);
    return(true);
} /* FField */


/* I S Y M   N E X T   F   B L O C K */

export long IsymNextFBlock(isym)
int4	isym;
{
    pSYMR	sym = SymFIsym(isym);
    pAUXU	aux;

    if (sym->st == stProc) {
	/* first AUX for proc is isymLast+1 of block */
	aux = AuxFIaux(sym->index);
	return(aux->isym);
    } else {
	return(sym->index);
    } /* if */
} /* IsymNextFBlock */


/* I S Y M   F	 E N C L O S I N G */

export long IsymFEnclosing(adr)
ADRT	adr;
{
    int4	isym, isymTemp;

    /* This routine finds the inner most enclosing scope
     * and returns the isym of the beginning block.
     */

    isym = v->rgPd[IpdFAdr(adr)].isym;

    SetNextSym(isym + 1);
    while (FNextSym(stBlock, stNil, stNil, stProc, true, sbNil, nil)) {
	if (v->sym->sc != scText) {
	    /* skip non-text blocks */
	    isymTemp = IsymNextFBlock(v->isym);
	    if (isymTemp == isymNil)
		return(isymNil);
	    SetNextSym(isymTemp);
	    continue;
	} /* if */

	if (adr < v->sym->value)
	    break;	/* everything from here on is too far in memory */
	
	/* Look at matching END symbol */
	SymFIsym(v->sym->index-1);
	if (adr >= v->sym->value ) {
	    /* we can completely skip this block - it's before us */
	    continue; /* this works because FNextSym moves to next symbol */
	} /* if */

	SymFIsym(v->sym->index);	/* point back to BLOCK */
	isym = v->isym;	/* remember this - it is currently the inner most */
    } /* while */

    return(isym);
} /* IsymFEnclosing */


/* I S Y M   F	 A D R */

export long IsymFAdr(adr, isymHint)
ADRT	adr;
int4	isymHint;
{
    int2	ipd;
    int4	isym;

    /* This routine finds the next outer enclosing scope
     * and returns the isym of the beginning block.
     */

    if (isymHint == isymNil) {
	/* they know nothing - get inner most */
	return( IsymFEnclosing(adr) );
    } /* if */

    /* We now look for block ends */

    /* Start with symbol following the block we last gave them */
    isym = IsymNextFBlock(isymHint);
    if (isym == isymNil)
	return(isymNil);

    SetNextSym(isym);

    while (FNextSym(stEnd, stBlock, stProc, stFile, true, sbNil, nil)) {

	if (v->sym->st == stProc) {
	    /* This may not be correct for PASCAL - 
	     * C Efficiency KLUDGE!
	     */
	    return(isymNil);
	} else if (v->sym->st == stEnd) {
	    if (v->sym->sc == scText)
		return(v->sym->index);	/* isym of matcing block */
	    continue;	/* non-text stuff is uninteresting */
	} else {
	    /* It is a BLOCK start.
	     * By definition, its entire contents are uninteresting.
	     */
	    SetNextSym(IsymNextFBlock(v->isym));
	} /* if */
	
    } /* while */
    return(isymNil);
} /* IsymFAdr */


/* S Y M   F   A D R  */

export pSYMR SymFAdr(adr, sb)
ADRT	adr;
SBT	sb;
{
    int4	isym;

    /* This routine uses 'adr' to establish a scope.
     * It then searches outward from that scope for a
     * variable whose name is *sb.
     */

    if (v->rgFd[IfdFAdr(adr)].lc == lcFortran)
	return((pSYMR)F_SymFAdr(adr, sb));

    isym = isymNil;
    while (true) {
	isym = IsymFAdr(adr, isym);	/* this finds our location in symbols */
	if (isym == isymNil)
	    break;	/* we ran out of scopes to look in */
	SymFIsym(isym);
	if (v->sym->st == stFile)
	    break;	/* we are beginning of file */

	while (true) {
	    if (FNextSym(stParam, stLocal, stStatic, stEnd, false, sb, FSbCmp))
		return(v->sym);
	    if ((v->sym == symNil) OR (v->sym->sc == scText))
		break;
	    /* this is an info block of some sort - continue */
	} /* while */

    } /* while */

    /* If we get here, sb was not the name of a Local, Param,
     * or local Static of any kind.
     */
    return(symNil);
} /* SymFAdr */


/* F   T Y P E   F   N A M E */

export FLAGT FTypeFName(seType, tk)
pSER	seType;
int2	tk;
{
    int2	ipd;
    SBT		sb;
    CNTXR	acntx;
    int4	isym, iss;
    
    iss = seType->asym.iss;;

    sb = SbFIss(iss);
    ipd = v->ipd;
    CntxFIpd(&acntx, &ipd, -1); /* get our start address */

    isym = isymNil;
    while (true) {
	isym = IsymFAdr(acntx.pc, isym);/* this finds our location in symbols */
	if (isym == isymNil)
	    break;	/* we ran out of scopes to look in */
	SetNextSym(isym+1); /* info starts 1 after block start */
	if (FNextSym(stTypedef, stBlock, stNil, stEnd, true, sb, FSbCmp))
	    goto Success;
    } /* while */

    /* not defined inside the current procedure, try the file */
    SetNextSym(v->rgFd[v->ifd].isym+1);
    if (FNextSym(stTypedef, stBlock, stNil, stFile, false, sb, FSbCmp))
	goto Success;

    if (! vfStrict) {
	/* try the entire symbol table */
	SetNextSym(1);
	if (FNextSym(stTypedef, stBlock, stNil, stNil, true, sb, FSbCmp))
	    goto Success;
    } /* if */

    return(false);	/* no luck */

Success:
    if (v->sym->st == stBlock) {
	/* then they were looking for a struct, union or enum */
	*seType = *vseInit;
	seType->asym.iss = iss;
	seType->isymRef = v->isym;
	switch (tk) {
	    default:		Panic("Unknown tk in FTypeFName: %d.", tk);
	    case tkStruct:	seType->ti.bt = btStruct;	break;
	    case tkUnion:	seType->ti.bt = btUnion;	break;
	    case tkEnum:	seType->ti.bt = btEnum;		break;
	} /* switch */
    } else {
	SeFSym(seType, v->sym, true);
    } /* if */
    return(true);
} /* FTypeFName */


/* F   E N U M */

export FLAGT FEnum(se)
pSER	se;
{
    int2	ipd;
    FLAGT	fDoLocalScope;
    SBT		sb;
    CNTXR	acntx;
    int4	isym, iss;
    
    iss = se->asym.iss;;
    sb = SbFIss(iss);
    ipd = v->ipd;

    if (v->pid == pidNil) {
	/* there is no process, look in current proc */
	acntx.pc = AdrFIpdIln(v->ipd, v->iln);
	fDoLocalScope = (acntx.pc == adrNil);
    } else {
	CntxFIpd(&acntx, &ipd, -1); /* get our start address */
	fDoLocalScope = ! acntx.fEndOfStack;
    } /* if */

    if (fDoLocalScope) {
	isym = isymNil;
	while (true) {
	    isym = IsymFAdr(acntx.pc, isym);/* find our location in symbols */
	    if (isym == isymNil)
		break;	/* we ran out of scopes to look in */
	    SetNextSym(isym+1); /* info starts 1 after block start */
	    while (FNextSym(stMember, stNil, stNil, stEnd, true, sb, FSbCmp)) {
		if (v->sym->index == 0)
		    goto Success;
	    } /* while */
	} /* while */
    } /* if */

    /* not defined inside the current procedure, try the file */
    SetNextSym(v->rgFd[v->ifd].isym+1);
    while (FNextSym(stMember, stNil, stNil, stFile, true, sb, FSbCmp)) {
	if (v->sym->index == 0)
	    goto Success;
    } /* while */

    return(false);	/* no luck */

Success:
    isym = v->isym;
    SeFSym(se, v->sym, true);
    se->val.valUInt = se->asym.value;
    se->ti.fConstant = true;
    se->ti.bt = btEnum;
    /* now we get the isym of the start by finding the end */
    SetNextSym(isym);
    FNextSym(stEnd, stNil, stNil, stNil, false, sbNil, nil);
    se->isymRef = v->sym->index; /* the end points back to the start */
    return(true);
} /* FEnum */


/* P A R A M   O F F S E T */

local int ParamOffset(pc)
ADRT	pc;
{
    SER		ase;
    /* this routine allows for procedures which return more than an INT */

    SeFSym(&ase, SymFIsym(v->rgPd[IpdFAdr(pc)].isym), true);
    if ((TqFSe(&ase, 1) == tqNil)
       AND ((ase.ti.bt == btUnion) OR (ase.ti.bt == btStruct))) {
	return(4);
    } else {
	return(0);
    } /* if */
} /* ParamOffset */


/* S E T   A D R */

export void SetAdr(se, cntx)
pSER	se;
pCNTXR	cntx;
{
    /* given an SE and and a properly set group of walking
     * variables in the 'v' record, make the address correct.
     */

    if (se->asym.sc == scRegister) {
	DoReg(se, cntx);	/* finds register values */
    } else if (se->asym.st == stParam) {
	se->asym.value += cntx->ap;
#if (STRUCT_RETURN_OFFSET)
	se->asym.value += ParamOffset(cntx->pc);
#endif /* (STRUCT_RETURN_OFFSET) */
	if ((v->os->fIntegerOffsets)
	   AND (v->cpu->byteSex == byteSexBig)
	   AND ((se->asym.value & (v->cpu->cbInt - 1)) == 0)) {
	    /* Args on the stack are NOT where they appear to be */
	    int cb = CbFSe(se);
	    if (cb < v->cpu->cbInt)
		se->asym.value += (v->cpu->cbInt - cb);
	} /* if */
    } else if (se->asym.st == stLocal) {
	se->asym.value += cntx->fp;
    } /* if */

    if (se->ti.tq0 == tqRef) {
	se->ti.tq0 = tqPtr;
	GetVal(se);
	se->asym.value = se->val.valAdr;
	RemoveTq(se);	/* get rid of qualifier*/
    } /* if */

    if ((se->ti.tq0 == tqArray) OR (se->ti.tq0 == tqProc))
	se->val.valAdr = se->asym.value;
} /* SetAdr */


/* F   L O C A L */

export FLAGT FLocal(se, ipd, cnt)
pSER	se;
int2	ipd, cnt;
{
    pSYMR	sym;
    SBT		sb = SbSafeFIss(se->asym.iss);
    FLAGT	fPhony = false;
    CNTXR	acntx;

    if ((v->pid == pidNil) AND (v->fnCore == fnNil)) {
	/* there is no process, look in current proc */
	fPhony = true;
	acntx.pc = AdrFIpdIln(v->ipd, v->iln);
	if (acntx.pc == adrNil)
	    UError("No symbolic information for current location.");
    } else {
	CntxFIpd(&acntx, &ipd, cnt);
	if (acntx.fEndOfStack)
	    return(false);
    } /* if */

    sym = SymFAdr(acntx.pc, sb);	/* find this variable, based on scope */
    if (sym == symNil)
	return(false);

    /* get all of the type goodies */
    SeFSym(se, sym, true);
    if (fPhony) {
	se->asym.value = adrNil;
    } else {
	SetAdr(se, &acntx);
    } /* if */
    return(true);
} /* FLocal */


/* F   G L O B A L */

export FLAGT FGlobal(se)
pSER	se;
{
    SBT		sb = SbSafeFIss(se->asym.iss);

    /* first we try statics in current file */

    if (v->ifd != ifdNil) {
	SetNextSym(v->rgFd[v->ifd].isym+1);
	if (FNextSym(stStatic, stNil, stNil, stFile, false, sb, FSbCmp)) {
	    SeFSym(se, v->sym, true);
	    return(true);
	} /* if */
    } /* if */

    /* no luck, try globals */
    SetNextSym(v->hdr.isymGl);
    if (FNextSym(stGlobal, stNil, stNil, stNil, false, sb, FSbCmp)) {
	SeFSym(se, v->sym, true);
	return(true);
    } /* if */
    return(false);
} /* FGlobal */


/* A D R   F   L A B E L */

export ADRT AdrFLabel(sbLabel)
SBT	sbLabel;
{
    int2	ipd = IpdFName(sbLabel);

    if (ipd != ipdNil)
	return(v->rgPd[ipd].adr);	/* it was in the proc table */

    /* look for a local label in the current file */
    if (v->ifd != ifdNil) {
	SetNextSym(v->rgFd[v->ifd].isym + 1);
	if (FNextSym(stLabel, stNil, stNil, stFile, true, sbLabel, FSbCmp))
	    return((ADRT)(v->sym->value));
    } /* if */
    return(adrNil);
} /* AdrFLabel */


/* C N T X   F   I P D */

export pCNTXR CntxFIpd(cntx, pipd, cnt)
pCNTXR	cntx;
int2	*pipd, cnt;
{
    int2	depth, ipdX, ipd = *pipd;
    FLAGT	fDontCare;

    if (fDontCare = (cnt == -1))
	cnt = 10000;

    depth = 0;
    *cntx = v->acntx;
    while (! cntx->fEndOfStack) {
	ipdX = IpdFAdr(cntx->pc);
	if (ipdX == ipd) {
	    if ((fDontCare) OR (cnt == depth))
		return;
	} /* if */
	if ((ipd == ipdNil) AND (cnt == depth)) {
	    *pipd= ipdX;
	    return;
	}
	if (cnt < depth)
	    UError("Procedure '%s' is not at that stack depth.", SbFIpd(ipd));
	depth++;
	NextFrame(cntx);
    } /* for */
} /* SetStackFIpd */


/* S B   F   R E G */

export SBT SbFReg(pc, reg)
ADRT	pc;
int	reg;
{
    int4	isym;

    /* This routine uses pc to establish a scope.
     * It then searches outward from that scope for a
     * register variable using the given regietr number.
     */

    if (v->rgPd[IpdFAdr(pc)].iline == 0)
	return(sbNil);	/* no symbols */

    isym = isymNil;
    while (true) {
	isym = IsymFAdr(pc, isym);	/* this finds our location in symbols */
	if (isym == isymNil)
	    break;	/* we ran out of scopes to look in */
	SetNextSym(isym+1); /* info starts 1 after block start */

	while (FNextSym(stParam, stLocal, stNil, stEnd, true, sbNil, nil)) {
	    if ((v->sym->sc == scRegister) AND (v->sym->value == reg))
		return(SbFIss(v->sym->iss));
	} /* while */
    } /* while */

    /* If we get here, sb was not a register thingiemabob */
    return(sbNil);
} /* SbFReg */


/* S B   F   S T A C K   O F F S E T */

export SBT SbFStackOffset(pc, fFp, offset)
ADRT	pc;
FLAGT	fFp;
int2	offset;
{
    int4	isym;

    /* This routine uses 'adr' to establish a scope.
     * It then searches outward from that scope for a
     * variable with the given stack offset.
     */

    isym = isymNil;
    while (true) {
	isym = IsymFAdr(pc, isym);	/* this finds our location in symbols */
	if (isym == isymNil)
	    break;	/* we ran out of scopes to look in */
	SetNextSym(isym+1); /* info starts 1 after block start */

	while (FNextSym(stParam, stLocal, stNil, stEnd, true, sbNil, nil)) {
	    if (v->sym->value == offset)
		return(SbFIss(v->sym->iss));
	} /* while */
    } /* while */

    /* If we get here, sb was not the offset of a Local or Param */
    return(sbNil);
} /* SbFStackOffset */


/* S B	 F   L A B E L	 A D R */

export SBT SbFLabelAdr(adr)
ADRT	adr;
{
    return(SbFNearest(adr, true, false, true));
} /* SbFLabelAdr */


/* S B	 F   N E A R E S T */

export SBT SbFNearest(adr, fText, fData, fProcOnly)
ADRT	adr;
FLAGT	fText, fData, fProcOnly;
{
    int2	ipd, ifd;
    SBT		sbT;
    ADRT	dadr;
    SYMR	asymNearest;
    static char sbLabel[100];

    /* given an address, find the NEAREST text OR data label to it.
     * return a formatted string containing the goodies.
     */

    asymNearest.iss = issNil;
    asymNearest.value = 0;

    ipd = ipdNil;
    if (fText) { 
	ipd = IpdFAdr(adr);
	asymNearest = * SymFIsym(v->rgPd[ipd].isym);

	if (!fProcOnly AND ((ifd = IfdFAdr(adr)) != ifdNil) ) {
	    /* check local text addresses for closer match */
	    SetNextSym(v->rgFd[ifd].isym + 1);
	    while (FNextSym(stLabel, stNil, stNil, stFile, true, sbNil, nil)) {
		if ((v->sym->value <= adr)
		   AND (v->sym->value >= asymNearest.value)) {
		    asymNearest = *(v->sym);
		} /* if */
	    } /* while */
	} /* if */
    } /* if */


    if ( fData				 /* they want us to check data labels */
       AND ( !fText			/* we didn't check text space */
	  OR !v->fPureText	/* text and data addresses might intermingle */
          OR (ipd == v->hdr.ipdMax-1)) ) {    /* we mapped to last procedure */
	/* check statics in this file */
	SetNextSym(v->rgFd[v->ifd].isym+1);
	while (FNextSym(stStatic, stLabel, stNil, stFile, true, sbNil, nil)) {
	    if ((v->sym->value <= adr)
	       AND (v->sym->value >= asymNearest.value)) {
		asymNearest = *(v->sym);
	    } /* if */
	} /* while */
	/* check global symbols */
	SetNextSym(v->hdr.isymGl);
	while (FNextSym(stGlobal, stNil, stNil, stNil, false, sbNil, nil)) {
	    if ((v->sym->value <= adr)
	       AND (v->sym->value >= asymNearest.value)) {
		asymNearest = *(v->sym);
	    } /* if */
	} /* while */
    } /* if */

    dadr = adr - asymNearest.value;
    sbT = SbFIss(asymNearest.iss);
    if (dadr == 0)
	sprintf(sbLabel, "%s", sbT);
    else if (dadr < vdadrMax)
	sprintf(sbLabel, "%s+%s", sbT, SbFAdr(dadr, true));
    else
	sprintf(sbLabel, "%s", SbFAdr(adr, true));
    return(sbLabel);
} /* SbFNearest */


/* D I S P   F R A M E */

export void DispFrame(cntx, cnt, fDoLocals, fList)
pCNTXR	cntx;
int2	cnt;
FLAGT	fDoLocals, fList;
{
    int2	ipd, ifd, iln, slop, i;
    int4	isym, isymStart;
    SER		aseTemp, ase, *se = &ase;

    /* First we set the cache for those going ovewr remote lines */

    ipd = IpdFAdr(cntx->pc);
    if (v->rgPd[ipd].iline == 0) {
	DispUnknown(cntx, ipd);
	return;
    } /* if */

    if (!fList)
	printf("%s(", SbFIpd(ipd));
    isymStart = isym = v->rgPd[ipd].isym + 1;
    i = 0;
    while (FNextLocal(false, true, isym)) {
	isym = v->isym + 1;
	SeFSym(se, v->sym, true);
	if (!fList AND i++)
	    printf(", ");
	SetAdr(se, cntx);
	if (!fList)
	    printf("%s=", SbFIss(se->asym.iss));
	DisplaySe(se, modeNil, fList, true, fList);
	if (fList) {
	    /* check to see if this is correct for current stop point */
	    aseTemp.asym.iss = se->asym.iss;
	    FLocal(&aseTemp, ipd, cnt);
	    if (aseTemp.asym.value != se->asym.value)
		printf("	*** Caution: variable is hidden or out of scope *** ");
	    printf("\n");
	} /* if */
    } /* while */

    if (!fList) {
	printf(")");
	if (v->cpu->fCalleeStackCleanup)
	    cntx->pc -= 1;
	ipd = IpdIlnFAdr(&iln, &slop, cntx->pc);
	ifd = IfdFAdr(cntx->pc);
	printf("	[%s:%d]\n", SbFIss(v->rgFd[ifd].iss), iln);
    } /* if */

    if (fDoLocals) {
	if (!fList)
	    vcNest++;	/* in case we have a structure to indent */
	if (v->rgFd[IfdFAdr(cntx->pc)].lc == lcFortran)
	    isym = isymStart;
	while (FNextLocal(true, false, isym)) {
	    isym = v->isym + 1;
	    SeFSym(se, v->sym, true);
	    if ( !fList
	       AND ((se->ti.tq0 != tqNil)
	          OR ((se->ti.bt != btStruct) AND (se->ti.bt != btUnion))) )
		printf("%s", "    ");
	    SetAdr(se, cntx);
	    DisplaySe(se, modeNil, true, true, true);
	    if (fList) {
		/* check to see if this is correct for current stop point */
		aseTemp.asym.iss = se->asym.iss;
		FLocal(&aseTemp, ipd, cnt);
		if (aseTemp.asym.value != se->asym.value)
		    printf("	*** Caution: variable is hidden or out of scope *** ");
	    } /* if */
	    printf("\n");
	} /* while */
	if (!fList)
	    vcNest--;
    } /* if */
} /* DispFrame */


/* S T A C K   T R A C E */

export void StackTrace(cnt, fDoLocals)
int2	cnt;
FLAGT	fDoLocals;
{
    int		i;
    ADRT	adrMin;
    CNTXR	acntx;

    if ((v->pid == pidNil) AND (v->fnCore == fnNil))
	UError("No process.");
    if (v->ps == psRunning)
	UError("Cannot do a stack trace on a running process.");

    MoreOn();

    acntx = v->acntx;
    adrMin = v->acntx.sp;
    for (i=0; ((i < cnt) AND (! acntx.fEndOfStack)); i++) {
	/* force the cache to the current frame */
	/* handles 95% of frames */
	SetCommCache(adrMin, Max((acntx.fp - adrMin) + 32, 128) );
	adrMin = acntx.fp; /* next loop's min value */

	if (cnt > 1)
	    printf("%2d ", i);
	DispFrame(&acntx, i, fDoLocals, false);
	if (NextFrame(&acntx) == true)
	    printf("----------- Command Line Procedure Call -----------\n");
    } /* for */
    MoreOff();
} /* StackTrace */


/* O P E N   S T A C K */

export void OpenStack(cnt)
int2	cnt;
{
    int2	cntIn = cnt;
    ADRT	adrMin;
    CNTXR	acntx;

    acntx = v->acntx;
    adrMin = v->acntx.sp;
    while (cnt-- AND (! acntx.fEndOfStack)) {
	adrMin = acntx.fp; /* next iteration's min stack value */
	NextFrame(&acntx);
    } /* while */

    if (acntx.fEndOfStack)
	UError("Stack isn't that deep!");

    if (v->cpu->fCalleeStackCleanup AND (cntIn != 0))
	acntx.pc -= 1;

    SetViewPosition(acntx.pc);
    ShowViewPosition(fmtFile+fmtProc+fmtLn+fmtLine+fmtEol);
    v->stackdepth = cntIn;

    /* force the cache to the current frame */
    SetCommCache(adrMin, (acntx.fp - adrMin) + 32); /* handles 95% of frames */
} /* OpenStack */


/* S T A C K   D E P T H */

export int StackDepth()
{
    int		cnt;
    CNTXR	acntx;

    cnt = 0;
    if (v->pid == pidNil)
	return(0);

    for (acntx = v->acntx; ! acntx.fEndOfStack; NextFrame(&acntx))
	cnt++;
    return(cnt);
} /* StackDepth */

