/* $Header: waitMulti.c,v 5.9 86/06/02 23:26:03 peter Exp $ */
/* (C) Copyright 1985 Third Eye Software, Inc. - All Rights Reserved */

/* This module contains most of the code used for MULTI-PROCESS debugging.
 * This module is mutually exclusive with swait.c.
 */


#include "cdb.h"

#include <signal.h>

#if (OP_SYS == BSD42)
#include <sys/time.h>
#include <sys/wait.h>
#endif (OP_SYS == BSD42)

#if (OP_SYS == SYSV)
#include <fcntl.h>
#endif (OP_SYS == SYSV)

#include <errno.h>

export int2	viprMax = 100;	/* allow lots of room */


/* P P I D   F   P I D */

export PpidFPid(pid)
int	pid;
{
    int		ppid;

    ppid = Ptrace(ptPpid, pid, 0, 0);
    if (ppid == -1)
	UError("Attempting to find parent of %d: %s.", SbFError(errno));
    return(ppid);
} /* PpidFPid */


/* S B   E X E C   F   P I D */

export SBT SbExecFPid(pid)
int	pid;
{
    static char	sbCmd[257];

    Ptrace(ptCmdName, pid, sbCmd, 0);
    return(sbCmd);
} /* SbExecFPid */


/* F R E E   P R O C E S S */

export void FreeProcess(pr)
pPRR	pr;
{
    if (pr == prNil)
	return;

    SetGlobalPr(pr);

    RemoveAllBp(true);
    /* setpgrp(v->pid, v->pgrpSave); /* */
    Ptrace(ptReleaseChild, pr->pid, (char *) 1, 0);
    if (pr->pidParent != vpidCdb) {
	DeallocPr(pr, false);
    } else {
	pr->pid = pidNil;
	pr->ps = psNoChild;
    } /* if */

    SetGlobalPr(prNil);
} /* FreeProcess */


/* C O P Y   P R */

export void CopyPr(pr1, pr2)
pPRR	pr1, pr2;
{
    int2	cb;

    /* Make pr1 identical to pr2 */

    *pr1 = *pr2;	/* this gets all of the normal fields */

    /* KLUDGE - need to create CopyOs, CopyCpu */

    /* KLUDGE - need to check ALL fields and make our own copies
     * so that DeallocPr is free to throw everything away.
     */
    
    pr1->fForkedProcess = true;

    /* setup tables */
    cb = pr1->ibpMax * cbBPR;	/* breakpoints */
    pr1->rgBp = (pBPR) malloc(cb);
    MoveBytes(pr1->rgBp, pr2->rgBp, cb);

    cb = pr1->ilvMax * cbSER;	/* local variables */
    pr1->rgLv = (pSER) malloc(cb);
    MoveBytes(pr1->rgLv, pr2->rgLv, cb);

    cb = pr1->iadMax * cbADR;	/* assertions */
    pr1->rgAd = (pADR) malloc(cb);
    MoveBytes(pr1->rgAd, pr2->rgAd, cb);

    pr1->seDot = (pSER)malloc(cbSER);
    pr1->sbArgsChild = SbFAlloc(pr2->sbArgsChild);
} /* CopyPr */


/* P R   F O R K */

export pPRR PrFFork(pidChild, fAttached)
int2	pidChild;
FLAGT	fAttached;
{
    int		pidParent;
    SBT		sbExec;
    pPRR	pr, prParent;

    /* A new child has come into the world.
     * Register it with the draft board.
     */

    /* First we have to find out who the parent is */
    pidParent = PpidFPid(pidChild);
    prParent = PrFPid(pidParent);

    if (fAttached AND (prParent != prNil)) {
	/* We know the family, but the kid has been
	 * away from home for a long time.
	 * Check his id at the door */
	sbExec = SbExecFPid(pidChild);
	if (! FSbCmp(sbExec, prParent->sbExec))
	    prParent = prNil;
    } /* if */

    if (prParent == prNil) {

	/* A stranger in a strange land */
	pr = PrFInitPr( SbExecFPid(pidChild) );

    } else {

	/* A friend of the family */
	pr = PrFAlloc();
	CopyPr(pr, prParent);

	if (pr->fRemoveBpsOnFork) {
	    SetGlobalPr(pr);
	    RemoveAllBp(true);
	    SetGlobalPr(prNil);
	} /* if */

	pr->fnComm = pr->fnCdb = pr->fnExec = pr->fnCore = fnNil;

    } /* if */

    pr->pid = pidChild;
    pr->pidParent = pidParent;
    pr->ps = psFork;
    if (vfNotify) {
	printf("New process\n");
	ListPr(pr);
    } /* if */

    ++vcActive;		/* count of Active proceesses */
    return(pr);
} /* PrFFork */


/* P R   F   A T T A C H   P R O C E S S */

export pPRR PrFAttachProcess(pid)
int2	pid;
{
    int		ret;
    SBT		sb;
    pPRR	pr, prSave;

    ret = Ptrace(ptMultiChild, pid, 0, 0);	/* make him ours */
    if (ret == -1)
	UError("Unable to attach to process %d: %s.", pid, SbFError(errno));

    /* Chicken and Egg problem: We can't call Waiter, because we
     * don't have a process record because we don't have the process
     * name because we haven't waited on the process yet.
     * Sooooooo, we send SIGINT, wait a couple of seconds and
     * hope that the process has stopped.
     */
    kill(pid, SIGINT);
    sleep(2);

    pr = PrFFork(pid, true);
    pr->ps = psRunning;

    prSave = v;
    SetGlobalPr(pr);
    ReadPc();
    SetCntx();
    SetViewPosition(pr->acntx.pc);
    v->pgrpSave = getpgrp(pid);
    /* setpgrp(pid, getpgrp(0)); /* make him the same process group as us */
    return(pr);
} /* PrFAttachProcess */


/* P R   F   E X E C */

export pPRR PrFExec(pr)
pPRR	pr;
{
    extern char * gets();
    int2	pidParent, pidChild = pr->pid;
    char	*sbExec, sbBuf[200];
    FLAGT	fStopOnExec;		/* stop immediately after exec */
    FLAGT	fStopOnFork;		/* stop BOTH process after fork */
    FLAGT	fRemoveBpsOnFork;	/* child does NOT get parent's BPs */
    FLAGT	fDebugChildren;		/* indicates we WANT the kids */
    pSER	rgLv;
    FLAGT	fDoingSingle;
    
    pidParent = pr->pidParent;	/* Remember Mom */

    sbExec = SbExecFPid(pr->pid);	/* Find out who we are */
    if (sbExec == sbNil) {
	printf("Unable to determine name of program for process #%d\n",
	    pidChild);
	printf("Please type name of executable: ");
	sbExec = gets(sbBuf);
	if ((sbExec == sbNil) OR (*sbExec == chNull)) {
	    Ptrace(ptReleaseChild, pidChild, (char *) 1, 0);
	    UError("Aborting debugging of process #%d.", pidChild);
	} /* if */
    } /* if */

    fStopOnExec = pr->fStopOnExec;
    fStopOnFork = pr->fStopOnFork;
    fRemoveBpsOnFork = pr->fRemoveBpsOnFork;
    fDebugChildren = pr->fDebugChildren;
    fDoingSingle = pr->fDoingSingle;
    rgLv = pr->rgLv;

    DeallocPr(pr, true);		/* kill old identity */

#if 0
    pr = PrFAlloc();		/* a clean slate */
    pr->sbExec = SbFAlloc(sbExec);	/* our new name */
    pr->sbCore = sbNil;
    pr->isymCacheMax = 0;
    InitPr(pr);
#endif
    pr = PrFInitPr(sbExec);
    pr->pid = pidChild;
    pr->pidParent = pidParent;

    pr->fStopOnExec = fStopOnExec;
    pr->fStopOnFork = fStopOnFork;
    pr->fRemoveBpsOnFork = fRemoveBpsOnFork;
    pr->fDebugChildren = fDebugChildren;
    pr->fDoingSingle = fDoingSingle;
    pr->rgLv = rgLv;
    pr->ps = psRunning;

    SetGlobalPr(pr);
    ReadPc();
    SetCntx();
#if 0
    SetGlobalPr(prNil);
#endif 0

    if (vfNotify) {
	printf("Process exec'ed\n");
	ListPr(pr);
    } /* if */
    return(pr);
} /* PrFExec */

#ifdef BILLB
int DeadChild, InGetchar;

/*
	This routine catches the child died signal and sets a flag to 
	that effect. If there is potential that the process is about
	to enter getchar after checking the DeadChild flag then an alarm 
	is started to prevent the hang.
*/
CatchChildSig()
{
	signal(SIGCLD,CatchChildSig);
	DeadChild++;
	if (InGetchar) alarm(1);
}
#endif BILLB



/* N O   H A N G   G E T S */
export SBT NoHangGets(sb, fRedoPrompt, fHang)
SBT	sb;
FLAGT	fRedoPrompt, fHang;
{
#if (OP_SYS == SYSV)
    int		ch;
    SBT		sbTemp;

    /* this routine ASSUMES that we already have O_NDELAY set */

    if (fHang) {
#ifndef BILLB
	if (fcntl(0, F_SETFL, fcntl(0, F_GETFL, 0) & ~O_NDELAY) == -1)
	    SysPanic("fcntl on ~O_NDELAY");
	sleep(1); /* to disturb the ether and make O_NDELAY go away */
#endif BILLB
	if (fRedoPrompt)
	    printf("\n%s", vsbPrompt);
	return(gets(sb));
    } else {
#ifdef BILLB
	/* check for a dead child already occuring */
	InGetchar++;
	if (DeadChild) {
		InGetchar = 0;
		alarm(0);
		return((SBT)-1);
	}
#endif BILLB

	/* wait for the first character */
	sbTemp = sb;
	ch = getchar();

#ifdef BILLB
	/* char retrieved or child died */
	InGetchar = 0;
	alarm(0);
#endif BILLB

	if (ch != EOF) {

	    /* There is something there.  Let them type in peace */
#ifdef BILLB
	    signal(SIGCLD,SIG_DFL);
#else
	    if (fcntl(0, F_SETFL, fcntl(0, F_GETFL, 0) & ~O_NDELAY) == -1)
		SysPanic("fcntl clearing O_NDELAY");
#endif BILLB

	    while ((*sb++ = ch) != '\n')
		ch = getchar();

	    sb[-1] = 0;	/* step on ending <cr> */

#ifdef BILLB
	    signal(SIGCLD,CatchChildSig);
#else
	    if (fcntl(0, F_SETFL, O_NDELAY | fcntl(0, F_GETFL, 0)) == -1)
		SysPanic("fcntl setting O_NDELAY");
#endif BILLB
	    return(sbTemp);

	} /* if */
	return((SBT)-1);
    } /* if */
#else
    int		nfd, readfd, writefd, exceptfd;
    struct timeval	atimeval, *ptimeval;

    nfd = 1;		/* used for the select call */
    writefd = 0;
    exceptfd = 0;
    atimeval.tv_sec = 0;
    ptimeval = &atimeval;

    readfd = 1;
    if (!fHang) {
	ptimeval->tv_usec = 500000;	/* 1/2 second */
    } else {
	ptimeval = 0;	/* we wait forever */
	if (fRedoPrompt)
	    printf("\n%s", vsbPrompt);
    } /* if */

    fflush(stdout);
    if (0 != select(nfd, &readfd, &writefd, &exceptfd, ptimeval))
	return(gets(sb));
    return((SBT)-1);	/* says we didn't see anything before timeout */
#endif /* (OP_SYS == SYSV) */
} /* NoHangGets */


/* N O   H A N G   W A I T */

export int NoHangWait(pstatus)
int	*pstatus;
{
#if (OP_SYS == SYSV)
    int		ret;

    alarm(1);
    ret = wait(pstatus);
#ifdef BILLB
    if ((ret == -1) AND (errno == EINTR)) {
	DeadChild = 0;	/* not cleared until we are sure no signals lost */
	return(0);	/* timed out */
    }
#else
    if ((ret == -1) AND (errno == EINTR))
	return(0);	/* timed out */
#endif BILLB
    return(ret);
#else
    return( wait3(pstatus, WNOHANG, nil) );
#endif /* (OP_SYS == SYSV) */
} /* NoHangWait */


/* W A I T E R */

export long Waiter(wt, pidWait)
int2	wt, pidWait;
{
    int		pid, ppid, waitReturn, status;
    FLAGT	fRunningChildren, fRedoPrompt;
    pPRR	pr, prSave = v;

    /* Wait for something to happen in a
     * MULTIPLE PROCESS environment
     */

    vfUseCache = false;	/* just in case they turned it on */
    fRedoPrompt = false;
    fRunningChildren = true;	/* assume we do until we fail on wait3 */

#if (OP_SYS == SYSV)
#ifndef BILLB
    if (fcntl(0, F_SETFL, O_NDELAY | fcntl(0, F_GETFL, 0)) == -1)
	SysPanic("fcntl on O_NDELAY");
#endif BILLB
#endif (OP_SYS == SYSV)

    while (1) {
	if (wt == wtTty) {
	    /* there are no children we need to deal with */
	    /* See if we should get something from the terminal */
	    if ((SBT)-1 != NoHangGets(vsbGet, fRedoPrompt, !fRunningChildren))
		break;
	    /* a negative return indicates no command line yet */
	} /* if */

	waitReturn = 0;
	if (fRunningChildren) {
	    if (vcActive <= 1)
		signal(SIGINT, SIG_IGN);	/* ignore INT */

	    waitReturn = NoHangWait(&status);

	    if (vcActive <= 1)
		signal(SIGINT, Fixer);	/* reconnect to the Fixer routine */
	    if (waitReturn == -1) {

		if (errno == ECHILD) {
		    /* it's no crime not to have children.... */
		    fRunningChildren = false;
		    waitReturn = 0;
		} else if (errno == EINTR) {
		    printf("PANIC! received signal: %d (%s).\n",
			status>>8, SbFSignal(status>>8));
		    /* what do we do here?? */
		    exit(1);
		} else {	/* some sort of error occurred */
		    perror(vsbDebugName);
		    exit(1);
		} /* if */
	    } /* if */
	} /* if */
	
	if (waitReturn > 0) {
	    pid = waitReturn;
	    pr = PrFPid(pid);
	    if (pr == prNil) {
		/* We weren't expecting this guy.
		 * The highest probability is that he
		 * is the result of a fork.  
		 */

		pr = PrFFork(pid, false);
		if (pr == prNil) {
		    /* PrFFork didn't like him */
		    /* What should we do here? Kill him? */
		    continue;
		} /* if */
	    } /* if */

	    if (pr->ps == psStopped)
		continue;

	    SetGlobalPr(pr);	/* make him the boss */

	    if (v->fStty)
		SetMyTty();
	    v->status = status;

	    AnalyzeStop();
	    fRedoPrompt = true;
	    fRunningChildren = true;	/* as a default */

	    if (v->ps == psContinue) {
		/* He's not supposed to stop yet, onward! */
		RunProcess(ptResume, psRunning, false);
		continue;
	    } /* if */

	    if ((wt == wtProcess) AND (pid == pidWait)) {
		break;	/* we were waiting for this guy */
	    } else if (vfNotify AND (vcActive > 1)) {
		pr = PrFPid(pid);
		if (pr != prNil) {
		    printf("Process stopped:\n");
		    ListPr(pr);
		} /* if */
	    } /* if */
	} /* if */
    } /* while */

    SetGlobalPr(prSave);
} /* Waiter */
