/* $Header */
/* (C) Copyright 1984 by Third Eye Software, Inc. - All Rights Reserved */

/* This module is the REMOTE end of the CDB Standard Remote Debugging package.
 * It is set up to run as a normal UNIX user process.  To modify it for
 * use in other environments, simply implement the functionality without
 * changing the interface.  A full description of the protocol may be
 * found in the file "xptrace.h".
 */

#include <signal.h>
#include <errno.h>
#include <stdio.h>
#include "xptrace.h"

#define O_RDONLY	0
#define chNull '\0'
typedef char *SBT;
#define sbNil ((SBT)0)
#define true 1
#define false 0
#define AND &&
#define OR ||

extern int errno;
extern void SendStatus();

#define DEBUG 1

#if DEBUG
#define dprint(x, y) {if (x <= vlevel) fprintf y; fflush(stderr);}
#else
#define dprint(x, y)
#endif

int	vpidChild = 0;	/* KLUDGE - this limits us to one sub-process */
int	vfCommDebug;	/* if true, causes echoing of all messages */
int	vlevel = 0;		/* debug print level for dprint */
FILE	*vfpTx, *vfpRx;	/* they may be the same in some cases */


#define cArgsMax	100	/* max number of args to child */
#define cbBufMax 1024	/* pick a number suited to the size of your system */
char	vsbBuf[cbBufMax];	/* buffer where we format messages */
SBT	vsbTx = vsbBuf;


#if 1
/* Change this to "if 0" when this is a subroutine of another program */
extern FILE *fdopen();


/* M A I N */

void main(argc, argv)
int	argc;
char	**argv;
{
    int		i, fUsedArg, cMainArgs;
    SBT		sbArg, sbTemp, sbUsage, sbTx, sbRx, sbTty;

    sbUsage = "Usage: cdbremote [-d#] [device-name]";

    sbTx = sbRx = sbNil;

    cMainArgs = 0;
    for (i=1; i<argc; ++i) {
	if (*(sbArg=argv[i]) != '-') {
	    if (cMainArgs == 0) {
		sbTty = sbArg;
	    } else {
		printf(sbUsage);
		exit(1);
	    } /* if */
	    cMainArgs++;
	} else {
	    sbArg++;	/* move past the '-' */

	    /* point to a potential argument */
	    sbTemp = (sbArg[1] != chNull) ? sbArg+1 : argv[i+1];
	    fUsedArg = true; /* set FALSE if option does NOT use arg */

	    switch (*sbArg) {
		default:
		    printf(sbUsage);
		    exit(1);
		    break;

		case 'T':
		    sbTx = sbTemp;
		    break;

		case 'R':
		    sbRx = sbTemp;
		    break;

		case 'd':
		    vlevel = atoi(sbTemp);
		    vfCommDebug = true;
		    break;

	    } /* switch */

	    if ((fUsedArg) AND (sbArg[1] == chNull))
		i++;	/* eat next word - it was the arg we used */
	} /* if */
    } /* for */

    if (sbTty != sbNil) {
	dprint(1, (stderr, "Opening %s for READ and WRITE\n", sbTty));
	vfpTx = fopen(sbTty, "r+");
	if (vfpTx == NULL) {
	    perror(sbTty);
	    exit(1);
	} /* if */
	vfpRx = vfpTx;
    } else {
	if (sbTx != sbNil) {
	    dprint(1, (stderr, "Using channel %s for WRITE\n", sbTx));
	    vfpTx = fdopen(atoi(sbTx), "w");
	    if (vfpTx == NULL) {
		perror(sbTx);
		exit(1);
	    } /* if */
	} else {
	    dprint(1, (stderr, "Using stdout for WRITE\n", sbTx));
	    vfpTx = stdout;
	} /* if */

	if (sbRx != sbNil) {
	    dprint(1, (stderr, "Using channel %s for READ\n", sbRx));
	    vfpRx = fdopen(atoi(sbRx), "r");
	    if (vfpTx == NULL) {
		perror(sbRx);
		exit(1);
	    } /* if */
	} else {
	    dprint(1, (stderr, "Using stdin for READ\n", sbTx));
	    vfpRx = stdin;
	} /* if */
    } /* if */

    DebugServer(vfpTx, vfpRx);
} /* main */
#endif


/* T X   S T R I N G */

int TxString(sb)
SBT	sb;
{
    dprint(1, (stderr, "TxRemote: %s", sb));

    fputs(sb, vfpTx);
    fflush(vfpTx);
} /* TxString */


/* R X   S B */

void RxSb(sb)
SBT	sb;
{
    SBT		sbRet;

    sbRet = fgets(sb, 200, vfpRx);
    if (sbRet == NULL) {
	/* the line went away */
	dprint(1, (stderr, "Comm line closed\n"));
	exit(1);
    } /* if */

    dprint(1,(stderr, "RxRemote:%s", sb));
} /* RxSb */


/* C D B   P A N I C */

int RemotePanic(sbFmt, arg1, arg2, arg3)
SBT	sbFmt;
int	arg1, arg2, arg3;
{
    fprintf(stderr, "Remote Panic - ");
    fprintf(stderr, sbFmt, arg1, arg2, arg3);
    fprintf(stderr, "\n");
    fflush(stderr);
} /* RemotePanic */


/* V A L   F   P S B   */

long ValFPsb(psb)
SBT	*psb;
{
    int		x = 0;
    char	ch, *sb = *psb;

    /* this routine eats a hex number from a string and advances
     * the string pointer to the next character AFTER the first
     * non-hex character.  E.g if we start with "1234|abcd",
     * we will return the integer value 0x1234 and leave the string
     * pointing to the 'a'.
     */

#define FHex(ch) (((ch >='0') AND (ch <= '9')) OR ((ch >= 'a') AND (ch <= 'f')))

    if ((sb == sbNil) OR (*sb == chNull))
	return(0);

    while (true) {
	ch = *sb++;
	if (! FHex(ch))
	    break;
	x = (x * 16) + ((ch <= '9') ? (ch - '0') : ((ch - 'a') + 10));
    } /* while */
    *psb = sb;
    return(x);
} /* ValFPsb */


/* R E A D   H E X   B Y T E S */

void ReadHexBytes(pid, adr, count, sb)
int	pid, count;
long	adr;
SBT	sb;
{
    char	ch1, ch2, *sbDest;
    short	val;

    /* used with the host-to-remote block transfer command */
    sbDest = (SBT) adr;
    while (count--) {
	ch1 = *sb++;
	ch2 = *sb++;
	val = ((ch1 - ((ch1 > '9') ? ('a'-10) : '0')) << 4)
	       + (ch2 - ((ch2 > '9') ? ('a'-10) : '0'));
	*sbDest++ = val;
    } /* while */
} /* ReadHexBytes */


/* W R I T E   H E X   B Y T E S */

void WriteHexBytes(pid, adr, count)
int	pid, count;
long	adr;
{
    SBT		sbTx, sbSrc;
    short	val;

    /* used with the remote-to-host block transfer command */

    sbSrc = (char *) adr;
    sbTx = vsbTx;
    sprintf(sbTx, "%x|%x|%x|", statusOk, pid, count);
    sbTx = sbTx + strlen(sbTx);
    while (count--) {
	val = (*sbSrc & 0xf0) >> 4;
	*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
	val = *sbSrc++ & 0x0f;
	*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
    } /* while */
    *sbTx = '\n';
    TxString(sbTx);
} /* WriteHexBytes */


/* L O C A L   P T R A C E */

void LocalPtrace(pt, pid, adr, value)
int	pt, pid, value;
long	adr;
{
    int		status;
    
    /* This routine does whatever is necessary to implement
     * the ptrace(2) system call.
     *
     * For UNIX systems, this means we just call ptrace itself.
     */

    /* On a majority of systems, Instruction and Data Space
     * are identical.  On systems where this is not true, you
     * will need to deal with them separatly.
     */

    dprint(10, (stderr, "pt: %d  adr: %x  value: %d\n", pt, adr, value));

    errno = 0;
    switch (pt) {
	default:	RemotePanic("Unknown ptrace request: %d.", pt);	break;

	case  ptReadI:
	case  ptReadD:
		/* Read memory */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptWriteI:
	case  ptWriteD:
		/* Write memory */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptReadUser:
		/* Read Registers */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptWriteUser:
		/* Write Registers */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptSingle:
		/* Single Step the child */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptResume:
		/* Continue an existing child */
		value = ptrace(pt, pid, adr, value);
		break;

	case  ptTerm:
		/* Terminate an existing child */
		value = ptrace(pt, pid, adr, value);
		vpidChild = 0;
		break;
    } /* switch */

    status = ((value == -1 ) AND (errno != EIO)) ? statusOk : errno;
    SendStatus(status, pid, value);
} /* LocalPtrace */


/* L O C A L    W A I T */

int LocalWait(pstatus)
int	*pstatus;
{
    /* This routine does whatever is necessary to implement
     * the wait(2) system call.
     *
     * For UNIX systems, this means we just call wait itself.
     */

    return(wait(pstatus));
} /* LocalWait */


/* S E N D   S T A T U S */

void SendStatus(status, pid, value)
int	status, pid, value;
{
    sprintf(vsbTx, "%x|%x|%x\n", status, pid, value);
    TxString(vsbTx);
} /* SendStatus */


/* N E X T   A R G */

char * NextArg(psbIn)
SBT	*psbIn;
{
    SBT		sbRet;

    /* skip white space */
    while (**psbIn == ' ' OR **psbIn == '\t')
	(*psbIn)++;

    if (**psbIn == chNull)
	return(sbNil);	/* that's all, folks! */

    sbRet = *psbIn;	/* set current arg pointer */
    while (**psbIn != chNull
       AND **psbIn != ' '
       AND **psbIn != '\t')
	(*psbIn)++;	/* skip all non-white space characters */

    **psbIn = chNull;	/* overwrite the tab, space, or 0 with a 0 */
    (*psbIn)++;
    return(sbRet);
} /* NextArg */


/* D O	 A R G S */

void DoArgs(sbProgram, sbArgs, argv, fReassign)
SBT	sbProgram, sbArgs;
char	**argv;
int	fReassign;
{
    int		i;
    SBT		sbFile, sbArgCur;
    char	sbBuffer[1024];

    for (i=0; i < cArgsMax; i++)
	argv[i] = sbNil;

    *argv++ = sbProgram; /* set the filename */

    if (sbArgs == sbNil)
	return;

    strcpy(sbBuffer, sbArgs);	/* copy it to a safe place to hack on it */
    sbArgs = sbBuffer;
    sbArgs[strlen(sbArgs)+1] = chNull; /* this puts a SECOND null at end */

    /* eat the arguments */
    while ((sbArgs != sbNil)
       AND (*sbArgs != chNull)) {
	sbArgCur = NextArg(&sbArgs);
	if (sbArgCur == sbNil)
	    break;

	/* check for indirection */
	if (!fReassign
	   OR ((*sbArgCur != '<') AND (*sbArgCur != '>')) ) {
	    *argv++ = sbArgCur;	/* we just pass it on */
	} else {
	    /* find that file name */
	    sbFile = sbArgCur + 1;  /* initially we assume `<foo' vs `< foo' */
	    if (*sbFile == chNull) {
		/* there was white space between < and sbFile */
		sbFile = NextArg(&sbArgs);
		if (sbFile == sbNil) {
		    printf("Illegal indirection\n");
		    exit(1);
		} /* if */
	    } /* if */

	    if (*sbArgCur == '<') {
		/* changing stdin */
		close(0);
		if (open(sbFile, O_RDONLY) < 0) {
		    perror(sbFile);
		    exit(1);
		} /* if */
	    } else if (*sbArgCur == '>') {
		/* changing stdout */
		close(1);
		if ( (open(sbFile, 2) < 0) 	/* O_WRONLY */
		   AND (creat(sbFile, 0666) < 0) ) {
		    perror(sbFile);
		    exit(1);
		} /* if */
	    } /* if */
	} /* if */
    } /* while */
    *argv = sbNil;
} /* DoArgs */


/* P I D   F   E X E C   C H I L D */

int PidFExecChild(sbExec, sbArgs)
SBT	sbExec, sbArgs;
{
    int		pid, fn, sig;
    char	*argvChild[cArgsMax];

    /* Parse the argument list passed into an argv array.
     * Create a child process, initialize it to a more or less virgin state,
     * (i.e. clean up the signals, close the files,....) and return the pid
     * to the caller.
     */

    if ((pid = fork()) == 0) {
	/* we are the CHILD process */

	for (sig=1; sig<NSIG; sig++)
	    signal(sig, SIG_DFL);	/* start out with default response */

	for (fn = 3; fn < _NFILE; fn++)
	    close(fn);

	DoArgs(sbExec, sbArgs, argvChild, true);
	ptrace(ptChild, 0, (char *)0, 0); /* Hi, Mom! */
	execv(sbExec, argvChild);

	/* if we get here, we are child BUT exec failed */
	perror(sbExec);
	RemotePanic("Exec failed.");
	exit(1);
    } /* if */

    if (pid == -1) {
	perror(sbExec);
	RemotePanic("Fork failed.");
    } /* if */

    return(pid);
} /* PidFExecChild */


/* D E B U G   S E R V E R */

int DebugServer(fpTx, fpRx)
FILE	*fpTx, *fpRx;
{
    int		pid, result, status, msg, pt, value, count;
    int		sig, sigExit = 0;
    long	adr, adr2;
    char	*sb, sbBuf[cbBufMax], sbName[cbBufMax], sbArgs[cbBufMax];

    /* This routine is the main command loop.
     * It acts as the Parent process to the target process.
     */

    dprint(1, (stderr, "Entering debug server\n"));

#ifdef unix
    signal(SIGINT, SIG_IGN);
#endif

    vfpRx = fpRx;
    vfpTx = fpTx;

    /* startup wait */
    result = LocalWait(&value);

    while (true) {
	RxSb(sbBuf);	/* get a message from the host */
	sb = sbBuf;
	msg = ValFPsb(&sb);	/* get message number */
	pid = ValFPsb(&sb);	/* get process id */
	dprint(1, (stderr, "msg = %d, pid = %d\n", msg, pid));

	status = statusOk;	/* to start with */

	switch (msg) {
	    default:
		SendStatus(statusRetry, pid, 0);
		break;

	    case msgPtrace:
		pt = ValFPsb(&sb);
		adr = ValFPsb(&sb);
		value = ValFPsb(&sb);

		LocalPtrace(pt, pid, adr, value);
		break;

	    case msgWait:
		result = LocalWait(&value);
		sigExit = (value >> 8) & 0xff;
		SendStatus(statusOk, pid, value);
		break;

	    case msgQuit:
		if (pid == 0) {
		    sprintf(vsbTx, "%x|%x\n", statusOk, pid);
		    TxString(vsbTx);
		    return;	/* all done */
		} /* if */
		LocalPtrace(ptTerm, pid, (char *)0, 0);
		break;


	    case msgExec:
		if (vpidChild != 0) {
		    /* KLUDGE - if you need more than one at a time,
		     * this will not work correctly.
		     */
		    LocalPtrace(ptTerm, vpidChild, (char *)0, 0);
		    vpidChild = 0;
		} /* if */

		/* the pid returned should either be that specified by the host
		 * OR one determined by the local environment.
		 */
		
		/* sb now points at "program|args" */

		sscanf(sb, "%[^|]|%[^|]", sbName, sbArgs);
		sbArgs[strlen(sbArgs)-1] = '\0'; /* clear the <nl> */
		dprint(1, (stderr, "Execing: %s, args:%s\n", sbName, sbArgs));

		vpidChild = PidFExecChild(sbName, sbArgs);
		if (pid != vpidChild) {
		    pid = vpidChild;
		    status = statusNewPid;
		} /* if */

		/* If we send a value of 0, it means that parent MUST send
		 * us msgWait.  A value of -1 says that it* must NOT send
		 * us the wait message.  What you send is based on how
		 * your PidFExecChild works.  The UNIX version needs the wait.
		 */
		SendStatus(status, pid, 0);
		break;

	    case msgInput:
		/* KLUDGE - this should be fed to the program, somehow */
		dprint(1, ("KLUDGE - inputing: %s", sb));
		SendStatus(statusOk, pid, 0);
		break;

	    case msgHtoR:
		adr = ValFPsb(&sb);
		count = ValFPsb(&sb);
		ReadHexBytes(pid, adr, count, sb);
		SendStatus(statusOk, pid, 0);
		break;

	    case msgRtoH:
		adr = ValFPsb(&sb);
		count = ValFPsb(&sb);
		WriteHexBytes(pid, adr, count);
		break;

	    case msgMove:
		adr = ValFPsb(&sb);
		adr2 = ValFPsb(&sb);
		count = ValFPsb(&sb);
		/* MoveBlock((char *)adr, (char *)adr2, count); */
		SendStatus(status, pid, 0);
		break;

	    case msgSignal:
		sig = ValFPsb(&sb);
		kill(pid, sig);
		SendStatus(statusOk, pid, 0);
		break;

	    case msgLevel:
		vlevel = ValFPsb(&sb);
		SendStatus(statusOk, pid, 0);
		break;

	    case msgDump:
		/* Fill in with any appropriate goodies */
		/* starting here ---->v			*/
		sprintf(vsbTx, "%x|%x|RemoteStat pc:%x fp:%x sp:%x\n",
		    statusPrint, pid, 0, 0, 0);
		TxString(vsbTx);
		SendStatus(statusOk, pid, 0);
	} /* switch */
    } /* while */
} /* DebugServer */
