#
/*
Name:
	TENTAPPOINT Calendar System Command

Function:
	An authorized user makes an appointment for a user of the calendar system

Algorithm:
	Determine if the user entering the appointment is an authorized user.
	If he is, then prompt him for the information needed for the appointment
	such as the date, the start and end time, and the appointment itself.
	Then add this information to the end of the calendar user's temporary
	appointment file.

Parameters:
	userid	the userid of the calendar user

Returns:
	ENOERR	Normal exit
	ENOENT	Bad user home directory or exec/appointments directory
	EFLAG	Unknown user entered for userid
	EBRK	Not an authorized user
	EFORM	Incorrect command line
	EACCES	Cannot open auth.inputters or temp.appts file
	ESPIPE	Illegal seek in temp.appts file

Files and Programs:
	calendar user's home-dir/exec/auth.inputters		authorized users list
	calendar user's home-dir/exec/appointments/(file)	appointments file
	calendar user's home-dir/exec/appointments/temp.appts	place where temporary
								appointments are stored

Installation Instructions:
	Compile as follows:
	cc -o $EXECDIR/tentappoint tentappoint.c library
	Change owner to root
	Deny write permissions except to owner

History:
	Oct 79,	Release 1 Version 1, G. Maltagliati, coded in shell
	Nov 80, Release 5 Version 1, G. Maltagliati, coded in C for efficiency
	Aug 82, Release 7 Version 2, J. Neyer...re-write exec support

*/
#include <stdio.h>
#include <pwd.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define APPOINT "exec/appointments"
#define APMAX 80		/* longest appointment allowed */
#define AUTHUSER "exec/auth.inputters"
#define NOTAUTH "You are not an authorized user\n"
#define TEMP "temp.appts"
#define USAGE "You must enter: tentappoint <userid of person you are making appointment for>\n"

char id[] = "%W% %H%";

/*

Name:
	main

Function:
	Does preliminary error checking, determines if the user entering the
	appointment is an authorized user and finally gets the needed information
	and adds the appointment to the end of the calendar user's temporary
	appointments file.

Algorithm:
	Determine if the userid entered on the command line is a valid userid.
	If it is valid, then determine if the user entering the appointment is
	on the calendar user's  authorized user list. If so, ask for the
	date of the temporary appointment. If any appointments exist on that
	day then output the appointments and prompt the user to see if he would
	like to continue. If so or if no appointments existed on that day, then
	call writetemp to ask the rest of the information and write it out.
	Otherwise, exit.

Parameters:
	userid	userid of the calendar user

Returns:
	ENOERR	Normal exit
	EFLAG	Unknown user entered for userid
	ENOENT	Bad user home directory or exec/appointments directory
	EBRK	Not an authorized user
	EFORM	Incorrect command line

Files and Programs:
	calendar user's home-dir/exec/appointments/(file)	appointments file


*/
main (argc, argv)
char   *argv[];						/* Pointer to list of argument pointers */
int     argc;						/* Count of command line arguments */
{
    char    date[7],					/* Holds the standardized date */
            file[15],
	    ans_buf[256],
	   *answer = ans_buf,                           /* Pointer to the answer supplied by the user */
            buffer[512];				/* I/O buffer */
    struct passwd *pw;		/* /etc/passwd data */
    register int    fd,					/* File descriptor for appointment file */
                    num_chars;				/* Number of characters read in buffer */


    if (argc == 2)					/* Were the correct number of arguments entered */
    {							/* Correct number entered */
	if ((pw = getpwnam(argv[1])) == NULL)
	{
	    printf ("tentappoint: unknown user: %s\n", argv[1]);
	    exit (EFLAG);
	}
	if (chdir (pw->pw_dir))			/* go to login directory */
	{
	    printf ("tentappoint: bad directory--%s\n", pw->pw_dir);
	    exit (ENOENT);
	}
	if (!authuser())        {
	    fprintf(stderr, NOTAUTH);
	    exit (EACCES);
	}
	if (chdir (APPOINT) == -1)			/* Move to exec/appointments directory */
	{
	    printf ("tentappoint: bad directory--%s\n", APPOINT);
	    exit (ENOENT);
	}
	printf ("Enter the date of the appointment-(mm/dd/yy): ");
							/* Get a valid date */
	while (!gets (answer) || !valid_date (answer, date, 0))
		printf ("Invalid date format\nEnter the date, format mm/dd/yy. (ex 9/29/79): ");
	copy ("appts.", file);				/* Create the name of the appointments file */
	copy (date, &file[6]);
	if ((fd = open (file, 0)) == -1)		/* Does the appointments file exist */
	    printf ("No appointments for %s on %s\n", argv[1], &file[6]);
	else
	{						/* File exists */
	    printf ("The appointments for %s on %s are:\n\n", argv[1], &file[6]);
	    while (num_chars = read (fd, buffer, 512))  /* Display all appointments */
		write (1, buffer, num_chars);
	    close (fd);
	    do
		printf ("\nDo you wish to continue? (y or n): ");
	    while (!gets(answer) || (*answer != 'y' && *answer != 'n'));
	    if (*answer == 'n')
	    {						/* User does not wish to continue so exit */
		exit (ENOERR);
	    }
	}
	writetemp (&file[6]);				/* write the new appointment out to the temporary file */
    }
    else
    {							/* Incorrect number of arguments were entered */
	fprintf(stderr, USAGE);
	exit (EFORM);
    }
    exit (ENOERR);
}
/*

Name:
	writetemp

Function:
	Get the information needed for the appointment and write the appointment
	at the end of the temporary appointments file

Algorithm:
	Prompt the user for a start and end time and the appointment. Build
	the appointment record as the information is entered. Now, add the 
	record built to the end of the temporary appointment file.

Parameters:
	date	the date of the appointment

Returns:
	EACCES	Cannot open temp.appts file
	ESPIPE	Bad seek in temp.appts file

Files and Programs:
	calendar user's home-dir/exec/appointments/temp.appts	temporary appointment file


*/
extern char *gettime();
extern char *itoa();
extern int   atoi();
writetemp (date)
char   *date;						/* Date of the appointment */
{
    char    ans_buf[256],
	   *answer = ans_buf,                           /* Pointer to the answer supplied by the user */
            appoint[APMAX],				/* Appointment record */
           *appt_ptr;					/* Pointer to appoint */
    register int    fd;					/* File descriptor for temp.appts file */
    int start_time, end_time;                    /* the time limits of the appt */
    char store_stm[5],                           /* store start time */
	 store_etm[5],                           /* store end time   */
	 *time_ptr,                              /* pointer to time  */
	 temp;
    char *ans;                                   /* pointer to answer*/
    int  ival,
	 temp_stm,
	 temp_etm,
	 time_flag;
    register int  i;                             /* index for copying time */

    appt_ptr = copy (date, appoint);			/* Start building the record */
    printf ("Enter the start time of the appointment: ");
    ans = gets(answer);                          /* enter start time */
    time_flag = 1;
    time_ptr = gettime(ans, time_flag);                     /* check time validity */

    for (i=0; *(time_ptr + i) != '\0'; i++) {     /* loop to copy after-checked time to an array */
	store_stm[i] = *(time_ptr + i);
	if (store_stm[i] == 'a' || store_stm[i] == 'p')
	   temp = store_stm[i];
    }

    for(;;){                                     /* loop to check start_time validity */
	temp_stm = atoi(store_stm);
	if (atoi(store_stm) == 1200 && temp == 'a')
	   temp_stm = 0;
	else
	   if (atoi(store_stm) == 1200 && temp == 'p')
	      temp_stm = 1200;
	if (temp_stm >= 0 && temp_stm <= 2400)   /* check start_time limits */
		break;
	else  {                                  /* if out-of-bound, enter again */
		printf("Invalid time format\nEnter the start time: ");
		ans = gets(answer);
		time_ptr = gettime(ans, time_flag);
		for (i=0; *(time_ptr + i) != '\0'; i++) {
		    store_stm[i] = *(time_ptr + i);
		    if (store_stm[i] == 'a' || store_stm[i] == 'p')
		       temp = store_stm[i];
		}
       }
   }
    ival = atoi(store_stm);
    if (temp == 'p' && ival >= 1300)
       ival -= 1200;
    appt_ptr = copy (itoa(ival,4), appt_ptr);
    *appt_ptr++ = temp;

    printf ("Enter the end time of the appointment: ");
    ans = gets(answer);                           /* enter end_time */
    time_flag = 0;
    time_ptr = gettime(ans, time_flag);                      /* check time validity */
    for (i=0; *(time_ptr + i) != '\0';i++) {       /* loop to copy time */
	store_etm[i] = *(time_ptr + i);
	if (store_etm[i] == 'a' || store_etm[i] == 'p')
	   temp = store_etm[i];
    }
    for(;;){                                      /* loop to check if end time is out-of-bound */
	temp_etm = atoi(store_etm);
	if (atoi(store_etm) == 1200 && temp == 'a')
	   temp_etm = 0;
	else
	   if (atoi(store_etm) == 1200 && temp == 'p')
	      temp_etm = 1200;

	if (temp_etm >  temp_stm && temp_etm <= 2400)
		break;
	else {                                     /* if out_of_bound, enter again */
		 printf("Invalid end time\nEnter the end time: ");
		 ans = gets(answer);
		 time_ptr = gettime(ans, time_flag);
		 for (i=0; *(time_ptr + i) != '\0';i++) {
		     store_etm[i] = *(time_ptr + i);
		     if (store_etm[i] == 'a' || store_etm[i] == 'p')
			temp = store_etm[i];
		 }
	     }
    }
    ival = atoi(store_etm);
    if (temp == 'p' && ival >= 1300)
       ival -= 1200;
    *appt_ptr++ = '-';

    appt_ptr = copy (itoa(ival,4), appt_ptr);
    *appt_ptr++ = temp;

/* Finish building the record except for the actual appointment */
    *appt_ptr++ = ':';
    *appt_ptr++ = ' ';
    *appt_ptr++ = ' ';
    *appt_ptr = '\0';
    do							/* Get the appointment */
	printf ("Enter the appointment on %s from %s\n", date, &appoint[5]);
    while (!fgets(answer, 256, stdin) || *answer == '\0');
/* Finish building the record */
    appt_ptr = copy (answer, appt_ptr);
    *appt_ptr = '\0';
    if ((fd = open (TEMP, 1)) == -1) {
	if ( (fd = creat(TEMP, 0666)) == -1) {
		printf("Cannot access or create %s\n", TEMP);
		exit (EACCES);
	}
    }
    else if (lseek (fd, 0, 2) < 0)				/* Move read/write pointer to EOF */
    {
	printf ("tentappoint: seek error \n");
	exit (ESPIPE);
    }
    write (fd, appoint, strlen (appoint));		/* Write the appointment out */
    close (fd);
}
