#
/*
Name:
	DAILYCAL Calendar System Command

Function:
	Output a user's daily appointment calendar

Algorithm:
	Gather all appointments for the day given placing them in a buffer.
	Output the buffer.

Parameters:
	date            the date of the calendar (default today)
	userid          the userid of the calendar user (default self)

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or home-dir/exec directory, cannot open 
		suspense file, overdue suspense file, standard appointment
		time file, three month calendar file
	EFORM	Bad date format, invalid userid, or wrong number of command line arguments
	EACCES	Cannot create temporary file
	EFLAG	Not an authorized user

Files and Programs:
	home-dir/exec/suspenses/(file)		suspense files
	home-dir/exec/auth.inputters		authhorized users list
	home-dir/exec/appointments/(file)	appointment file

Installation Instructions:
	Compile as follows:
	cc -o $EXECDIR/dailycal dailycal.c library
	Change owner to root
	Deny write permissions except for owner

History:
	Oct 79,	Release 1, Version 1, G. Maltagliati, coded in shell
	Jun 80,	Release 3, Version 1, Sgt T. Irish, added wrap-around
		feature for dolist items
	Sep 80,	Release 4, Version 1, Sgt T. Irish, added wrap-around
		for all items
	Nov 80,	Release 5, Version 1, G. Maltagliati, coded in C for efficiency
	Jul 82, Release 7, Version 1, L. Reed for Perkin-Elmer
	Aug 82, Release 7, Version 2, J. Neyer...rewrite exec support


*/
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include "../includes/error.h"
#include <time.h>
#include <pwd.h>
#include "../includes/getdef.h"
#define START "\nAppointments calendar for user %s for date %s\n\n"
#define APPOINT "appointments/appts."
#define AUTHUSER "exec/auth.inputters"
#define EXEC "exec"
#define USAGE "You must enter: dailycal <date> <userid>\n"
#define DATEERR "dailycal: Invalid date format\n"
#define USERINVL "dailycal: Invalid userid\n"
#define NOTAUTH "dailycal: You are not an authorized user\n"

char id[] = "%W% %H%";


/*

Name:
	main

Function:
	Perform error checking on the command line arguments and create
	a buffer containing the calendar for the date and output the buffer.

Algorithm:
	Check to make sure the correct number of arguments were entered.
	If dailycal has been called by todaycal, skip the following.
	Standardize the date entered on the command line. If the date
	was the only argument entered, then find the userid of person logged
	on. Otherwise, check the userid entered and then check to make
	sure the user logged on is an authorized user of the userid 
	entered on the command line. Call dailycal to create the buffer
	with the appointments for the day.
	Finally call nroff to output the buffer.

Parameters:
	date		the date of the calendar
	userid		the userid of the calendar user

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or home-dir/exec directory
	EFORM	Bad date format, invalid userid, or wrong number of command line arguments
	EFLAG	Not an authorized user

Files and Programs:
	None


*/
extern char *copy();
main (argc, argv, envp)
char   *argv[];						/* Pointer to list of argument pointers */
int     argc;						/* Count of command line arguments */
char **envp;		/* the environment */
{
    struct  passwd  *pw;                                /* Password file structure */
    char    name[10];                                   /* Saves name from password struc so authuser doesn't clobber it */
    char   *execdir;                                    /* will point to $EXECDIR */
    struct  tm *tp;                                     /* Ctime return struct */
    char    date_array[8];                              /* The date in the week for dailycal to start in */
    char   *date;                                       /* Points to date */
    char   *vd;                                         /* Date validity indicator */
    char    *user;                                      /* Points to user name */
    int     tvec[2];                                    /* Time vector for time system call */
    int     index,i;
    char    tdate[10];                                  /* Holds default date */
    char   *tempdate = tdate;
    char   *cp;                                         /* Temp char pointer */

    extern struct tm *localtime();
    execdir = findenv("EXECDIR", envp);         /* the programs are here */
    date = user = NULL;
    switch(argc)        {
	case 1:
	    break;
	case 2:                                         /* One parameter passed */
	    date = date_array;
	    vd = valid_date (argv[1], date, 0);            /* Is it the date? */
	    if (!vd)    {
		date = NULL;
		if ((pw = getpwnam(argv[1])) == NULL)       {
		    fprintf(stderr, USAGE);
		    exit (EFORM);
		}
		else
		    user = argv[1];
	    }
	    break;
	case 3:                                         /* Two parameters passed */
	    date = date_array;
	    vd = valid_date (argv[1], date, 0);             /* Is the first parameter the date? */
	    if (!vd)    {
		date = NULL;
		if ((pw = getpwnam(argv[1])) == NULL)       {
		    fprintf(stderr, USAGE);
		    exit (EFORM);
		}
		else
		    user = argv[1];
	    }
	    if (!vd)      {                                 /* If first parameter was not date */
		vd = valid_date(argv[2], date, 0);
		if (!vd)        {
		    fprintf(stderr, DATEERR);               /* Second parameter not date either */
		    exit(EFORM);
		}
	    }
	    else                                            /* First parameter was date */
		if ((pw = getpwnam(argv[2])) == NULL)       {
		    fprintf(stderr, USERINVL);
		    exit (EFORM);
		}
		else
		    user = argv[2];
	    break;
	default:
	    fprintf(stderr, USAGE);
	    exit(EFORM);
    }
    if (!date)          {                                   /* If date was defaulted */
	date = date_array;
	time(tvec);
	tp = localtime(tvec);
	tp->tm_yday += 1;
	for (i=0; i<11; i++)
	  tdate[i] = '';

	tempdate = copy(itoa(tp->tm_year,2), tempdate);
	tempdate = copy(itoa(tp->tm_yday,3), tempdate);
	 if (!valid_date (tdate, date, 1)){
	    printf(stderr, DATEERR);
	    fprintf(stderr, "dailycal: serious system error in date conversion\n");
	    exit(EFORM);
	 }
    }
    if (!user)
	pw = getpwnam(findenv("USER", envp));
    if (chdir (pw->pw_dir))         {            /* move to $HOME */
	printf ("dailycal: bad directory--%s\n", pw->pw_dir);
	exit (ENOENT);
    }
    cp = pw->pw_name;
    for (index = 0; *cp != '\0'; index++)               /* Save authuser name since authuser() call getpwuid which clobbers old results */
	name[index] = *cp++;
    name[++index] = '\0';
    if (user && !(authuser ()))
    {                                               /* Not an authorized user */
	fprintf(stderr, NOTAUTH);
	exit (EFLAG);
    }
    if (chdir (EXEC) == -1)				/* Move to the exec directory */
    {
	printf ("dailycal: bad directory--%s\n", EXEC);
	exit (ENOENT);
    }
    printf(START, name, date);                   /* Output the header information */
    dailycal (date, envp);                              /* Output the buffer with appts, suspenses, and dolist items */
    exit (ENOERR);
}
/*

Name:
	dailycal

Function:
	Output a buffer of the given dates appointments for a given user.

Algorithm:
	Find the julian date for the current date and the number of days
	in the year. If the date is not a Monday then decrement the julian
	date by one, else decrement it by three. Find the standardized date
	for the julian date and place all suspenses for that date on the 
	overdue suspense list.  Now call update to update the dolist and
	overdue suspense list.  Call get_appt to put the appointments for
	the day in the buffer. Call get_sus_do to put the suspenses and
	dolist items in the buffer.

Parameters:
	date		the date of the calendar

Returns:
	None

Files and Programs:
	None


*/
dailycal (date, envp)
char   *date;						/* The date of the calendar */
char **envp;
{
#define BUFLEN 1024

    char    buffer[BUFLEN],
	   *buf_ptr = buffer;
    char   *line,					/* Pointer to line extracted from the files */
            appt_line[256],				/* Holds a line from the appointment file */
            time_line[256];				/* Holds a line from the standard appointment time file */
    register int    aflag = 1,				/* Indicates more lines are left in the standard time file */
                    bflag = 1;				/* Indicates more lines are left in the appointment fiile */
    char    appt_file[25],
	   *apptimes;               /* standard appointment times file */
    int     temp1,                  /* holds temporary appointment time */
	    temp2;                  /* holds temporary standard table time */
    FILE  *fp1, *fp2;


    apptimes = findenv("APPTIMES", envp);
    copy (APPOINT, appt_file);
    copy (date, &appt_file[19]);
    if ((fp1 = fopen(apptimes, "r")) == NULL )
    {
	printf ("dailycal: cannot open %s\n", apptimes);
	exit (ENOENT);
    }
    if ((fp2 = fopen(appt_file, "r")) == NULL)
    {							/* Appointment file for date given does not exist */
	while (line = getline (fp1))		/* Copy the standard time file to the buffer */
	{
	    buf_ptr = copy (line, buf_ptr);
	    buf_ptr = copy ("\n", buf_ptr);
	}
    }
    else
    {							/* The appointment file exists */
	if (line = getline (fp1))			/* Get a line from the standard time file */
	    copy (line, time_line);
	else
	    aflag = 0;
	if (line = getline (fp2))			/* Get a line from the appointment file */
	    copy (line, appt_line);
	else
	    bflag = 0;
	while (aflag && bflag)				/* While there is info left in both files */
	{

	  temp1 = atoi(time_line);
	  temp2 = atoi(appt_line);

      /* convert time after 1200p in the appointment file to military time */

	  if (temp1 == 1200 && time_line[4] == 'a')
	      temp1 = 0;
	  else
	     if (temp1 < 1200 && time_line[4] == 'p')
		 temp1 += 1200;
	     else
		if (temp1 == 1200 && time_line[4] == 'p')
		    temp1 = 1200;

      /* convert time after 1200p in the standard time table to military time */

	  if (temp2 == 1200 && appt_line[4] == 'a')
	      temp2 = 0;
	  else
	     if (temp2 < 1200 && appt_line[4] == 'p')
		 temp2 += 1200;
	     else
		if (temp2 == 1200 && appt_line[4] == 'p')
		    temp2 = 1200;


	    if ( temp1  ==  temp2)
	    {						/* The start times are equal so put the appointment in the buffer */
		buf_ptr = copy (appt_line, buf_ptr);
		buf_ptr = copy ("\n", buf_ptr);
		if (line = getline (fp1))
		    copy (line, time_line);
		else
		    aflag = 0;
		if (line = getline (fp2))
		    copy (line, appt_line);
		else
		    bflag = 0;
	    }
	    else
		if (temp1 > temp2)
		{					/* The start time of the standard time file line is greater than */
							/* the start time of the appointment file line so put */
							/* the appointment line in the buffer and get a line */
							/* from the appointment file */
		    buf_ptr = copy (appt_line, buf_ptr);
		    buf_ptr = copy ("\n", buf_ptr);
		    if (line = getline (fp2))
			copy (line, appt_line);
		    else
			bflag = 0;
		}
		else
		{					/* The start time of the appointment file line is greater than the start */
							/* time of the standard time file so put the standard time */
							/* file line in the buffer and get a line from the standard  */
							/* time file */
		    buf_ptr = copy (time_line, buf_ptr);
		    buf_ptr = copy ("\n", buf_ptr);
		    if (line = getline (fp1))
			copy (line, time_line);
		    else
			aflag = 0;
		}
	}
	if (aflag)
	{						/* Lines remain in the standard time file, so place them */
							/* in the buffer */
	    line = time_line;
	    do
	    {
		buf_ptr = copy (line, buf_ptr);
		buf_ptr = copy ("\n", buf_ptr);
	    }
	    while (line = getline (fp1));
	}
	else
	    if (bflag)
	    {						/* Lines remain in the appointment file so copy */
							/* them to the buffer */
		line = appt_line;
		do
		{
		    buf_ptr = copy (line, buf_ptr);
		    buf_ptr = copy ("\n", buf_ptr);
		}
		while (line = getline (fp2));
	    }
	fclose (fp2);
    }
    fclose (fp1);
    fwrite (buffer, sizeof(*buffer), (int) buf_ptr - (int) buffer, stdout);
}
