h62365
s 00003/00013/00451
d R 1.4 83/05/12 15:10:36 jan 4 3
c USENIX Distribution version
e
s 00000/00000/00464
d D 1.3 83/03/23 10:05:53 anna 3 2
c no change
e
s 00044/00008/00420
d D 1.2 83/02/01 17:30:11 jan 2 1
c change date/time format
e
s 00428/00000/00000
d D 1.1 82/10/06 14:43:36 jan 1 0
e
u
U
t
T
I 1
#
/*
Name:
	CANCELAPPT Calendar System Command

Function:
	Deletes an appointment from a user's calendar file system.

Algorithm:
	If only one appointment is to be cancelled, remove the appointment from
	the user's appointment file and print out the remaining appointments on
	that day. If more than one appointment is to be cancelled, remove the
	appointment for the consecutive number of days specified.

Parameters:
	m	Indicates multiple appointments to be deleted
	date	the date the appointment is to be deleted. If multiple appointments
		then it is the starting date.
	#_days	the number of consecutive days to delete the appointment (for 
		multiple appointments only )
	start-time	the starting time of the appointment to be deleted

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or exec/appointments directory
	EFORM	Invalid date or time format
	EBRK	Too many consecutive days (max = 31)

Files and Programs:
	home-dir/exec/appointments/filename	appointments files

Installation Instructions:
	Compile as follows: cc cancelappt.c /hom/a/gls/maltagli/lib/datecheck.o
	Put in /usr/bin/exec.dir/cancelappt
	Change owner to root
	Deny write permissions except for owner
	Link to /usr/bin/exec.dir/cancel

History:
	Oct 79,	Release 1 Version 1,G. Maltagliati, coded in shell
	Nov 80,	Release 5 Version 1,G. Maltagliati, coded in C for efficiency
		and added idea for deletion of multiple appointments.


*/
#include <stdio.h>
#include <sys/types.h>	/* used with fstat system call */
#include <sys/stat.h>
#include <signal.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define APPOINT "exec/appointments"
#define DATEERR "appoint: invalid date format\n"
#define TIMEERR "appoint: invalid time format\n"
#define TOOMANY "You cannot cancel an appointment for more than 31 days at a time\n"
#define USAGE "You must enter: cancelappt <date> <start-time>\n		cancelappt m <start-date> <# of days> <start-time>\n"

char id[] = "%W% %H%";

char delfil[14] = "del.x";    /* temp name for file,
						will include process id */
	FILE *stnew;		/* stream i/o for new file */
/*

Name:
	main

Function:
	Delete an appointment for one day or multiple days from a user's calendar
	file system.

Algorithm:
	Determine if one or multiple appointments are to be entered. Check
	to see that the correct number of arguments have been entered, and then
	if multiple appointments are to be entered,check to see if the consecutive number
	of days are less than the maximum number allowed (31 days). If any errors have been 
	found, note them and exit. Otherwise execute the subroutine cancel to
	delete the appointments and then exit.

Parameters:
	m	Indicates multiple appointments to be deleted
	date	the date the appointment is to be deleted. If multiple appointments
		then it is the starting date.
	#_days	the number of consecutive days to delete the appointment (for 
		multiple appointments only )
	start-time	the starting time of the appointment to be deleted

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or exec/appointments directory
	EFORM	Not enough arguments or too many consecutive days

Files and Programs:
	None


*/
main (argc, argv, envp)
int     argc;						/* Count of command line arguments */
char   *argv[];						/* Pointer to list of argument pointers */
char   **envp;						/* environment */
{
D 2


E 2
    if ( chdir (findenv("HOME", envp)))			/* move to $HOME */
    {
	printf ("bad home directory\n");
	exit (ENOENT);
    }
    if (chdir (APPOINT) == -1)				/* Move to exec/appointments directory */
    {
	printf ("bad directory--%s\n", APPOINT);
	exit (ENOENT);
    }
    if (argc >= 3)					/* Check the number of arguments */
	if (*argv[1] != 'm')
	    cancel (argv[1], 1, argv[2]);	/* remove the appointment */
	else						/* Multiple date cancellations */
	    if (argc >= 5)				/* Were the correct number of arguments entered */
		if (atoi (argv[3]) <= 31)		/* Is number of days greater than maximum number */
		    cancel (argv[2], atoi (argv[3]), argv[4]);
		else					/* Invalid number of consecutive days */
		{
		    write (2, TOOMANY, sizeof (TOOMANY));
		    exit (EFORM);
		}
	    else					/* Not enough arguments were entered */
	    {
		write (2, USAGE, sizeof (USAGE));
		exit (EFORM);
	    }
    else						/* Not enough arguments were entered */
    {
	write (2, USAGE, sizeof (USAGE));
	exit (EFORM);
    }
    exit (ENOERR);
}
/*

Name:
	cancel

Function:
	Delete an appointment from a user's appointment file system for a number
	of consecutive days, which could be one (for cancelling just one appointment)

Algorithm:
	First, check the date to insure a valid date and get a standardized date.'
	Then, standardize the times to the four digit military time form (hhmm).
	Build the appointment record, by first placing the start time, dash, and
	then the end_time. Find the julian date for the start date of the cancel-
	lations. For num_days number of days, do the following: Check to see if
	any appointments exist on the day. If there are none then print a message
	stating so. Otherwise delete the appointment on that day. Finally,
	increment the julian date by 1, checking to see that the end of year has
	not been  reached. Now get the standardized date for that julian date in
	the format ddmmm and repeat the above process.

Parameters:
	checkdate	start date for the range of appointments
	num_days	the number of consecutive days for the appointment to be deleted
	stime		the starting time of the appointment to be deleted

Returns:
	EFORM	Bad time format

Files and Programs:
	None


*/
I 2
extern char *gettime();
E 2
cancel (checkdate, num_days, stime)
char   *checkdate,					/* The starting date for cancellation of appointments */
       *stime;						/* The starting time of the appointment */
int     num_days;					/* The number of consecutive days the appointment is */
							/* to be deleted */
{
    char    date[15],					/* The standardized date */
            *file_ptr,					/* pointer into "file" */
           *month,					/* The number of the month of the date given */
	    stantime[5],				/* hold the standard time */
	   *stan = stantime,
            file[12],					/* Holds the name of the file */
	   *end_ptr,					/* points to stantime */
            jdate[6];					/* Julian date in ascii */
    register int i;
    register int num_char;	/* returned from delete_appt */

    int flag;		/* indicates whether to print appointments */
    int     julian,					/* Julian date (integer) */
            year,					/* Year (yy) */
            days;					/* Number of days in the year */

I 2
    char    store_stm[5],
	    *time_ptr,
	    temp;
    int     ival,
	    time_flag,
	    temp_stm,
	    temp_etm;
E 2

    if (!(month = valid_date (checkdate, date, 0)))
    {							/* Invalid date */
	write (2, DATEERR, sizeof (DATEERR));
	exit (EFORM);
    }
D 2
    if (atoi (stime) > 0 && atoi (stime) < 2400)    {   /* Standardize start time */
	if (atoi(stime) < 1000)
E 2
I 2
    time_flag = 1;
    time_ptr = gettime(stime, time_flag);
    for (i=0; *(time_ptr + i) != '\0'; i++){
	store_stm[i] = *(time_ptr + i);
	if (store_stm[i] == 'a' || store_stm[i] == 'p')
	   temp = store_stm[i];
    }
	temp_stm = atoi(store_stm);
	if (atoi(store_stm) == 1200 && temp == 'a')
	   temp_stm = 0;
	else
	   if (atoi(store_stm) == 1200 && temp == 'p')
	      temp_stm = 1200;
    if (temp_stm >= 0 && temp <= 2400)    {   /* Standardize start time */
	ival = atoi(store_stm);
	if (temp == 'p' && ival >= 1300)
	   ival -= 1200;
	if (ival < 1000)
E 2
	    *stan++ = '0';
D 2
	end_ptr = copy (itoa (atoi (stime)), stan);
E 2
I 2
	stan = copy (itoa(ival), stan);
	*stan++ = temp;
E 2
    }
    else						/* Invalid time format */
    {
	write (2, TIMEERR, sizeof (TIMEERR));
	exit (EFORM);
    }
D 2
    *end_ptr = '\0';
E 2
I 2
    *stan = '\0';
E 2
    file_ptr = copy ("appts.", file);
    year = atoi (get_year ());				/* Get the current year */
    julian = atoi (julian_date (date, atoi (month), year));
							/* Get the julian date */
    if ((year / 4) * 4 == year && year != 0)		/* Find the number of days in the year */
	days = 366;
    else
	days = 365;
    flag = num_days;					/* Set flag to number of days. If it is 1, then appointments */
							/* will be printed after the appointment is deleted */
    while (num_days-- > 0)
    {
	copy (date, file_ptr);				/* Create filename of appointment file */
	 num_char = delete_appt(stantime, date, file);		/* delete it */
	if (flag != 1 && num_char > 0)		/* must close open stnew */
		fclose(stnew);
	if (++julian > days)				/* Check for end of year and increment julian date */
	{						/* New year */
	    julian = 1;
	    if (++year == 100)				/* Only occurs if it is the year 2000 */
		year = 0;
	}
/* Create the julian date in the form yyjjj */
	copy (itoa (year), jdate);
D 2
	copy (itoa(julian), &jdate[2]);
	valid_date (jdate, date, 1);			/* Get the new date, format ddmmm */
E 2
I 2
	if (julian < 10) {
	   copy("00", &jdate[2]);
	   copy(itoa(julian), &jdate[4]);
	}
	else
	   if (julian < 100) {
	      copy("0", &jdate[2]);
	      copy(itoa(julian), &jdate[3]);
	   }
	   else
	      copy (itoa(julian), &jdate[2]);
	if (!valid_date (jdate, date, 1))  {                    /* Get the new date, format ddmmm */
	   printf(stderr, "appoint: date conversion error\n");
	   exit(EFORM);
	}
E 2
    }
    if (flag == 1)					/* Check flag to see if appointments are to be printed */
    {							/* Print appointments */
	if ( num_char == -1 )		/* was the last appointment deleted */
	    printf ("\nNo remaining appointments on %s\n\n", file + 6);
	else if (num_char > 0)	/* print remaining appoinments */
	{
	    printf ("\nThe following appointments remain on %s:\n\n", file + 6);
	    lseek(stnew->_file, 0, 0);
	    for(i =1; i <= num_char; i++)
		putchar(getc(stnew));
	    fclose(stnew);
	}
    }
}
/*

Name:
	delete_appt

Function:
	Delete an appointment from a file.

Algorithm:
	Copying the data to the output file,
	read the appointment file one line at a time.  Stop when either
	an exact match of the line in apbuffer is found, if this
	is a subsequent day of a multiple delete, or when a start time
	match is found otherwise.  In the latter case, read in all
	lines with the same start time.  Finally, write out the
	undeleted buffered lines, and copy the rest of the file.
	Then use link and unlink to rename the new file to the old
	and delete the old file.

Parameters:
	time		the start time of the compare string
	date		the date of the appointment
	file		name of the appointment file

Returns:
	-2		matching appointment not found; all files closed
	-1		all files closed and deleted; no appts left that day
	other		the number of characters in the new file
	exit(ENOENT)	called if an error occurs

Files and Programs:
	home-dir/exec/appointments/(file)	file to place appointment in


*/
delete_appt( time, date, file)

char *time;		/* time of appointment, eg. 1000-1200  */
char *date;		/* date of appointment, eg. 12may	*/
char *file;		/* file name to manipulate */

{

#define WBUF 512		/* i/o buffer for lines */
#define APSIZE 80		/* size of apbuffer (maximum)  */

	FILE *storig;		/* stream i/o for original file */

	char buffer[WBUF];	/* holds the lines */
	register char *bp;	/* used to manipulate buffer data */

	static int count = 0;	/* not zero after first good call */
	static int aplen;	/* number of characters used in apbuffer */
	static char apbuffer[APSIZE];	/* exact line to delete */
	register int i, j, k;
	int more;		/* set to one in data after matches */
	char *ap;		/* used to point to apbuffer */
	extern int sigint();	/* closes files on SIGINT */
	char ans_buf[10];
	char *answer = ans_buf; /* points to user answer */
	struct stat st_buf;	/* information returned by fstat */

	if ((storig = fopen(file, "r")) == NULL) {
		printf("No appointments on %s\n", date);
		if (count)
			return(-2);
		else {
			printf("No appointments cancelled\n");
			exit (ENOENT);
		}
	}
	sprintf(&delfil[4], "%d", getpid());	/* unique file */
	signal(SIGINT, SIG_IGN);	/* don't respond */
	stnew = fopen(delfil, "w");
	signal(SIGINT, sigint);		/* close files and exit */
	k = -1;
	if (count)	/* subsequent time thru--look for exact match */
		while (fgets(buffer, WBUF, storig)
				&& (k = strncmp(buffer, apbuffer, aplen)))
			for(bp = buffer; putc(*bp++, stnew) != '\n';);
	else
		while( fgets(buffer, WBUF, storig)
				&& (k = strncmp(buffer, time, 4)) < 0)
			for(bp = buffer; putc(*bp++, stnew) != '\n';);
	if (k) {
		printf("\nAppointment not found on %s\n", date);
		signal(SIGINT, SIG_IGN);	/* don't mess this up */
		fclose(stnew);
		unlink(delfil);
		fclose(storig);
		if (count) {
			signal(SIGINT, SIG_DFL);
			return(-2);
		}
		else {
			printf("No appointments cancelled\n");
			exit(ENOENT);
		}
	}
	more = 1;		/* assume there's more to read */
	if (count++ == 0) {	/* first day */
		printf("\n%s appointment(s) on %s are:\n\n", time, date);
		bp = buffer;
		i = 0;
		while( k == 0) {
			printf("(%d) ", ++i);
			while (putchar(*bp++) != '\n');
			k = !(more = fgets(bp, WBUF, storig))
					|| strncmp(bp, time, 4);
		}
		putchar('\n');
		if (i > 1) {
			printf("Delete which one? (1-%d, 0 for none) ", i);
			while (!(fgets(answer, 256, stdin)) || (k = atoi(answer))
					< 0 || k > i)
				printf("Answer 0-%d only ", i);
		}
		else {
			printf("Delete it? (y or n) ");
			while (!(fgets(answer, 256,stdin)) || (*answer != 'y'
					&& *answer != 'n'))
				printf("Answer y or n only ");
			k = (*answer == 'y') ? 1 : 0;
		}
		if (k == 0) {
			printf("No appointments cancelled\n");
			sigint();		/* simulate SIGINT */
		}
		bp = buffer;
		for(j = 2; j <= k; j++)
			while(putc(*bp++, stnew) != '\n');
		for (ap = apbuffer; (*ap++ = *bp++) != '\n' 
				&& ap < &apbuffer[APSIZE];);
		aplen = ap - apbuffer;
		if (more) i++;		/* dump non-matching data, too */
		while( j++ <= i)
			while( putc(*bp++, stnew) != '\n');
	}
	signal(SIGINT, SIG_IGN);	/* don't mess up close */
	if (more)
		while ( (i = getc(storig)) != EOF)
			putc(i, stnew);
	fclose(storig);
	unlink(file);
	fflush(stnew);			/* write out the data */
	fstat( stnew->_file, &st_buf);	/* get data on open file */
	if (k = st_buf.st_size)		/* wrote something ? */
		link(delfil, file);
	else {
		fclose(stnew);
		k = -1;
	}
	unlink(delfil);
	signal(SIGINT, SIG_DFL);	/* previous status */
	return(k);
}
/*

Name:
	sigint

Function:
	Handle SIGINT interrupts.  This is needed to close and unlink
	the temporary file used.

Returns:
	None:  will simply do an exit after cleaning up.  Note that
	opened files are close by exit, thus we need only do unlink.

*/
sigint()
{
	unlink(delfil);		/* remove dir. entry; file removed
						by exit */
	exit(ENOENT);
}
E 1
