h42093
s 00000/00000/00373
d D 1.2 83/05/12 15:11:37 jan 2 1
c USENIX Distribution version
e
s 00373/00000/00000
d D 1.1 82/10/06 12:23:45 jan 1 0
e
u
U
t
T
I 1
#
/*
Name:
	PMON_APT_CAL Calendar System Command

Function:
	Print all appointments for n weeks

Algorithm:
	Find the date for the Monday of the week of the date given. For each
	date, write any appointments  for that date to the standard output.

Parameters:
	date		the date in the week to start monthcal
	userid		the userid of the calendar system user (optional)
	num_weeks       number of weeks to print calendar for (optional - 4 week default)

Returns:
	ENOERR	Normal exit
	EFORM	Invalid date format, invalid userid, or incorrect number of command line arguments
	EFLAG	Not an authorized user
	ENOENT	Bad home directory or exec directory
	EACCES  Cannot create temporary tabpr file

Files and Programs:
	/BIN/SORT		used to sort files
	/BIN/TABPR                used to output the 4 week calendar

Installation Instructions:
	Compile as follows:
	cc -o $EXEC.DIR/monthcal pmon_apt_cal.c library
	Change owner to root
	Deny write permissions except for ower

History:
	Oct 79,	Release 1 Version 1, G. Maltagliati, coded in shell
	Dec 79,	Release 2 Version 1, G. Maltagliati, output format changed
	Sep 80,	Release 4 Version 1, Sgt T. Irish, added idea of 5 or 7 day calendar
	Nov 80,	Release 5 Version 1, G. Maltagliati, coded in C for efficiency
	Jul 82, Release 7, Version 1, L. Reed for Perkin-Elmer
	Aug 82, Release 7, Version 2, J. Neyer...re-write exec support

*/
#include <ctype.h>
#include <stdio.h>
#include <pwd.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define APPOINT "exec/appointments/appts."
#define USAGE "pmon_apt_cal: You must enter: pmon_apt_cal date <number of weeks> <userid>\n"
#define DATEERR "pmon_apt_cal: Invalid date format\n"
#define USERINVL "pmon_apt_cal: Invalid userid\n"
#define NOTAUTH "pmon_apt_cal: You are not an authorized user\n"
#define READ 0
#define WRITE 1
#define LINELEN 256

char id[] = "%W% %H%";
/*

Name:
	main

Function:
	Gather all arguments and perform error checking on the arguments

Algorithm:
	Insure that the correct number of arguments were entered. The first argument must
	always be the start date for the report. The second and third argument
	can optionally be the number of weeks to report for (default is 4) and
	the user for whom the report is to be done. These arguments are parsed
	and validated. Change to the target user's home directory. Check the
	authorized users file for permission. If ok, begin the loop to check
	for n weeks of appointments writing them as you go.

Parameters:
	date		the date in the week to start monthcal
	userid		the userid of the calendar system user (optional)
	num_weeks       number of weeks to report

Returns:
	ENOERR	Normal exit
	EFORM	Invalid date format, invalid userid, or incorrect number of command line arguments
	EFLAG	Not an authorized user
	ENOENT	Bad home directory or exec directory

Files and Programs:
	None


*/
main (argc, argv, envp)
char   *argv[];						/* Pointer to list of argument pointers */
int     argc;						/* Count of command line arguments */
char **envp;
{
    struct  passwd  *pw;
    char    date[8];					/* The date in the week for monthcal to start in */
    int     weeks;                                      /* Number of weeks to report */
    register int    curr_item = 1;
    int p[2], popen_pid;
    FILE *pfp;
    extern int authuser();

    if (argc > 4 || argc == 1)
    {							/* Incorrect number of command line arguments were entered */
	fprintf(stderr, USAGE);
	exit (EFORM);
    }
    if (!valid_date (argv[1], date, 0))             /* Standardize the date format */
    {
	fprintf(stderr, DATEERR);
	exit (EFORM);
    }
    if (argc == 3 && isdigit(*argv[2]))  {
	weeks = atoi(argv[2]);                          /* Number of weeks...default user */
	pw = getpwnam(findenv("USER", envp));
    }
    else if (argc == 3) {
	weeks = 4;                                      /* Default weeks...user id supplied */
	if ((pw = getpwnam(argv[2])) == NULL)   {
	    fprintf(stderr, USERINVL);
	    exit (EFORM);
	}
    }
    else if (argc == 2)         {
	weeks = 4;
	pw = getpwnam(findenv("USER", envp));
    }
    else if (argc == 4 && isdigit(*argv[2]))     {
	weeks = atoi(argv[2]);                          /* User id and weeks passed */
	if ((pw = getpwnam(argv[3])) == NULL)   {
	    fprintf(stderr, USERINVL);
	    exit (EFORM);
	}
    }
    else        {
	weeks = atoi(argv[3]);
	if ((pw = getpwnam(argv[2])) == NULL)   {
	    fprintf(stderr, USERINVL);
	    exit (EFORM);
	}
    }
    if ((chdir(pw->pw_dir)) == -1)      {
	fprintf(stderr, "pmon_apt_cal: cannot access directory %s\n", pw->pw_dir);
	exit (EACCES);
    }
    if (!authuser())      {
	fprintf(stderr, NOTAUTH);
	exit(EFORM);
    }

    fputs(APPTHEAD, stdout);                         /* Write out header */
    pmon (date, weeks, stdout);                          /* Generate the 4 week calendar */
    exit (ENOERR);
}
/*

Name:
	pmon

Function:
	Generate the dates needed for the n weeks  and call get_appt_sus to place
	all appointments to stdout.

Algorithm:
	Find the julian date for the Monday of the current week and the number of 
	days in the year. Now do the following (weeks * 7) times:

	Standardize the julain date to the format ddmmm. Find the standardized
	date for the end of the week.  Call get_appt_sus to place any appointments
	for that date to the stdout.
	Add 1 to the julian date. Now check to make sure that the
	julian date is valid (not past the end of the year or before the  beginning
	of the year). If it is not then adjust the julian date to a valid one and
	find the number of days in the new year.

Parameters:
	date            the date in a week to generate the calendar
	weeks           indicates the number of weeks to report for

Returns:
	None

Files and Programs:
	None


*/
pmon (date, weeks, file)
char   *date;						/* The date in the week that the calendar is desired */
int     weeks;                                          /* Indicates number of weeks to report for */
FILE *file;
{
    int     year,					/* The current year */
            julian,					/* The julian date */
            month,					/* The number of the current month */
            days,					/* The number of days in the year */
            weekj,					/* The julian date for the end of a week */
            tempyear;					/* The year the end of the week ends in */
    register int    i,					/* Counter for outer loop */
		    j;                                  /* Counter for inner loop */
    char    newdate[7],					/* Holds the date in the form yyjjj */
            end_week[8],				/* Holds the date for the end of the week in the format yymmm */
            dummy[8];					/* Dummy buffer */


/* Find the current year, the month and the julian date for the given date */
    year = atoi (get_year ());
    month = atoi (valid_date (date, dummy, 0));
    julian = atoi (julian_date (date, month, year));
    if ((year / 4) * 4 == year && year != 0)		/* Find the number of days in the current year */
	days = 366;
    else
	days = 365;
    julian - = day_of_week (julian);			/* Find the julian date for the Monday of the week */
    if (julian < 1)
    {							/* Julian date for the Monday of the week is in the previous year */
	year--;
	if ((year / 4) * 4 == year && year != 0)	/* Find the number of days in the new year */
	    days = 366;
	else
	    days = 365;
	julian = days + julian;				/* Find the julian date for the new year */
    }
    for (j = 1; j <= (weeks * 7); j++)   {
	copy (itoa (year, 2), newdate);     /* Get the standardized date for the julian date */
	copy (itoa (julian, 3), &newdate[2]);
	valid_date (newdate, date, 1);
	tempyear = year;
	weekj = julian + 6;
	if (weekj > days)
	{
	    weekj - = days;
	    tempyear++;
	}
	copy (itoa (tempyear, 2), newdate);
	copy (itoa (weekj, 3), &newdate[2]);
	valid_date (newdate, end_week, 1);
	get_appt_sus (date, file);   /* Place any appointments on the day in the stdout */
	julian++;                       /* increment the julian date to the next Monday */
	if (julian > days)
	{                                           /* Julian date is past the end of the year */
	    julian - = days;                        /* Find the julian date in the next year */
	    year++;                                 /* Increment the year and find the number of days in that year */
	    if ((year / 4) * 4 == year && year != 0)
		days = 366;
	    else
		days = 365;
	}
	if (julian < 1)
	{                                           /* Julian date is before the beginning of the year */
	    year--;                                 /* Decrement the year and find the number of days in that year */
	    if ((year / 4) * 4 == year && year != 0)
		days = 366;
	    else
		days = 365;
	    julian = days + julian;                 /* Find the julian date in the new year */
	}
    }
}
/*

Name:
	get_appt_sus

Function:
	Format all appointments for the date given and write it to the stdout.

Algorithm:
	Create the names of the appointment file. If this is the first
	time through, format the "DATE:TIME:APPOINTMENT" header.
	If there is an appointment file for the day, then add it to the stdout.

Parameters:
	date            the date to check for appointments
	file            fd for tabpr

Returns:
	None

Files and Programs:
	None


*/
get_appt_sus (date, file)
char   *date;                                           /* The date to check for appointments */
FILE   *file;

{
	char *line;
	char appt_file[40];
	char *appt = appt_file;
	FILE *fp_appt;

/* Create the name of the appointment file */
    strcat (appt, APPOINT);
    strcat (appt, date);
    *(date + 2) - = ' ';				/* Capitalize the first letter of the month */
    if ((fp_appt = fopen(appt_file, "r")) != NULL)      {
      while (line = getline(fp_appt))
	  fprintf(file, "%s:%s\n", date, line);
      fclose (fp_appt);
    }
}
/*

Name:
	day_of_week

Function:
	Find the day of the week for a date (0 = Monday, 6 = Sunday)

Algorithm:
	Find the absolute value of the difference between the current date and
	the date given. Now find the remainder of dividing that diffence by 7 (the
	number of days in a week). This will give the difference between the day
	of the week for the date given and the day of the week for the current date.
	If the julian date for the date given is greater than the julian date of
	the current date, then add to the day of the week for the current date the 
	result of the previous division and subtract 1. Make a correction if the
	result is greater than 6. Otherwise, subtract the result of the division from
	the day of the week for the current date and then subtract 1. If that difference
	is less than zero, then add 7 to it. (The reason for the subtractions of 1
	are to convert the day of the week of the current day that the system gives 
	us with 0=Sunday and 6=Saturday to 0=Monday and 6=Sunday

Parameters:
	julian		the julian date of the date desired

Returns:
	diff		0=Monday, 1=Tuesday, ..., 6=Sunday

Files and Programs:
	None


*/
day_of_week (julian)
int     julian;						/* The julian date for the date desired */
{
    int     tvec[2],					/* Time vector */
           *time_ptr,					/* Pointer to information about current date */
            day_of_year,				/* The julian date for the current date */
            diff;					/* The diffence between the two julian dates and finally the day_of_the_week */


    time (tvec);
    time_ptr = localtime (tvec);			/* Get information about current date */
    day_of_year = time_ptr[7] + 1;			/* Get the julian date for the current date */
    if (julian >= day_of_year)				/* Find the absolute value of the difference between the two julian dates */
	diff = julian - day_of_year;
    else
	diff = day_of_year - julian;
    diff - = (diff / 7) * 7;				/* Find the difference between the day of the week for the current date */
							/* and the date given */
/* Now find the day of the week for the date given making the correction for the */
/* system day of the week with 0=Sunday and 6=Saturday */
    if (julian > day_of_year && diff != 0)
    {
	diff = time_ptr[6] + diff - 1;
	if (diff > 6)
	    diff - = 7;
    }
    else
    {
	diff = time_ptr[6] - diff - 1;
	if (diff < 0)
	    diff + = 7;
    }
    return (diff);
}
E 1
