#
/*
 * prtb
 *
 *
 * Call:
 *	% prtb list filesystem
 *
 *	where list is:
 *		-i[d] block# item# [item#]
 *			Specify an indirect block number, and an item number
 *			in that block.  If the 'd' option on this flag is
 *			specified, the second item# field must be specified,
 *			and a double indirect will be performed.
 *		-f inumber
 *			This prints out the entire logical file made up of
 *			the set of blocks contained in that inode.
 *		block
 *			This prints out just the information in block.
 *		-h
 *			Labels every block printed.
 *		-n
 *			Turn off block labels.
 *
 *
 * Description:
 *	Prtb lists a given block in binary, ie, as it resides on disk.
 * To actually get this in an intelligible form, use od (I).  Note that blocks
 * are numbered beginning at 0 for the boot block, and 1 for the super block.
 *
 *
 * CSL	v10.03.77(0) - r10.03.77(0)	Hannes Beinert
 *	v10.03.77(0) - r01.03.77(1)
 *	v10.03.77(0) - r02.17.78(2)
 */

#include	"/usr/sys/ino.h"	/* include inode structure	*/

struct	inode	i_node;

struct	{
	char	minor;
	char	major;
	int	inumber;
	int	flags;
	char	nlinks;
	char	uid;
	char	gid;
	char	size0;
	int	size1;
	int	addr[8];
	int	actime[2];
	int	modtime[2];
}	statnode;			/* for file status		*/

int	*fsize;				/* highest unavailable block	*/
int	*cmp;				/* used for unsigned compare	*/
int	isize;				/* highest available inumber	*/
int	buf[256];			/* block buffer			*/
char	**argv;				/* global arg vector		*/
int	argc;				/* global arg count		*/
int	err;				/* error indicate		*/
int	hflag;				/* header switch		*/
int	file;

main(narg,argp)
int	narg;
char	**argp;
{
	register int	i;
	register int	f;

	/*
	 * make argp, narg global;  check arg count.
	 */

	argc = narg;
	argv = argp;
	if (argc < 3) {
		printf("Usage:  %s list filesystem\n",argv[0]);
		exit(0);
	}

	/*
	 * check if device is block special file, or normal file.
	 */

	stat(argv[argc-1],&statnode);	/* get device status	*/
	if ((f = (statnode.flags&IFMT)) && f != IFBLK) {
		printf("Device not block-type special, or normal.\n");
		exit(0);
	}

	if ((file = open(argv[argc-1],0)) < 0) {
		printf("Can't open %s.\n",argv[argc-1]);
		exit(0);
	}

	/*
	 * get highest block number in super block.
	 */

	seek(file,1,3);		/* set pointer to superblock		*/
	read(file,&isize,2);
	read(file,&fsize,2);

	/*
	 * process each block in list.
	 */

	--argc;		/* pop off program name	*/
	while(--argc) {
		if (**++argv == '-') {
			switch (*++*argv) {
				case 'h':	/* print header	*/
					hflag++;
					break;
				case 'n':	/* no headers	*/
					hflag = 0;
					break;
				case 'i':	/* indirect	*/
					prtidr();
					break;
				case 'f':	/* inumber	*/
					prtfil();
					break;
				default:
					printf("Bad switch: %c\n",*argv);
			}
		}
		else	if ((i = getoct(*argv)), !err) {
				if (hflag)	printf("\nBlock ");
				prtblk(i);
			}
	}
}


/*
 * getblk:	gets a block from the device, check if it is in range.
 */

getblk(block)
int	block;
{
	/*
	 * see if block is legal.
	 */

	cmp = block;
	if (cmp >= fsize) {
		printf("\n\tBad block: %o\n",block);
		return(-1);
	}

	/*
	 * get the block into the buffer.
	 */

	seek(file,block,3);
	read(file,&buf,512);

	return(0);
}


/*
 * getdec:	gets a decimal number.
 */

getdec(string)
char	*string;
{
  register char	*p, c;
  register int	num;

	err = 0;
	p = string;
	num = 0;
	while(c = *p++)
		if (c >= '0' && c <= '9') {
			num =* 10;
			num =+ c - '0';
		}
		else	{
			printf("Bad number: %s\n",string);
			err = 1;
			return(0);
		}
	return(num);
}


/*
 * getoct:	gets an octal number.
 */

getoct(string)
char	*string;
{
	register char	*ptr, cc;
	register int	num;

	err = 0;
	num = 0;
	ptr = string;
	while (cc = *ptr++)
		if (cc <= '7' && cc >= '0') {
			num =<< 3;
			num =| cc - '0';
		}
		else {
			printf("Bad number: %s\n",string);
			err = 1;
			return(0);
		}
	return(num);
}


/*
 * prtblk:	prints an arbitrary block on the given filesystem.
 */

prtblk(blk)
int	blk;
{
	if (hflag)	printf("%o:\n",blk);
	if (getblk(blk) >= 0) write(1,&buf,512);
}


/*
 * prtidr:	prints an indirect block, or double indirect block,
 *		depending on the passed string.
 */

prtidr()
{
	register int	item1, item2, block;
	int	 dflag;

	dflag = 0;

	/*
	 * collect arguments.
	 */

	if (*++*argv == 'd') dflag++;
	if (--argc, (block = getoct(*++argv)), err) return;
	if (--argc, (item1 = getdec(*++argv)), err) return;
	if (dflag)
		if (--argc, (item2 = getdec(*++argv)), err) return;

	/*
	 * perform indirections required.
	 */

	if (dflag) {
		if (hflag) {
			printf("\nDouble Indirect Block %o-->",block);
			printf("%o-->",(block = indir(block,item1)));
		}
		item1 = item2;
	}
	else	if (hflag)	printf("\nIndirect Block %o-->",block);

	/*
	 * indirect and do print.
	 */

	prtblk(indir(block,item1));
}


/*
 * prtfil:	prints all the blocks in an arbitrary inode.
 */

prtfil()
{
	register int	i, j, k;
	int	 inum, lflag;

	lflag = 0;

	/*
	 * get specified inumber, and see if it is legal.
	 */

	--argc;
	if ((inum = getdec(*++argv)), err) return;

	if (inum <= 0 || inum > 16 * isize) {
		printf("Bad inode inumber: %d\n",inum);
		return;
	}

	/*
	 * check if it is allocated, and large.
	 */

	seek(file,(inum+31)/16,3);		/* point at block of inode */
	seek(file,32*((inum+31)%16),1);		/* point at inode in block */
	read(file,&i_node,sizeof i_node);	/* read inode		   */

	if ((i_node.i_mode&IALLOC) != IALLOC) {
		printf("Inode %l not allocated.\n",inum);
		return;
	}

	if ((i_node.i_mode&ILARG) == ILARG) lflag = 1;

	/*
	 * start doing the indirections.
	 */

	for (i = 0;i < 8;i++) {
		if (i_node.i_addr[i] == 0) continue;
		if (lflag) {
			if (i != 7) inprt(i_node.i_addr[i]);
			else {
				for (j = 0;j < 256;j++)
					inprt(indir(i_node.i_addr[7],j));
			}
		}
		else	{
			if (hflag)	printf("\nBlock ");
			prtblk(i_node.i_addr[i]);
		}
	}
}


/*
 * indir:	gets an indirect block number.
 */

indir(block,item)
int	block;
int	item;
{
	/*
	 * get the block, and return proper word.
	 */

	if (getblk(block) < 0) return(0);
	return(buf[item]);
}


/*
 * inprt:	prints an entire indirect block.
 */

inprt(block)
int	block;
{
	register int	i, j;

	if (!block) return;

	for (i = 0;i < 256;i++) {
		if (!(j = indir(block,i))) continue;
		if (hflag)	printf("\nBlock ");
		prtblk(j);
	}
}
