#

/*
 * seti: fix a mangled inode
 * Seti allows the (super) user to modify any field of an inode to be what
 * he wants.  This is usefull when a system crash has left something in an
 * inconsistent state.  For example the link count may be wrong and need to
 * be changed.
 * The syntax of the command is:
 *	seti <inumber> <device> <field> <value> [ <field> <value> ... ]
 * field names are:
 *	mode		file mode (all 16 bits used), converted in octal
 *	nlink		link count
 *	uid		user id
 *	gid		group id
 *	size		file size
 *	atime		access time
 *	mtime		modify time
 *	ctime		create time
 *
 * The value will be converted with a size (in bits) appropriate to the
 * size of the inode field it is being inserted into.
 *
 * Written 9/16/79 Randy Schulz CSL.
 * modified back to version 6 unix by David E. Miran 10/30/80
 */

#include <sys/param.h>
#include <sys/filsys.h>
#include <sys/ino.h>
#include <stdio.h>
#define	INOPB	16	/* number of inodes per block */
#define	BSIZE	512	/* disk block size */

char *fname[] = {
	"mode",
	"nlink",
	"uid",
	"gid",
	"size",
	"atime",
	"mtime",
	"addr",
	(char *) 0
};

struct filsys sblock;
struct inode dino;

main(argc, argv)
int argc;
char *argv[];
{
	extern long atol();
	register int i;
	unsigned int aindx, alladdr = 0;
	unsigned int addr;
	int inum, dev;
	int mode;
	int nlink, uid, gid;
	long size;
	long atime, mtime, ctime;
	struct inode inarr[INOPB];
	int foff;

	if(argc < 5) {
		printf("Usage: %s <inum> <dev> <field> <val> [ <field> <val> ... ]\n", *argv);
		printf("field options: mode, nlink, uid, gid, size,");
		printf(" atime, mtime, addr <index>\n");
		exit(1);
	}

	if((dev = open(argv[2], 2)) < 0) {
		printf("Cannot open %s\n", argv[2]);
		exit(1);
	}

	seek(dev, 1, 3);
	if (read(dev, &sblock, 512) != 512) {
		printf("Error reading super block of %s\n", argv[1]);
		exit(1);
	}

	sscanf(argv[1], "%d", &inum);
	if(inum > (sblock.s_isize - 2) * INOPB) {
		printf("Inode number out of range\n");
		exit(1);
	}

	foff = (inum - 1)/INOPB +2;
	seek(dev, foff, 3);
	if(read(dev, (char *)inarr,BSIZE) != BSIZE){
		printf("Error reading inode from %s\n", argv[1]);
		exit(1);
	}
	dino = inarr[(inum-1)%INOPB];

	for(i = 3;i < argc;i += 2) {
		if(i + 1 >= argc) { 
			printf("Field name %s has no value indicated\n", argv[i]);
			exit(1);
		}

		sscanf(argv[i + 1], "%o", &mode);
		sscanf(argv[i + 1], "%d", &nlink);
		uid = gid = nlink;
		size = atol(argv[i + 1]);
		atime = mtime = size;

		switch(cmdnum(argv[i])) {

		case 0:		/* mode */
			dino.i_mode = (short) mode;
			break;

		case 1:		/* nlink */
			dino.i_nlink = (short) nlink;
			break;

		case 2:		/* uid */
			dino.i_uid = (short) uid;
			break;

		case 3:		/* gid */
			dino.i_gid = (short) gid;
			break;

		case 4:
			inssize(size, &dino.i_size0);
			break;

		case 5:		/* atime */
			dino.i_atime = atime;
			break;

		case 6:		/* mtime */
			dino.i_mtime = mtime;
			break;

		case 7:		/* addr */
			if(*argv[i + 1] == '*')
				alladdr++;
			else
				if(*argv[i + 1] < '0' || *argv[i + 1] > '9') {
					printf("Illegal address subscript\n");
					exit(1);
				}
				else
					aindx = atoi(argv[i + 1]);

			if(aindx > 7) {
				printf("Address subscript not in range 0-7\n");
				exit(1);
			}
			addr = atol(argv[i + 2]);
			if(addr > sblock.s_fsize) {
				printf("Illegal disk address\n");
				exit(1);
			}
			if(!alladdr)
				dino.i_addr[aindx] = addr;
			else
				for(aindx = 0;aindx < 8;aindx++)
				dino.i_addr[aindx] = addr;
			i++;
			break;

		default:
			printf("Unknown field name %s\n", argv[i]);
			exit(1);
		}
	}
	inarr[(inum-1)%INOPB] = dino;
	seek(dev, foff, 3);
	if(write(dev, (char *)inarr, BSIZE) != BSIZE){
		printf("Write error on %s\n", argv[1]);
		exit(1);
	}
	exit(0);
}


cmdnum(cmd)
char *cmd;
{
	register int i;

	for(i = 0;fname[i] != (char *) 0;i++)
		if(seq(cmd, fname[i]))
			return(i);
	return(-1);
}


seq(s1, s2)
register char *s1, *s2;
{
	while(*s1++ == *s2)
		if(*s2++ == '\0')
			return(1);
	return(0);
}


inssize(isize, svec)
long isize;
char *svec;
{
	*svec++ = (isize >> 16)  & 0377;
	*svec++ = (isize) & 0377;
	*svec++ = (isize >> 8) & 0377;
}
