/*
 * smsd [-s n] [-s n] ... [-cm] [-ca] [-va] [-f] [-cv] [-ms] [-ct] file ncol [start [end] ]
 * read specified file and produce a mean, s.d., and c.v. of all
 * columns not specifically excluded.
 * lines with fewer than ncol columns are padded to ncol columns.
 * columns are separated by ':' and the first column is 1.
 * only data on lines start through end are included.
 * missing data and data with '*' at the end of the field are skipped.
 * options available are:
 * -s n   - skip column n
 * -cm    - generate cross column means (i.e. row means).
 * -ca    - generate cross column absolute means
	    (if -ca or -cm is selected then all data means and
	     means of cross column absolute means are printed also.)
 * -ct    - generate cross column totals (sum)
 * -va    - generate vertical absolute means
	    (includes mean, s.d., and c.v.)
 * -cv    - print coefficient of variation (for regular means)
 * -ms    - do not print the regular mean and s.d. (or c.v.)
 * -sd    - do not print the regular s.d.
 * -f     - vertical full report (add sum of x, sum of x squared)
 *
 * by David E. Miran
 * version of 1/28/83
 */

#include	<math.h>
#include	<stdio.h>
#define	TOTC	41
#define	MAXCOL	40
int fid, ncol, nrd, iptr;
long stline,enline,curline,nskip,nmiss;
char buf[512], lbuf[512];
double sum[TOTC], sumsq[TOTC], mean[TOTC], sd[TOTC], val[TOTC], cv[TOTC];
double asum[TOTC], amean[TOTC], asd[TOTC], acv[TOTC];
double rsum, rsuma, rmean, rmeana;
double gsum, gsuma, gmean, gmeana, gsumsq, gsd, gcv, gsda, gcva;
double msum, mmean, msumsq, msd, mcv, zd;
int rn, gn, nlab;
int	allow[TOTC], pres[TOTC], n[TOTC];
int	caflag, cmflag, vaflag, fxflag, cvflag, msflag, sdflag, ctflag;
char	*ifile;

main(argc,argv)
int argc;
char ** argv;
{
register int i, ic, cp;
int	sc;
char	c, prevc;
extern double fabs(), sqrt();
extern float atof();
extern long atol();
	if (argc < 3) {
		fprintf(stderr,"usage:  smsd  file ncols start end\n");
		exit(1);
	}
	iptr = nrd = 0;
	nskip = nmiss = 0;
	curline = stline = 0;
	enline = 1000000;
	ctflag = msflag = caflag = cmflag = vaflag = fxflag = cvflag = 0;
	sdflag = 0;
	gsda = gsum = gsuma = gsumsq = gmean = gmeana = gsd =gcv = 0;
	msum = msumsq = mmean = msd = mcv = zd = 0;
	gn = nlab = 0;
	for (i=0; i< TOTC; i++) allow[i] = 1;
	argc--;
	argv++;
arglp:
	if (argc < 1) goto pshort;
	if (**argv != '-') goto filarg;  /* file name finally */
	if (strcmp(*argv, "-s") == 0) {
		argc--;
		argv++;
		if (argc < 1) goto pshort;
		i = atoi(*argv);
		allow[i] = 0;
		goto again;
	}
	if (strcmp(*argv, "-cm") == 0) {
		cmflag++;
		goto again;
	}
	if (strcmp(*argv, "-ca") == 0) {
		caflag++;
		goto again;
	}
	if (strcmp(*argv, "-ct") == 0) {
		ctflag++;
		goto again;
	}
	if (strcmp(*argv, "-va") == 0) {
		vaflag++;
		goto again;
	}
	if (strcmp(*argv, "-cv") == 0) {
		cvflag++;
		goto again;
	}
	if (strcmp(*argv, "-sd") == 0) {
		sdflag++;
		goto again;
	}
	if (strcmp(*argv, "-ms") == 0) {
		msflag++;
		goto again;
	}
	if (strcmp(*argv, "-f") == 0) {
		fxflag++;
		goto again;
	}
	fprintf(stderr,"smsd: unrecognized option %s\n",*argv);
	exit(1);
again:
	argc--;
	argv++;
	if (argc > 0) goto arglp;
pshort:
	fprintf(stderr,"smsd:  Too few parameters\n");
	exit(1);
filarg:
	ifile = *argv;
	argc--;  argv++;
	if (argc < 1) goto pshort;
	ncol = atoi(*argv);
	argc--;  argv++;
	if (argc < 1) goto ardone;
	stline = atol(*argv);
	argc--;  argv++;
	if (argc < 1) goto ardone;
	enline = atol(*argv);
ardone:
	fid = open(ifile, 0);
	if (fid < 0) {
		fprintf(stderr,"smsd: cannot open file %s\n",ifile);
		exit(1);
	}
lp:
	if (getl()) goto comp;  /* end of data */
/* cp is column pointer,  ic is character pointer.
   ':' is field separator,  line ends with '\0', not '\n'
   first column is 1 (from user viewpoint).
 */

	for (cp=0; cp<TOTC; cp++) pres[cp] = 0; /* no data yet */
	sc =ic = 0;  /* sc is start char pos of a field */
	cp = 1;
clp:
	prevc = c;
	c = lbuf[ic++];
	if ((c == ':') || (c == '\0')) goto fldmrk;
	goto clp;
fldmrk:
	if ((prevc == ':') || (prevc == '*')) goto fldfin;
	if (!allow[cp]) goto fldfin;  /* skipped column */
	val[cp] = atof(&lbuf[sc]);
	pres[cp] = 1;
fldfin:		/* process end of a field */
	if (c == '\0') goto recfin;
	cp++;
	sc = ic;
	goto clp;
recfin:
	rsum = rsuma = rmean = rmeana = 0;
	rn = 0;
	for (i=0; i<TOTC; i++) {
		if (i>ncol) break;
		if (pres[i]) {
			sum[i] += val[i];
			sumsq[i] += val[i]*val[i];
			asum[i] += fabs(val[i]);
			n[i]++;
			rsum += val[i];
			rsuma += fabs(val[i]);
			rn++;
			gsuma += fabs(val[i]);
			gsum +=	val[i];
			gsumsq += val[i] * val[i];
			gn++;
		}
	}
	if (rn>0) {
		rmean = rsum/rn;
		rmeana = rsuma/rn;
		msum += rmeana;
		msumsq += rmeana * rmeana;
		nlab++;
		if (cmflag) printf(":%8.2f",rmean);
		if ((!cmflag) && (caflag)) fputs(":",stdout);
		if (caflag) printf(":%8.2f",rmeana);
		if (ctflag) printf(":%8.2f",rsum);
	}
	fputs("\n", stdout);
	goto lp;

comp:
	close(fid);
	for (i=1; i<=ncol; i++) {
		if (n[i] < 2) {
			amean[i] = mean[i] = sd[i] = cv[i] = 0.0;
			continue;
		}
		mean[i] = sum[i]/n[i];
		amean[i] = asum[i]/n[i];
		sd[i] = sqrt(fabs((sumsq[i] - (n[i]*mean[i]*mean[i]))/(n[i]-1)));
		asd[i] = sqrt(fabs((sumsq[i] - (n[i]*amean[i]*amean[i]))/(n[i]-1)));
		if (fabs(mean[i]) > 0.01)
			cv[i] = (sd[i]/mean[i]) * 100.0;
		if (amean[i] > 0.01)
			acv[i] = (asd[i]/amean[i]) * 100.0;
	}
	if (gn > 1) {
		gmean = gsum/gn;
		gmeana = gsuma/gn;
		gsd = sqrt(fabs((gsumsq - (gn*gmean*gmean))/(gn-1)));
		gsda = sqrt(fabs((gsumsq - (gn*gmeana*gmeana))/(gn-1)));
		gcv = (gsd/gmean) * 100.0;
		gcva = (gsda/gmeana) * 100.0;
	}
	if (nlab > 1) {
		mmean = msum/nlab;
		msd = sqrt(fabs((msumsq - (nlab*mmean*mmean))/(nlab-1)));
		mcv = (msd/mmean) * 100.0;
	}

	if (caflag || cmflag) {
		for (i=0; i<ncol; i++) fputs(":",stdout);
		fputs("ALLDATA:A MEANS\n",stdout);
	}
	if (!msflag) {
		rpr(mean,"mean",gmean,mmean);
		if (!sdflag)
			rpr(sd,"s.d.",gsd,msd);
		if (cvflag) rpr(cv,"c.v.",gcv,mcv);
	}
	if (vaflag) {
		rpr(amean,"mean abs",gmeana,mmean);
		rpr(asd,"s.d. abs",gsda,msd);
		rpr(acv,"c.v. abs",gcva,mcv);
	}
	npr(n,"n",gn,nlab);
	if (fxflag) {
		rpr(sum,"sum x",gsum,msum);
		if (vaflag) rpr(asum, "sum abs",gsuma, zd);
		rpr(sumsq,"sum x*x", gsumsq, msumsq);
	}
	exit(0);
}

rpr(arr,name,x, y)
double arr[];
double x, y;
char *name;
{
register i, j;
	j=1;
	if (!allow[1]) {
		printf("%s:",name);
		j=2;
	}
	for (i=j; i<=ncol; i++) {
		if (allow[i])
			printf("%7.2f:",arr[i]);
		else
			fputs(":",stdout);
	}
	if (caflag || cmflag)
		printf("%7.2f:%7.2f",x,y);
	fputs("\n",stdout);
}
npr(arr,name,nn,nm)
int arr[];
int nn, nm;
char *name;
{
register i, j;
	j=1;
	if (!allow[1]) {
		printf("%s:",name);
		j=2;
	}
	for (i=j; i<=ncol; i++) {
		if (allow[i])
			printf("%7d:",arr[i]);
		else
			fputs(":", stdout);
	}
	if (caflag || cmflag)
		printf("%7d:%7d",nn,nm);
	fputs("\n",stdout);
}

getl()
{
register int i, cc;
register char ch;
extern char mget();

strt:
	i = cc = 0;
clp:
	ch = mget();
	if (ch == '\0') return(1);
	lbuf[i++] = ch;
	if ((ch == ':') || (ch == '\n')) cc++;
	if (ch == '\n') {
		curline++;
		if (curline < stline) goto strt;
		if (curline > enline) {
			curline--;
			return(1);
		}
		i--;
		while (cc<ncol) {
			lbuf[i++] = ':';
			cc++;
		}
		lbuf[i] = '\0';
		fputs(lbuf,stdout);
		return(0);
	}
	goto clp;
}

char mget()
{
	if (iptr == nrd) {
		nrd = read(fid, buf, 512);
		if (nrd < 1) {
			return(0);
		}
		iptr = 0;
	}
	return(buf[iptr++]);
}
