/* xcterm.c		XCOMM terminal mode
 */

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "xcomm.h"

/* globals */
int	cbsiz		= DEFCBUF * K;	/* capture buffer size in bytes	*/
char	captfile[NMSIZE]  = CAPTFILE,	/* capture file's name		*/
	phonefile[NMSIZE] = PHFILE;	/* phone number file's name	*/

/* locals */
static FILE
	*cfp;				/* capture file pointer		*/
static int
	ctop,				/* index top of capture buffer	*/
	capture		= FALSE;	/* are we capturing or not ?	*/
static char
	*cbuffp		= NULL;		/* capture buffer pointer	*/

/* forward declarations */
int toggle(), cleanup();

terminal()
{
    int c, pid;

    if(verbose)
	printf("Entering Terminal Mode\r\n");
    /* split into read and write processes */
    if((pid = forkem()) == 0){
	/* child, read proc: read from port and write to tty */
	cfp = NULL;
	signal(SIGALRM, toggle);
	signal(SIGTERM, cleanup);
	while(1){
	    while((c = readbyte(0)) == -1)
		;
	    putchar(c);
	    if(capture && ctop < cbsiz){ /* handle capture buffer */
		cbuffp[ctop++] = c;
		if(ctop == (8 * cbsiz) / 10)
		   printf("\r\n< capt buff 80\% full >\r\n");
		else if(ctop == cbsiz)
		  printf("\r\n< capt buff 100\% full, rest lost >\r\n");
	    }
	}
	/*NOTREACHED*/
    }
    /* parent, write proc: read from tty and write to port */
    do {
	switch(c = getchar()) {
	case ENDCHAR:		    /* end terminal mode */
		kill(pid, SIGTERM); /* signal read proc to cleanup and exit */
		break;
	case TOGCHAR:		    /* signal read process to toggle	*/
		kill(pid, SIGALRM); /* capture buffer open/closed	*/
		break;
	case DIVCHAR:		    /* divert a file into the modem port */
		divert();
		break;
	case PSELECT:		    /* select and dial a phone number	*/
		pselect();
		break;
	default:		    /* just send the character to the port */
		sendbyte(c);
		break;
	}
    } while(c != ENDCHAR);
    while(wait((int *) 0) >= 0)	    /* wait for the read process to die	*/
	;
    if(verbose)
	printf("\r\nLeaving Terminal Mode\r\n");
    else
	printf("\r\n");
}

/* The next three functions are only run by the port read process (child).
 * They handle the capture buffer.
 *
 * toggle capture status, flush buffer
 */
toggle()
{
    char *malloc();

    if(capture){
	if(ctop > 0)
	    putcapt();
	capture = FALSE;
    }
    else if(cbuffp == NULL && (cbuffp = malloc((unsigned) cbsiz)) == NULL){
	if(verbose)
	    printf("\r\n< can't alloc capture buff space >\r\n");
	else
	    printf("<alloc err>");
    }
    else if(cfp == NULL && (cfp = fopen(captfile, "a")) == NULL){
	if(verbose)
	    printf("\r\n< can't open capture file >\r\n");
	else
	    printf("<open err>");
    }
    else {
	ctop = 0;
	capture = TRUE;
	if(verbose)
	    printf("\r\n< %dk capture buff open >\r\n", cbsiz / K);
	else
	    printf("<buff op, %dk>", cbsiz / K);
    }
    signal(SIGALRM, toggle); /* set signal for next toggle */
}

/* cleanup, flush and exit
 */
static cleanup()
{
    if(capture && ctop > 0)
	putcapt();
    if(cfp != NULL){
	fclose(cfp);
	if(verbose)
	    printf("\r\n< capture buff saved, closed >\r\n");
	else
	    printf("<buff sav, cl>");
    }
    exit(0);
}

/* save the contents of the capture buffer
 */
static putcapt()
{
    int i, c;

    for(i = 0; i < ctop; i++){
	if((c = cbuffp[i]) == '\r')
	    continue;
	if(!isascii(c)){
	    putc('~', cfp);
	    c = toascii(c);
	}
	if(iscntrl(c) && c != '\n'){
	    putc('^', cfp);
	    c += '@';
	}
	putc(c, cfp);
    }
    if(verbose)
	printf("\r\n< capture buff closed, %dk written >\r\n", ctop / K);
    else
	printf("<buff cl, %dk>", ctop / K);
}

/* The next two subroutines are used by the port write process (parent).
 * They produce other sources of input for the port than the keayboard.
 *
 * Select phone number and dial it. (Hayes - type modem)
 * Phone number is stored in a file, one number per line, with the phone
 * number first, followed by some space and then any comments.
 */
pselect()
{
    FILE *fp;
    char buffer[WBSIZE];
    int c, i;
    struct stat sbuf;

    if(stat(phonefile, &sbuf) || sbuf.st_size == 0){
	printf("File %s does not exist or is empty.\r\n", phonefile);
	return;
    }
    if((fp = fopen(phonefile, "r")) == NULL){
	printf("Can't open phone file '%s'.\r\n", phonefile);
	return;
    }
    printf("Type 'd' to dial, 'x' to exit, any other key for next line.\r\n");
    do {
	if(fgets(buffer, WBSIZE, fp) == NULL){
	    rewind(fp);
	    if(fgets(buffer, WBSIZE, fp) == NULL){
		printf("Bad phonelist file.\r\n");
		return;
	    }
	}
	erasln();
	for(i = 0; i < WBSIZE && (c = buffer[i]) != '\n'; i++)
	    putchar(c);
	putchar('\r');
    } while((c = mklow(getchar())) != 'd' && c != 'x');
    erasln();
    fclose(fp);
    if(c == 'd'){
	for(i = 0; i < WBSIZE && !isspace(c = buffer[i]) && c; i++)
	    ;
	buffer[i] = '\0';
	dial(buffer);
    }
}

/* Divert file into input stream, with optional delay after each newline.
 */
divert()
{
    FILE *fp;
    char buffer[NMSIZE];
    int c, i;

    if(verbose)
	printf("\r\nDiverted file's name? ");
    else
	printf(" File? ");
    getsome(buffer);
    if((fp = fopen(buffer, "r")) == NULL){
	printf("\r\nCan't open %s for input\r\n", buffer);
	return;
    }
    if(verbose > 1){
	printf("\r\nThe default delay of %d can be ", DRIBBLE);
	printf("changed by typing the new delay after the 'y'");
    }
    if(verbose)
	printf("\r\nDelay after every newline (y/n)? ");
    else
	printf(" Delay? ");
    getsome(buffer);
    if(verbose)
	printf("\r\n");
    i = 0;
    if(mklow(buffer[0]) == 'y' && (i = atoi(buffer + 1)) == 0)
	i = DRIBBLE;
    while((c = getc(fp)) != EOF)
	if(c != '\n')
	    sendbyte(c);
	else {
	    sendbyte('\r');
	    if(i)
		sleep((unsigned) i);
	}
    fclose(fp);
}

