/* xcxmdm.c		XMODEM protocol send and receive
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "xcomm.h"

/* XMODEM constants, don't change! */
#define SOH	001	/* ^A */
#define EOT	004	/* ^D */
#define ACK	006	/* ^F */
#define NAK	025	/* ^U */
#define CAN	030	/* ^X */
#define CPMEOF	032	/* ^Z */
#define WANTCRC	'C'
#define OK	 0
#define TIMEOUT  -1	/* -1 is returned by readbyte() upon timeout */
#define ERROR    -2
#define WCEOT    -3
#define RETRYMAX 10
#define SECSIZ  128

/* globals */
int
	badline		= FALSE,	/* bad telephone line?		*/
	crcheck		= TRUE;		/* CRC check enabled?		*/

/* locals */
static FILE
	*xfp;				/* buffered file pointer	*/
static int
	firstsec,			/* first sector of file or not?	*/
	textmode	= TRUE;		/* Text translations enabled?	*/
static char wcbuff[SECSIZ];		/* Ward Christensen sector buffer */
static unsigned short updcrc();

xreceive(c, p)
int c;
char *p;
{
    long bytes, ftime;
    short fmode;
    char *ctime();

    if(setmode(c))
	return;
    crcheck = (bitmask == 0xFF);
    if(*p){
	if(verbose)
	    printf("XCOMM: ready to receive single file %s\n", p);
	if(wcrx(p) == ERROR)
	    canit();
	return;
    }
    if(verbose)
	printf("XCOMM: ready to recive series.\n");
    while(TRUE){
	purge();	/* get rid of whatever was sent */
	firstsec == TRUE;
	if(verbose)
	    printf("File name sync...\n");
	sendbyte(crcheck ? WANTCRC : NAK);
	if((c = wcgetsec(10)) != 0){
	    printf("XCOMM: File name fetch failed.\n");
	    if(verbose > 1)
		printf("XCOMM: received sector %d instead.\n", c);
	    canit();
	    return;
	}
	sendbyte(ACK);
	if(wcbuff[0] == '\0'){
	    if(verbose)
		printf("XCOMM: series received.\n");
	    return;
	}
	if(verbose > 1)
	    printf("Incoming file's name: %s\n", wcbuff);
	else if(verbose)
	    printf("File: %s\n", wcbuff);
	p = wcbuff + strlen(wcbuff) + 1;
	if(*p){	/* file coming from a UNIX type system */
	    if(verbose > 1){
		sscanf(p, "%ld%lo%o", &bytes, &ftime, &fmode);
		printf("                size: %ld\n", bytes);
		printf("                mode: %o\n", fmode);
		printf("                time: %s\n", ctime(&ftime));
		printf("From a UNIX - type system.\n\n");
	    }
	}
	else {	/* file coming from a CP/M type system */
	    if(verbose > 1)
		printf("From a CP/M - type system.\n\n");
	    for(p = wcbuff; *p; p++){
		*p = mklow(*p);
		if(*p == '/')
		    *p = '_';
	    }
	    if(*--p == '.')
		*p = '\0';
	}
	if(wcrx(wcbuff) == ERROR){
	    canit();
	    return;
	}
    }
}

xsend(c, p)
int c;
char *p;
{
    FILE *pfp;
    int attempts;
    struct stat f;
    char *file, *next, wrkbuff[WBSIZE], *rindex();

    if(!*p){
	printf("XCOMM: Must have filenames for send option.\n");
	printf("XCOMM: Send option aborted.\n");
	canit();
	return;
    }
    if(setmode(c))
	return;
    sprintf(wrkbuff, "echo %s", p);
    if((pfp = popen(wrkbuff, "r")) == NULL){
	printf("Can't expand %s\n", p);
	return;
    }
    p = wrkbuff;
    while((c = getc(pfp)) != EOF)
	*p++ = c;
    pclose(pfp);
    *p = '\0';
    if(verbose)
	printf("Send file(s): %s\n", wrkbuff);
    next = wrkbuff;
    skipspace(&next);
    while(*next){
	file = next;
	skipitem(&next);
	if(*next){
	    *next++ = '\0';
	    skipspace(&next);
	}
	if((p = rindex(file, '/')) != NULL)
	    p++;
	else
	    p = file;
	zerobuff();
	strcpy(wcbuff, p);
	p = wcbuff + strlen(wcbuff) + 1;
	if(stat(file, &f)){
	    if(verbose)
		printf("Can't get status of file '%s'.\n", file);
	    return;
	}
	sprintf(p, "%lu %lo %o", f.st_size, f.st_mtime, f.st_mode);
	if(verbose)
	    printf("Awaiting pathname NAK for %s\n", file);
	firstsec = TRUE;
	attempts = 0;
	while((c = readbyte(30)) != NAK && c != WANTCRC && c != CAN)
	    if(c == TIMEOUT && ++attempts > RETRYMAX)
		break;
	if(c == TIMEOUT){
	    if(verbose)
		printf("Timeout on pathname NAK.\n");
	    canit();
	    return;
	}
	if(c == CAN){
	    if(verbose)
		printf("Receiver CANcelled on pathname NAK.\n");
	    return;
	}
	crcheck = (c == WANTCRC);
	if(wcputsec(0) == ERROR){
	    if(verbose)
		printf("Error on pathname send.\n");
	    return;
	}
	if(wctx(file) == ERROR){
	    if(verbose)
		printf("Error in transmitting file %s\n", file);
	    return;
	}
    }
}

/* receive a file using XMODEM protocol
 */
static wcrx(name)
char *name;
{
    register int sectnum, sectcurr, sendchar, eofseen;

    if(!mungmode && !access(name, 0)){
	canit();
	if(verbose)
	    printf("Receive of %s aborted due to pre-exsistence.\n");
	return(ERROR);
    }
    if((xfp = fopen(name, "w")) == NULL){
	canit();
	if(verbose)
	    printf("Receive of %s aborted due to inabilty to open.\n");
	return(ERROR);
    }
    firstsec = TRUE;
    eofseen = FALSE;
    sectnum = 0;
    sendchar = crcheck ? WANTCRC : NAK;
    if(verbose)
	printf("Sync...\n");
    while(TRUE){
	if(badline)
	    purge();
	sendbyte(sendchar);
	sectcurr = wcgetsec(6);
	if(sectcurr == ((sectnum + 1) & bitmask)){
	    sectnum++;
	    if(eofseen){
		if(verbose)
		    printf("Received sector after eof #%d   \r", sectnum);
	    }
	    else {
		eofseen = putsec();
		if(verbose)
		    printf("Received sector #%d             \r", sectnum);
	    }
	    sendchar = ACK;
	    continue;
	}
	if(sectcurr == sectnum & bitmask){
	    if(verbose)
		printf("Received duplicate sector #%d   \r", sectnum);
	    sendchar = ACK;
	    continue;
	}
	putchar('\n');
	fclose(xfp);
	if(sectcurr == WCEOT){
	    if(verbose)
		printf("File recived OK\n");
	    sendbyte(ACK);
	    return(OK);
	}
	if(sectcurr == ERROR)
	    return(ERROR);
	if(verbose)
	    printf("Sync error ... expected %d(%d), got %d\n",
				(sectnum + 1) & bitmask, sectnum, sectcurr);
	return(ERROR);
    }
}

/* transmit file using XMODEM protocol
 */
static wctx(name)
char *name;
{
    register int sectnum, eoflg, c, attempts;

    if((xfp = fopen(name, "r")) == NULL){
	if(verbose)
	    printf("Can't open file %s for reading.\n", name);
	return(ERROR);
    }
    firstsec = TRUE;
    attempts = 0;
    if(verbose)
	printf("Sync...\n");
    while((c = readbyte(30)) != NAK && c != WANTCRC && c != CAN)
	if(c == TIMEOUT && ++attempts > RETRYMAX){
	    if(verbose)
		printf("Receiver not responding.\n");
	    fclose(xfp);
	    return(ERROR);
	}
    if(c == CAN){
	if(verbose)
	    printf("Receiver CANcelled.\n");
	fclose(xfp);
	return(ERROR);
    }
    crcheck = (c == WANTCRC);
    if(verbose)
	printf("%s error checking requested.\n",
					crcheck ? "CRC check" : "Checksum");
    sectnum = 1;
    do {
	eoflg = getsec();
	if(verbose)
	    printf("Transmitting sector #%d\r", sectnum);
	if(wcputsec(sectnum) == ERROR){
	    fclose(xfp);
	    return(ERROR);
	}
	sectnum++;
    } while(eoflg);
    putchar('\n');
    fclose(xfp);
    attempts = 0;
    sendbyte(EOT);
    while(readbyte(15) != ACK && attempts++ < RETRYMAX)
	sendbyte(EOT);
    if(attempts >= RETRYMAX){
	if(verbose)
	    printf("Receiver not responding to completion.\n");
	return(ERROR);
    }
    if(verbose > 1)
	printf("Transmission complete.\n");
    return(OK);
}

/* wcgetsec inputs a Ward Christensen type sector.
 * Returns the sector number encountered; or ERROR if valid sector is not
 * received, or CAN received; or WCEOT if eot sector.
 * maxtime is the timeout for the first character, set to 6 sec for retrys.
 * No ACK is sent if the sector is received ok. This must be done by the
 * caller when it is ready to receive the next sector.
 */
static wcgetsec(maxtime)
int maxtime;
{
    register checksum, j, c;
    register unsigned short oldcrc;
    int sectcurr, sectcomp, attempts;

    for(attempts = 0; attempts < RETRYMAX; attempts++){
	do {
	    c = readbyte(maxtime);
	} while(c != SOH && c != EOT && c != CAN && c != TIMEOUT);
	switch(c){
	case SOH:
	    sectcurr = readbyte(3);
	    sectcomp = readbyte(3);
	    if((sectcurr + sectcomp) == bitmask){
		oldcrc = checksum = 0;
		for(j = 0; j < SECSIZ; j++){
		    if((c = readbyte(3)) < 0)
			goto timeout;
		    wcbuff[j] = c;
		    if(crcheck)
			oldcrc = updcrc(c, oldcrc);
		    else
			checksum += c;
		}
		if((c = readbyte(3)) < 0)
		    goto timeout;
		if(crcheck){
		    oldcrc = updcrc(c, oldcrc);
		    if((c = readbyte(3)) < 0)
			goto timeout;
		    if(updcrc(c, oldcrc)){
			if(verbose > 1)
			    printf("\nCRC error\n");
			break;
		    }
		}
		else if(((checksum - c) & bitmask) != 0){
		    if(verbose > 1)
			printf("\nChecksum error\n");
		    break;
		}
		firstsec = FALSE;
		return(sectcurr);
	    }
	    else if(verbose > 1)
		printf("\nSector number garbled 0%03o 0%03o\n",
							    sectcurr, sectcomp);
	    break;
	case EOT:
	    if(readbyte(3) == TIMEOUT)
		return(WCEOT);
	    break;
	case CAN:
	    if(verbose)
		printf("\nSender CANcelled.\n");
	    return(ERROR);
	case TIMEOUT:
	    if(firstsec)
		break;
timeout:
	    if(verbose > 1)
		printf("\nTimeout\n");
	    break;
	}
	if(verbose > 1)
	    printf("\nTrying again on ths sector.\n");
	purge();
	if(firstsec)
	    sendbyte(crcheck ? WANTCRC : NAK);
	else {
	    maxtime = 6;
	    sendbyte(NAK);
	}
    }
    if(verbose)
	printf("\nRetry count exceeded.\n");
    canit();
    return(ERROR);
}

/* wcputsec outputs a Ward Christensen type sector.
 * it returns OK or ERROR
 */
static wcputsec(sectnum)
int sectnum;
{
    register checksum, j, c, attempts;
    register unsigned short oldcrc;

    oldcrc = checksum = 0;
    for(j = 0; j < SECSIZ; j++){
	c = wcbuff[j];
	oldcrc = updcrc(c, oldcrc);
	checksum += c;
    }
    oldcrc = updcrc(0, updcrc(0, oldcrc));

    for(attempts = 0; attempts < RETRYMAX; attempts++){
	sendbyte(SOH);
	sendbyte(sectnum);
	sendbyte(-sectnum - 1);
	for(j = 0; j < SECSIZ; j++)
	    sendbyte(wcbuff[j]);
	if(badline)
	    purge();
	if(crcheck){
	    sendbyte((int) (oldcrc >> 8));
	    sendbyte((int) oldcrc);
	}
	else
	    sendbyte(checksum);
	switch(c = readbyte(10)){
	case CAN:
	    if(verbose)
		printf("\nReceiver CANcelled.\n");
	    return(ERROR);
	case ACK:
	    firstsec = FALSE;
	    return(OK);
	case NAK:
	    if(verbose > 1)
		printf("\nGot a NAK on sector acknowledge.\n");
	    break;
	case TIMEOUT:
	    if(verbose > 1)
		printf("\nTimeout on sector acknowledge.\n");
	    break;
	default:
	    if(verbose > 1)
		printf("\nGot 0%03o for sector acknowledge.\n", c);
	    do {
		if((c = readbyte(3)) == CAN){
		    if(verbose)
			printf("\nReceiver CANcelled.\n");
		    return(ERROR);
		}
	    } while(c != TIMEOUT);
	    if(firstsec)
		crcheck = (c == WANTCRC);
	    break;
	}
    }
    if(verbose)
	printf("\nRetry count exceeded.\n");
    return(ERROR);
}

/* update the cyclic redundantcy check value
 */
static unsigned short updcrc(c, crc)
register c;
register unsigned crc;
{
    register int i;

    for(i = 0; i < 8; i++){
	if(crc & 0x8000){
	    crc <<= 1;
	    crc += (((c <<= 1) & 0400) != 0);
	    crc ^= 0x1021;
	}
	else {
	    crc <<= 1;
	    crc += (((c <<= 1) & 0400) != 0);
	}
    }
    return(crc);
}

static zerobuff()
{
    register int i;

    for(i = 0; i < SECSIZ; i++)
	wcbuff[i] = '\0';
}

/* send 10 CAN's to try to get the other end to shut up */
static canit()
{
    register int i;

    for(i = 0; i < 20; i++)
	sendbyte(CAN);
}

static setmode(c)
int c;
{
    switch(mklow(c)) {
    case 't':
	textmode = TRUE;
	goto whatmode;
    case 'b':
	textmode = FALSE;
    case ' ':
whatmode:
	if(verbose)
	    printf("%s file transfer mode.\n", textmode ? "Text" : "Binary");
	break;
    default:
	printf("Unknown mode char '%c' (%d).\n", c, c);
	printf("XCOMM: Options 'r' and 's' must have a mode of 't' or 'b'\n");
	return(1);
    }
    return(0);
}

/* fill the CP/M sector buffer from the UNIX file
 * do text adjustments if necessary
 * return 1 if more sectors are to be read, or 0 if this is the last
 */
static getsec()
{
    register int i, c;

    i = 0;
    while(i < SECSIZ && (c = getc(xfp)) != EOF){
	if(textmode && c == '\n'){
	    wcbuff[i++] = '\r';
	    if(i >= SECSIZ){		/* handle a newline on the last byte */
		ungetc(c, xfp);		/* of the sector		     */
		return(1);
	    }
	}
	wcbuff[i++] = c;
    }
    /* make sure that an extra blank sector is not sent */
    if(c != EOF && (c = getc(xfp)) != EOF){
	ungetc(c, xfp);
	return(1);
    }
    /* fill up the last sector with ^Z's if text mode or 0's if binary mode */
    while(i < SECSIZ)
	wcbuff[i++] = textmode ? CPMEOF : '\0';
    return(0);
}

/* Put received WC sector into a UNIX file
 * using text translations in nessesary.
 */
static putsec()
{
    register int i, c;

    for(i = 0; i < SECSIZ; i++){
	c = wcbuff[i];
	if(textmode){
	    if(c == CPMEOF)
	        return(1);
	    if(c == '\r')
		continue;
	}
	putc(c, xfp);
    }
    return(0);
}

