/* xcport.c	XCOMM modem port interface routines
 */
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <sgtty.h>
#include "xcomm.h"

/* The dial routine uses these two defines. DIALSTR is a sprintf format
 * string that assumes a Hayes-compatabile modem. MDELAY is the delay in
 * the output required because (on my modem) the command would be ignored
 * if sent at full speed. Change for other setups. (I am using a U.S.Robotics
 * Password modem right now.)
 */
#define DIALSTR "\rATDT %s\r"	/* format string for Hayes-type modem */
#define MDELAY	20000		/* delay for output to modem itself */

int bitmask = 0xFF;		/* modem port i/o data mask	*/

static int mfd = -1;		/* modem port file descriptor	*/
static struct sgttyb pmode =	/* default 1200 baud rate	*/
    { B1200, B1200, 0, 0, RAW };
static char port[NMSIZE];	/* modem port device file string */

char *mport(s)	/* get/set port string */
char *s;
{
    if(s != NULL && mfd == -1)
	strcpy(port, s);
    return(port);
}

/* Get/set the baud rate of the modem port. If the port hasn't been opened yet,
 * just store in pmode for mopen() to use when it opens the port.
 */
int mbaud(s)
char *s;
{
    int baud;

    if(s != NULL){
	/* this gives a more limited, realistic range than in sgtty.h */
	switch (atoi(s)) {
	case  300: baud = B300;  break;
	case 1200: baud = B1200; break;
	case 2400: baud = B2400; break;
	case 9600: baud = B9600; break;
	default:   return(-1);
	}
	pmode.sg_ispeed = pmode.sg_ospeed = baud;
	if(mfd != -1)
	    ioctl(mfd, TIOCSETN, &pmode);
    }
    switch(pmode.sg_ispeed){
    case  B300: return(300);
    case B1200: return(1200);
    case B2400: return(2400);
    case B9600: return(9600);
    }
    printf("Impossible error in baud rate.\n");
    return(0);
}

/* Opens the modem port and configures it. If the port string is
 * already defined it will use that as the modem port;  otherwise it
 * gets the environment variabele MODEM. Returns 0 for sucess
 * or -1 on error.
 */
int mopen()
{
    char *p, *getenv();

    if(port[0] == '\0'){
	if((p = getenv("MODEM")) == NULL)
	    return(-1);
	strcpy(port, p);
    }
    if((mfd = open(port, O_RDWR)) < 0)
	return(-1);
    ioctl(mfd, TIOCEXCL, (struct sgttyb *) 0);
    ioctl(mfd, TIOCSETP, &pmode);
    return(0);
}

/* Attach standard input and output to the modem port. This only gets called
 * after a fork by the child process; which then exec's a program that uses
 * standard i/o for some data transfer protocol. (To put this here is actually
 * a kludge, but I wanted to keep the modem-specific stuff in a black box.)
 */
mattach()	/* attach standard i/o to port */
{
    dup2(mfd, 0);	/* close local stdin and connect to port	*/
    dup2(mfd, 1);	/* close local stdout and connect to port	*/
			/* (stderr is still connected to the local tty)	*/
    close(mfd);		/* close the old port descriptor		*/
}

/* alarm clock timeout function - do nothing, just return
 */
static alrm(){}

/* Read one or more bytes. Buffering is performed so that a data burst
 * can be caught up with quickly and less kernal calls need to be made.
 * If there are bytes still in the buffer it returns the next one;
 * otherwise a read call is made and all available bytes (in the system's
 * tty queue) are placed in the buffer. If there are no bytes available
 * after a wait of 'seconds' the read call is aborted by the alarm clock
 * and a -1 is returned. If 'seconds' is zero then no abort is called
 * for and the read call waits (forever, or until some other signal is
 * received). All data from the modem is input through this routine.
 */
int readbyte(seconds)
int seconds;
{
    static char rxbuf[BUFSIZ], *p;	/* BUFSIZ is defined in stdio.h */
    static int count = 0;

    if(count > 0){
	count--;
	return(*p++ & bitmask);
    }
    if(seconds > 0){
	signal(SIGALRM, alrm);
	alarm((unsigned)seconds);
    }
    if((count  = read(mfd, p = rxbuf, BUFSIZ)) < 1)
	return(-1);
    if(seconds > 0)
	alarm(0);
    count--;
    return(*p++ & bitmask);
}

/* Output a byte to the modem port, filtering with bitmask.
 * All data sent to the modem is output through this routine.
 */
sendbyte(ch)
int ch;
{
    char c;

    c = ch & bitmask;
    write(mfd, &c, 1);
}

/* Dial a phone number, using proper format and delay.
 */
dial(s)
char *s;
{
    int i;
    char buffer[WBSIZE];

    sprintf(buffer, DIALSTR, s);
    s = buffer;
    while(*s){
	sendbyte(*s++);
	/* This busy waiting, normally a bad idea on a multi-tasking system,
	 * was used because sleep(1) is way too much delay.
	 */
	for(i = 0; i < MDELAY; i++)
	    ;
    }
}
