;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; expa.asm
; Explosion support routines for EXPLOD.C.  For HGC and CGA.
;
; Make sure one of DLC or TURBOC is defined below.
; This has been tested with the Arrowsoft Assembler 1.00d and MASM 5.0
; C compilers other than Datalight C and Turbo C will probably require 
; changes to the segment naming.
;
; 89/06/24 Dennis Lo    Initial release.
; 89/07/03 Dennis Lo    Added ifdefs for linking with Turbo C
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;***** uncomment this for Datalight C
;DLC     equ     1
;***** uncomment this for Turbo C
TURBOC  equ     1

ifdef DLC
pgroup  group   prog
prog    segment byte public 'prog'
        assume  cs:pgroup
prog    ends

dgroup  group   data
data    segment word public 'data'
        assume ds:dgroup
data    ends
endif


ifdef TURBOC
        name    t
_TEXT   segment byte public 'CODE'
DGROUP  group   _DATA,_BSS
        assume  cs:_TEXT,ds:DGROUP,ss:DGROUP
_TEXT   ends
_DATA   segment word public 'DATA'
_d@     label   byte
_DATA   ends
_BSS    segment word public 'BSS'
_b@     label   byte
_BSS    ends
endif


        public  _GetKey
        public  _ChkKey
        public  _gr_card
        public  _gr_setcard
        public  _gr_gmode
        public  _gr_tmode
        public  _gr_fill
        public  _gr_addr
        public  _gr_value
        public  _gr_frplot


;;;;;;;;;;;;;;;;;;;;;;;;; CONST ;;;;;;;;;;;;;;;;;;;;;;;;; 
;       port addresses
config  equ     03bfh
index   equ     03b4h
cntrl   equ     03b8h

;       control codes
scrn_on equ     8
grph    equ     2
text    equ     20h

par     equ     4               ;stack offset of 1st C call parameter 


;;;;;;;;;;;;;;;;;;;;;;;;; DATA ;;;;;;;;;;;;;;;;;;;;;;;;;
ifdef DLC
data    segment word public 'data'
endif
ifdef TURBOC
_DATA   segment word public 'DATA'
endif
 
Vidtype db      0               ;video card type 
Vidseg  dw      0               ;video mem segment
Vidsize dw      0               ;video mem size

;       6845 config tables
gtable  db      35h,2dh,2eh,07h
        db      5bh,02h,57h,57h
        db      02h,03h,00h,00h

ttable  db      61h,50h,52h,0fh
        db      19h,06h,19h,19h
        db      02h,0dh,0bh,0ch

;       Table of bit values for plotting points
;oldmasks db    128,64,32,16,8,4,2,1
masktable db    192,96,48,24,12,6,3,1   ;2-pixel wide points

ifdef DLC
data    ends
endif
ifdef TURBOC
_DATA   ends
endif

;;;;;;;;;;;;;;;;;;;;;;;;; CODE ;;;;;;;;;;;;;;;;;;;;;;;;;
ifdef DLC
prog    segment byte public 'prog'
endif
ifdef TURBOC
_TEXT   segment byte public 'CODE'
endif

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Blocking keyboard read - returns ASCII char
;
_GetKey  proc    near
        push    si
        push    di

        mov     ax,0
        int     16h
        mov     ah,0

        pop     di
        pop     si
        ret
_GetKey  endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Nonblocking keyboard read - returns ASCII char
;
_ChkKey  proc    near
        push    si
        push    di

        mov     ah,1
        int     16h
        jz      nokey           ;if no key then return
        mov     ax,0            ;else get the key out of the buffer
        int     16h
        mov     ah,0
        jmp     chkret
nokey:
        mov     ax,0
chkret:
        pop     di
        pop     si
        ret
_ChkKey  endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Guess video card type
;
_gr_card proc    near
        push    bp
        push    si
        push    di

        mov     ah,0fh
        int     10h
        mov     bl,al
        mov     ax,'c'          ;Assume CGA unless it looks
        cmp     bl,07h          ; like a Monochrome Adapter,
        jne     card_done       ; in which case assume it is
        mov     ax,'h'          ; a HGC. (won't work for EGA)
card_done:
        pop     di
        pop     si
        pop     bp
        ret
_gr_card endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Set video card type
;
_gr_setcard proc near
        push    bp
        mov     bp,sp

        mov     ax,par[bp]
        mov     Vidtype,al
        cmp     al,'h'          ;set video mem segment
        jne     s_cga

        mov     ax,0b000h
        mov     cx,8000h
        jmp     s_end

s_cga:  mov     ax,0b800h
        mov     cx,4000h

s_end:  mov     Vidseg,ax
        mov     Vidsize,cx

        pop     bp
        ret
_gr_setcard endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Set video card for graphics mode
; No params
_gr_gmode proc   near
        push    bp
        push    es
        push    ds
        push    si

        mov     al,Vidtype
        cmp     al,'h'
        jne     g_cga
g_hgc:  call    hgc_gmode
        jmp     g_end

g_cga:  call    cga_gmode

g_end:
        pop     si
        pop     ds
        pop     es
        pop     bp
        ret
_gr_gmode        endp


;Set HGC graphics mode, page 0
hgc_gmode:
        mov     dx,config
        mov     al,3
        out     dx,al

        mov     al,grph
        lea     si,gtable
        mov     bx,0
        mov     cx,4000h
        call    setmd
        ret


;Set CGA graphics mode
cga_gmode:
        mov     ah,00h          ;set 320x200 4-colour mode
        mov     al,04h          
        int     10h

        mov     ah,0bh          ;set green-red-brown colour palette
        mov     bx,0100h
        int     10h
        ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Set text mode.  No params.
_gr_tmode proc  near
        push    bp
        push    es
        push    ds
        push    si

        mov     al,Vidtype
        cmp     al,'h'
        jne     t_cga
t_hgc:  call    hgc_tmode
        jmp     t_end
t_cga:  call    cga_tmode
t_end:
        pop     si
        pop     ds
        pop     es
        pop     bp
        ret
_gr_tmode        endp


;Set HGC text mode
hgc_tmode:
        mov     dx,config
        mov     al,0
        out     dx,al

        mov     al,text
        lea     si,ttable
        mov     bx,720h
        mov     cx,2000h
        call    setmd
        ret


;Set CGA text mode
cga_tmode:
        mov     ah,00h
        mov     al,02h          ;80x25 b/w text
        int     10h
        ret


;;;;;;;;;;;;;;;;;;;;;;;;; setmd
; Set display mode to graphics or text.
; Entry param:
;       al = value to be output to 6845 control port 
;       si = 6845 param table
;       cx = # of words to be cleared
;       bx = blank value
setmd   proc    near
        push    di
        push    ds
        push    es
        push    ax
        push    bx
        push    cx

;       change mode but without scrn_on
        mov     dx,cntrl
        out     dx,al

;       initialize the 6845
        mov     ax,ds
        mov     es,ax           ;also point es:si to param table

        mov     dx,index
        mov     cx,12           ;12 params to be output
        xor     ah,ah           ;starting from reg. 0

parms:  mov     al,ah
        out     dx,al           ;output 6845 reg. index

        inc     dx
        lodsb
        out     dx,al           ;output 6845 reg. data

        inc     ah
        dec     dx
        loop    parms

;       clear the display buffer
        pop     cx
        mov     ax,0b000h
        cld

        mov     es,ax
        xor     di,di
        pop     ax
        rep     stosw

;       scrn_on, page 0
        mov     dx,cntrl
        pop     ax
        add     al,scrn_on
        out     dx,al

        pop     es
        pop     ds
        pop     di
        ret
setmd   endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Fill graphics screen.
; Params: int16 fill_value
_gr_fill proc   near
        push    bp
        mov     bp,sp
        push    es
        push    si
        push    di

        mov     cx,Vidsize
        mov     ax,Vidseg
        mov     es,ax
        cld                     ;set dir fwd
        xor     di,di           ;set up dest
        mov     ax,par[bp]      ;ax := parameter1
        rep     stosw           ;mem fill

        pop     di
        pop     si
        pop     es
        pop     bp
        ret
_gr_fill endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Returns video mem addr of a (x,y) point
; Params: int16 x,y
par_x   equ     par     ;in:  x
par_y   equ     2+par   ;in:  y

_gr_addr proc   near
        push    bp
        mov     bp,sp
        push    si
        push    di

        mov     al,Vidtype
        cmp     al,'h'
        jne     a_cga
a_hgc:  call    hgc_addr
        jmp     a_end
a_cga:  call    cga_addr
a_end:
        pop     di
        pop     si
        pop     bp
        ret
_gr_addr endp


;Calc HGC addr = 90 * (y / 4)  +  2000h * (y & 3)  +  x / 8
hgc_addr:
        ;loc = (y / 4) * 90
        MOV     AX,par_y[BP]
        MOV     CX,AX                   ;save y in CX
        SHR     AX,1
        SHR     AX,1

        ;ax * 90 = ax*2 + ax*8 + ax*16 + ax*64
        shl     ax,1                    ;*2
        mov     bx,ax
        shl     ax,1                    ;*4
        shl     ax,1                    ;*8
        add     bx,ax
        shl     ax,1                    ;*16
        add     bx,ax
        shl     ax,1                    ;*32
        shl     ax,1                    ;*64
        add     ax,bx
        MOV     DI,AX                   ;store sum in DI

        ;loc += (y & 3) * 2000H
        AND     CX,3
        MOV     AX,2000H
        MUL     CX                      ;(msb ignored)
        ADD     DI,AX

        ;loc += (x / 8)
        MOV     AX,par_x[BP]
        shr     ax,1
        shr     ax,1
        shr     ax,1
        ADD     ax,di
        ret


;Calc CGA addr = 80 * (y / 4)  +  2000h * (y & 3)  +  x / 8
cga_addr:
        ;loc = (y / 2) * 80
        MOV     AX,par_y[BP]
        MOV     CX,AX                   ;save y in CX
        SHR     AX,1

        MOV     BL,80
        MUL     BL
        MOV     DI,AX                   ;store sum in DI

        ;loc += (y & 1) * 2000H
        AND     CX,1
        MOV     AX,2000H
        MUL     CX                      ;(msb ignored)
        ADD     DI,AX

        ;loc += (x / 8)
        MOV     AX,par_x[BP]
        shr     ax,1
        shr     ax,1
        shr     ax,1
        ADD     ax,di
        ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Returns byte value of a (x,y) point
; Params: int16 x,y

_gr_value proc   near
        push    bp
        mov     bp,sp

;       Calc bitval = 2 << [7 - (x & 7)]
        MOV     AX,par_x[BP]
        AND     AX,7
        MOV     BX,AX
        MOV     AL,masktable[BX]

        pop     bp
        ret
_gr_value endp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
; Plot one frame of 3-byte point records (offset, value) 
; _gr_frplot (num_points, &frame_table[i], centre_addr)
;
fpar_nump       equ     par             ;No. of points per frame
fpar_ftable     equ     par+2           ;Pointer to frame
fpar_centre     equ     par+4           ;addr of explosion centre 
DEADPOINT       equ     32767

_gr_frplot proc near
        push    bp
        mov     bp,sp
        push    es
        push    si
        push    di

        mov     ax,Vidseg
        mov     es,ax
        mov     cx,fpar_nump[bp]
        mov     si,fpar_ftable[bp]
        mov     dx,fpar_centre[bp]
        mov     di,3

frloop:
        mov     bx,[si]                 ;get point offset
        cmp     bx,DEADPOINT            ;check if point is dead
        je      frnext
        add     bx,dx                   ;add centre to offset
        mov     al,[si+2]               ;get point's byte value
        xor     al,es:[bx]              ;xor into point's addr
        mov     es:[bx],al
frnext:
        add     si,di
        loop    frloop

        pop     di
        pop     si
        pop     es
        pop     bp
        ret
_gr_frplot      endp


ifdef DLC
prog    ends
endif
ifdef TURBOC
_TEXT   ends
endif
        end
