/* Object.c -- implementation of class Object

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2017
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-5363
	uucp: {decvax!}seismo!elsie!cecil!keith
	September, 1985

Function:

Operations applicable to all objects.
	
Modification History:

15-Feb-87	K. E. Gorlen

1.  Fix output of class name length in Object::storer(fileDescTy&)
to be 1 byte.

25-Nov-86	K. E. Gorlen

1.  Remove call to derivedClassResponsibility from Object::isA() to
eliminate recursion on error.

1-Nov-86	K. E. Gorlen

1.  Added classNumber.

27-Oct-86	K. E. Gorlen

1.  Remove object numbers from ASCII and binary Object I/O.
2.  Remove version number suppression option.
3.  Add class references to Object I/O.

26-Oct-86	K. E. Gorlen

1.  Binary Object I/O cleanup.
2.  Add option to suppress version number and object number in ASCII
Object I/O.

06-Oct-86	S. M. Orlow

1.  Added readFrom(fileDescTy&,...), suppressStoreOn(bool,bool)
      and members of Class Object:  storeOn(fileDescTy&,...), 
       reader(fileDescTy&,...), storer(fileDescTy&,...), 

2.  Constructor for class Class changed to out-of-line because additional
      argument causes inline limit to be exceeded.

3.  Moved checkRead, syntaxError, read_Cstring, store_Cstring to oopsIO.c

1-Oct-86	K. E. Gorlen

1. C++ Release 1.1: restore use of default argument in readFrom.

26-Sep-86	K. E. Gorlen

1.  C++ Release 1.1: make baseClass(), className(), and classVersion() inline.
2.  OK to use String in readFrom.

1-Aug-86	K. E. Gorlen

1.  Skip whitespace on entry to read_Cstring.

30-Apr-86	K. E. Gorlen

1.  Modify deepCopy() to call Object::shallowCopy() instead of
shallowCopy().

29-Apr-86	K. E. Gorlen

1.  Add Object::baseClass().
2.  Modify Object::isKindOf() to use baseClass().
3.  Change default implementation of copy() from shallowCopy()
to deepCopy().
4.  Change implementation of Object::deepenShallowCopy() from {} to
derivedClassResponsibility().
	
7-Mar-86	K. E. Gorlen

1.  Modify storeOn/readFrom to use { and } instead of ( and ) to
mark beginning and end of objects on stream.
2.  Modify store_Cstring to translate { and } to \[ and \] on output.
3.  Modify read_Cstring to translate \[ and \] to { and } on input.

These modifications will allow a future version of readFrom to skip over
an unknown object by counting {s and }s.

5-Mar-86	K. E. Gorlen

1.  Add Object::destroyer().

26-Feb-86	K. E. Gorlen

1.  Modify readFrom to not use class String to work around compiler bug.

25-Feb-86	K. E. Gorlen

1.  Combined Object.c and readFrom.c; made possible by repair of compiler bug.

10-Feb-86	K. E. Gorlen

1.  Fix Object::dependents to return empty list instead of KEYNOTFOUND error
when dependDict exists, but this object has no dependents.

17-Jan-86	K. E. Gorlen

1.  Increase initial capacity of readTable to 1024.
2.  Expand readTable by 1024 when it becomes full.

10-Jan-86	K. E. Gorlen

1.  Add strm to syntaxErr calling sequence.

1.  Increase initial size of storeOnDict to 256.

18-Dec-85	K. E. Gorlen

1.  Modified Object::dependents to return an empty OrderedCltn instead
of nil if the object has no dependents.

*/

#include <ctype.h>
#include "oopsconfig.hxx"
#include "Object.hxx"
#include "Dictionary.hxx"
#include "LookupKey.hxx"
#include "IdentDict.hxx"
#include "String.hxx"
#include <string.h>
#include "Assoc.hxx"
#include "AssocInt.hxx"
#include "OrderedCltn.hxx"
#include "oopsIO.hxx"

extern const int OOPS_DRVDCLASSRSP,OOPS_ILLEGALMFCN,OOPS_BADARGCL,
	OOPS_BADARGSP,OOPS_BADARGCLM,OOPS_BADARGSPM,OOPS_BADCLASS,
	OOPS_BADSPEC,OOPS_NOISA,OOPS_RDEOF,OOPS_RDFAIL,OOPS_RDSYNERR,
	OOPS_RDREFERR,OOPS_RDWRONGCLASS,OOPS_RDUNKCLASS,OOPS_RDVERSERR,
	OOPS_RDBADTYP,OOPS_RDABSTCLASS,OOPS_STROV;

#define	THIS	Object
#define	BASE
/* DEFINE_CLASS(); */
overload Object_reader;
static void Object_reader(istream&, Object&)
{
	setOOPSerror(OOPS_RDABSTCLASS,DEFAULT,"Object");
}
static void Object_reader(fileDescTy&,Object&)
{
	setOOPSerror(OOPS_RDABSTCLASS,DEFAULT,"Object");
}

Class class_Object = Class( *(Class*)0, "Object", 1, sizeof(Object), Object_reader, Object_reader, NULL, NULL);

extern Dictionary classDictionary;	// Dictionary of all classes 

static IdentDict* dependDict =0;	// object ID -> dependents list 
static IdentDict* deepCopyDict =0;	// object ID -> object copy dictionary for deepCopy 
static IdentDict* storeObjDict =0;	// object ID -> object number dictionary for storeOn 
static OrderedCltn* storeClassTable =0;	// table of classes stored
static OrderedCltn* readObjTable =0;	// table of objects for readFrom()
static OrderedCltn* readClassTable =0;	// table of classes for readFrom()
static char clsName[256];		// class name storage for readFrom()
	
void OOPSIoErrCleanup() {

    if (storeObjDict) {
	DO(*storeObjDict,AssocInt*,asc) delete asc; OD
	delete storeObjDict;  storeObjDict = 0;
    }

    if (storeClassTable) {
	DO(*storeClassTable,Class*,cl) cl->classNumber(0); OD
	delete storeClassTable;  storeClassTable = 0;
    }

    if (readObjTable)   delete readObjTable;  readObjTable = 0;
    if (readClassTable) delete readClassTable;  readClassTable = 0;
}

bool Object::isKindOf(const Class& clid)
{
	const Class* cl = isA();
	do {
		if (cl == &clid) return YES;
		cl = cl->baseClass();
	} while (cl != (Class*) 0);
	return NO;
}

UNSIGNED Object::size() { return 0; }		// # of objects in Array.container subclass 

UNSIGNED Object::capacity() { return 0; }	// subClass.capacity 

const Class* Object::isA()
{
	setOOPSerror(OOPS_NOISA,FATAL);
	exit(1);
	return 0;
}

const const Class* Object::species()	{ return isA(); }

bool Object::isEqual(const Object&) { derivedClassResponsibility("isEqual"); return NO; }

UNSIGNED Object::hash() { derivedClassResponsibility("hash"); return 0; }

int Object::compare(const Object&) { derivedClassResponsibility("compare"); return 0; }

obid Object::copy() { return deepCopy(); }

obid Object::shallowCopy()
{
#ifdef LOG2_PTR
	register UNSIGNED i = (isA()->size()+(sizeof(void*)-1)) >> LOG2_PTR;
#else
	register UNSIGNED i = (isA()->size()+(sizeof(void*)-1))/sizeof(void*);
#endif
	register void** p = (void**)this;
	register void** q = new void*[i];
	register obid o = (obid)q;
	while (i--) *q++ = *p++;
	return o;
}

obid Object::deepCopy()
{
	bool firstObject = NO;
	if (deepCopyDict == 0) {
		deepCopyDict = new IdentDict;
		deepCopyDict->add(*new Assoc(*nil,*nil));
		firstObject = YES;
	}
	Assoc* asc = (Assoc*)&deepCopyDict->assocAt(*this);
	if (asc == nil) {			// object has not been copied 
		obid copy = Object::shallowCopy();	// make a shallow copy 
		deepCopyDict->add(*new Assoc(*this,*copy));	// add to dictionary 
		copy->deepenShallowCopy();	// convert shallow copy to deep copy 
		if (firstObject) {		// delete the deepCopy dictionary 
			DO(*deepCopyDict,Assoc*,asc) delete asc; OD
			delete deepCopyDict;
			deepCopyDict = 0;
		}
		return copy;
	}
	else return asc->value();	// object already copied, just return object reference 
}

void Object::deepenShallowCopy()	{ derivedClassResponsibility("deepenShallowCopy"); }

// error reporting 

void Object::derivedClassResponsibility(const char* fname)
{
	setOOPSerror(OOPS_DRVDCLASSRSP,DEFAULT,this,className(),fname);
}

void Object::destroyer() {}

void Object::shouldNotImplement(const char* fname)
{
	setOOPSerror(OOPS_ILLEGALMFCN,DEFAULT,this,className(),fname);
}

void invalidArgClass(const Object& ob, const Class& expect, const char* fname)
{
	setOOPSerror(OOPS_BADARGCL,DEFAULT,fname,expect.className(),fname,ob.className());
}

void invalidArgSpecies(const Object& ob, const Class& expect, const char* fname)
{
	setOOPSerror(OOPS_BADARGSP,DEFAULT,fname,expect.className(),fname,ob.species()->className());
}

void Object::invalidArgClass(const Object& ob, const Class& expect, const char* fname)
{
	setOOPSerror(OOPS_BADARGCLM,DEFAULT,className(),fname,expect.className(),className(),fname,ob.className());
}

void Object::invalidArgSpecies(const Object& ob, const Class& expect, const char* fname)
{
	setOOPSerror(OOPS_BADARGSPM,DEFAULT,className(),fname,expect.className(),className(),fname,ob.species()->className());
}

void invalidClass(const Object& ob, const Class& expect)
{
	setOOPSerror(OOPS_BADCLASS,DEFAULT,expect.className(),ob.className());
}

void invalidSpecies(const Object& ob, const Class& expect)
{
	setOOPSerror(OOPS_BADSPEC,DEFAULT,expect.className(),ob.species()->className());
}

// Object I/O

void Object::printOn(ostream&) { derivedClassResponsibility("printOn"); }

void Object::scanFrom(istream&) { derivedClassResponsibility("scanFrom"); }

void Object::storeOn(ostream& strm)
{
	bool firstObject = NO;
	if (storeObjDict == 0) {
		storeObjDict = new IdentDict(256);
		storeObjDict->add(*new AssocInt(*nil,0));
		storeClassTable = new OrderedCltn(classDictionary.size());
		firstObject = YES;
	}
	AssocInt* asc = (AssocInt*)&storeObjDict->assocAt(*this);
	if (asc == nil) {			// object has not been stored 
		storer(strm);			// call storer for this object 
		strm << "}\n";
		if (firstObject) {		// delete the storeOn dictionaries
			DO(*storeObjDict,AssocInt*,asc) delete asc; OD
			delete storeObjDict;  storeObjDict = 0;
			DO(*storeClassTable,Class*,cl) cl->classNumber(0); OD
			delete storeClassTable;  storeClassTable = 0;
		}
	}
	else strm << "@" << ((Integer*)asc->value())->value() << " ";	// object already stored, just output object reference 
}

void Object::storer(ostream& strm)	// store class Object 
{
	UNSIGNED objectNum = storeObjDict->size();
	storeObjDict->add(*new AssocInt(*this,objectNum));
	if (isA()->classNumber() == 0) { // object of this class has not been previously stored 
		storeClassTable->add(*isA());
		isA()->classNumber(storeClassTable->size());
		strm << ":" << className() << "." << classVersion();
	}
	else				// object of this class already stored, just output class reference
		strm << "#" << isA()->classNumber();
	strm << "{";
}

void Object::clearForReadFrom() {}

obid readFrom(istream& strm, const char* expectedClass, Object& where)
{
	bool firstObject = NO;			// first object flag 
	char delim;				// delimiter character 
	Class* readClass =0;			// class descriptor pointer
	obid target = &where;			// target object address 
	
// read first character of next object on input stream 
	strm >> delim;
	if (strm.eof()) return nil;		// no objects remaining on input stream 
	checkRead(strm);
	
// allocate table if necessary 
	if (readObjTable == 0) {
		readObjTable = new OrderedCltn(1024);
		readObjTable->add(*nil);	// nil is @0 
		readClassTable = new OrderedCltn(classDictionary.size());
		readClassTable->add(*nil);
		firstObject = YES;
	}

// parse object reference, class reference, or class name & version
	switch (delim) {
		case '@': {			// read object reference 
			UNSIGNED objectNum;	// object reference number 
			strm >> objectNum;
			checkRead(strm);
			if (target != 0) setOOPSerror(OOPS_RDREFERR,DEFAULT,(readObjTable->at(objectNum))->className(),objectNum);
			return readObjTable->at(objectNum);
		}
		case '#': {			// read class reference
			UNSIGNED classNum;	// class reference number
			strm >> classNum >> delim;
			checkRead(strm);
			if (delim != '{') syntaxErr(strm,"{",delim);
			readClass = (Class*)readClassTable->at(classNum);
			break;
		}
		case ':': {			// read class name and class version number
			UNSIGNED classVerNum;	// class version # 
			strm.get(clsName, sizeof clsName, '.');  strm >> delim;
			checkRead(strm);
			if (delim != '.') syntaxErr(strm,".",delim);
			strm >> classVerNum >> delim;
			checkRead(strm);
			if (delim != '{') syntaxErr(strm,"{",delim);
			if ((strlen(expectedClass) > 0 || target != 0) && (strcmp(expectedClass,clsName) != 0))
				setOOPSerror(OOPS_RDWRONGCLASS,DEFAULT,expectedClass,clsName);

// lookup class name in dictionary 
			LookupKey* asc = &classDictionary.assocAt(String(clsName));
			if (asc == nil)	setOOPSerror(OOPS_RDUNKCLASS,DEFAULT,clsName);
			readClass = (Class*)asc->value();

// validate class version number, add class to readClassTable
			if (readClass->classVersion() != classVerNum)
					setOOPSerror(OOPS_RDVERSERR,DEFAULT,clsName,readClass->classVersion(),classVerNum);
			readClassTable->add(*readClass);
			break;
		}
		default: syntaxErr(strm,":, # or @",delim);
	}

	// allocate storage for object if needed, add pointer to readObjTable
	if (target == 0) {
	    target = (obid)new char[readClass->size()];
	} else {
	    // Handle case of already constructed object about to be
	    // constructed again...
	    // This causes memory leaks because certain classes "new" some
	    // space when they are constructed (e.g. Arraychar && Arrayobid)
	    // and when they are read into 
	    // (i.e. readFrom(strm, "Arrayobid",AlreadyConstructedArrayobid))
	    // they do not get a chance to delete the space they previously
	    // new'd. This is their chance.
	    target->clearForReadFrom();
	}
	UNSIGNED readObjTableSize = readObjTable->size();
	if (readObjTableSize == readObjTable->capacity()) readObjTable->reSize(readObjTableSize+1024);
	readObjTable->add(*target);

// call class object reader to read object 
	readClass->reader(strm,*target);
	strm >> delim;
	checkRead(strm);
	if (delim != '}') syntaxErr(strm,"}",delim);
	if (firstObject) {
		delete readObjTable;  readObjTable = 0;
		delete readClassTable;  readClassTable = 0;
	}
	return target;
}

// binary object I/O

void Object::storeOn(fileDescTy& fd)
{
	bool firstObject = NO;
	if (storeObjDict == 0) {
		storeObjDict = new IdentDict(256);
		storeObjDict->add(*new AssocInt(*nil,0));
		storeClassTable = new OrderedCltn(classDictionary.size());
		firstObject = YES;
		oopsResetIO();
	};
	AssocInt* asc = (AssocInt*)&storeObjDict->assocAt(*this);
	if (asc == nil) {		// object has not been stored 
		storer(fd);		// call storer for this object 
		if (firstObject) { 	// delete the storeOn dictionary 
			oopsIOflush(fd);
			DO(*storeObjDict,AssocInt*,asc) delete asc; OD
			delete storeObjDict;  storeObjDict = 0;
			DO(*storeClassTable,Class*,cl) cl->classNumber(0); OD
			delete storeClassTable;  storeClassTable = 0;
		};
	}
	else {
       		char recordType = storeOnObjectRef;	// object already stored, just output object reference 
		storeBin(fd,(char*)&recordType,1);
		storeBin(fd,((Integer*)asc->value())->value());
	};
}

void Object::storer(fileDescTy& fd)	// store Object on file descriptor
{
	UNSIGNED objectNum = storeObjDict->size();
	UNSIGNED short classNum;
	storeObjDict->add(*new AssocInt(*this,objectNum));
	if (isA()->classNumber() == 0) { // object of this class has not been previously stored 
		storeClassTable->add(*isA());
		isA()->classNumber(storeClassTable->size());
		char recordType = storeOnClass;
		storeBin(fd,(char*)&recordType,1);
		unsigned char classNameLen = strlen(className());
		storeBin(fd,(char)classNameLen);
		storeBin(fd,(char *) className(),classNameLen);
		storeBin(fd,classVersion());
	}
	else {				// object of this class already stored, just output class reference
		char recordType = storeOnClassRef;
		storeBin(fd,(char*)&recordType,1);
		classNum = isA()->classNumber();
		storeBin(fd,classNum);
	}
}

obid readFrom(fileDescTy& fd,const char* expectedClass,Object& where)
{
	bool firstObject = NO;			// first object flag 
	char recordType =-1;			// object I/O record type
	Class* readClass =0;			// class descriptor pointer
	obid target = &where;			// target object address 

// allocate tables if necessary 
	if (readObjTable == 0) {
		readObjTable = new OrderedCltn(1024);
		readObjTable->add(*nil);	// nil is @0 
		readClassTable = new OrderedCltn(classDictionary.size());
		readClassTable->add(*nil);
		firstObject = YES;
		oopsResetIO();
	}

// read record type byte from file descriptor
	readBin(fd,&recordType,1);
	
// parse object reference, class reference, or class name & version
	switch (recordType) {
		case storeOnObjectRef: {	// read object reference
			UNSIGNED objectNum;	// object reference number 
			readBin(fd,objectNum);
			if (target != 0) setOOPSerror(OOPS_RDREFERR,DEFAULT,
				(readObjTable->at(objectNum))->className(),objectNum);
#ifdef DEBUG_OBJIO
cerr << "readFrom: ref to object #" << objectNum
	<< ", class " << (readObjTable->at(objectNum))->className() << "\n";
#endif
			return readObjTable->at(objectNum);
		}
		case storeOnClassRef: {		// read class reference
			UNSIGNED short classNum;  // class reference number
			readBin(fd,classNum);
			readClass = (Class*)readClassTable->at(classNum);
#ifdef DEBUG_OBJIO
cerr << "readFrom: ref to class #" << classNum
	<< ", class " << *readClass << "\n";
#endif
			break;
		}
		case storeOnClass: {		// read class name and class version number
			UNSIGNED classVerNum;	// class version # 
			unsigned char namelen = 0;
			readBin(fd,namelen);
			readBin(fd,clsName,namelen);
			clsName[namelen] = '\0';
			readBin(fd,classVerNum);
			if ((strlen(expectedClass) > 0 || target != 0) && (strcmp(expectedClass,clsName) != 0))
				setOOPSerror(OOPS_RDWRONGCLASS,DEFAULT,expectedClass,clsName);
				
// lookup class name in dictionary 
			LookupKey* asc = &classDictionary.assocAt(String(clsName));
			if (asc == nil)	setOOPSerror(OOPS_RDUNKCLASS,DEFAULT,clsName);
			readClass = (Class*)asc->value();

// validate class version number, add class to readClassTable
			if (readClass->classVersion() != classVerNum)
				setOOPSerror(OOPS_RDVERSERR,DEFAULT,clsName,readClass->classVersion(),classVerNum);
			readClassTable->add(*readClass);
#ifdef DEBUG_OBJIO
cerr << "readFrom: class " << *readClass << "\n";
#endif
			break;
		}
		default: setOOPSerror(OOPS_RDBADTYP,DEFAULT,recordType,fd);
	}

	// allocate storage for object if needed, add pointer to readObjTable
	if (target == 0) {
	    target = (obid)new char[readClass->size()];
	} else {
	    // Handle case of already constructed object about to be
	    // constructed again...
	    // This causes memory leaks because certain classes "new" some
	    // space when they are constructed (e.g. Arraychar && Arrayobid)
	    // and when they are read into 
	    // (i.e. readFrom(strm, "Arrayobid",AlreadyConstructedArrayobid))
	    // they do not get a chance to delete the space they previously
	    // new'd. This is their chance.
	    target->clearForReadFrom();
	}
	UNSIGNED readObjTableSize = readObjTable->size();
	if (readObjTableSize == readObjTable->capacity()) readObjTable->reSize(readObjTableSize+1024);
	readObjTable->add(*target);

// call class object reader to read object 
#ifdef DEBUG_OBJIO
cerr << "readFrom: read object #" << readObjTable->size()-1 << "\n";
#endif
        readClass->reader(fd,*target);
	if (firstObject) {
		delete readObjTable;  readObjTable = 0;
		delete readClassTable;  readClassTable = 0;
	}
	return target;
}

// Object Dependence Relationships 

obid Object::addDependent(const Object& ob)
{
	if (dependDict == 0) dependDict = new IdentDict;
	if (!(dependDict->includesKey(*this)))
		dependDict->add(*new Assoc(*this,*new OrderedCltn));
	((OrderedCltn*)dependDict->atKey(*this))->add(ob);
	return &ob;
}

obid Object::removeDependent(const Object& ob)
{
	if (dependDict == 0) return nil;
	OrderedCltn* depList = (OrderedCltn*)dependDict->atKey(*this);
	if (depList == nil) return nil;
	obid dependent = depList->removeId(ob);
	if (depList->size() == 0) release();
	return dependent;
}
	
OrderedCltn& Object::dependents()
{
	if (dependDict == 0) return *new OrderedCltn(1);
	Assoc* asc = (Assoc*)&dependDict->assocAt(*this);
	if (asc == nil) return *new OrderedCltn(1);
	return *new OrderedCltn(*(OrderedCltn*)asc->value());
}
	
void Object::release()
{
	if (dependDict != 0 && dependDict->includesKey(*this)) {
		Assoc* asc = (Assoc*)&dependDict->removeKey(*this);
		delete (OrderedCltn*)asc->value();
		delete asc;
		if (dependDict->size() == 0) {
			delete dependDict;
			dependDict = 0;
		}
	}
}

void Object::changed(const Object& param)
{
	OrderedCltn* depList = &dependents();
	DO(*depList,obid,depob) depob->update(*this,param); OD
	delete depList;
}

void Object::changed()	{ changed(*nil); }

void Object::update(const Object& /*dependent*/, const Object& /*param*/)
{
	derivedClassResponsibility("update");
}
